;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

#+(or cltl2 lispworks clisp)
(progn
  (defpackage :utilities (:nicknames :utils))
  (in-package :utilities))

#-(or cltl2 lispworks clisp)
(in-package :utilities :nicknames '(:utils))
 
#-(or cltl2 lispworks loop)
(require :loop)

(export '(defresource using-resource allocate-resource deallocate-resource 
	      fast-deallocate-resource initialize-resource object resource
          default-resource-finder basic-resource-finder *default-resource-size*
		  make-holder holder-object holder-index holder-object-parameters 
          map-resource resource-free-objects resource-used-objects
		  deallocate-all-objects))

(defvar *default-resource-size* 32.)

(defstruct resource
  freelist			; vector of resource objects
  size				; length of freelist vector
  (count 0)			; number of elements in freelist
  (free 0)			; index of first free (0 to 1-free are in use)
  constructor			; funargs: (resource &rest parameters)
  finder			; funargs: (resource &rest parameters)
  matcher			; funargs: (resource object &rest parameters)
  initializer			; funargs: (resource object &rest parameters)
  deinitializer			; funargs: (resource object)
  reinitializer			; funargs: (resource object &rest parameters)
  parameters)

;;;
;;; the resource freelist is a sorted vector of resource objects. the array
;;; locations from 0 < free hold allocated objects.  the locations from 
;;; free < count hold unallocated objects. the locations from count < size
;;; are nil.  a sorted freelist yields shorter searches in allocation and
;;; deallocation at the price of an element swap during the actual
;;; enqueue/dequeue of an object.  each entry in a resource freelist is held
;;; in a "structure" called a holder, which contains the object itself, a
;;; freelist index if the object has been allocated, and the resource
;;; parameter values (if any) that were used to construct the object.  There
;;; is no defstruct for holders, they are simply lists in the form:
;;; (datum index . object-parameters). the dotted representation avoids
;;; "wasting" cons cells for resouces that have no parameters. Even though a
;;; holder is just a dotted list, it is probably best if the elements are
;;; accessed using the constuctor and accessors defined below, which implement
;;; all the functionality of a true defstuct, except that the tail of the
;;; structure is used as an element.  this extra hair to save on cons cells
;;; is probably not worth it, but what the hell, its only my life...
;;;

(proclaim '(inline make-holder holder-object holder-index 
	    holder-object-parameters))

(defun make-holder (&key object index object-parameters)
  (list* object index object-parameters))

(defun holder-object (holder)
  (car holder))

(defun holder-index (holder)
  (cadr holder))

(defun holder-object-parameters (holder)
  (cddr holder))

(defsetf holder-object (holder) (object)
  `(setf (car ,holder) ,object))

(defsetf holder-index (holder) (index)
  `(setf (cadr ,holder) ,index))

(defsetf holder-object-parameters (holder) (parameters)
  `(setf (cddr ,holder) ,parameters))

;;;
;;; using new-freelist to occasionally grow the simple vector freelist
;;; is faster than always adding elements using vector-push-extend.
;;;

(defun new-freelist (resource size)
  (let ((new (make-array size))
	(old (resource-freelist resource)))
    (when old
      (replace new old))
    (setf (resource-size resource) size)
    (setf (resource-freelist resource) new)
    new))


(defun create-resource (&optional (size *default-resource-size*))
  (let ((r (make-resource :size size)))
    (new-freelist r size)
    r))

;;;
;;; a default resource finder uses an optional matcher function to find the
;;; first available unused object.  If no matcher was supplied to the resource,
;;; the parameters values are checked for equivalence.
;;;

(defun default-resource-finder (resource &rest parameters)
  (let ((free (resource-free resource))
	(count (resource-count resource))
	(matcher (resource-matcher resource))
	(freelist (resource-freelist resource)))

    (loop for index from free below count
	  for holder = (svref freelist index)
	  when (if matcher
		   (apply matcher resource (holder-object holder)
			  parameters)
		   (equal parameters (holder-object-parameters holder)))
	    return  	 
              ;; Found object. Reinitialize and return.
	      (let ((object (holder-object holder))
		    (reinitializer (resource-reinitializer resource)))
		(when reinitializer
		  (apply reinitializer resource object parameters))
		;; Put the found resource at the end of the used portion
		;; of the freelist by swapping with the item at the free
		;; index and incrementing the free index.
		(setf (svref freelist index) (svref freelist free))
		(setf (svref freelist free) holder)		
		(setf (holder-index holder) free)
		(incf (resource-free resource))		
		(values object holder))
	  finally
	    ;; Didn't find a usable object. Create one and initialize.
	    (flet ((add-resource-object (resource object parameters)
		     (let ((freelist (resource-freelist resource))
			   (size (resource-size resource))
			   (holder (make-holder 
				    :object object
				    :object-parameters parameters)))
		       ;; Enlarge freelist if necessary.
		       (unless (< (incf (resource-count resource))
				  size)
			 (setf freelist (new-freelist resource (* size 2))))
		       ;; Put the new resource at the end of the used portion
		       ;; of the freelist by swapping with the item at the 
		       ;; free index and then incrementing the free index.
		       (setf (svref freelist count) (svref freelist free))
		       (setf (svref freelist free) holder)
		       (setf (holder-index holder) free)
		       (incf (resource-free resource))
		       holder)))
	      (let ((object (apply (resource-constructor resource)
				   resource parameters))
		    (initializer (resource-initializer resource)))
		(when initializer
		  (apply initializer resource object parameters))
		(return (values object (add-resource-object
					 resource object parameters))))))))
;;;
;;; a basic finder is provided to simply return the next free object. no
;;; searching or parameter matching is done.
;;;

(defun basic-resource-finder (resource &rest parameters)
  (let ((free (resource-free resource))
	(count (resource-count resource)))
    (if (< free count)
	(let* ((holder (svref (resource-freelist resource) free))
	       (object (holder-object holder))
	       (reinitializer (resource-reinitializer resource)))
	  (when reinitializer
	    (apply reinitializer resource object parameters))
	  (setf (holder-index holder) free)
	  (incf (resource-free resource))		
	  (values object holder))
      (flet ((add-resource-object (resource free object parameters)
	       (let ((freelist (resource-freelist resource))
		     (size (resource-size resource))
		     (holder (make-holder :object object
					  :object-parameters parameters)))
		 (unless (< (incf (resource-count resource))
			    size)
		   (setf freelist (new-freelist resource (* size 2))))
		 (setf (svref freelist free) holder)
		 (setf (holder-index holder) free)
		 (incf (resource-free resource))
		 holder)))
	(let ((object (apply (resource-constructor resource)
			     resource parameters))
	      (initializer (resource-initializer resource)))
	  (when initializer
	    (apply initializer resource object parameters))
	  (values object 
		  (add-resource-object resource free object parameters)))))))

(defun allocate-resource (resource &rest parameters)
  (let ((r (get resource ':resource)))
    (unless r
      (error "~S is not a defined resource." resource))
    (apply (resource-finder r) r parameters)))

(defun deallocate-resource (resource object &optional holder)
  (let ((r (get resource ':resource)))
    (unless r
      (error "~S is not a defined resource." resource))
    (let ((deinitializer (resource-deinitializer r))
	  (last (1- (resource-free r)))
	  (freelist (resource-freelist r)))
      (unless holder
	(setf holder (or (loop for i from 0 below (resource-free r)
			       for h = (svref freelist i)
			       when (eq (holder-object h) object)
				 return h)
			 (error "Object ~S was not allocated from resource ~S."
				object resource))))
      (when deinitializer
	(funcall deinitializer r object))
      (let ((swap (svref freelist last))
	    (index (holder-index holder)))
	(setf (svref freelist index) swap)
	(setf (holder-index swap) index)
	(setf (svref freelist last) holder)
	(setf (holder-index holder) 'nil)
	(decf (resource-free r))
	(values)))))

;;;
;;; this one does no checking and no deinitialization
;;;

(defun fast-deallocate-resource (resource holder)
  (let ((deinitializer (resource-deinitializer resource))
        (last (1- (resource-free resource)))
        (freelist (resource-freelist resource)))
    (when deinitializer
      (funcall deinitializer resource (holder-object holder)))
	(let ((swap (svref freelist last))
          (index (holder-index holder)))
      (setf (svref freelist index) swap)
      (setf (holder-index swap) index)
      (setf (svref freelist last) holder)
      (setf (holder-index holder) 'nil)
      (decf (resource-free resource))
      (values))))
	  
(defun initialize-resource (resource number &rest args)
  (let ((holders (loop repeat number
		       collect
		   (multiple-value-bind (object holder) 
		       (apply #'allocate-resource resource args)
		     (declare (ignore object))
		     holder))))
    (when holders
      (loop for holder in holders
	    do
	(deallocate-resource resource (holder-object holder) holder)))
    (values)))
		    
(defmacro defresource (name parameters 
		       &key constructor finder matcher
			    initializer deinitializer reinitializer
			    (size '*default-resource-size*))
  (unless constructor
    (error "~S required for ~S" :constructor 'defresource))
  (let ((r (gensym)))
    `(let ((,r (setf (get ',name ':resource) (create-resource ,size))))
       (setf (resource-parameters ,r) ',parameters)
       (setf (resource-constructor ,r) ,constructor)
       (setf (resource-finder ,r) ,(or finder
				       '(function default-resource-finder)))
       ,@(when initializer
	   `((setf (resource-initializer ,r) ,initializer)))
       ,@(when deinitializer
	   `((setf (resource-deinitializer ,r) ,deinitializer)))
       ,(if reinitializer
	    `(setf (resource-reinitializer ,r) ,reinitializer)
	  `(setf (resource-reinitializer ,r) (resource-initializer ,r)))
       ,@(when matcher
	   `((setf (resource-matcher ,r) ,matcher)))
       ',name)))

(defmacro using-resource ((variable resource &rest parameters) &body body)
  (let ((holder (gensym)))
    `(multiple-value-bind (,variable ,holder)
	 (allocate-resource ,resource ., parameters)
       (unwind-protect
	   (progn ., body)
	 (deallocate-resource ,resource ,variable ,holder)))))

;;;
;;; debugging routines.
;;;
							     
(defun map-resource-1 (fn resource &optional section)
  (let ((r (get resource ':resource))
	lb ub)
    (unless r
      (error "~S is not a defined resource." resource))
    (case section
      (:used
	(setf lb 0 ub (resource-free r)))
      (:free
	(setf lb (resource-free r) ub (resource-count r)))
      (t
	(when section
	  (error "section value ~s not :USED or :FREE" section))
	(setf lb 0 ub (resource-count r))))
    (loop with freelist = (resource-freelist r)
	  for i from lb below ub
	  for holder = (svref freelist i)
	  do
      (funcall fn (holder-object holder) holder))
    (values)))

(defun map-resource (fn resource &optional section)
  (map-resource-1 #'(lambda (object holder)
		      (declare (ignore holder))
		      (funcall fn object))
		  resource section))

(defun resource-free-objects (resource)
  (let ((list '()))
    (map-resource-1 #'(lambda (o h)
			(declare (ignore h))
			(push o list))
		    resource ':free)
    (nreverse list)))

(defun resource-used-objects (resource)
  (let ((list '()))
    (map-resource-1 #'(lambda (o h)
			(declare (ignore h))
			(push o list))
		    resource ':used)
    (nreverse list)))

(defun deallocate-all-objects (resource)
  (let ((used (resource-used-objects resource)))
    (loop for object in used 
          do
      (deallocate-resource resource object))))
