;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(defconstant +source-type+ #-dos "lisp" #+dos "LSP")
(defconstant +binary-type+ #+(or excl mcl) "fasl"
                           #+kcl "o"
                           #+aclpc "fsl"
                           #+(and lispworks mips) "mfasl"
                           #+(and clisp dos) "FAS"
                           #+(and clisp (not dos)) "fas")
(defconstant +midi-type+   #-dos "midi" #+dos "MID")
(defconstant +text-type+   #-dos "text" #+dos "TXT")
(defconstant +stella-type+ #-dos "stella" #+dos "STL")
(defconstant +sound-type+  "snd")

;;;
;;; part status constants used by scheduler
;;;

(eval-when (load eval compile)
  (defconstant +killed+ (ash 1 0))	; no output and not rescheduled
  (defconstant +ending+ (ash 1 1))	; output but not rescheduled
  (defconstant +resting+ (ash 1 2))	; no output
  (defconstant +chording+ (ash 1 3))	; time not increment
  (defconstant +unqueued+ (ash 1 4))	; not rescheduled
  (defconstant +removed+ (logior +killed+ +unqueued+))
  (defconstant +normal-mask+ (logior +resting+ +chording+)))

(defvar *cm-state* (make-random-state)
  "The random state object for Common Music.")

(defmacro without-interrupts (&body body)
  `(#+excl excl:without-interrupts 
    #+mcl ccl:without-interrupts
    #+lispworks mp:without-interrupts
    #-(or excl mcl lispworks clisp) progn
    ,@body))

(defun date-string (&optional stream)
  (multiple-value-bind (second minute hour day month year)
      (get-decoded-time)
    (format stream "~d-~a-~d ~d:~d:~d"
	    day (svref '#("Jan" "Feb" "Mar" "Apr" "May" "Jun" 
			  "Jul" "Aug" "Sep" "Oct" "Nov" "Dec")
		       (1- month))
            (- year 1900) hour minute second)))

(defmacro defprop (sym prop value)
  `(setf (get ',sym ',prop) ,value))

(defmacro defsubst (name parameters &body body)
  `(progn (proclaim '(inline ,name))
	  (defun ,name ,parameters ,@body)))

(defun string-append (&rest strings)
  (apply #'concatenate 'string (mapcar #'string strings)))

(defun quotify (token)
  (concatenate 'string "\"" (string token) "\""))

(defun quote-if-necessary (x)
  (if (constantp x)
      x
      `(quote ,x)))

(defun objc-string (token)
  (let* ((s (string-downcase (string token)))
         (p (position #\- s)))
    (if p
      (remove #\- (string-capitalize s :start (1+ p)))
      s)))

(defmacro delay-eval (&body body)
  `(function (lambda () ,@body)))

;;;
;;; A constructor and deinitializer for the various part resources
;;;

(defun instance-resource-constructor (resource class)
  (declare (ignore resource))
  (allocate-instance (find-class class)))

(defun instance-resource-deinitializer (resource object)
  (declare (ignore resource))
  (dolist (s (class-slots (class-of object)))
    (slot-makunbound object (slot-definition-name s))))

(defun instance-resource-matcher (resource object class)
  (declare (ignore resource))
  (typep object class))

;;;
;;;
;;;

(defun slots-makunbound (instance)
  (dolist (s (class-slots (class-of instance)))
    (slot-makunbound instance (slot-definition-name s)))
  instance)

(defun compute-slot-and-accessor-pairs (class)
  (let ((pairs '()))
    (flet ((get-accessors (slots)
	     (flet ((slot-and-accessor (slotd)
		      (let ((name (slot-definition-name slotd)))
			(if (not (assoc name pairs))
			    (loop for r in (slot-definition-readers slotd)
			     when (member r (slot-definition-writers slotd)
					  :test #'eq :key #'cadr)
			     return (list name r))
			  nil))))
	       (loop with pair for slot in slots
		when (setf pair (slot-and-accessor slot))
		collect pair))))
      (dolist (c (class-precedence-list class))
        (setf pairs (nconc pairs (get-accessors (class-direct-slots c)))))
      pairs)))

(defun find-slot (class name)
  (loop for slot in (class-slots class)
	when (eq (slot-definition-name slot) name)
	  return slot))

(defun class-slot-names (class)
  (mapcar #'slot-definition-name (class-slots class)))

;;;
;;;                         Readtable hacking
;;;

(defvar *common-music-readtable*
	(copy-readtable  #+excl top-level::*readtable*
			 #+lispworks common-lisp-user::*readtable*))

;;;
;;; [] is a read time macro for creating chord streams, and expands into the
;;; CHORD macro call. Pains are taked to insure that the "]" is only a legal
;;; macro terminator if a "[" has actually been typed.
;;;

(defvar *chord-readtable* 
  (copy-readtable #+excl top-level::*readtable*
		  #+lispworks common-lisp-user::*readtable*))

(defun chord-reader (stream char &optional arg)
  (declare (ignore char arg))
  (let ((*readtable* *chord-readtable*)
         members)
    (setf members (read-delimited-list #\] stream t))
    (list* 'chord members)))

;;;
;;; #@ is a read time macro for referencing named motives.
;;;

(set-macro-character #\[ #'chord-reader nil *common-music-readtable*)
(set-syntax-from-char #\] #\) *chord-readtable* 
		      #+ExCl top-level::*readtable*
		      #+lispworks system::*std-lisp-readtable*)

#-mcl
(defun motive-reader (stream subchar arg)
  (declare (ignore subchar arg))
  (values `(motive ,(read stream t nil t))))

#+mcl
(defun motive-reader (stream subchar arg)
  (if (char= (peek-char nil stream t nil t) #\()
      (funcall (get-dispatch-macro-character #\# #\@ 
                                             ccl::%initial-readtable%) 
               stream subchar arg)
    (values `(motive ,(read stream t nil t)))))

(set-dispatch-macro-character #\# #\@ #'motive-reader
                              *common-music-readtable*)

;;;
;;; #i reader for item stream definition.
;;;

(defun item-stream-reader (stream subchar arg)
  (declare (ignore subchar arg))
  (let ((p (peek-char nil stream))
        d i)
    (cond ((alpha-char-p p)
           (read-char stream)
           (setf d (peek-char nil stream))
           (cond ((alpha-char-p d)
                  (read-char stream))
                 ((char= d #\() 
                  (setf d #\x))
                 (t (error "Bogus #i character: \"~C\"" d))))
          ((char= p #\()
           (setf p #\c d #\x))
          (t (error "Bogus #i character: \"~C\"" p)))
    (cond ((char-equal p #\c) (setf p 'cycle))
          ((char-equal p #\r) (setf p 'random))
          ((char-equal p #\h) (setf p 'heap))
          ((char-equal p #\g) (setf p 'graph))
          ((char-equal p #\s) (setf p 'sequence))
          ((char-equal p #\a) (setf p 'accumulation))
 	  ((char-equal p #\p) (setf p 'palindrome)) 
          ((char-equal p #\f) (setf p 'function)) 
          ((char-equal p #\o) (setf p 'rotation)) 
          (t (error "\"~C\" not one of C,R,H,G,S,A,P, F or O." p)))
    (cond ((char-equal d #\x) (setf d 'items))
          ((char-equal d #\n) (setf d 'notes))
          ((char-equal d #\r) (setf d 'rhythms))
          ((char-equal d #\p) (setf d 'pitches))
          ((char-equal d #\d) (setf d 'degrees))
          ((char-equal d #\i) (setf d 'intervals))
          ((char-equal d #\s) (setf d 'steps)) 
          ((char-equal d #\a) (setf d 'amplitudes)) 
          (t (error "\"~C\" not one of X,N,R,P,D,I,S,A." d)))
    (setf i (read stream t nil t))
   `(,d ,@ i in ,p)))
         
(set-dispatch-macro-character #\# #\i #'item-stream-reader 
                              cm::*common-music-readtable*)

#+excl(setf top-level::*readtable* *common-music-readtable*)
#-excl(setf *readtable* *common-music-readtable*)

(defun shell (string)
  #+excl (excl:shell string)
  #+lispworks (foreign:call-system string "/bin/csh")
  #+kcl (lisp:system string)
  #+clisp (lisp:shell string)
  #-(or excl kcl lispworks clisp)
    (warn "No shell calling implemented in this lisp. Can't do: ~s" 
          string))

;;;
;;; site initialization. if a file called cminit.lisp or cminit.fasl exists
;;; in either *common-music-directory* or the directory that contains the
;;; booted image, then the system loads it when it first starts up.  This can
;;; be used to auto load patch files and system extensions, and also provides 
;;; a mechanism for resetting important variables like *common-music-directory*
;;; if the image is used on a different filesystem than it was created on.
;;;

(defun load-cminit ()
  (flet ((maybe-load (dir)
         (let* ((pdir #-lispworks dir
		      #+lispworks (pathname dir))
		(fasl (make-pathname :defaults pdir
				    :name "cminit" 
                                    :type +binary-type+))
		(lisp (make-pathname :defaults pdir :name "cminit" 
                                    :type +source-type+)))
           (if (probe-file fasl)
               (load fasl :verbose nil)
             (if (probe-file lisp)
                 (load lisp :verbose nil)
               nil)))))
    (or (maybe-load *common-music-directory*)
        (maybe-load (image-directory)))))

#+mcl
(defun image-directory () (translate-logical-pathname "HOME:"))

#+akcl
(defun image-directory () (truename "./"))

#+clisp
(defun image-directory () (truename (lisp:cd)))

#+excl
(defun image-directory ()
  (let ((name (system:command-line-argument 0)))
    (if (or (char= (elt name 0) #\/)
            (char= (elt name 0) #\.))
        (let ((pos (position #\/ name :from-end t)))
          (if (not pos)
              "./"
            (subseq name 0 (1+ pos))))
      (let ((path (system:getenv "PATH")))
        (loop with beg = 0 and len = (length path) 
              while (< beg len)
              for end = (or (position #\: path :start beg) len)
              for dir = (subseq path beg end)
              do (unless (char= (elt dir (1- (length dir))) #\/)
                   (setf dir (concatenate 'string dir "/")))
              when (probe-file (make-pathname :defaults dir :name name))
              return dir
              do (setf beg (1+ end )))))))

#+lispworks
(defun image-directory ()
  (let ((name (car sys::*line-arguments-list*)))
    (if (or (char= (elt name 0) #\/)
            (char= (elt name 0) #\.))
        (let ((pos (position #\/ name :from-end t)))
          (if (not pos)
              "./"
            (subseq name 0 (1+ pos))))
      (let ((path (system::getenv "PATH")))
        (loop with beg = 0 and len = (length path) 
              while (< beg len)
              for end = (or (position #\: path :start beg) len)
              for dir = (subseq path beg end)
              do (unless (char= (elt dir (1- (length dir))) #\/)
                   (setf dir (concatenate 'string dir "/")))
              when (probe-file (make-pathname :defaults (pathname dir) 
					      :name name))
              return dir
              do (setf beg (1+ end )))))))

(defun save-cm (&optional pathname (package :common-music) syntax)
  (when (null pathname)
    (setf pathname (merge-pathnames "cm" *common-music-directory*)))
  (let ((file (namestring pathname)))
    #+excl
    (excl:dumplisp :name file :read-init-file t :checkpoint nil
		   :restart-function
		   #'(lambda (&rest args)
                       (declare (ignore args))
                       (tpl:setq-default *readtable* 
                         cm::*common-music-readtable*)
                       (tpl:setq-default *package*
                         (find-package package))
                      (when syntax
                        (funcall (find-symbol "IN-SYNTAX" package)
                                 syntax))
		      (cm::load-cminit)))				 
    #+lispworks
    (lispworks:save-image file 
			  :gc t     ; Garbage collect
			  :type :user ; Clear global variables
			  :normal-gc t     ; normal-gc
			  :restart-function 
			  #'(lambda ()
			      (setf *package* (find-package package))
			      (setf *readtable* *common-music-readtable*)
			      (when syntax
				    (funcall (find-symbol "IN-SYNTAX" package)
					     syntax))
			      (cm::load-cminit)))			
    #+kcl(si:save-system file)					
    #+mcl
    (progn 
      (setf ccl:*lisp-startup-functions* 
        (append ccl:*lisp-startup-functions* 
                (list #'(lambda ()
                          (ccl:egc t)
                          (setf *package* (find-package package))
                          (setf *readtable* *common-music-readtable*)
                          (when syntax
                            (funcall (find-symbol "IN-SYNTAX" package)
                                     syntax))))))
      ;; image size is 6144 K suggested, 2048 K minimum
      (ccl:save-application file :size '(6291456 2097152)
                                 :init-file "cminit"))					
    #+aclpc
    (progn
      (setf allegro:*session-init-fns*
        (append allegro:*session-init-fns*
                (list #'(lambda ()
                          (setf *package* (find-package package))
                          (setf *readtable* *common-music-readtable*)
                          (when syntax
                            (funcall (find-symbol "IN-SYNTAX" package)
                                     syntax))))))
      (allegro:save-image :image-file file))
    #+clisp (lisp:saveinitmem file :quiet t
              :init-function
              #'(lambda ()
                  (setf *package* (find-package package))
                  (setf *readtable* *common-music-readtable*)
                  (when syntax
                    (funcall (find-symbol "IN-SYNTAX" package) syntax))))
    #-(or excl kcl mcl aclpc lispworks clisp)
      (warn "save-cm: No image saving implemented for this lisp.")))

