;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92, 93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; send to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

;;;
;;; sundry state flags. (0-7) are taken up by states already defined
;;; in cm kernel
;;;

(defconstant +hidden+		(ash 1 8))	; element is skipped
(defconstant +frozen+		(ash 1 9))	; generator uses thread
(defconstant +dumped+		(ash 1 10))	; saved bit for archiving
(defconstant +edited+		(ash 1 11))	; object needs save.
(defconstant +deleted+		(ash 1 12))	; element is deleted
(defconstant +ephemeral+  	(ash 1 13))	; unset flushes subelements
(defconstant +system+  		(ash 1 14))     ; element belongs to system
(defconstant +no-link+ 		(ash 1 15))     ; subelements sans backpointer
(defconstant +start-unset+      (ash 1 16))     ; flush start time on unset
(defconstant +marked+           (ash 1 17))     ; general marking bit
(defconstant +compile+          (ash 1 18))     ; force compile on definition

(defconstant +slot-unset+ "-unset-" "Stella's print value for unbound slots.")
(defconstant +unbound-tokens+ '(none unset unbound :none :unset :unbound))

(defvar *object-table* (make-hash-table)
  "Table for objects with associated id.")

(defun find-object (id &optional errorp)
  (or (gethash id *object-table*)
      (and errorp (error "No element with id: ~S" id))))

(defun find-element (&rest args)	; need for backward compatibility
  (apply #'find-object args))

(defsetf find-object (id) (element)
  `(setf-find-object ,element ,id))

(defun setf-find-object (element id)
  (when (slot-exists-p element 'id)
    (setf (slot-value element 'id) id))
  (setf (gethash id *object-table*) element))

(defun list-all-objects (&key top-level system)
  (declare (ignore top-level system))
  (let ((list ()))
    (maphash #'(lambda (key val)
    	         (declare (ignore key))
                 (push val list))
  	     *object-table*)
    list))

(defun name (name &optional (new nil))
  (let ((sym (etypecase name (symbol name)(string (intern name)))))
    (if new (gentemp (symbol-name name)) sym)))

(defun object-namestring (obj &optional (pretty t) )
  (if obj
    (let ((id obj))
      (when (typep obj 'id-mixin)
        (setf id (object-name obj)))
      (if pretty
        (string-capitalize (string id))
        (princ-to-string id)))
    nil))

;;;
;;; clos utilities
;;;

(defun class-subclasses (class &optional include-self)
  (let ((kids (loop for kid in (class-direct-subclasses class)
                    nconc (class-subclasses kid t))))
    (if include-self (cons class kids) kids)))

(defmacro print-slot-if-bound (stream instance slot &optional string)
  `(when (slot-boundp ,instance ,slot)
     (progn
       ,@(and string `((write-string ,string ,stream)))
       (princ (slot-value ,instance ,slot) ,stream))))

(defun class-slot-p (class slot)
  (find slot (class-slots class) :key #'slot-definition-name))

(defun expand-initargs (class inits)
  #+aclpc (unless (class-finalized-p class) (finalize-inheritance class))
  (loop with slots = (class-slots class) 
        and saved = inits and arg and init and value
        while inits
        do (setf init (pop inits))
           (setf arg (if (constantp init) init `(quote ,init)))
           (if inits 
               (setf value (pop inits))
             (error "Uneven initialization list: ~S" saved))
        if (find init slots :key #'slot-definition-initargs :test #'member)	  
        nconc (list arg value) into parsed
        else 
        do (error "~S is not a legal initialization argument for ~A."
                  init (class-name class) )
        finally (return parsed)))
				 
(defmacro slot-value-or-default (instance slot &optional (default nil))
  (let ((i (gensym))
  	(s (gensym)))
   `(let ((,i ,instance)
          (,s ,slot))
       (if (slot-boundp ,i ,s) (slot-value ,i ,s) ,default))))

(defmacro careful-slot-value (obj slot &optional default)
  (let ((ov (gensym))(sv (gensym))(dv (gensym)))
    `(let ((,ov ,obj)(,sv ,slot)(,dv ,default))
       (and (slot-exists-p ,ov ,sv)
            (slot-boundp ,ov ,sv)
            (or (slot-value ,ov ,sv) ,dv)))))
	    
(defmethod object-slots ((object t))
  nil)

(defmethod object-slots ((object standard-class))
  (class-slots object))

(defmethod object-slots ((object standard-object))
  (class-slots (class-of object)))

(defmethod object-private-slots ((object standard-object))
  nil)

(defmethod object-public-slots ((object standard-object))
  (class-slots (class-of object)))

;;;
;;; a bag is a list structure optimized for appending ala interlisp's TCONC
;;; list constructor, and for cycling through the elements over and over again.
;;; a bag looks like:  (cache tail . current), where cache holds all the
;;; elements added since the bag was created, tail points to the last cons
;;; cell in the cache, and current are the current bag elements. popping
;;; elements removes them from current, not from the cache.
;;;
     
(defun make-bag (&optional elements (copy-first t)) 
  (when copy-first (setf elements (copy-list elements)))
  (list* elements (last elements) elements))

(defmacro bag-elements (bag)
  `(cddr ,bag))

(defmacro bag-tail (bag)
  `(cadr ,bag)) 

(defmacro bag-cache (bag)
  `(car ,bag))

(defmacro push-bag (x bag)
  ;; push new thing onto cache, don't update elements.
  (let ((var (gensym)))
    `(let ((,var ,bag))
       (push ,x (bag-cache ,var))
       (when (null (bag-tail ,var))
         (setf (bag-tail ,var) (bag-cache ,var)))
       ,var)))

(defmacro append-bag (x bag)
  ;; append new thing to cache, don't update elements.
  (let ((var (gensym))
        (new (gensym)))
    `(let ((,var ,bag)
           (,new (list ,x)))
       (if (null (bag-cache ,var))
           (setf (bag-cache ,var) (setf (bag-tail ,var) ,new))
         (setf (bag-tail ,var)
	       (cdr (rplacd (bag-tail ,var) ,new))))
       ,var)))

(defmacro insert-bag (element bag pos)
  (let ((ev (gensym))(pv (gensym))(bv (gensym))(tv (gensym)))
    `(let ((,ev ,element)
           (,pv ,pos)
	   (,bv ,bag))
       (if (or (null (bag-cache ,bv))
               (= ,pv 0))
           (push-bag ,ev ,bv)
         (if (< ,pv (length (bag-cache ,bv)))
             (let ((,tv (nthcdr (1- ,pv) (bag-cache ,bv))))
               (setf (cdr ,tv) (cons ,ev (cdr ,tv)))
	       ,bv)
	   (append-bag ,ev ,bv))))))
	   
(defmacro pop-bag (bag)
  (let ((var (gensym)))
    `(let ((,var ,bag))
       (pop (bag-elements ,var)))))
  
(defmacro reset-bag (bag)
  (let ((var (gensym)))
    `(let ((,var ,bag))
       (setf (bag-elements ,var) (bag-cache ,var))
       ,var)))
	
(defmacro forget-bag (bag)
  (let ((var (gensym)))
    `(let ((,var ,bag))
       (setf (bag-tail ,var) nil 
             (bag-cache ,var) nil
	     (bag-elements ,var) nil)
      ,var)))

;;;
;;; bags as score queues...
;;;

(defmacro bag-enqueue (element bag)
  (let ((evar (gensym))
        (bvar (gensym)))
    `(let ((,evar ,element)
           (,bvar ,bag))
       (cond ((null (bag-elements ,bvar))
              (push ,evar (bag-elements ,bvar))
              (setf (bag-tail ,bvar) (bag-elements ,bvar)))
             (t 
	      (let ((time (object-time ,evar)))
                 (cond ((>= time (object-time (car (bag-tail ,bvar))))
                        (setf (bag-tail ,bvar)
	                  (cdr (rplacd (bag-tail ,bvar) (list ,evar)))))
                       ((< time (object-time (car (bag-elements ,bvar))))
                        (push ,evar (bag-elements ,bvar)))	       
                       (t 
	                (loop with tail = (bag-elements ,bvar)
	                      for head on (cdr tail)
                              while (<= (object-time (car head)) time)
                              do (setf tail head)
		              finally (rplacd tail (cons ,evar head))))))))
      ,evar)))

(defmacro bag-dequeue (bag)
  `(pop-bag ,bag))
  
;;;
;;; readtable hackery.
;;;
  
(defun find-object-reader (stream subchar arg)
  (declare (ignore subchar arg))
  (let ((thing (read stream t nil t)))
    (values `(find-object ',thing t))))
    
(set-dispatch-macro-character #\# #\! #'find-object-reader 
                              cm::*common-music-readtable*)

;;;
;;; instance printing - stolen from pcl
;;;

#-pcl
(defmacro printing-random-thing ((thing stream) &body body)
  (let ((s (gensym)))
    `(let ((,s ,stream))
       (format ,s "#<")
       ,@body
       (format ,s " ")
       #+mcl   (prin1 (ccl:%address-of ,thing) ,s)
       #+excl  (format ,s "~O" (excl::pointer-to-fixnum ,thing))
       #+kcl   (format ,s "~O" (si:address ,thing))
       #+lucid (format ,s "~O" (%pointer ,thing))
       #+clisp (format ,s "~O" (sys::address-of ,thing))
       (format ,s ">"))))

;;;
;;; defobject
;;;

(defvar *defobjects* (make-hash-table)
  "Defobject class information.")

(defmacro defobject (name supers slots &rest options)
  (let (classoptions stellaforms pars)
    (dolist (o options)
      (case (first o)
        (:parameters
         (setf pars (cdr o)))
        (t
         (push o classoptions))))
    (push `(define-object-parameters (find-class ',name)
             ',(or pars t))
          stellaforms)
    `(progn
       (defclass ,name ,supers 
         ,(loop for x in slots
                when (consp x)
                collect
             (let* ((inits (copy-list x))
                    (slot (first inits))
                    (keyword (intern (string slot) :keyword))
                    sym? key?)
               (loop for (key val) on (cdr inits) by #'cddr
                     do 
                 (when (eq key ':initarg) 
                   (if (eq val slot) (setf sym? t)
                     (if (eq val keyword) (setf key? t)))))
               (unless sym?
                 (push slot (cdr inits)) 
                 (push ':initarg (cdr inits)))
               (unless key?
                 (push keyword (cdr inits))
                 (push ':initarg (cdr inits)))
               inits)
                else 
                collect (list x ':initarg x 
                              ':initarg (intern (string x) :keyword)))
         ,@classoptions)
       ,@stellaforms
       (find-class ',name))))

;;;
;;; we store parameter declarations in a class table. a parameter
;;; for defobject is just a slot that a user manuiplates directly.
;;; it would be ideal to use the MOP to define metaclasses for this,
;;; ie. class-with-parameters and a parameter-definition, but many
;;; clos implementations are still too weak for this...
;;;

(defstruct (parameter (:type list))
  name)

(defmethod object-parameters ((object standard-class))
  (gethash object *defobjects*))

(defmethod object-parameters ((object t))
  (gethash (class-of object) *defobjects*))

(defun define-object-parameters (class &optional (specs t))
  (flet ((canonicalize-spec (spec)
           (if (symbolp spec)
             (list spec)
             spec)))
    (cond ((eq specs t)
           (setf (gethash class *defobjects*)           
                 (loop for super in (class-precedence-list class)
                       thereis (object-parameters super))))
          (t
           #+(and excl cltl2) (clos:finalize-inheritance class)
           #+aclpc (unless (class-finalized-p class)
                     (finalize-inheritance class))
           (let ((internal (mapcar #'canonicalize-spec 
                                   specs))
                 (slots (class-slots class)))
             (dolist (spec internal)
               (unless (find (first spec)
                             slots :key #'slot-definition-name)
                 (error "Attempt to define ~A as a parameter ~
                         but ~A has no such slot." 
                        (first spec) (class-name class))))
             (setf (gethash class *defobjects*)           
                   internal))))))
            
(defun element-classes (&optional mode)
  (declare (ignore mode))
  (let ((list ()))
    (maphash #'(lambda (key val)
                 (declare (ignore val))
                 (when (typep (class-prototype key)
                              'element)
                   (push key list)))
             *defobjects*)
    (nreverse list)))

(defun container-classes (&optional mode)
  (declare (ignore mode))
  (let ((list ()))
    (maphash #'(lambda (key val) 
                 (declare (ignore val))
                 (when (typep (class-prototype key)
                              'startable-element)
                   (push key list)))
             *defobjects*)
    (nreverse list)))

(defun stream-classes (&optional mode)
  (declare (ignore mode))
  (let ((list ()))
    (maphash #'(lambda (key val) 
                 (declare (ignore val))
                 (when (typep (class-prototype key)
                              'event-stream)
                   (push key list)))
              *defobjects*)
    (nreverse list)))
