;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92, 93 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; send to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

;;;
;;; this file contains low level routines for parsing and processing
;;; command input data.
;;;

(defvar .io-streams.
  (make-instance 'container :id 'IO-streams 
                 :flags +system+))
(defvar .top-level. 
  (make-instance 'container :id 'Top-Level 
                 :flags (logior +system+ +no-link+)))
(defvar .syntaxes. 
  (make-instance 'container :id 'Syntaxes
                 :flags (logior +system+ +no-link+)))
(defvar .pasteboard.
  (make-instance 'container :id 'Pasteboard 
                 :flags (logior +system+ +no-link+)))
(defvar .scratch-score.
  (make-instance 'score :id 'Scratch-Score
                 :flags (logior +system+ +no-link+ +ephemeral+)))
(defvar .scratch-thread. 
  (make-instance 'thread :id 'Scratch-Thread
                 :flags (logior +system+ +ephemeral+ +no-link+)))
(defvar .toc. 
  (make-instance 'container :id 'Toc
                 :flags (logior +system+ +no-link+)))

(eval-when (load eval)
  (add-objects (list .top-level. .io-streams. .syntaxes. .pasteboard.
                     .scratch-score. .scratch-thread.)
               .toc. nil :copy-first nil))

;;;
;;; io streams and output functions. ports can customize these to
;;; take advatage of graphic capabilities beyond terminal interaction.
;;; our default definitions just use format.
;;;

(defvar *tell-stream* *standard-output*)
(defvar *warn-stream* *standard-output*)
(defvar *list-stream* *standard-output*)
(defvar *show-stream* *standard-output*)
(defvar *graph-stream* *standard-output*)

(defun tell-user (&rest args) 
  (apply #'format *tell-stream*   args))

(defun warn-user (&rest args)
  (apply #'format *warn-stream* args))

;;;
;;; *command-prompting* controls whether or not commands prompt 
;;; for missing/illegal data.
;;;

(defparameter *command-prompting* t "True if Stella should prompt for input.")

;;;
;;; most of the commands accept "pair arguments". Pairs are a generalization 
;;; of Lisp's keywords and can be either :keyword or symbols.
;;; values or slot initializations, depending on the context. 
;;;

(defmacro dopairs ((optionvar valuevar pairs) &body body)
  (let ((pvar (gensym)) )
    `(let ((,pvar ,pairs) ,optionvar ,valuevar)
       (loop while ,pvar
             do (setf ,optionvar (pop ,pvar))
                (setf ,valuevar (pop ,pvar))
                ,@body))))

;;; 
;;; canonicalize-pairs normalizes pairs supplied by the user to either slot or
;;; keyword names. it also can perform careful evaluation of values.
;;;

(defun canonicalize-pairs (pairs &optional eval (destruct t) (package :stella))
  (unless (packagep package)
    (setf package (find-package package)))
  (unless destruct (setf pairs (copy-list pairs)))
  (loop for slot = t then (not slot)
        for tail on pairs
        if (and slot (not (eq package (symbol-package (car tail)))))
        do (setf (car tail)
             (let ((name (symbol-name (car tail))))
               (or (find-symbol name package) (intern name package))))
        else if (and eval (not slot))
        do (setf (car tail)
             (careful-eval (car tail)))
        finally (return pairs)))

(defun pair-value (name pairs &optional default)
  (getf pairs name default))

;;;
;;; careful-eval and careful-quote support "careful evaluation", which
;;; allows unquoted/unbound symbols and unquoted lists without functional
;;; CAR's.
;;; 

(defun careful-eval (x)
  ;; given a form, evaluate it carefully
  (if (symbolp x)
      (if (boundp x) (symbol-value x)
          x)
    (if (constantp x) x
      (if (and (consp x) (symbolp (car x)) (fboundp (car x)))
          (eval x)
         x))))

(defun careful-quote (x)
  ;; produce careful form suitable for eval
  (cond ((constantp x) x)
        ((symbolp x)
         (if (boundp x) x `(quote ,x)))
        ((and (consp x) (symbolp (car x)) (fboundp (car x)))
          x)
        ((stringp x)
         x)
        (t
         `(quote ,x))))
;;;
;;; argfield finds the limits of the next argument in an input string, taking
;;; care to parse object references, pathnames, sequences and lisp forms
;;; correctly. an argument may include whitespace if it uses type delimitors:
;;; :pathname  "", :reference [], :form (), :sequence ,.
;;;

(defun argfield (string &optional (start 0) end type)
  (unless type (setf type ':form))
  (let* ((beg start)
         (end (or end (length string)))
         (lev 0)
         (bag '(#\space #\tab #\,))
         (white '(#\space #\tab))
         (seq (or (eq type ':sequence)
                  (eq type ':reference)))
         #-(or kcl aclpc) (char #\null)
         #+kcl (char (code-char 0))
         #+aclpc (char #\return)     ; no #\null or (char-code 0) in aclpc!
         pos next done)
    (loop while (< beg end)
          while (member (setf char (elt string beg))
                        white)
          do (incf beg))
    (if (char= char #\,)
      (error "Illegal ,")
      (setf pos beg))
     (case type
      ((:reference :range :position)
       (setf seq (if (eq type :reference) t nil))
       (loop while (< pos end) do 
             (setf char (elt string pos))
             (cond ((eq char #\[)
                    (incf lev))
                   ((eq char #\])
                    (if (= (decf lev) 0)
                      (setf done t pos (1+ pos))))
                   ((and (= lev 0) (member char bag))
                    (setf done t)))
             until done
             do (incf pos))
       (unless (= lev 0)
         (error "Unbalanced ~:[]~;[~]" (> lev 0))))
      (:pathname
       (loop while (< pos end) do
             (setf char (elt string pos))
             (cond ((eq char #\")
                    (if (= pos beg)
                      (incf lev)
                      (if (= (decf lev) 0)
                        (setf done t pos (1+ pos)))))
                   ((and (= lev 0) (member char bag))
                    (setf done t)))
             until done
             do (incf pos))
       (unless (= lev 0) (error "Unbalanced \"")))
      ((:form :careful-form :symbol :sequence :container :containers) 
       (setf seq (if (member type '(:sequence :containers)) t nil))
       (loop while (< pos end) do
             (setf char (elt string pos))
             (cond ((eq char #\()
                    (if (or (= pos beg) (> lev 0))
                      (incf lev)
                      (setf done t)))
                   ((eq char #\))
                    (if (= (decf lev) 0)
                      (setf done t pos (1+ pos))))
                   ((and (= lev 0)
                         (member char bag))
                    (setf done t)))
             until done
             do (incf pos))
       (unless (= lev 0)
         (error "Unbalanced ~:[)~;(~]" (> lev 0))))
      ((:pairs :raw :rest)
       (setf pos 
             (1+ (position-if-not 
                  #'(lambda (x) 
                      (member x bag))
                  string :start beg :from-end t))))
      (t 
       (error "Don't know how to parse ~S args." type)))
    ;; pos is now 1 past the arg. find the pos
    ;; of the next arg, if any, and recurse if
    ;; we hit a comma and sequences are allowed.
    (unless (= pos end)
      (let ((n pos))
        (loop while (< n end)
              while (member (setf char (elt string n))
                            white)
              do (incf n))
        (when (< n end)
          (setf next n)
          (when (char= char #\,)
            (if seq
              (multiple-value-bind 
                (x y z) 
                (argfield string (1+ next) end type)
                (unless x (error "Dangling ,"))
                (setf pos y next z))
              (error "Illegal ,"))))))    
    (if (= beg end)
      (values nil nil nil)
      (values beg pos next))))

;;;
;;; with-args parses an input string according to the specified specs.
;;; each spec is (var &optional type init) where var is a variable that
;;; receives the parsed argument, type is the argument's type and
;;; init is an initial value for var.  There are currenty 8 types:
;;;    t (default)    a string
;;;    :form          lisp form 
;;;    :careful-form  lisp form prepared for careful evaluation
;;;    :pathname      a file name, can include spaces if delimited by ""
;;;    :reference     a reference, can include spaces if delimited by []
;;;    :sequence      a series of elements delimited by ,
;;;    :rest          the rest of the args returned as list of :forms
;;;    :pairs         the rest of the args returned as list of {symbol :form}
;;;    :raw           the rest of the input returned as string
;;;    

(defmacro with-args (input (&rest specs) &body body)
  (let ((svar (gensym))
        (bvar (gensym))
        (evar (gensym))
        (nvar (gensym))
        (lvar (gensym))
        (argnum (length specs))
        string argerr user types syntax nullok)
    (if (consp input)
        (setf string (car input)
              argerr (if (getf (cdr input) ':argchecking)
                         (format nil "Command input ~~S contains more than ~A ~
                                 argument~@[s~]."
                                 argnum (> argnum 1)))
              syntax (getf (cdr input) ':syntax)
              nullok (getf (cdr input) ':nullok))
      (setf string input))
    (when (and argerr syntax)
      (setf argerr
            (concatenate 'string argerr
                         (format nil "~&Syntax: ~A" syntax))))
    (loop with var and type and init
          for spec in specs
          when (consp spec)
          do 
      (setf var (car spec) type (second spec) init (third spec))
          else do
      (setf var spec type nil init nil)
          collect (list var init) into vlist
          collect type into tlist
          finally (setf user vlist types tlist))
    `(let* ((,svar (or ,string ""))
            (,bvar 0)
            (,lvar (length ,svar))
            ,evar ,nvar ,@ user )
       (block parse-block
         ,@(if nullok
             `((when (or (null ,svar) (nullstringp ,svar))
               (return-from parse-block nil))))
         ,@(if syntax
             `((when (and (not *command-prompting*)
                          (or (null ,svar) (nullstringp ,svar)))
                 (tell-user ,syntax)
                 (cmdreturn nil))))
         ,.(loop for binding in user for type in types
                 for n from 1
            if (eq type :raw)
            collect `(setf ,(first binding) (subseq ,svar ,bvar))
            else if (or (eq type :rest) (eq type :pairs))
            collect 
              (let ((local (gensym))
                    (results (gensym))
                    (pairerr (format nil "Uneven pair arguments in: ~~S.~@[~~&Syntax: ~A~]"
                                     syntax)))
                `(let (,local ,results)
                   (loop while ,bvar
                         do
                     ,(parse-clause 
                        svar local bvar evar nvar lvar 
                        (if (eq type :rest) :form :symbol)
                        (if (eq type :rest) nil
                         `(cmderror ,pairerr ,svar) )
                        nil)
                     (push ,local ,results)
                    ,@(and (eq type :pairs)
                           (list (parse-clause svar local bvar evar nvar lvar 
                                              :form t nil)
                                 `(push ,local ,results)))

                     )
                   (setf ,(first binding) (nreverse ,results))))
            else
            collect (parse-clause svar (first binding)
                                  bvar evar nvar lvar type
                                  '(return-from parse-block nil)
                                  (and argerr (= n argnum)
                                       `(cmderror ,argerr ,svar))
                                  )))
       ,@body)))

(defun parse-clause (strvar valvar begvar endvar nxtvar lenvar type
                            if-no-next if-next)
  `(progn
     (multiple-value-setq (,begvar ,endvar ,nxtvar)
                          (argfield ,strvar ,begvar ,lenvar , type))  
     (when ,begvar
       ,(cond ((or (eq type ':symbol) (eq type ':form))
               `(setf ,valvar 
                  (read-from-string ,strvar nil :eof
                                    :start ,begvar :end ,endvar)))
              ((eq type ':careful-form)  ; not currently called.
               `(setf ,valvar 
                  (careful-quote
                     (read-from-string ,strvar nil :eof
                                       :start ,begvar :end ,endvar))))
              (t
               `(setf ,valvar (subseq ,strvar ,begvar ,endvar))))
       (setf ,begvar ,nxtvar))
     ,@ (if if-no-next `((unless ,begvar , if-no-next)) nil)
     ,@ (if if-next `((when ,begvar , if-next)) nil)
      ))


(defun split-input (string &optional type)
  (multiple-value-bind (b e n) (argfield string 0 nil type)
      (if b (values (subseq string b e)
                    (if n (subseq string n) "")))))

(defun sequence-to-list (string &key (beg 0) (end (length string)) 
                                     (read t))
  ;; the sequence elements must be delimited by comma
  (setf beg (tl:next-token-start string :start beg :end end))
  (let ((e end))
    (loop while beg
          do
         (setf e (or (tl:next-whitespace-start string :start beg :end end
                                              :whitespace '(#\,))
                       end))
          collect (if read 
                      (multiple-value-bind (x ee)
                        (read-from-string string nil :eof :start beg :end e)
                        (when (< ee e) 
                          (return-from sequence-to-list (values nil nil)))
                        x)
                     (let ((p e))
                       (loop while (char= (elt string (1- p)) #\space)
                             do (decf p))
                       (subseq string beg p))) 
          into results until (= e end)
          do (setf beg
               (tl:next-token-start string :start e :end end
                                   :whitespace '(#\, #\space #\tab)))
          finally (return (values results (= e end))))))


#|
(with-args "hoo bar z" ((x :reference) 
                                (y :pathname "test.snd")
                                z 
                                (a nil 123))
  (format t "Args: x=~S  y=~S  z=~S  a=~S"  x y z a))

  
(with-args "foo[1,2,4] 0 port a syntax midi"
  ((r :reference) 
   (o )
   (l :rest))
  (format t "Args: r=~S  o=~S  l=~S"  r o l))  
|#

;;;
;;; command functions
;;;

(defmacro defcmd (name lambda &body body)
  (let (comment declare)
    (when (stringp (first body))
      (setf comment (list (pop body))))
    (when (eq (first (first body)) 'declare)
       (setf declare (list (pop body))))
    `(defun ,name ,lambda
       ,@comment
       ,@declare
       (block command-function ,@body))))

(defmacro cmderror (&rest warnforms)
  (if warnforms
      `(progn 
         (warn-user ,@warnforms)
         (return-from command-function ':error))
    `(return-from command-function ':error)))

(defmacro cmdreturn (&optional value)
  `(return-from command-function ,value))

(defmacro cmdabort ()
  '(cmdreturn ':aborted))

(defmacro cmdabort? (form)
  `(if (eq ,form ':aborted) (cmdabort)))

(defmacro insure-arg (var &body form)
  (let ((ivar (gensym))
        (var1 (gensym))
        (var2 (gensym))
        (inputv nil))
    (when (consp var)
      (setf inputv (getf ':input var)
            var (car var)))
    `(let ((,ivar ,(or inputv var)))
       (multiple-value-bind (,var1 ,var2) 
                            (,@(car form) :input (or ,ivar "")
                                    :error-return (not *command-prompting* ))
         ,var2   ; not using for now.
         (when (find ,var1 '(:aborted :error))
           (return-from command-function ,var1))
         (setf ,var ,var1)))))
