;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92, 93 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; send to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(defparameter *syntax* nil "The current output syntax.")

(defclass syntax () 
  ((name :accessor syntax-name :initarg :name)
   (nicknames :initarg :nicknames :accessor syntax-nicknames :initform nil)
   (pathname :initarg :pathname)
   (stream-types :initarg :stream-types)
   (listener :accessor syntax-listener)))

(defmethod initialize-instance :after ((object syntax) &rest args)
  (declare (ignore args))
  (dolist (n (cons (slot-value object 'name)
                   (slot-value object 'nicknames)))
    (setf (get n ':syntax) object)
    (setf (get (intern (string n) ':keyword) ':syntax) object)))

(defmethod print-object ((object syntax) stream)
  (format stream "#<Syntax: ~A>" (slot-value object 'name)))

(defmacro defsyntax (name &rest args)
  `(progn
     (eval-when (compile load eval) (defclass ,name (syntax) nil))
     (add-object (make-instance ',name :name ',name ,@args)
                .syntaxes.)))

(defun find-syntax (syntax &optional (mode :error))
  (if (typep syntax 'syntax)
      syntax
    (let ((name (if (symbolp syntax) (symbol-name syntax) syntax)))
      (or (loop for s in (container-objects .syntaxes.)
                when (or (string-equal name (string (syntax-name s)))
                         (find name (syntax-nicknames s) 
                               :test #'string-equal :key #'symbol-name))
                return s)
          (case mode
	    (:error (error "~S is not the name of a syntax." syntax))
  	    (:ask (ask-syntax :abort-ok nil))
            (t nil))))))

(defun in-syntax (syntax &optional verbose)
  (setf syntax (find-syntax syntax :error))
  (when verbose (format t "Syntax set to ~A~%" (syntax-name syntax)))
  (setf *syntax* syntax))

(defun list-all-syntaxes ()
  (car (slot-value (find-object 'syntaxes) 'elements)))

;;;
;;; pathname-syntax returns two values: the file's syntax and class of stream
;;; 

(defun pathname-syntax (pathname &optional syntax)
  (let ((type (pathname-type pathname)))
    (if syntax 
      (loop for entry in (slot-value syntax 'stream-types)
            when (find type (cdr entry) :test #'string-equal)
            return (values syntax (find-class (car entry))))
      (loop for syntax in (container-objects .syntaxes.)
            for class = (loop for entry in (slot-value syntax 'stream-types)
                              when (find type (cdr entry) :test #'string-equal)
                              return (car entry))
            when class return (values syntax (find-class class))))))

(defmethod syntax-pathname ((syntax syntax))
  (make-pathname :directory (pathname-directory (user-homedir-pathname))
                 :defaults (namestring (slot-value syntax 'pathname))))

(defun stream-for-type (type &optional (syntax *syntax*) (test #'equal))
  (loop for entry in (slot-value syntax 'stream-types)
        when (member type (cdr entry) :test test)
        return (find-class (car entry))))
