;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(defparameter *commands* nil)

(tl:defcommand "?" ?-cmd "Show this help." *commands* 
               :help-file "qmark")
(tl:defcommand "ADD" add-cmd "Add objects to container." *commands*)
(tl:defcommand "ARCHIVE" archive-cmd "Archive objects to file." *commands*)
(tl:defcommand "CL" cl-cmd "Enter lisp command loop." *commands*)
(tl:defcommand "CLOSE" close-cmd "Close listener if open." *commands*)
(tl:defcommand "CHANGE" change-cmd "Change the class of objects." *commands*)
(tl:defcommand "COPY" copy-cmd "Copy objects to pasteboard." *commands*)
(tl:defcommand "CUT" cut-cmd "Cut objects to pasteboard." *commands*)
(tl:defcommand "DELETE" delete-cmd "Mark objects as deleted." *commands*)
(tl:defcommand "EDIT" edit-cmd "Edit object." *commands*)
(tl:defcommand "EXPUNGE" expunge-cmd "Destroy deleted objects." *commands*)
(tl:defcommand "FREEZE" freeze-cmd "Mark generator cache as wanted." 
                *commands*)
(tl:defcommand "GO" go-cmd "Move current focus to new object." *commands*)
(tl:defcommand "HELP" help-cmd "Show lotsa help." *commands*)
(tl:defcommand "HUSH" hush-cmd "Stop listener processing." *commands*)
(tl:defcommand "HIDE" hide-cmd "Mark objects as unavailable." *commands*)
(tl:defcommand "IMPORT" import-cmd "Translate/Load score file into stella." 
               *commands*)
(tl:defcommand "INCREMENT" increment-cmd "Increment slot values." 
               *commands*)
(tl:defcommand "INVERT" invert-cmd "Invert slot values." *commands*)
(tl:defcommand "LIST" list-cmd "List objects." *commands*)
(tl:defcommand "LISTEN" listen-cmd "Listen to objects." *commands*)
(tl:defcommand "LOAD" load-cmd "Load file into Stella." *commands*)
(tl:defcommand "MAP" map-cmd "Map over slot values." *commands*)
(tl:defcommand "MOVE" move-cmd "Move objects to new container." *commands*)
(tl:defcommand "OPEN" open-cmd "Open listener for syntax." *commands*)
(tl:defcommand "NEW" new-cmd "Create new object." *commands*)
(tl:defcommand "NEXT" next-cmd  "Go to next object in same level." *commands*)
(tl:defcommand "PASTE" paste-cmd "Paste contents of copy buffer." *commands*)
(tl:defcommand "PLAY" play-cmd "Play scorefile." *commands*)
(tl:defcommand "PREVIOUS" last-cmd "Go to preceding object." *commands*)
(tl:defcommand "QUIT" tl:top-level-quit "Quit Stella." *commands*)
(tl:defcommand "REHASH" rehash-cmd "Update top-level contents." *commands*)
(tl:defcommand "RUN" run-cmd "Run object." *commands*)
(tl:defcommand "RECEIVE" receive-cmd "Receive midi for recording, etc." 
               *commands*)
(tl:defcommand "REMOVE" remove-cmd "Remove object from container." *commands*)
(tl:defcommand "RENAME" rename-cmd "Rename an object." *commands*)
(tl:defcommand "RETROGRADE" retrograde-cmd
               "Reverse the order of container objects." *commands*)
(tl:defcommand "SCALE" scale-cmd "Scale slot values." *commands*)
(tl:defcommand "SCRIPT" script-cmd "Load a script file." *commands*)
(tl:defcommand "SET" set-cmd "Set slot values." *commands*)
(tl:defcommand "SHOW" show-cmd "Show information about object." *commands*)
(tl:defcommand "SHUFFLE" shuffle-cmd
               "Randomly reorder container objects." *commands*)
(tl:defcommand "SLISTEN" slisten-cmd "Sequentially listen to objects." 
                *commands*)
;(tl:defcommand "SWAP" swap-cmd "Switch positions of objects in container." 
;                *commands*)		
(tl:defcommand "SWRITE" swrite-cmd "Sequentially write objects to score file." 
                *commands*)
(tl:defcommand "SYNTAX" syntax-cmd "Set output syntax." *commands*)
(tl:defcommand "TLIST" tlist-cmd "List objects in time line." *commands*)
(tl:defcommand "TRANSPOSE" transpose-cmd "Transpose slot values."
               *commands*)
(tl:defcommand "UNDELETE" undelete-cmd "Undelete objects." *commands*)
(tl:defcommand "UNFREEZE" unfreeze-cmd "Mark generator cache as unwanted." 
               *commands*)
(tl:defcommand "UNHIDE" unhide-cmd "Mark objects as avaiable." *commands*)
(tl:defcommand "UNSET" unset-cmd "Unset slot values." *commands*)
(tl:defcommand "UP" up-cmd  "Go to current focus' container." *commands*)
(tl:defcommand "WRITE" write-cmd "Write objects to scorefile." *commands*)	     

;;;
;;; fake command table for topic help parsing/listing
;;;

(defparameter *topics* '(("REFERENCING") ("INTERPRETER")("EXPRESSIONS")))

;;;
;;;
;;;

(defvar *focus* nil)
(defvar *focus-container* nil)

(defun set-focus (object)
  (if (or (null object) (eq object .top-level.))
      (setf *focus* .top-level. *focus-container* .top-level.)
    (if (typep object 'startable-element)
        (setf *focus-container* object *focus* object)
      (setf *focus-container* (the-container object) *focus* object)))
  *focus*)

(defun focus-container (&optional ignore-top-level)
  (if (and (eq *focus-container* .top-level.) ignore-top-level)
      nil
    *focus-container*))

(defmacro check-top-level (&optional (container '.top-level.))
  `(unless (subobjectsp ,container)
     (cmderror "~A is currently empty."
               (object-namestring ,container))))

;;;
;;; ? 
;;; HELP {topic}
;;;

(defcmd ?-cmd (input)
  (tl:top-level-help input)
  (tell-user "Use the HELP command for detailed help about the commands ~&~
              summarized above, or to see overviews about the following ~&~
              general topics:~:{ ~A~}.~&" *topics*))

(defcmd help-cmd (input)
  (with-args (input :argchecking t :syntax "HELP {topic}")
             (cmd)
    (insure-arg cmd
      (ask-user :prompt "Help topic: "
                :error "~:@(~A~) is not a help topic.~&"
                :predicate #'(lambda (x)
                               (values
                                (or (tl:find-command x *commands*)
                                    (tl:find-command x *topics*))))
                :help nil :abort-ok t :dont-read t
                :predicate-is-filter t))
    (let* ((topic (tl::command-name cmd))
           (file (or (tl::command-help-file cmd) topic))
           (path
            (concatenate 'string cm::*common-music-directory*
                         #+unix
                         (format nil "stella/help/~(~A~).text" file)
                         #+apple
                         (format nil "stella:help:~(~A~).text" file)
                         #+dos
                         (format nil "stella\\help\\~(~A~).txt"
                                 (if (> (length topic) 8)
                                     (subseq topic 0 8) file))
                         #-(or aclpc unix apple dos)
                         (error "help-cmd not fixed for this OS."))))
      (if (probe-file path)
        (unless (tl:show-documentation path :indent 2)
          (tell-user "~:@(~A~) is not a help topic.~&" topic))
        (warn-user "Can't find help file: ~A~&" (namestring path))))))

;;;
;;; OPEN {stream} {option value}*
;;;

(defcmd open-cmd (input)
  (with-args (input :syntax "OPEN {stream} {option value}*")
             ((ref :pathname) 
              (inits :pairs))
    (insure-arg ref
      (ask-user :prompt "Open stream: " :abort-ok t :dont-read t))
    (open-cmd-aux ref (loop for (slot form) on inits by #'cddr
                            collect slot collect (careful-eval form)))))

(defun open-cmd-aux (name inits)  
  (let ((syntax (find-syntax name nil))
        (stream nil))
    (setf stream
          (if syntax 
            (apply #'open-listener syntax inits)
            (apply #'open-file name :edit nil :open nil inits)))
    (tell-user "Initialized ~S." stream)))

;;;
;;; CLOSE
;;;

(defcmd close-cmd (input)
  (with-args (input :argchecking t :nullok t
                      :syntax "CLOSE {syntax}*")
               (syntax)
    (insure-arg syntax
      (ask-syntax :prompt "Close listener for syntax: " :null-ok t
                  :default *syntax*))
    (if syntax
      (let ((stream (slot-value-or-default syntax 'listener)))
        (if stream
            (close-listener stream ':force)
          (tell-user "No listener currently open for syntax ~:(~A.~)"
                     (syntax-name syntax))))
    (tell-user "No syntax currently set.")))) 

;;;
;;; REHASH
;;;

(defcmd rehash-cmd (input)
  (declare (ignore input))
  (tell-user "REHASH is no longer necessary and will soon disappear.")
  (update-top-level :update))

;;;
;;; CL {input}*
;;;

(defcmd cl-cmd (input)
  (if (and input (not (nullstringp input)))
    (eval-from-string input :commands tl::*lisp-commands* )
    (top-level :commands tl::*lisp-commands* :prompt "Lisp> ")))

;;;
;;; EDIT {object}
;;;

(defcmd edit-cmd (input)
  (with-args (input :argchecking t :syntax "EDIT {object}")
             ((object :reference))
    (insure-arg object
      (ask-objects :prompt "Object to edit: " :focus (focus-container) 
                   :default (if (systemp *focus*) nil *focus*) 
                   :only-one t :include-containers nil))
    (edit-cmd-aux (car (reflist object)))))

(defun edit-cmd-aux (object)
  (flet ((slot-edit-spec (slot)
           (case slot
             (id '(id :parser #'(lambda (s &aux x)
                                  (setf x (read-from-string s))
                                  (and (not (find-object x nil))
                                       x))))
             (t nil))))
    (let* ((slots (setable-slots object))
           (specs (loop for s in slots
                        for e = (slot-edit-spec s)
                        when e collect e)))
      (tl:edit-object object :slots slots :slot-commands specs))))

;;;
;;; LIST {objects}
;;;

(defparameter *listing-length* 50)

(defcmd list-cmd (input)
  (with-args (input :argchecking t :syntax "LIST {objects}"
                    :nullok t) 
             ((refs :reference))
    (let ((focus (focus-container)))
      (insure-arg refs 
        (ask-objects :prompt "List objects: " 
                     :input (or refs
                                (string (object-name focus)))
                     :top-ok t :focus focus :default focus)))
    (dolist (ref refs)
      (when (indexref? ref) 
        (let ((obj (refobject ref)))
          (when (typep obj 'container)
            (setf ref obj))))
      (list-object ref))))

;;;
;;; SHOW {object}
;;;

(defcmd show-cmd (input)
  (with-args (input :argchecking t :syntax "SHOW {objects*}")
             ((ref :reference))
    (if ref 
      (insure-arg ref
        (ask-objects :prompt "Show objects: "
                     :focus (focus-container) 
                     :default *focus* :top-ok t))
      (setf ref *focus*))
    (mapref #'(lambda (x) (show-object x t)) ref)))



;;;
;;; GO {object | UP | NEXT | PREVIOUS}
;;;

(defcmd go-cmd (input)
  (with-args (input :argchecking t :syntax "GO {object}")
             ((ref :reference))
    (check-top-level)
    (cond ((string-equal ref "NEXT")
           (go-cmd-aux *focus* ':next))
          ((string-equal ref "PREVIOUS")
           (go-cmd-aux *focus* ':previous))
          ((string-equal ref "UP")
           (go-cmd-aux *focus* ':up))
          (t
           (insure-arg ref
             (ask-objects :prompt "Go to object: " 
                          :focus (focus-container) 
                          :only-one t :top-ok t :default *focus*))
           (setf ref (refobject ref))
           (set-focus ref)
           (show-object ref)))))

(defcmd next-cmd (str)
  (declare (ignore str))
  (check-top-level)
  (go-cmd-aux *focus* ':next))
  
(defcmd last-cmd (str)
  (declare (ignore str))
  (check-top-level)
  (go-cmd-aux *focus* ':previous))

(defcmd up-cmd (str)
  (declare (ignore str))
  (check-top-level)
  (go-cmd-aux *focus* ':up))

(defun go-cmd-aux (obj pos)
  (block command-function
    (if (eq *focus-container* .top-level.)
        (cmderror "Can't move relative from top level.")
      (let (super new cache-position cache)
        (if (typep obj 'startable-element)
            (let ((supers (object-containers obj)))
              (when (> (length supers) 1)
                (cmderror "Can't move ~A from ~S: more than one container!" 
                          pos obj))
              (setf super (car supers)))
          (setf super (focus-container t)))
        (when (and (null super) (eq pos ':up))	; no back link to top-level
	    (setf super .top-level.))
        (if (not super)
            (cmderror "No ~A element from ~S.~%" pos obj)
          (ecase pos
            ((:previous :next)
             (setf cache (container-objects super))
             (setf cache-position (position obj cache))
             (if (eq pos ':previous)
                 (if (= cache-position 0)
                     (setf cache-position nil)
                   (setf cache-position (1- cache-position)))
               (if (= cache-position (1- (length cache)))
                   (setf cache-position nil)
                 (setf cache-position (1+ cache-position))))
             (if cache-position
                 (setf new (nth cache-position cache))
               (cmderror "No ~A element from ~S in ~S.~%" pos obj super)))
            (:up
             (if (consp super)
                 (let ((len (length super)))
                   (setf new (car super))
                   (unless (= len 1)
                     (tell-user "~A elements UP from ~A. You get ~A.~%"
                                len (object-name obj t) (object-name new t))))		
               (setf new super)))))
        (when new
          (set-focus new)
          (show-object new))))))

;;;
;;; DELETE {objects} {RECURSE [t | nil]}* {EXPUNGE [t | nil]}*
;;;

(defcmd delete-cmd (input)
  (with-args (input :syntax "DELETE {objects} {option value}*")
             ((ref :reference)
              (pairs :pairs))
    (check-top-level)
    (insure-arg ref
      (ask-objects :prompt "Delete objects: " :focus *focus*
                   :predicate #'(lambda (o)(not (systemp o)))
                   :predicate-error "~S references one or more system objects."))
    (let ((rec nil) (exp nil))
      (dopairs (opt val pairs)
        (case opt 
          (recurse (setf rec val))
          (expunge (setf exp val))
          (t (tell-user "Ignoring unknown {option value}: ~A ~S." opt val))))
      (mapref #'(lambda (x) (delete-object x rec)) ref)
      (when exp (expunge-cmd-aux ref)))))

;;;
;;; UNDELETE {object[,object]*} {recurse [t | nil]}*
;;;

(defcmd undelete-cmd (input)
  (with-args (input :argchecking t :syntax "UNDELETE {objects} {option value}*")
             ((ref :reference)
              (pairs :pairs))
    (check-top-level)
    (insure-arg ref
      (ask-objects :prompt "Undelete objects: " :focus *focus*))
    (let ((rec nil))
      (dopairs (opt val pairs)
        (case opt 
          (recurse (setf rec val))
          (t (tell-user "Ignoring unknown {option value}: ~A ~S." opt val))))
      (mapref #'(lambda (x) (undelete-object x rec)) ref))))

;;;
;;; EXPUNGE {object[,object]*}
;;;

(defcmd expunge-cmd (input)
  (with-args (input :argchecking t
                    :syntax "Syntax: EXPUNGE {objects}") 
             ((ref :reference))
    (check-top-level)
    (insure-arg ref
      (ask-objects :prompt "Expunge objects: " :focus *focus*))
    (expunge-cmd-aux ref)))

(defun expunge-cmd-aux (refs)
  ;; we process a list to guarantee touching all objects.
  ;; this may not be necessary...
  (dolist (obj (reflist refs)) (expunge-object obj))
  (update-top-level t))

;;;
;;; HIDE {objects]*}
;;;

(defcmd hide-cmd (input)
  (with-args (input :argchecking t 
                    :syntax "Syntax: HIDE {objects}") 
             ((ref :reference))
    (check-top-level)
    (insure-arg ref
      (ask-objects :prompt "Hide object: " :focus *focus*
                   ))
    (mapref #'hide-object ref)))

;;;
;;; UNHIDE {objects}
;;;

(defcmd unhide-cmd (input)
  (with-args (input :argchecking t :syntax "Syntax: UNHIDE {objects}") 
             ((ref :reference))
    (check-top-level)
    (insure-arg ref
      (ask-objects :prompt "Unhide objects: " :focus *focus*))
    (mapref #'unhide-object ref)))

;;;
;;; RENAME {object} {newname}
;;;

(defcmd rename-cmd (input)
  (with-args (input :argchecking t :syntax "RENAME {object} {newname}") 
             ((ref :reference)  name)
    (check-top-level)
    (insure-arg ref
      (ask-objects :prompt "Rename object: "
                   :focus *focus-container*
                   :only-one t :typep 'id-mixin ))
    (insure-arg name
      (ask-name :prompt "New Name: " 
                :error "\"~:@(~A~)\" is already a name."))
    (let ((obj (refobject ref)))
      (rename-object obj name (object-name obj)))))

;;;
;;; MAP {objects} {clause}+
;;;

(defcmd map-cmd (input)
  (let (width)
    (with-args (input :syntax "MAP {objects} {clause}*") 
               ((ref :reference) (spec :raw))
      (check-top-level)
      (insure-arg ref
        (ask-objects :prompt "Map objects: " :focus (focus-container) 
                     :default (focus-container t)
                     ))
      (when (and (not (cdr ref)) (groupref? (first ref)))
        (setf width (cdar (first ref))))
      (insure-arg spec
        (ask-clause :prompt "Map expression: " :width width))
      (mapslots ref spec width))))

;;;
;;; SET       {objects} {slot expr}+
;;; UNSET     {objects} {slot}+
;;; INCREMENT {objects} {slot expr}+
;;; SCALE     {objects} {slot expr}+
;;; TRANSPOSE {objects} {slot expr}+
;;; INVERT    {objects} {slot expr}+
;;; CHANGE    {objects} {newclass} {newslot oldslot}*
;;;

(defcmd set-cmd (input)       (set-cmd-aux ':set nil input))
(defcmd unset-cmd (input)     (set-cmd-aux ':unset nil input))
(defcmd increment-cmd (input) (set-cmd-aux ':increment nil input))
(defcmd scale-cmd (input)     (set-cmd-aux ':scale nil input))
(defcmd transpose-cmd (input) (set-cmd-aux ':transpose nil input))
(defcmd invert-cmd (input)    (set-cmd-aux ':invert nil input))
(defcmd change-cmd (input)    (set-cmd-aux ':change nil input))


(defcmd set-cmd-aux (op objects input)
  ;; objects is a list of instances, a class to instantiate and set, or nil
  (unless objects (check-top-level))
  (let ((syntax (cond ((eq op :change)
                       "CHANGE {class} {newclass} {slot expr}*")
                      ((eq op :unset) "UNSET {slot}+")
                      ((eq op :new)
                       "NEW {class} {name|count}* {positon}* {slot expr}*")
                      (t (format nil "~A {slot expr}+" op))))
        help create)
    (if objects
      (cond ((typep objects 'standard-class)
             (setf help objects create t))
            ((consp objects)
             (setf help (class-of (car objects)) create nil))
            (t (setf help (class-of objects) create nil)))
      (with-args (input :argchecking t :syntax syntax)
                 ((ref :reference)
                  (clause :raw))
        (insure-arg ref
          (ask-objects :prompt (format nil "~(~A~) objects: " op)
                       
                       :focus (focus-container) :any-level t))
        (setf objects ref input clause)))
    (insure-arg input
      (ask-clause :op op :terminate create :class help :only-one t))
    ;; we only do the work if the user specified exactly one clause.
    (if (consp input)
      (let ((fun (make-clause-closure input t))
            (lev (mapping-level))
            (mod (mapping-mode t)))
        (if create
          (let (($done$ nil) 
                i)
            (declare (special $done$))
            (setf objects    
                  (loop do (setf i (make-instance objects))
                        (funcall fun i nil)
                        collect i
                        until $done$)))
          (mapref #'(lambda (o c) (map-object-aux o fun c mod 0 lev))            
                  objects t t)))
      (if create (setf objects nil)))
    objects))

;;;
;;; Listen, Slisten, Tlist, Write, Swrite, Run
;;;

(eval-when (compile) ; only need this when compile the commands.
(defmacro listen-internal (op input)
  ;; this expands into the proper input parsing form for the above commands.
  (let (typep filep seqp syntax prompt)
    (ecase op
      ((:listen :tlist :slisten :run)
       (setf typep (eq op ':listen) filep nil)
       (setf seqp (eq op ':slisten))
       (setf syntax
         (format nil "~A {object[,object]*} {start[,start]*} ~
                     {option value}*" op))
       (setf prompt
         (if (eq op :listen) "Listen to objects: "
           (format nil "~:(~A~) objects: " op))))
      ((:write :swrite)
       (setf filep t typep nil)
       (setf typep (eq op ':write))
       (setf seqp (eq op ':slisten))  
       (setf syntax
         (format nil "~A {object[,object]*} {start[,start]*} ~
                     {file} {option value}*" op))
       (setf prompt
         (if (eq op :listen) "Listen to objects: "
           (format nil "~:(~A~) objects: " op)))))
    `(progn
       (let (objects offsets)
         (with-args (,input :syntax ,syntax)
                    ((ref :reference)
                     ,@(if filep '((file :pathname)))
                     (off :sequence) 
                     (pairs :pairs))
           (check-top-level)
           (insure-arg ref
             (ask-objects :prompt ,prompt :default (focus-container t)
                         ,@ (if typep 
                              `(:typep 'startable-element
                                :error ,(format nil "~:(~A~) expects startable objects. ~
                                                     Maybe you want to use S~(~A~)."
                                                op op))
                              '( ))
                          :focus (focus-container)))
           (setf objects ref)
           ,@(and filep
                  `((let ((d (if *syntax* 
                               (namestring (syntax-pathname *syntax*)))))
                      (insure-arg file
                        (ask-file :prompt "Output file: " :default d 
                                  :direction :write :pathname-defaults d)))))
           (setf off
             (ask-user :prompt "Start time offset:" :abort-ok t
                       :default-input-match "None" 
                       :default "0.0" :default-prompt "(<cr>=None)"
                       :dont-read t :input off
                       :null-ok (if *command-prompting* :ask t)))
           (cmdabort? off)
          (multiple-value-bind (x y) (ask-starts off objects ,seqp)
             (case x
               (:error (cmderror y))
               (:aborted (cmdabort))
               (t (setf offsets x))))
           (when pairs
             (setf pairs (canonicalize-pairs pairs t)))
           ;; keep it backward compatible. prompt for repeat and rest if not
           ;; supplied
           ,@ (if seqp
                '((when *command-prompting*
                    (unless (listp offsets)
                      (setf offsets (list offsets)))
                    (unless (second offsets)
                      (let (repeat)
                        (insure-arg repeat 
                          (ask-number :prompt "Number of times to sequence: "
                                      :help "Type the number of times to sequence."
                                      :default 1))
                        (when (> repeat 1)
                          (setf offsets (append offsets (list repeat))))))
                    (unless (third offsets)
                      (let (pause)  
                        (insure-arg pause 
                          (ask-number 
                           :prompt "Length of pause between selections: "
                           :help "Seconds of pause, or None."
                           :default 'none))
                        (when (numberp pause)
                          (setf offsets
                                (if (second offsets)
                                  (append offsets (list pause))
                                  (append offsets (list 1 pause))))))))))
          ,(if filep `(values objects file  offsets pairs)
             `(values objects offsets pairs)))))))
)

(defun ask-starts (input objects sequential?)
  (let (starts offsets ok)
    (multiple-value-setq (starts ok) (sequence-to-list input))
    (unless ok
      (return-from ask-starts
        (values nil (format nil "Offset ~S is not a sequence." input))))
    (when (eq (car starts) '*)
      (when sequential?
        (return-from ask-starts 
          (values :error 
                  "Offset \"*\" meaningless for sequential processing.")))
      (when (= (length starts) 1)
        (when (not *command-prompting*)
          (return-from ask-starts 
            (values :error "Offset \"*\" impossible without command prompting."))))
        (let ((l ()))
          (flet ((ask-start (o)
                 (let (a)
                   (setf a
                     (ask-user :prompt (format nil "Start time offset for ~A: " 
                                         (object-namestring o))
                               :null-ok ':ask :default 'none :default-value 0.0 
                               :default-prompt "(<cr>=None)" 
                               :default-input-match "none"
                               :abort-ok t :help "Start time delay."))
                   (when (eq a ':aborted)
                     (return-from ask-starts (values ':aborted nil)))
                   (push a l))))
             (mapref #'ask-start objects))
           (setf starts (nreverse l))))
;     (when (and sequential? (> (length offsets) 1))
;       (return-from ask-starts 
;         (values :error 
;                 "Bad offset \"A\": one offset for sequential processing.")))
      (setf offsets
        (loop for val in starts
              for tmp = (careful-eval val)
              unless (and (numberp tmp) (>= tmp 0.0))
              do (return-from ask-starts
                   (values :error (format nil "Bad offset: ~A" tmp)))
              else collect tmp))
     (values (if (cdr objects) 
                 (if (cdr offsets) offsets (car offsets))
               offsets)
             nil)))

(defcmd listen-cmd (input)
  (multiple-value-bind (objects offsets pairs) 
      (listen-internal :listen input)
    ;(format t "~%~S ~S ~S" objects offsets pairs)
    (when objects (apply #'listen-object objects offsets pairs))))

(defcmd slisten-cmd (input)
  (multiple-value-bind (objects offsets pairs) 
      (listen-internal :slisten input)
    ;(format t "~%~S ~S ~S" objects offsets pairs)
    (when objects (apply #'slisten-object objects offsets pairs))
    ))

(defcmd tlist-cmd (input)
  (multiple-value-bind (objects offsets pairs) 
    (listen-internal :tlist input)
    (when objects (apply #'tlist-object objects offsets pairs))))

(defcmd write-cmd (input)
  (multiple-value-bind (objects file offsets pairs) 
      (listen-internal :write input)
  ;(format t "~%~S ~S ~S ~S" objects file offsets pairs)
  (when objects (apply #'write-object objects file offsets pairs))))

(defcmd swrite-cmd (input)
  (multiple-value-bind (objects file offsets pairs) 
      (listen-internal :swrite input)
    ;(format t "~%~S ~S ~S ~S" objects file offsets pairs)
    (when objects (apply #'swrite-object objects file offsets pairs))))

(defcmd run-cmd (input)
  (multiple-value-bind (objects offsets pairs) 
      (listen-internal :run input)
    (when objects (apply #'run-object objects offsets pairs))))

;;;
;;; HUSH {syntax}*
;;;

(defcmd hush-cmd (input)
  ;; let {syntax} be optional if at all possible...
  (when (and (nullstringp input) *syntax* (not *command-prompting*))
    (setf input (string (syntax-name *syntax*))))
  (with-args (input :argchecking t :syntax "HUSH {syntax}*") 
             (syntax)
    (insure-arg syntax
      (ask-syntax :prompt "Hush listener for: " :default *syntax*
                  :error "Not a syntax: \"~A\"."
                  :abort-ok t :null-ok (and *syntax* (nullstringp input))))
    (print syntax)))

;;;
;;; PLAY {file} {option value}*
;;;

(defcmd play-cmd (input)
  (with-args (input :argchecking t :syntax "PLAY {file} {option value}*") 
             ((file :pathname) (pairs :pairs))
    (let ((default (and *syntax* (syntax-pathname *syntax*))))
      (insure-arg file
        (ask-file :prompt "Play file: " :default default)))
    (apply #'play-file file pairs)))

;;;
;;; SCRIPT {file} {option value*}
;;;

(defcmd script-cmd (input)
  (with-args (input :argchecking t :syntax "SCRIPT {file} {option value}*") 
             ((file :pathname)
              (pairs :pairs))
    (insure-arg file
      (ask-file :prompt "Script file: " :default "test.tl"))
    (let ((p (format nil 
                     "~~:[~~&~~;~~%~~]Stella(~A.~A) [~~:(~~A~~)]: " 
                     (pathname-name file) (pathname-type file))))
      (apply #'script-file file :output *tell-stream*
             :prompt #'(lambda (stream)
                         (format stream p tl:*prompt-newline*
                                 (object-name *focus-container*))) pairs))))

;;;
;;; LOAD {file} {option value*}
;;;

(defcmd load-cmd (input)
  (with-args (input :argchecking t :syntax "LOAD {file} {option value}*") 
             ((file :pathname)
              (pairs :pairs))
    (insure-arg file
      (ask-file :prompt "Load file: " :default "test.stella"))
    (apply #'load-file file pairs)
    (update-top-level :update)))

;;;
;;; IMPORT {file} {option value*}
;;;

(defcmd import-cmd (input)
  (with-args (input :argchecking t :syntax "IMPORT {file} {option value}*")
             ((file :pathname)
              (pairs :pairs))
    (insure-arg file
      (ask-file :prompt "Import file: " ))
    (apply #'import-file file pairs)
    (update-top-level :update)))

;;;
;;; ARCHIVE {objects} {file} {option value*}
;;;

(defcmd archive-cmd (input)
  (with-args (input :argchecking t 
                    :syntax "ARCHIVE {objects} {file} {option value}*")
             ((ref :reference)
              (file :pathname) 
              (pairs :pairs))
    (check-top-level)
    (insure-arg ref
      (ask-objects :prompt "Archive objects: "
                   :any-level nil :top-ok nil :null-ok nil 
                   :focus .top-level. :include-containers t
                   :default (focus-container t)))
    (insure-arg file
      (ask-file :prompt "Archive file: " :direction ':write
                :default (merge-pathnames (user-homedir-pathname)
                                          #+aclpc "test.stl"
                                          #-aclpc "test.stella")))
    (when (and (probe-file file)
               *command-prompting*
               (not (y-or-n-p "~A exists. Overwrite? "
                              file)))
      (cmdabort))
    (let ((objects (reflist ref))
          (syntax *syntax*))
      (dopairs (n v pairs)
        (case n
          ((syntax :syntax) (setf syntax v))
          (t (tell-user "Ignoring unknown option and value ~A. ~A"
                        n v))))
      (when syntax (push (find-syntax syntax) objects))
      (archive-object objects file))))

;;;
;;; SYNTAX {syntax}*
;;;

(defcmd syntax-cmd (input)
  (unless (tl:nullstringp input)
    (with-args (input :argchecking t :syntax "SYNTAX {syntax}*")
               (syntax)
      (insure-arg syntax
        (ask-syntax :prompt "Set syntax to: " :default *syntax*))
      (setf *syntax* syntax)))
  (if *syntax*
    (tell-user "Current syntax is ~A." (syntax-name *syntax*))
    (tell-user "No syntax set.")))

;;;
;;; NEW {type} {name|number}* {position}* {slot value}*
;;;

(defcmd new-cmd (input)
  (with-args (input :argchecking 3
                    :syntax "NEW {type} {name|number}* {position}* {slot value}*")
             (type narg (pos :reference) (str :raw))
    (let (class new)    
      (insure-arg type
        (ask-class :prompt "Type of new object: " ))
      (setf class (find-class type))
      #+aclpc (unless (class-finalized-p class) (finalize-inheritance class))
      (if (typep (class-prototype class) 'id-mixin)
        (let ((default (gentemp (format nil "~A-" type))))
          (if (and (not *command-prompting*) (not narg))
            (setf narg default)
            (insure-arg narg                      
              (ask-name :prompt (format nil "Name for new ~:(~A~): " type) 
                        :default default :null-ok :ask)))
          (setf new (make-instance class :id narg)))
        (progn
          (if (and (not *command-prompting*) (not narg))
            (setf narg 1)
            (insure-arg narg
              (ask-number :prompt (format nil "Number of ~(~A~)s to create: "
                                          type)
                          :type '(integer 1) :default '*)))
          (if (numberp narg)
            (setf new (loop repeat narg collect (make-instance class)))
            (if (and (not *command-prompting*) (eq narg '*))
              (setf new (list (make-instance class)))
              (setf new class)))))
      ;; new is a single new container, a list of data, or a class of data to
      ;; instantiate by the {slot value} pairs. if *command-prompting* is nil
      ;; and we have no pairs we don't call on set. if it is nil and new is a
      ;; class then we can't instantiate anything.
      (if (and (not *command-prompting*) 
               (or (null str) (nullstringp str)))
        (when (eq new class)
          (cmderror "No objects in *, missing {slot value} pairs."))
        (when (or str (not (typep new 'id-mixin)))
          (let ((res (set-cmd-aux ':new new str)))
            (if (or (null res) (eq res ':aborted) (eq res ':error))
              (if (and (eq new class) (null res))
                (cmderror 
                 "New: no objects in *, missing {slot value} pairs.")
                (cmdreturn res))
              (setf new res)))) )
      ;; add the new element to some existing container. defaults to the
      ;; top level if the new object is itself a container, otherwise the
      ;; current focus container, or else the pasteboard.
      (let ((default (if (typep new 'id-mixin)
                       (focus-container)
                       (or (focus-container t) .pasteboard.)))
            tell con)
        (if (and (not *command-prompting*) (not pos))
          (setf pos default tell t)
          (insure-arg pos
            (ask-position :focus (focus-container)
                          :prompt "New object position: "
                          :include-containers :absolute-ok
                          :abort-ok nil :default default :null-ok :ask)))
        (if (consp pos)
          (setf con (cdr pos) pos (car pos))
          (setf con pos pos nil))
        (if (consp new) 
          (add-objects new con pos)
          (add-object new con pos))
        (when (and (not (eq con .top-level.))
                   (typep new 'id-mixin))
          (add-object new .top-level.))
        (when tell 
          (let ((len (if (consp new) (length new) 1)))
            (tell-user "Added ~A new object~@[s~] to ~:(~A~)"
                       len (/= 1 1) (slot-value con 'id)))))
      new)))

;;;
;;; ADD {objects} {position}
;;;

(defcmd add-cmd (input)
  (with-args (input :argchecking t :syntax "ADD {objects} {position}")
             ((ref :reference) (pos :reference))
    (check-top-level)
    (let ((focus (focus-container)))
      (insure-arg ref
        (ask-objects :prompt "Add objects: " :focus focus))
      (insure-arg pos
        (ask-position :prompt "Position for objects: "
                      :focus focus :default (focus-container t)
                      :only-one t :include-containers :absolute-ok))
      (let ((sys? (sysref? pos)))   ; its ok to add to pasteboard...
        (when (and sys? (not (eq sys? .pasteboard.)))
          (cmderror "Can't explicitly add objects to ~:(~A~)." 
                    (object-name sys?))))
      (add-cmd-aux (reflist ref) pos "Added"))))

(defun sysref? (ref)
  (mapref #'(lambda (o c) 
              (if (systemp o) (return-from sysref? o)
                 (if (and c (systemp c))
                     (return-from sysref? c))))
               
          ref t)
  nil)

(defun add-cmd-aux (objects target &optional verbose)
  (let ((len (length objects)) con pos err ids)
     (if (consp target)
         (setf con (cdr target) pos (car target))
       (setf con target pos nil))
     ;; this check should really be recursive.
     (mapc #'(lambda (x)
               (when (eq con x)  (setf err t))
               (when (typep x 'id-mixin)
                 (push (object-name x) ids)))
           objects)
     (when ids (setf ids (nreverse ids)))
     (if (find con objects)
         (warn-user "Can't add ~A to itself!" (object-namestring con))
       (add-objects objects con pos :copy-first nil))

     (when (and verbose (not err))
       (if (= len (length ids))
           (tell-user "~A ~:(~A~)~@[~{,~:(~A~)~}~] to ~:(~A~)."
                      verbose (car ids) (cdr ids) (object-name con))
         (tell-user "~A ~A ~:[object~;objects~] to ~:(~A~)."
                    verbose len (/= len 1) (object-name con))))
     objects))

;;;
;;; REMOVE {positions}
;;; CUT    {positions}
;;; MOVE   {positions} {position}
;;;

(defcmd remove-cmd (input)
  (with-args (input :argchecking t :syntax "REMOVE {positions}") 
             ((ref :reference))
    (insure-arg ref
      (ask-position :prompt "Remove positions: " :only-one nil
                    :include-containers :with-index
                    :focus (focus-container )
                    :default (focus-container t)))
    (let ((sys? (sysref? ref)))
      (when sys?
        (cmderror "Can't remove from ~:(~A~)." (object-name sys?))))     
    (remove-cmd-aux ref)))

(defcmd cut-cmd (input)
  (with-args (input :argchecking t  :syntax "CUT {positions}") 
             ((ref :reference))
    (insure-arg ref
      (ask-position :prompt "Cut positions: " :only-one nil
                    :include-containers :with-index
                    :focus (focus-container )
                    :default (focus-container t)))
    (let ((sys? (sysref? ref)))
      (when sys?
        (cmderror "Can't cut from ~:(~A~)." (object-name sys?))))     
    (remove-all-objects .pasteboard.)
    (add-cmd-aux (remove-cmd-aux ref) .pasteboard. "Cut")))

(defcmd move-cmd (input)
  (with-args (input :argchecking t :syntax "MOVE {positions} {position}") 
             ((ref :reference) (pos :reference))
    (insure-arg ref
      (ask-position :prompt "Move positions: " :only-one nil
                    :include-containers :with-index
                    :focus (focus-container ) 
                    :default (focus-container t)))
    (insure-arg pos
      (ask-position :prompt "Position to move to: "
                    :focus (focus-container) 
                    :default (focus-container t)
                    :only-one t :include-containers :absolute-ok))
    (let ((from (sysref? ref))
          (to   (sysref? pos)))
      (when (or from to)
        (cmderror "Can't move ~:[from~;to~] system object ~:(~A~)." 
                  to (object-name (or from to)))) )
    (add-cmd-aux (remove-cmd-aux ref) pos "Moved")))
      
(defun remove-cmd-aux (ref)
  ;; remove the references (in their reverse order) and return 
  ;; the removed objects in ascending order
  (let ((todo '())
        (done '()))
    (mappos #'(lambda (p c)
                (let ((bag (assoc c todo)))
                  (unless bag 
                    (push (setf bag (list c)) todo))
                  (push p (cdr bag))))
            ref)
    (map nil 
         #'(lambda (l) 
             (setf (cdr l) (sort (cdr l) #'>))
             (let ((c (car l)))
               (dolist (p (cdr l))
                 (push (remove-object p c) done))))
         todo)
    done))

;;;
;;; COPY {objects}*
;;;

(defcmd copy-cmd (input)
  (when (nullstringp input)
    (remove-all-objects .pasteboard.)
    (tell-user "Pasteboard cleared.")
    (cmdreturn nil))
  (with-args (input :argchecking t :syntax "COPY {objects}*")
             ((ref :reference))
    (insure-arg ref
      (ask-objects :prompt "Copy objects: " :default nil
                   :focus (focus-container)))
    (remove-all-objects .pasteboard.)
    (add-cmd-aux (reflist ref) .pasteboard. "Copied")))

;;;
;;; PASTE {position}*
;;;

(defcmd paste-cmd (input)
  (with-args (input :argchecking t :syntax "PASTE {position}* {name}*")
             ((pos :reference)
              name)
    (check-top-level)
    (unless (subobjectsp .pasteboard.)
      (tell-user "~:(~A~) is empty." (object-name .pasteboard.))
      (cmdreturn nil))
    (let ((count 0)
           default names)
      (mapc #'(lambda (x)
                (incf count)
                (when (typep x 'id-mixin)
                  (push (object-name x) names)))
            (container-objects .pasteboard.))
      (if (= (length names) count)
        (setf default .top-level. names (nreverse names))
        (setf default (focus-container t) names nil))
      (insure-arg pos
        (ask-position :prompt "Paste position: " :only-one :reference
                      :include-containers :absolute-ok
                      :focus (focus-container ) 
                      :default default))
      (unless (or (singleref? pos) (rangeref? pos))
        (cmderror "Paste position must be index or range."))
      (when (and *command-prompting* (= 1 (length names)))
        (insure-arg name
          (ask-name :prompt (format nil "Paste ~:(~A~) to new name: " 
                                    (car names))
                    :error "\"~:@(~A~)\" is already a name.")))
      (let ((con (refcontainer pos))
            new)
        ;; if we paste to a range we remove the current objects and set
        ;; our insert position to the low bound, if possible.
        (when (rangeref? pos)
          (remove-cmd-aux pos)
          (setf pos 
                (if (>= (caar pos) (object-count con)) 
                  con (makeref con (caar pos)))))
        (setf new
              (mapcar
               #'(lambda (o &aux (c (copy-object o)))
                   (when (typep c 'id-mixin)
                     (rename-object c
                                    (gentemp (format nil "~A-COPY-"
                                                     (object-name o)))
                                    (object-name c)))
                   c)
                (container-objects .pasteboard.)))
        (when name
          (if (and (= (length new) 1)
                   (typep (first new) 'id-mixin))
            (rename-object (first new) name)
            (let ((thread (make-instance 'thread :id name)))
              (add-objects new thread)
              (setf new (list thread)))))
         (add-cmd-aux new pos "Pasted")
        (unless (eq con .top-level.)
          (update-top-level :update))))))

;;;
;;; FREEZE {generators}
;;;

(defcmd freeze-cmd (input &optional unfreeze)
  (with-args (input :argchecking t
                    :syntax (format nil "~@[UN~]FREEZE {generators}"
                                    unfreeze)) 
             ((ref :reference))
    (check-top-level)
    (let* ((focus (focus-container))
           (default (if (typep focus 'generator) focus))
           (count 0))
      (insure-arg ref
        (ask-objects :prompt (format nil "~:[F~;Unf~]reeze: " unfreeze)
                     :focus focus :default default
                     :typep 'generator))
      (mapref #'(lambda (o)
                  (if unfreeze 
                    (unfreeze-object o)
                    (freeze-object o))
                  (incf count))
              ref)
      (tell-user "~A object~@[s~] ~@[un~]frozen."
                 count (/= count 1) unfreeze))))

;;;
;;; UNFREEZE {generators}
;;;

(defcmd unfreeze-cmd (input)
  (freeze-cmd input t))

;;;
;;; RETROGRADE {ranges}
;;; SHUFFLE {ranges}
;;;

(defcmd shuffle-cmd (input)
  (rearrange input ':shuffle))

(defcmd retrograde-cmd (input)
  (rearrange input ':retrograde))

(defcmd rearrange (input op)
  (with-args (input :argchecking t 
                    :syntax (format nil "~A {ranges}" op))
             ((ref :reference))
    (check-top-level)
    (let* ((focus (focus-container))
           (default (if (typep focus 'thread) focus)))
      (insure-arg ref
        (ask-objects :prompt "Retrograde positions: "
                     :focus focus :default default
                     :mapper #'mapc 
                     :error "~S is not a thread or subrange."
                     :predicate #'(lambda (r) 
                                    (or (and (singleref? r)
                                             (typep (refobject r) 'thread))
                                        (and (rangeref? r)
                                             (typep (refcontainer r)
                                                    'thread))))))
      (if (eq op ':retrograde)
        (mapc  #'(lambda (r) 
                   (if (singleref? r)
                     (retrograde-object (refobject r))
                     (retrograde-object (refcontainer r) 
                                        (reflb r) (refub r))))
               ref)
        (mapc #'(lambda (r)
                  (if (singleref? r)
                    (shuffle-object (refobject r))
                    (shuffle-object (refcontainer r)
                                    (reflb r) (refub r))))
              ref)))))


;;;
;;; top level stella. 
;;;

(defvar *stella* ':never "T if Stella is running.")

(defun stella-p () (eq *stella* t))

#-mcl
(progn

  (defun stella () (stella-internal))

  (defun stella-initialize ()
    (when (eq *stella* ':never)
      (tell-user "Hang on a second...~%"))
    (update-top-level t))

  (defun stella-deinitialize ()
    (remove-all-objects .top-level.))
)

#+mcl
(progn

  (defun install-stella-menubar ())    ; stub out for now.
  (defun deinstall-stella-menubar ())
  (defun update-stella-menubar ())

  (defun stella ()
    (push (ccl:%set-toplevel 
           #'(lambda () 
               (stella-internal)                 
               (ccl:%set-toplevel (pop tl::*top-level-stack*))))
          tl::*top-level-stack*)
    (ccl:toplevel))
  
  (defun stella-initialize ()
    (install-stella-menubar)
    (update-top-level t))
  
  (defun stella-deinitialize ()
    (remove-all-objects .top-level.)
    (deinstall-stella-menubar)
    ;    (ccl:%set-toplevel (pop tl::*top-level-stack*))
    (ccl:set-window-package ccl:*top-listener* 
                            (package-name *package*))
    (ccl:mini-buffer-update ccl:*top-listener*))
  
  )

(defun update-top-level (mode &rest objects)
  (unless (or (eq mode ':update) (eq mode ':add))
    (remove-all-objects .top-level.))
  (let ((old (and (eq mode ':update) (container-objects .top-level.)))
        (new (if (eq mode ':add) (copy-list objects) nil)))
    
    (unless new
      (maphash #'(lambda (key val)
                   (declare (ignore key))
                   (unless (or (systemp val)
                               (and old (find val old)))
                     (push val new)))
               *object-table*))
    (when new 
      (add-objects (nreverse new) .top-level. 
                   nil :copy-first nil))
    #+mcl(update-stella-menubar)
    (values)))     

(defun stella-internal ()
  (stella-initialize)
  (unwind-protect
    (let ((*package* (find-package :stella))
          (*focus* .top-level.)
          (*readtable* cm:*common-music-readtable*)
          (*focus-container* .top-level.))
      (setf *stella* t)
      (tl:top-level :commands *commands* 
                    :prompt #'(lambda (s)     
                                (format s "~:[~&~;~%~]Stella [~:(~A~)]: " 
                                        tl:*prompt-newline*
                                        (object-name *focus-container*)))))
    (stella-deinitialize)
    (setf *stella* nil))
  (values))

#+excl(top-level:alias ("stella" 5) () (stella))


;;;
;;; not yet converted.
;;;

(defcmd receive-cmd (input)
  (let (type prompt stop value lambda)
    (setf type 
      (ask-type-or-token :input input :default 'play :null-ok :ask
                         :prompt "Receive mode (Record, Play, Catch): "
                         :type '(member record play catch)))
    (cmdabort? type)
    (when (eq type 'catch)
      (setf lambda
        (ask-function :prompt "Function to catch Midi with: "
                      :default 'midi-print-message
                      :help "Specify the name of a compiled ~
			     function with 2 arguments: (msg time)~&"))
      (cmdabort? lambda))
    (setf prompt (format nil "Stop ~(~A~)ing by key, time or count? "  type))
    (setf stop 
      (ask-type-or-token :type '(member key time count)
                         :prompt prompt :default 'key))
    (cmdabort? stop)
    (setf prompt (format nil "~@(~A~) value: " stop))
    (setf value
      (if (eq stop 'key)
          (tl:ask-user :prompt prompt :abort-ok t :predicate #'degree
                       :predicate-is-filter t :null-ok :Ask :default 'a0)
        (ask-number :prompt prompt :min 0 :default nil)))
    (cmdabort? value)
    (case type
      (record
        (setf (fill-pointer *record-buffer*) 0)
	(setf lambda #'scratch-record))
      (play (setf lambda #'midi-write-message)))
    (let (success stream)
      (unwind-protect
        (progn
	  (setf stream (open-listener (find-syntax ':midi)))
	  (when stream
            (initialize-stream-for-processing stream)
	    (setf prompt
	      (format nil "Type <cr> to begin ~(~A~)ing messages: " type))
            (midi-receive-messages lambda :prompt prompt :start 0 
                                   :keynum (and (eq stop 'key) (degree value))
                                   :length (and (eq stop 'count) (floor value))
                                   :end (and (eq stop 'time) (float value)))
            (deinitialize-stream-for-processing stream))
	  (setf success t))   ; all ok, clear flag
        (when (not success)
          (close-listener stream :force)))
	(when (and success (eq type 'record))
	  (handle-recording *record-buffer*)))))

(defcmd handle-recording (buffer)
  (let (name type class proto thread channel-slot keynum-slot keynum-type
        amplitude-slot amplitude-type duration-slot rhythm-slot prompt)
    (setf name 
      (y-or-n-or-token :yes-ok nil
                       :prompt "Save recording? (No, or name of new thread): "
		       :predicate #'(lambda (x) (not (find-element x nil)))))
    (unless name (return-from handle-recording nil))
    (cmdabort? name)
    (setf type (ask-class :prompt "Element type for recorded data: "
                          :default 'midi-note))
    (cmdabort? type)
    (setf prompt "No, or slot to set")
    (setf class (find-class type))
    #+aclpc (unless (class-finalized-p class) (finalize-inheritance class))
    (setf proto (class-prototype class))
    ;; set defaults if midi-note
    (when (typep proto 'midi-note)
      (setf channel-slot 'channel keynum-slot 'note amplitude-slot 'amplitude
            duration-slot 'duration  rhythm-slot 'rhythm))
    ;; ask for the slots to set.
    (setf channel-slot
      (ask-slot class :prompt (format nil "Keep channel info? (~A): " prompt)
                :none-ok t :default channel-slot :slot t))
    (cmdabort? channel-slot)
    (setf keynum-slot
      (ask-slot class :prompt (format nil "Keep keynum info? (~A): " prompt)
	        :none-ok t :default keynum-slot :slot t))
    (cmdabort? keynum-slot)
    (setf keynum-type (ask-member '(note degree pitch)
                                  :prompt "Keynums become: " :default 'note))
    (cmdabort? keynum-type)
    (setf amplitude-slot
      (ask-slot class :prompt (format nil "Keep velocity info? (~A): " prompt)
	        :none-ok t :default amplitude-slot :slot t))
    (cmdabort? amplitude-slot)
    (setf amplitude-type
      (if (ask-y-or-n :prompt "Convert velocity to amplitude?: " 
                      :default :yes)
          'amplitude 'velocity))
    (setf duration-slot
      (ask-slot class :prompt
                      (format nil "Use On/Off pairs for duration? (~A): "
                              prompt)
	        :none-ok t :default duration-slot :slot t))	
    (cmdabort? duration-slot)
    (setf rhythm-slot
      (ask-slot class :prompt (format nil "Use On/On pairs for rhythm? (~A): "
                                      prompt)
	        :none-ok t :default 'rhythm :slot t))	
    (cmdabort? rhythm-slot)
    (cmdabort? (setf thread (new-cmd (format nil "THREAD ~A" name)))
              )
    (record-buffer-to-thread buffer thread class rhythm-slot 
                             duration-slot channel-slot keynum-slot
			     keynum-type amplitude-slot amplitude-type)))
