;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

;;;
;;; score-unset should be called on an element once it has been
;;; selected and processed.  this happens automatically for 
;;; containers within the score-selection function itself.
;;;

(defgeneric score-unset (object &optional recursive))

(defmethod score-unset ((object t) &optional recursive)
  #-aclpc (declare (ignore recursive))
  object)

(defmethod score-unset :after ((object startable-element) &optional recursive)
  #-aclpc (declare (ignore recursive))
  (let ((flags (slot-value object 'flags)))
    (when (logtest flags +start-unset+)
      (setf (slot-value object 'flags) (logandc2 flags +start-unset+))
      (slot-makunbound object 'start)))
  (slot-makunbound object 'time)
  (when (slot-boundp object 'finalizer)
    (funcall (slot-value object 'finalizer))))

(defmethod score-unset :after ((object container) &optional recursive)
  (slot-makunbound object 'offset)	     ; flush current start time offset
  (when recursive (map-subobjects #'(lambda (x) (score-unset x t)) object))
  (let ((flags (slot-value object 'flags)))
    (when (logtest flags +ephemeral+)
      (remove-all-objects object))))

(defmethod score-unset :after ((object merge) &optional recursive)
  #-aclpc (declare (ignore recursive))
  (when (logtest (slot-value object 'flags) +ephemeral+)
    (slot-makunbound object 'starts))  
  (let ((bag (slot-value object 'elements)))
    (setf (bag-elements bag) nil)
    (setf (bag-tail bag) (last (bag-cache bag)))))

(defmethod score-unset :after ((object algorithm) &optional recursive)
  #-aclpc (declare (ignore recursive))
  (object-makunbound-using-class (slot-value object 'class)
                                 (slot-value object 'instance)))
  
;;;
;;; score-reset is called on an element if it is marked as
;;; unset. as of now this is indicated by the time slot having
;;; no value (unbound). this may not be the best solution.
;;;

(defgeneric score-reset (object &optional time))

(defmethod score-reset ((object t) &optional time)
  #-aclpc (declare (ignore time))
  object)

(defmethod score-reset ((object element) &optional time)
  #-aclpc (declare (ignore time))
  (slot-makunbound object 'time)
  object)

#|
;; original
(defmethod score-reset ((object startable-element) &optional time)
  ;; cache the current time as the object's time offset and
  ;; then initialize object's time its local start time or 0.0
  (setf (slot-value object 'offset) (or time 0.0))
  (setf (slot-value object 'time)
    (slot-value-or-default object 'start 0.0)))
|#

(defmethod score-reset ((object startable-element) &optional time)
  ;; cache the current time as the object's time offset and
  ;; then initialize object's time its local start time or 0.0
  (setf (slot-value object 'offset) (or time 0.0))
  (setf (slot-value object 'time)
    (slot-value-or-default object 'start 0.0))
  (when (slot-boundp object 'initializer)  
    (funcall (slot-value object 'initializer)))
;  (FORMAT T "~%score-reset (startable) ~S ~S: 
;	start=~S offset=~S time=~S~&" 
;         object time 
;         (slot-value-or-default object 'start)
;         (slot-value-or-default object 'offset)
;         (slot-value-or-default object 'time))
  )

(defmethod score-reset :after ((object thread) &optional time)
  #-aclpc (declare (ignore time))
  (reset-bag (slot-value object 'elements)))

(defmethod score-reset :before ((object heap) &optional time)
  #-aclpc (declare (ignore time))
  (unless (logtest (slot-value object 'flags) +frozen+)
    (shuffle-object object)))

(defmethod score-reset :after ((object algorithm) &optional time)
  #-aclpc (declare (ignore time))
  (setf (slot-value object 'count) 0)
  (setf (slot-value object 'status) 0)
  (setf (slot-value object 'last-status) 0)
  ;; we check for +frozen+ because object may be a generator
  ;; a normal algorithm should never be marked frozen. 
  (unless (logtest +frozen+ (slot-value object 'flags))
    (setf (slot-value object 'function)
      (funcall (slot-value object 'function-init)))
    
    (let ((inst (funcall (slot-value object 'instance-init)
                         (slot-value object 'instance))))
      (setf (slot-value object 'instance) inst)
      (setf (object-container inst) object)))
  object)

(defmethod score-reset :after ((object generator) &optional time)
  #-aclpc (declare (ignore time))
  (unless (logtest +frozen+ (slot-value object 'flags))
    (forget-bag (slot-value object 'elements))))

;; this method overrides the method on startable-element
#|
(defmethod score-reset ((object merge) &optional time)
  ;; top-level merge can have a start offset.
  (unless time (setf time (slot-value-or-default object 'start 0.0)))
  (setf (slot-value object 'offset) Time) ;(or time 0.0)
  (setf time 0.0)				; inferiors in local time	
  (setf (slot-value object 'time) time)	
  (let ((*merge* object)	; allow dynamic queuing during reset
        (queue (slot-value object 'elements)))
    (setf (bag-elements queue) nil)
    (setf (bag-tail queue) nil)
    (dolist (obj (bag-cache queue))
      (score-reset obj time)
      (bag-enqueue obj queue)))
   object)
|#

(defmethod score-reset ((object merge) &optional time)
  ;; top-level merge can have a start offset.
  (if (slot-boundp object 'start)
      (setf time (slot-value object 'start))
    (setf time (or time 0.0)))
  (setf (slot-value object 'offset) Time) 
  (setf time 0.0)				; inferiors in local time	
  (setf (slot-value object 'time) (slot-value object 'offset))	

;  (FORMAT T "~%score-reset (merge) ~S ~S: 
;  	start=~S offset=~S time=~S~&" 
;           object otime 
;           (slot-value-or-default object 'start)
;           (slot-value-or-default object 'offset)
;           (slot-value-or-default object 'time))

  (let ((*merge* object)	; allow dynamic queuing during reset
        (queue (slot-value object 'elements)))
    (setf (bag-elements queue) nil)
    (setf (bag-tail queue) nil)
    ;; a layout merge maintains the start times of its objects
    (when (slot-boundp object 'starts)   
      (dolist (s (slot-value object 'starts))
        (setf (slot-value (car s) 'start) (cdr s))
        (setf (slot-value (car s) 'flags) 
          (logior (slot-value (car s) 'flags) +start-unset+))))
    (dolist (obj (bag-cache queue))
      (score-reset obj time) ;(if (typep obj 'merge) nil time)
      (bag-enqueue obj queue)))
   object)

;;;
;;; score-select is the main function for getting the next score event
;;; to process. it is called during the score scheduling process on
;;; the score's merge, and recursively, on threads and basic elements. 
;;; score-select returns nil if there are no more elemenets to be selected. 
;;;

(defgeneric score-select (object time))

(defmethod score-select ((object t) time)
  (declare (ignore time))
  object)
  
;;;
;;; an around method checks to see if the candidate element is
;;; hidden.  if so, the element will not be selected.
;;;

(defmethod score-select :around ((object flags-mixin) time)
  #+clisp (declare (ignore time))
  (if (logtest (slot-value object 'flags) +hidden+)
      nil
    (call-next-method)))

;;;
;;; a before method for basic elements sets the time of the
;;; element to the current selection time.
;;;

(defmethod score-select :before ((object element) time)
  (setf (slot-value object 'time) time))
  
;;;
;;; score-select for the various containers involves recursion on its 
;;; elements. if the container is currently unset, then it is initialized
;;; before attempting to read any elements. during initialization, the current
;;; time is cached in the container's offset slot and its sub elements are
;;; then initialized to local time 0.0.  when an output element is
;;; subsequently read from the container this offset is added back into the
;;; element's time. this means that events inside a container need only be
;;; concerned with time local to the surrounding container, and not the
;;; container's temporal position in the composition.
;;;

(defmethod score-select ((object thread) time)
  (let ((elements (slot-value object 'elements))
        dequeued element)
    (unless (slot-boundp object 'time)
      (score-reset object time))
    (setf time (slot-value object 'time))    
    (loop while (bag-elements elements)
          until element
          do (setf dequeued (car (bag-elements elements)))
             (setf element (score-select dequeued time))
          if (or (null element) (eq dequeued element))
          do (pop-bag elements))   
    (if element
        (progn
          (incf (slot-value element 'time) (slot-value object 'offset))
          (score-increment object dequeued))
      (score-unset object))
    (setf (slot-value object 'selected) element)
    element))

(defmethod score-select ((object merge) time)
  (let ((*merge* object)	; allow dynamic queuing during selection
        (queue (slot-value object 'elements))
  	dequeued element)
    (unless (slot-boundp object 'time)
      (score-reset object time))
    (loop while (setf dequeued (bag-dequeue queue))
          do (setf time (slot-value dequeued 'time))
          until (setf element
	 	  (score-select dequeued time))
          do (score-increment object (car (bag-elements queue))))
    ;; need to add the local offset back into to the element's time
    (cond (element
   	   (incf (slot-value element 'time)
	     (slot-value object 'offset))
	   (unless (eq element dequeued)
	     (bag-enqueue dequeued queue))
	   (let ((next (car (bag-elements queue))))
	     (score-increment object next)))
   	  (t
	   (score-unset object)))
    (setf (slot-value object 'selected) element)
    element))

;;;
;;; enqueue-element is a wrapper that may be used during element processing
;;; to dynamically enqueue an element into a specified merge. if real-time
;;; is true the element's time is shifted to reflect merge's local time.
;;;

(defun enqueue-element (element merge real-time)
  (let ((bag (container-bag merge)))
    (when real-time
      (let ((offset (container-offset merge)))
        (when (> offset 0)
          (decf (object-time element) offset))))
    (bag-enqueue element bag)
    element))

;;;
;;; hack object for generating rest objects from algorithm rests...
;;;

(defvar *algorithm-rest* (make-instance 'rest))

;;;
;;; this method checks for an algorithm being "frozen" or not.  the check
;;; doesn't really belong at this class level do because freezing an algorithm
;;; make sense only in the case that it is a generator. however, by putting the 
;;; check here and assuming that a basic algorithm is never marked as frozen,
;;; we avoid having to implement really ugly dispatching code in the around
;;; method for generators, ie we skip having to find and call the appropriate
;;; main method (there are two of them for generators -- algorithm and thread) 
;;; based on the generator being frozen or not.
;;;

(defmethod score-select ((object algorithm) time)
  (if (logtest +frozen+ (slot-value object 'flags))
      (call-next-method)	; call the main method on thread
    (let (status)
      (unless (slot-boundp object 'time)
        (score-reset object time))
      (setf status (slot-value object 'status))
	  (setf time (slot-value object 'time))    
      (cond ((or (logtest status +killed+)
                 (> time (slot-value object 'end))
                 (not (< (slot-value object 'count)
                         (slot-value object 'length))))
             (setf (slot-value object 'status) (logior status +killed+))
    	     (score-unset object)
    	     nil)
    	    (t
    	     (let ((instance (slot-value object 'instance))
    	     	   element)
    	       (funcall (slot-value object 'function) object instance)
    	       (setf status (slot-value object 'status))
    	       (setf (slot-value object 'last-status) status)
    	       (setf element 
    	         (if (logtest +resting+ status)
                     (progn 
                       (setf (slot-value *algorithm-rest* 'rhythm)
                         (slot-value instance 'rhythm))
    		 		   *algorithm-rest*)
    		       instance))
    	       ;; this naked time setf might be dangerous, maybe i should
    	       ;; use score-select on the element and not add offset until
    	       ;; the container's time has been incremented...
               (setf (object-time element)
    	         (+ time (slot-value object 'offset)))
               (if (logtest +chording+ status)
                   (setf (element-flags element)
                     (logior +chording+ (element-flags element) ))
                   ;(setf (slot-value element 'rhythm) 0.0)
                 (progn
                   (setf (element-flags element)
                     (logandc2 (element-flags element) +chording+))
                   (when (logtest +ending+ status)
                     (setf status +killed+))
                   (score-increment object element)
                   (unless (= status +killed+)
                     ;(score-increment object element)
                     (incf (slot-value object 'count) 1))) )
               (setf (slot-value object 'status) 
                 (logandc2 status +normal-mask+))
               (setf (slot-value object 'selected) element)
               element))))))

;;;
;;; this around method on generator uses call-next-method to return the
;;; next object from either the main method on algorithm (if the generator
;;; is "unfrozen") or from the main method on thread (if the generator is
;;; "frozen". If the object came from the generator, it is copied and cached,
;;; otherwise it is simply returned.
;;;
 
(defmethod score-select :around ((object generator) time)
  #+clisp (declare (ignore time))
  (let ((thing (call-next-method))
        (flags (slot-value object 'flags)))
    (when (logtest (slot-value object 'status) +killed+) 
      (freeze-object object))
    (if thing
        (if (logtest +frozen+ flags)
            thing
          (let ((new (if (eq thing *algorithm-rest*) ; hack!
                         (copy-object thing)
                       (copy-object-using-class (slot-value object 'class) 
                                                 thing)))
                (eflags (element-flags thing)))
            (when (logtest +chording+ eflags)
              (setf (element-flags thing) (logandc2 eflags +normal-mask+)))
            (add-object new object)
            new))
      nil)))

;;; 
;;; score-increment is called to increment the current container's time after
;;; an element has been selected. time then reflects the "next run time".
;;;

(defgeneric score-increment (obj1 obj2))
(defmethod score-increment ((obj1 container) (obj2 t))
  nil)
#|
(defmethod score-increment ((obj1 startable-element) (obj2 rhythmic-element))
  (if (logtest +chording+ (slot-value obj2 'flags))
      (progn (write-line "chording") 0.0)
    (progn (write-line "not chording") (incf (slot-value obj1 'time) (slot-value obj2 'rhythm)))))
  
(defmethod score-increment ((obj1 thread) (obj2 startable-element))
  (setf (slot-value obj1 'time)
        (+ (slot-value obj2 'time) (slot-value obj2 'offset))))

(defmethod score-increment ((obj1 merge) (obj2 timed-object))
  (setf (slot-value obj1 'time) (slot-value obj2 'time)))
|#

;;;
;;; threads and algorithms are incremented by the rhythm of selected element,
;;; if the element is not a chord member.
;;;

(defmethod score-increment ((obj1 thread) (obj2 rhythmic-element))
  (unless (logtest +chording+ (slot-value obj2 'flags))
    (incf (slot-value obj1 'time) (slot-value obj2 'rhythm))))

(defmethod score-increment ((obj1 algorithm) (obj2 rhythmic-element))
  (unless (logtest +chording+ (slot-value obj2 'flags))
    (incf (slot-value obj1 'time) (slot-value obj2 'rhythm))))

;;;
;;; threads are set to the current time of objects that are not
;;; elements, ie threads, merges or algorithms
;;;

(defmethod score-increment ((obj1 thread) (obj2 startable-element))
  (setf (slot-value obj1 'time)
        (+ (slot-value obj2 'time) (slot-value obj2 'offset))))

;;; merge time already has the offset added in, so we use its current time

(defmethod score-increment ((obj1 thread) (obj2 merge))
  (setf (slot-value obj1 'time) (slot-value obj2 'time)))

;(defmethod score-increment ((obj1 thread) (obj2 container))
;  (setf (slot-value obj1 'time)
;        (+ (slot-value obj2 'time) (slot-value obj2 'offset))))

;;;
;;; since merges are schedulers, their time is always set to
;;; the current time of the selected (next) object.
;;;

#|
(defmethod score-increment ((obj1 merge) (obj2 timed-object))
  (setf (slot-value obj1 'time) (slot-value obj2 'time)))
|#

(defmethod score-increment ((obj1 merge) (obj2 timed-object))
   (setf (slot-value obj1 'time)
         (+ (slot-value obj1 'offset)
            (slot-value obj2 'time))))

;;;
;;; peek-elements returns at the next element in the container's
;;; processing queue.
;;;

(defmethod peek-element ((object container))
  (car (bag-elements (slot-value object 'elements))))

;;;
;;; default-merge destructively merges the default object's data with
;;; the object passed to produce the fully specified object.
;;;

;(defgeneric default-merge (default class object))
;
;(defmethod default-merge ((default default) class object)
;  (let ((i (slot-value default 'instance)))
;    (dolist (s (slot-value default 'slot-map))
;      (slot-makunbound-using-class class i (slot-definition-name s)))
;    (merge-instances class i object)))
;    
;(defmethod default-merge ((default default) class (object default))
;  (declare (ignore class))
;  object)
;
;(defmethod default-merge ((default default) class (object rest))
;  (declare (ignore class))
;  object)

