;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

;;;
;;; The RT syntax currently supports only a file based connection to
;;; Paul Lansky's rt.app driver, ie only Stella's Write and SWrite
;;; commands work.  It would be trivial to implement Listening given
;;; some way of sending messages or piping data directly to the driver,
;;; but file interaction may actually be faster so ill wait until 
;;; someone asks for it.  rt.driver is envoked via the the shell script
;;; unix/playrt located in the common music  directory. This script looks
;;; for rt.app in ~Apps and /LocalApps for rt.app/rt.driver, so if you
;;; have installed rt.app somewhere else be sure to edit this script.
;;;
;;; The following slots may be set in an rt stream via the Open command:
;;;   infiles - the list of files to mix. see the infiles function below
;;;   outfile - the output .snd file name, if any
;;;   from    - start time in mix, defaults to 0
;;;   to      - end time in mix, defaults to 9999 
;;;   tick    - rt's "timescale" parameter
;;;
;;; Use the PlayNote object to hold mixing info. PlayNote slots are
;;; are named after their playnote(...) equivalents, ie
;;; (setf gain .1) outputs as "gain=.1. The one exception is the
;;; at= time value, which receives the object's current time and is
;;; incremented automatically from rhythm statements. Values for rhythm,
;;; snd and track slots are required and an error signaled output if
;;; stella detects a playnote with any of these parameters missing
;;; values at output time. All other playnote slots are optional. Slots
;;; holding envelope data, ie. ampl or gliss data, may contain either
;;; strings already in rt format "x,y x,y", or as Lisp lists
;;; (x y x y ...) that will be automatically conveted at output time.
;;; You can set rt's "secret" overlap= and last= values, but these don't
;;; override the order in which stella normally schedules an object, or
;;; affect current time in any way.
;;;
;;; the file stella/examples/barnyard.stella contains a simple 
;;; example that writes a mix of 8 standard NeXT sounds.
;;;

(defsyntax rt 
   :pathname (pathname "test.rt")
   :stream-types '((rt-file "rt")))

(defmethod open-event-stream ((syntax rt) &rest args)
  (apply #'open-event-stream (find-class 'rt-file) args))

(defun infiles (string &rest others)
  ;; helper for input file specification. takes either a single string
  ;; with files names delimited by "," or a list of file names.
  ;; directories and extensions are sticky, ie: "~/foo, bar"
  ;; finds "/user/hkt/foo.snd" and "/user/hkt/bar.snd"
  (let (list file)
    (if others (setf list (cons string others))
      (setf list (sequence-to-list string :read nil)))
    (setf file (merge-pathnames (pop list) (pathname "~/test.snd")))
    (cons file
          (loop for f in list
                collect (setf f (merge-pathnames f file))
                do (setf file f)))))

;;;
;;; rt interaction is file based for now; use the Write or Swrite commands,
;;; not Listen or SListen. To implement rt "listening", we would inherit
;;; from event-stream and subclass to rt-file and rt-listener, and then
;;; define write-event methods for the listener that sends data directly
;;; to the rt.driver somehow (via NeXT app messages, piping, etc).
;;;

(defclass rt-file (event-file)
  ((infiles :initarg infiles :initarg :infiles :initform nil)
   (outfile :initarg outfile :initarg outfile)
   (tick :initarg tick :initarg :tick :initform .01) ; timescale already used.
   (from :initarg from :initarg :from :initform 0)
   (to :initarg to :initarg :to :initform 9999)
   (syntax :initform (find-syntax ':rt))))
  
(defmethod initialize-stream-for-processing ((stream rt-file))
  (let ((file (slot-value stream 'stream)))
    (format file "// Stella output from ~A~%" (cm::date-string))
    (dolist (f (slot-value stream 'infiles))
      (format file "infile=~A~%" (namestring f)))
    (format file "gain(1,1,1,1,1,1,1,1,1)~%~
                  timescale=~A~%~
                  tracks=8~%~
                  turntrackson(1,1,1,1,1,1,1,1,0)~%~
                  setlefttrackgain(1,1,1,1,1,1,1,1,0)~%~
                  setrighttrackgain(1,1,1,1,1,1,1,1,0)~%~
                  setoutputgain(1,1,0)~%" 
            (slot-value stream 'tick))
    (when (slot-boundp stream 'outfile)
      (format file "outfile=~A~%"
              (namestring (merge-pathnames (slot-value stream 'outfile)
                                           "test.snd"))))))

(defmethod deinitialize-stream-for-processing ((stream rt-file))
  (let ((file (slot-value stream 'stream)))
    (format file "mix(~S,~S)~%" 
            (slot-value stream 'from) (slot-value stream 'to))))

(defmethod play-using-syntax ((syntax rt) file &rest pairs)
  (declare (ignore pairs))
  (let ((command (format nil "~Aunix/playrt ~A"
                             *common-music-directory*
                             (namestring (truename file)))))
    (tell-user "Playing: ~A~%" command)
    (cm::shell command)
    file))

;;;
;;; The playnote object.
;;;

(defobject playnote (note)
  ((time :initarg at :initarg :at)
   (rhythm :initform 0.0)
   snd track gain transp skip dur end rev last overlap 
   amp gliss ampl ampr pan))

(defmethod print-object ((object playnote) stream)
  (printing-random-thing (object stream)
    (formatting-slots (object stream :default +slot-unset+)
                      rhythm track)))

(defun rt-envelope-printer (env &optional stream)
   ;; print an rt envelope. env can already be a string in "x,y,x,y"
   ;; format or else a lisp list (x y x y) to be translated.
   (unless (stringp env) 
     (setf env (apply #'concatenate 'string
       (loop while env for v = (pop env)
             collect (if (member v '(stickpoint :stickpoint)) 
                         "|" (prin1-to-string v)) 
             when env collect ","))))
   (format stream env))

(defmethod write-event ((object playnote) (stream rt-file))
  ;; output prefers last= or overlap= values over at=time.
  ;; this complicates things enough so that we dont try to use
  ;; formatting-slots on these values. 
  (let ((file (slot-value stream 'stream))) 
    (write-string "playnote(" file)
    (let ((last (careful-slot-value object 'last))
          (overlap (careful-slot-value object 'overlap)))
      (cond (last
             (when overlap 
               (error "LAST and OVERLAP are mutually exclusive."))
             (format file "last=~S," last))
            (overlap
             (when last
               (error "LAST and OVERLAP are mutually exclusive."))
             (format file "overlap=~S," last))
            (t (format file "at=~S," (slot-value object 'time)))))
    (formatting-slots (object file :postamble ")" :eol t 
                       :prefix (lambda (slot) (format nil "~(~A~)=" slot))
                       :delimiter #\, :print-if :value)
      (snd :print-if t) (track :print-if t) 
      gain transp skip dur end rev last overlap pan
      (amp   :prefix "amp("   :suffix ")" :printer rt-envelope-printer)
      (ampl  :prefix "ampl("  :suffix ")" :printer rt-envelope-printer)
      (ampr  :prefix "ampr("  :suffix ")" :printer rt-envelope-printer)
      (gliss :prefix "gliss(" :suffix ")" :printer rt-envelope-printer))
    ))

