;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92, 93 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; send to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

;;;
;;; parse-references should probably be rewritten now that i understand 
;;; referencing better. until then, only-one can be either NIL, T or :REFERENCE.
;;; if NIL then any number of references are ok. If T, then only a single object
;;; is legal, but its reference can be by name or index. if :REFERENCE, then only
;;; a single reference is legal, but this reference can be of any type.
;;; include-containers can be either NIL, T or :WITH-INDEX.  If nil then
;;; nothing special is done. if T, then we attempt to return individual objects
;;; as locations relative to a container. if :WITH-INDEX, the user must
;;; have actually specified that reference an index in some container.
;;;;

(defun parse-references (str &optional focus &key only-one any-level root 
                                                  (include-containers t))
  (let ((len (length str))
        (beg (refbeg str))
        (refs ())
        end tok ref rng subs)
    (when (and (< beg len) (char= (elt str beg) #\[))
      (incf beg)
      (loop while (> (decf len) beg)
            for c = (elt str len)
            until (char= c #\])
            while (or (char= c #\space) (char= c #\tab))))
    (loop while (< beg len)
          do
      (unless (setf end (refend str beg len))
      (return-from parse-references nil))
      (setf tok (subseq str beg end))
      ;; splitref destructures "foo[1,2]" into "foo" and "[1,2]". either
      ;; portion may be missing, but not both. if the second is present, we
      ;; recurse.
      (multiple-value-setq (ref rng) (splitref tok))
      (if rng
          (let (foc)
            (if ref
                (unless (setf foc (parseref ref focus t nil any-level))
                  (return-from parse-references nil))
            (setf foc focus)) 
            (if (setf subs (parse-references rng foc :only-one only-one
                                           :any-level any-level :root root))
                (setf refs (nconc refs subs))
              (return-from parse-references nil)))
        (if (setf subs (parseref ref focus (if (eq only-one :reference) NIL only-one) include-containers
                               any-level))
            (if (and refs (eq only-one :reference)) (return-from parse-references nil)
              (setf refs (nconc refs (list subs))))
          (return-from parse-references nil)))
      (setf beg (refbeg str end len))
      (when (and only-one (< beg len)) (return-from parse-references nil)))
    refs))

;;;
;;; parseref parses an external reference into its internal form. there are 4
;;; types of refrences: number, range, iteration and grouping.  number simply
;;; references a single object. range references all objects between a lower
;;; and upper bound.  iteration is like range, except that a "step" amount
;;; supplies an explicit increment value for the range. (a normal range is an
;;; iteration with a step amount = 1). a grouping is like an iteration execpt
;;; that it also has an explicit width associated with it. objects within
;;; the width window are processed "in parallel" as a group.
;;; the external notation for ranges are numeric fields delimited by ":", with
;;; the full form being "start:end:step:width". any or all of the full form
;;; may be expressed.  omitting a field value but including its delimiter
;;; is a short short hand whichs accepts the default value for the ommitted
;;; field. Defaults are  1:*:1:1. In addition, the wildcard symbols * and End
;;; have special significance for the first two fields, and are illegal in the
;;; step or width fields.  The token End references the last object, no matter
;;; what its numeric postion is. The token * can be used in the first field
;;; to reference ALL of the objects, and is exactly like typeing 1:*. If * is
;;; used in the second field, it references the last object, exactly like End.
;;; parseref returns one of the following internal forms of the reference:
;;;
;;; single without container:	object
;;; single with container:	(pos . container)
;;; range:			((lb . ub) . container)
;;; interation:			(((lb . ub) . step) . container)
;;; group:			((((lb . ub) . step) . width) . container)
;;;

(defun parseref (string container only-one? include-container? allow-any? )
  (let ((len (length string))
        (max 0)
        (beg 0)
        (field 0)
        (colons 0)
        (error (string= string ""))
        (subs ())
        end neg colon tok value lb ub step width foreign)
    (when (subobjectsp container)
      (setf subs (container-objects container)
	    max (length subs)))
    (loop while (< beg len)
          until error
          do
       (setf end beg)
       (loop while (< end len)
             until (setf colon (char= (elt string end) #\:))
             do (incf end))
       (when (> end beg)
         (setf tok (subseq string beg end))
         (cond ((or (string= tok "*")
                    (and (>= (- end beg) 3)
                         (string-equal tok "END" :end1 3)
                         (or (= (- end beg) 3)
                             (and (setf neg (parse-integer tok :start 3
                                                           :junk-allowed t))
                                  (> (+ neg max) 0)))))
                (cond ((= field 0)
                       (if (string= tok "*")
                           (progn (setf ub ':end) (incf field))
                         (setf lb (if neg (+ (1- neg) max) ':end))))
                      ((= field 1)
                       (setf ub (if neg (+ (1- neg) max) ':end)))
                      (t (setf error t)))                 ; not lb or ub
                (setf neg nil))
               ((integerp (setf value 
                            (multiple-value-bind (n i) 
                                (parse-integer tok :junk-allowed t)
                              (and n 
                                   (or (= i (length tok))
                                       (member (elt tok i) 
                                               '(#\space #\: #\, #\])))
                                   n))))
                (unless (<= 1 value max) 
                  (setf error t))                        ; bad num or no subs
                (cond ((= field 0) 
                       (setf lb (1- value))
                       (unless include-container?
                         (setf lb (nth lb subs))))
                      ((= field 1) 
                       (setf ub (1- value))
                       (when (and (integerp lb) (< ub lb))
                         (setf error t)))               ; ub>max or lb>ub
                      ((= field 2) (setf step value))
                      ((= field 3) (setf width value))))     
               ((and (= field 0) (not (eq include-container? :with-index))
                     (setf value (find-symbol (string-upcase tok))))
                (let ((obj (if (eq value '!) 
                               container
                             (find-object value nil)))
                      pos)
                  (when (eq include-container? :absolute-ok)
                    (setf include-container? nil))
                  (if obj 
                      (if (setf pos (position obj subs))
                          (setf lb 
                            (if include-container? pos obj))
                        (if allow-any?
                            (setf lb obj foreign t)     ; ok but not sub
                          (setf error t)))              ; not a sub
                     (setf error t))))                  ; not a name
               (t (setf error t))))                     ; not a num or name
       (setf beg (1+ end))
       (when colon
         (when (> (incf colons) 3) (setf error t)) 
         (setf colon nil)
         (unless (< beg len) (setf error t))            ; dangling ":"
         (if only-one?                                  ; too many refs
             (setf error t)          
           (incf field))))
    (if error
        nil
      (let (result)
        ;; fill in default values for fields specified in short form
        (unless lb (setf lb 0))
        (when (> field 1) (unless ub (setf ub ':end)))
        (when (> field 2) 
          (unless step (setf step 1))
          (when (eq ub ':end) (setf ub (1- max)))) ; group needs max
        (when foreign
          ; if absolute ref not in subs, try to return its container anyway.
          (let ((can (if (eq include-container? :absolute-ok)
                         nil (the-container lb))))
             (if can 
                 (progn (setf container can )
                        (when include-container?
                          (setf lb (object-position lb container))))
               (setf include-container? nil))))
        (setf result
          (cond ((= field 0) (if (eq lb ':end) (1- max) lb)) 
                ((= field 1) (cons lb ub))
                ((= field 2) (cons (cons lb ub) step))
                ((= field 3) (cons (cons (cons lb ub) step) width))))
        (if include-container?
            (cons result container)
           result)))))

;;;
;;;
;;;

(defun refend (str &optional (beg 0) end)
  (let ((len (or end (length str)))
	(lev 0)
	(pos beg)
	(flg nil))
    (loop until (or (= pos len) flg)
	  for chr = (elt str pos)
	  do
	  (cond ((char= chr #\[) 
	         (incf lev)
		 (incf pos))
	        ((char= chr #\])
		 (decf lev)
		 (incf pos)
		 (if (< lev 0) 
		     (setf pos len)
		   (if (= lev 0) (setf flg pos))))
		((or (char= chr #\,) (char= chr #\space) (char= chr #\tab))
		 (if (= lev 0)
		     (setf flg pos)
		   (incf pos)))
		(t (incf pos)))
	  finally (when (and (null flg) (= lev 0))
		    (setf flg pos)))
    (if flg (values flg (subseq str beg flg) (subseq str flg)))))

(defun refbeg (str &optional (beg 0) end)
  (unless end (setf end (length str)))
  (let ((pos beg))
    (loop while (< pos end)
          for c = (elt str pos)
	  unless (or (char= c #\,) (char= c #\space) (char= c #\tab))
 	  return pos
	  do (incf pos)
	  finally (return end))))

(defun splitref (str)	; str is syntactically correct ref
  (let (left right)
    (if (setf left (position #\[ str))
        (progn 
          (setf right (position #\] str :start left))
          (if (= left 0)
              (values nil str)
	    (values (subseq str 0 left) (subseq str left (1+ right)))))
      (values str nil))))      

;;;
;;; mapref maps a function over the element selection returned
;;; by ask-elements, which may be a single element, a range, or list. 
;;;

(defun mapref (fn sel &optional pass-container pass-group)
  (labels ((doone (fn x c) 
             (if pass-container (funcall fn x c) (funcall fn x)))
           (dorng (fn lb ub can)
             (if (eq ub ':end) 		; optimize for :end
                 (dolist (x (nthcdr lb (container-objects can)))
                   (doone fn x can))
               (loop for e in (nthcdr lb (container-objects can))
                     until (> lb ub)
                     do (doone fn e can)
                        (incf lb))))
           (doitr (fn lb ub by can)
             (let ((list (nthcdr lb (container-objects can))))
               (if (eq ub ':end)               ; optimize for :end
                   (loop while list
                         do (doone fn (car list) can)
                            (setf list (nthcdr by list)))
                 (loop while list
                       until (> lb ub)
                       do (doone fn (car list) can)
                          (setf list (nthcdr by list))
                          (incf lb by)))))
           (dogrp (fn lb ub by width can)
             (let ((list (nthcdr lb (container-objects can))))
               (loop while list
                     until (> (+ lb (1- width)) ub)
                     do (if pass-group
                            (loop for i below width 
                                  collect (nth i list) into grp
                                  finally (doone fn grp can))
                          (loop for i below width
                                do (doone fn (nth i list) can)))
                        (incf lb by)
                        (setf list (nthcdr by list))))))
    (cond ((null sel) nil)
          ((consp sel)
           (if (listp (cdr sel)) 
               (progn (mapref fn (car sel) pass-container pass-group)
                      (mapref fn (cdr sel) pass-container pass-group))
             (if (listp (car sel))
                 (if (listp (caar sel))
                     (if (listp (caaar sel))
                         (dogrp fn (caaaar sel) (cdaaar sel) (cdaar sel) 
                                (cdar sel) (cdr sel))
                       (doitr fn (caaar sel) (cdaar sel) (cdar sel) (cdr sel)))
                   (dorng fn (caar sel) (cdar sel) (cdr sel)))
               (let ((x (if (eq (car sel) ':end)       ; optimize for :end
                            (car (last (container-objects (cdr sel))))
                          (nth (car sel) (container-objects (cdr sel))))))
                 (doone fn x (cdr sel))))))
          (t (doone fn sel nil)))))

;;;
;;; form of mapref that passes positions instead of objects.
;;;

(defun mappos (fn sel &optional (single-mode ':error))
  (labels ((doone (fn x c) (funcall fn x c))
           (doitr (fn lb ub by can)
             (when (eq ub ':end) 		; optimize for :end
               (setf ub (1- (object-count can))))
             (loop until (> lb ub)
                   do (doone fn lb can)
                      (incf lb by)))
           (dogrp (fn lb ub by width can)
             (let ((list (nthcdr lb (container-objects can))))
               (loop while list
                     until (> (+ lb (1- width)) ub)
                     do (loop for i below width
                                do (doone fn i can))
                        (incf lb by)
                        (setf list (nthcdr by list))))))
    (cond ((null sel) nil)
          ((consp sel)
           (if (listp (cdr sel)) 
               (progn (mappos fn (car sel) single-mode)
                      (mappos fn (cdr sel)) single-mode)
             (if (listp (car sel))
                 (if (listp (caar sel))
                     (if (listp (caaar sel))
                         (dogrp fn (caaaar sel) (cdaaar sel) (cdaar sel) 
                                (cdar sel) (cdr sel))
                       (doitr fn (caaar sel) (cdaar sel) (cdar sel) (cdr sel)))
                   (doitr fn (caar sel) (cdar sel) 1 (cdr sel)))
               (let ((x (if (eq (car sel) ':end)       ; optimize for :end
                            (1- (object-count (cdr sel)))
                          (car sel))))
                 (doone fn x (cdr sel))))))
          ((eq single-mode ':error)
           (error "Found non-positional reference"))
          (single-mode (doone fn nil nil))
          (t nil))))
;;;
;;; utils
;;;

(defun reflist (ref)
  (let ((list '()))
    (mapref #'(lambda (o) (push o list)) ref)
    (nreverse list)))

(defun refobject (ref)
  (if (objectref? ref)
      ref
    (if (indexref? ref)
        (nth-object (car ref) (cdr ref))
      (and (sequenceref? ref) (not (cdr ref))
           (refobject (car ref))))))

(defun refcontainer (ref)
  (and (consp ref)
       (not (listp (cdr ref)))
       (cdr ref)))

(defun reflb (ref)
  (cond ((objectref? ref) nil)
        ((indexref? ref) (car ref))
        ((rangeref? ref) (caar ref))
        ((iterationref? ref) (caaar ref))
        ((groupref? ref) (caaaar ref))
        (t nil)))

(defun refub (ref)
  (cond ((objectref? ref) nil)
        ((indexref? ref) nil)
        ((rangeref? ref) (cdar ref))
        ((iterationref? ref) (cdaar ref))
        ((groupref? ref) (cdaaar ref))
        (t nil)))

(defun objectref? (ref)
  (not (consp ref)))

(defun indexref? (ref)
  (and (consp ref)
       (not (listp (cdr ref)))
       (not (consp (car ref)))))

(defun rangeref? (ref)
  (and (consp ref)
       (not (listp (cdr ref)))
       (consp (car ref))
       (not (consp (caar ref)))))
  
(defun iterationref? (ref)
  (and (consp ref)
       (not (listp (cdr ref)))
       (consp (car ref))
       (consp (caar ref))
       (not (consp (caaar ref)))))
  
(defun groupref? (ref)
  (and (consp ref) 
       (not (listp (cdr ref)))
       (consp (car ref))
       (consp (caar ref))
       (consp (caaar ref))))

(defun singleref? (ref)
  (or (objectref? ref)
      (indexref? ref)
      (and (sequenceref? ref) 
           (not (cdr ref))
           (singleref? (car ref)))))

(defun sequenceref? (ref)      ; list of one or more
  (and (consp ref) (listp (cdr ref))))

(defun multipleref? (ref)      ; list of more than one
  (and (consp ref) (consp (cdr ref))))

(defun makeref (obj &optional lb ub ic wi)
  (if (not lb) obj
    (progn (when ub (setf lb (cons lb ub)))
           (when ic (setf lb (cons lb ic)))
           (when wi (setf lb (cons lb wi)))
           (cons lb obj))))
  
(defun destructureref (ref)
  (if (not (consp ref))
      ref
    (let ((pos (car ref))
          (can (cdr ref)) lb ub it gr)
      (if (not (consp pos))                    ; index
          (setf lb pos)
        (if (not (consp (car pos)))            ; range
            (setf lb (car pos) ub (cdr pos))
          (if (not (consp (caar pos)))         ; iter
              (setf lb (caar pos) ub (cdar pos) it (cdr pos))
            (setf lb (caaar pos) ub (cdaar pos) it (cdar pos) 
                  gr (cdr pos)))))
      (values can lb ub it gr))))

