;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(defun make-record-buffer (&optional (size 64))
  (let ((array (make-array size :adjustable t :fill-pointer t)))
    (setf (fill-pointer array) 0)
    array))

(defvar *record-buffer* (make-record-buffer))

(defun scratch-record (message time)
  (declare (fixnum message time) (optimize (speed 3) (safety 0)))
  (midi-write-message message)
  (vector-push-extend (cons message time) *record-buffer*))

(defun record-buffer-length (buffer)
  (length buffer))

(defun map-record-buffer (function buffer &optional (start 0) end)
  (unless end (setf end (length buffer)))
  (loop for i from start below end
  	do (let ((e (aref buffer i)))
	      (funcall function (car e) (cdr e)))))

(defun map-record-buffer2 (function buffer &optional (start 0) end)
  (let ((len (length buffer))
        (p1 start)
        p2 e1 e2 n1 n2 t1 t2)
    (unless end (setf end len))
    (loop while (< p1 end)
          do (setf e1 (aref buffer p1) n1 (car e1) t1 (cdr e1))
	  until (and (note-on-p n1)(> (note-on-velocity n1) 0))
	  do (incf p1))    
    (loop while (< p1 end)
          do
      (let ((chan (note-on-channel n1))
            (key (note-on-key n1)))
         (setf p2 (1+ p1))	    
	 (loop while (< p2 len)
	       do (setf e2 (aref buffer p2) n2 (car e2) t2 (cdr e2))
	       until (and (or (note-off-p n2)
	                      (= (note-on-velocity n2) 0))
			  (= (note-on-key n2) key)
  	                  (= (note-on-channel n2) chan))
               do (incf p2)))			  
      (if n2
          (funcall function n1 t1 n2 t2)
	(error "Missing note off for: ~S ~S" n1 t1))
      (incf p1)
      (loop while (< p1 end)
            do (setf e1 (aref buffer p1) n1 (car e1) t1 (cdr e1))
	    until (and (note-on-p n1)(> (note-on-velocity n1) 0))
	    do (incf p1)))))


(defun record-buffer-to-thread (buffer thread class rhythm-slot duration-slot
                                channel-slot keynum-slot keynum-type
                                amplitude-slot amplitude-type)
  (let ((list ())(this nil)(last nil) (t0 0))
    (map-record-buffer2
      #'(lambda (on t1 off t2)
          (declare (ignore off))
          (setf this (make-instance class))
          (when rhythm-slot
            (when last 
	      (setf (slot-value-using-class class last rhythm-slot)
	            (- (real-time t1) (real-time t0)))))
          (when duration-slot
	    (setf (slot-value-using-class class this duration-slot)
              (real-time (- t2 t1))))
	  (when channel-slot
	    (setf (slot-value-using-class class this channel-slot) 
	      (note-on-channel on)))
	  (when keynum-slot
	    (setf (slot-value-using-class class this keynum-slot)
	      (if (eq keynum-type 'note)
	          (scale-note (note-on-key on) *standard-scale*)
                (if (eq keynum-type 'pitch)
		    (scale-pitch (note-on-key on) *standard-scale*)
		  (note-on-key on)))))		  
	  (when amplitude-slot
	    (setf (slot-value-using-class class this amplitude-slot)
	      (if (eq amplitude-type 'amplitude)
	          (/ (note-on-velocity on) 127.0)
                (note-on-velocity on))))
          (push this list)
	  (setf last this t0 t1))
      buffer)
    (if (and last rhythm-slot)
      (setf (slot-value-using-class class last rhythm-slot) 
        (if duration-slot
            (slot-value-using-class class last duration-slot)
          0.0)))
    (add-objects (nreverse list) thread nil :copy-first nil)
    thread))










