;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92, 93 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; send to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(shadow '(cm::part cm::part-name))

(defsyntax music-kit 
   :pathname (pathname "test.score")
   :nicknames '(musicKit mk)
   :stream-types '((music-kit-event-file "score" )))

(defmethod open-event-stream ((syntax music-kit) &rest args)
  (apply #'open-event-stream (find-class 'music-kit-event-file) args))

;;;
;;; music kit score files
;;;

(defclass music-kit-event-file (event-file header-mixin)
  ((syntax :initform (find-syntax ':music-kit))
   (part-file :initarg :part-file)
   (file-time)))

(defmethod initialize-instance :after ((object music-kit-event-file)
                                       &rest args)
   (declare (ignore args))
   (unless (slot-boundp object 'part-file)
     (setf (slot-value object 'part-file)
        (let ((path (slot-value object 'pathname)))
	  (make-pathname :defaults path :type "parts")))))

(defmethod initialize-stream-for-processing ((stream music-kit-event-file))
  (setf (slot-value stream 'file-time) -1)
  (let ((file (slot-value stream 'stream)))
    (format file "/* Stella output: ~A */~%" (cm::date-string))
    (when (slot-boundp stream 'header)
      (format file "~A~%"  (slot-value stream 'header)))
    (format file "include \"~A\";~%BEGIN;~%" 
            (namestring (slot-value stream 'part-file)))))

(defun print-part (part file)
  (let ((name (part-name part)))
    (format file "part ~(~A~);~%~(~A~)~{ ~A~};~%"
            name name (slot-value part 'info))))

(defmethod deinitialize-stream-for-processing ((stream music-kit-event-file))
  (write-line "END;" (slot-value stream 'stream))
  (with-open-file (file (slot-value stream 'part-file)
                   :direction ':output :if-exists ':supersede)
    (format file "/* Included by \"~A\" */~%"
            (namestring (slot-value stream 'pathname)))
    (map nil #'(lambda (p)
                 (when (slot-value p 'include)
                   (print-part p file)
                   (setf (slot-value p 'include) nil)))
         *parts*)))

;;;
;;; playing music kit score files.
;;;

(defparameter *play-in-background* nil)
(defparameter *.score-player* nil)

(defmethod play-using-syntax ((syntax music-kit) file &rest pairs)
  (let ((background *play-in-background*)
        (player (or *.score-player*
                    (setf *.score-player*
                      (concatenate 'string
                                   (namestring *common-music-directory*)
                                   "unix/playscore"))))
        (args nil)
        command)
    (dopairs (n v pairs)
      (case n
        ((player :player) (setf player v))
        (args (setf args v))
        ((background :background) (setf background v))))
    (setf command
      (format nil "~a~@[ -~a~] ~a~@[ &~]"
              player args (namestring (truename file)) background))
    (tell-user "Playing: ~A~%" command)
    (cm::shell command)
    file))

;;;
;;; importing music kit score files.
;;;

(defmethod import-using-syntax ((syntax music-kit) file &rest pairs)
  (setf file (namestring (truename file)))
  (let ((program (concatenate 'string *common-music-directory*
                              "stella/mk2stella"))
        (archive (make-pathname :type "stella" :defaults file)))
    (unless program (error "Can't find conversion program: ~S." program))
    (dopairs (n v pairs)
      (case n
        ((archive :archive) (setf archive (namestring v)))))
    (setf archive (namestring archive))
    (tell-user "Translating ~A to ~A and loading.~%" file archive)
    (cm::shell (format nil "~A ~A ~A~%" program file archive))
    (load-file archive)))


;;;
;;; music kit "parts". these hold information relevant to the Music Kit. each
;;; instance of music-kit-note has a pointer to some part object which
;;; holds, at minimum, the print name of the note and its synthPatch.
;;;

(defparameter *parts* () "Music Kit scorefile part information.")
		       
(defclass part (flags-mixin)
  ((name :initarg :name :initarg name :accessor part-name)
   (include :initform nil :accessor part-include)
   (info :initarg :info :initarg info)))

(defmethod initialize-instance :after ((obj part) &rest args)
  (declare (ignore args))
  (unless (listp (slot-value obj 'info))
    (setf (slot-value obj 'info) (list (slot-value obj 'info))))
  (unless (find obj *parts*)
    (setf *parts* (nconc *parts* (list obj)))))

(defmethod print-object ((obj part) stream)
  (printing-random-thing (obj stream)
    (format stream "part ~(~A~)" (slot-value obj 'name))))

(defun make-part (&rest args &key name &allow-other-keys)
  (let ((i (find-part name)))
     (if i 
         (if args
             (apply #'initialize-instance i args)
           i)
       (apply #'make-instance 'part args))))

(defun partInfo (name &rest args)
  (make-part :name name
             :info (loop for (msg val) on args by #'cddr
                         collect (format nil "~A:~S" msg val))))

(defmethod archive-form ((object part))
  (if (logtest (slot-value object 'flags) +system+)
      `(find-part ',(slot-value object 'name))
    `(make-part :name ,(slot-value object 'name)
                :info ',(slot-value object 'info))))

(defun find-part (name &optional (parts *parts*))
  (find name parts :key #'part-name))

;;;
;;; create default system parts.  Music Kit notes that are created
;;; without a specified part will point to the system part 
;;; appropriate for the class of note.
;;;

(eval-when (load eval)
  (make-part :name 'pluck :flags +system+ 
             :info  '("synthPatch:\"Pluck\""))
  (make-part :name 'dbfm1vi :flags +system+ 
             :info '("synthPatch:\"DBFm1vi\""))
  (make-part :name 'dbwave1vi :flags +system+  
             :info  '("synthPatch:\"DBWave1vi\""))
  (make-part :name 'dbwave2vi :flags +system+
             :info  '("synthPatch:\"DBWave2vi\""))
  (make-part :name 'fm1i :flags +system+
             :info  '("synthPatch:\"Fm1i\""))
  (make-part :name 'fm1vi :flags +system+ 
             :info  '("synthPatch:\"Fm1i\""))
  (make-part :name 'fm2cnvi :flags +system+
             :info  '("synthPatch:\"Fm2cnvi\""))
  (make-part :name 'fm2cvi :flags +system+
             :info  '("synthPatch:\"Fm2cvi\""))
  (make-part :name 'fm2pnvi :flags +system+
             :info  '("synthPatch:\"Fm2pnvi\""))
  (make-part :name 'fm2pvi :flags +system+
             :info  '("synthPatch:\"Fm2pvi\""))
  (make-part :name 'mkmidi :flags +system+
             :info  '("synthPatch:\"midi\""))
  (make-part :name 'mixsounds :flags +system+
             :info  '("synthPatch:\"sounds\""))
  (make-part :name 'wave1i :flags +system+  
             :info '("synthPatch:\"Wave1i\""))
  (make-part :name 'wave1vi :flags +system+ 
             :info '("synthPatch:\"Wave1vi\""))
  
 )

;;;
;;; music kit notes
;;;

(defclass music-kit-note (note)
  ((tag :initarg :tag :initarg tag)
   (type :initarg :type :initarg type)
   (duration :initarg :duration :initarg duration)
   (part :initarg :part :initarg part)))

(defmethod print-object ((note music-kit-note) stream)
  (printing-random-thing (note stream)
    (let ((rhythm (slot-value-or-default note 'rhythm +slot-unset+)))
      (format stream "~(~A~) ~A ~A " 
              (if (slot-boundp note 'part)
                  (part-name (slot-value note 'part))
                (class-name (class-of note)))
              rhythm
             (let ((type (slot-value-or-default note 'type ':duration)))
               (if (eq type ':duration)
                   (slot-value-or-default note 'duration rhythm)
                 type))))
    (when (slot-boundp note 'tag)
      (format stream " tag:~A" (slot-value note 'tag)))
    (mk-print-parameters note stream)))
      
(defmethod initialize-instance :after ((obj music-kit-note) &rest args)
  (declare (ignore args))
  (unless (slot-boundp obj 'part)
    (setf (slot-value obj 'part)
      (find-part (class-name (class-of obj))))))

;;;
;;; mk-make-prefix and *mk-parameter-info* must exist at
;;; compile time in order to macroexpand the mk-print-parameters
;;; method bodies.
;;;

(eval-when (compile load eval)
(progn

(defun mk-make-prefix (slot)
  (let ((e (assoc slot *mk-parameter-info*)))
    (unless e (error "No parameter entry for ~S." slot))
    (second e)))

;;; missing realtime entries AMPENVTIMESCALE, MIDICHAN
(defparameter *mk-parameter-info*
  '((aftertouch "afterTouch:" 2 mklisp_addintparam)
    (aftertouchsensitivity "afterTouchSensitivity:" 20 mklisp_addintparam)
    (amp "amp:" 30 mklisp_addfloatparam)
    (amp0 "amp0:" 82 mklisp_addfloatparam)
    (amp1 "amp1:" 30 mklisp_addfloatparam)
    (ampatt "ampAtt:" 83 mklisp_addfloatparam)
    (ampenv "ampEnv:" 81 mklisp_addstringparam)
    (ampEnvTimeScale "ampEnvTimeScale:" 0 error)
    (amprel "ampRel:" 84 mklisp_addfloatparam)
    (balancesensitivity "balanceSensitivity:" 25 mklisp_addintparam)
    (basicchan "basicChan:" 13 mklisp_addintparam)
    (bearing "bearing:" 31 mklisp_addfloatparam)
    (breathsensitivity "breathSensitivity:" 22 mklisp_addintparam)
    (bright "bright:" 32 mklisp_addfloatparam)
    (chanmode "chanMode:" 11 mklisp_addintparam)
    (controlchange "controlChange:" 3 mklisp_addintparam)
    (controlval "controlVal:" 14 mklisp_addintparam)
    (cratio "cRatio:" 37 mklisp_addfloatparam)
    (decay "decay:" 64 mklisp_addfloatparam)
    (freq "freq:" 29 mklisp_addfloatparam)
    (freq0 "freq0:" 78 mklisp_addfloatparam)
    (freq1 "freq1:" 29 mklisp_addfloatparam)
    (freqatt "freqAtt:" 79 mklisp_addfloatparam)
    (freqenv "freqEnv:" 77 mklisp_addstringparam)
    (freqrel "freqRel:" 80 mklisp_addfloatparam)
    (keynum "keyNum:" 18 mklisp_addintparam)
    (keypressure "keyPressure:" 1 mklisp_addintparam)
    (lowestfreq "lowestFreq:" 227 mklisp_addfloatparam)
    (m1ind "m1Ind:" 47 mklisp_addfloatparam)
    (m1ind0 "m1Ind0:" 104 mklisp_addfloatparam)
    (m1ind1 "m1Ind1:" 47 mklisp_addfloatparam)
    (m1indatt "m1IndAtt:" 105 mklisp_addfloatparam)
    (m1indenv "m1IndEnv:" 103 mklisp_addstringparam)
    (m1indrel "m1IndRel:" 106 mklisp_addfloatparam)
    (m1phase "m1Phase:" 49 mklisp_addfloatparam)
    (m1ratio "m1Ratio:" 46 mklisp_addfloatparam)
    (m1waveform "m1Waveform:" 48 mklisp_addstringparam)
    (m2ind "m2Ind:" 51 mklisp_addfloatparam)
    (m2ind0 "m2Ind0:" 108 mklisp_addfloatparam)
    (m2ind1 "m2Ind1:" 51 mklisp_addfloatparam)
    (m2indatt "m2IndAtt:" 109 mklisp_addfloatparam)
    (m2indenv "m2IndEnv:" 107 mklisp_addstringparam)
    (m2indrel "m2IndRel:" 110 mklisp_addfloatparam)
    (m2phase "m2Phase:" 53 mklisp_addfloatparam)
    (m2ratio "m2Ratio:" 50 mklisp_addfloatparam)
    (m2waveform "m2Waveform:" 52 mklisp_addstringparam)
    (midiChan "midiChan:" 0 error)
    (monochans "monoChans:" 15 mklisp_addintparam)
    (noiseamp "noiseAmp:" 75 mklisp_addfloatparam)
    (noiseamp0 "noiseAmp0:" 136 mklisp_addfloatparam)
    (noiseamp1 "noiseAmp1:" 75 mklisp_addfloatparam)
    (noiseampatt "noiseAmpAtt:" 137 mklisp_addfloatparam)
    (noiseampenv "noiseAmpEnv:" 135 mklisp_addstringparam)
    (noiseamprel "noiseAmpRel:" 138 mklisp_addfloatparam)
    (pansensitivity "panSensitivity:" 26 mklisp_addintparam)
    (phase "phase:" 36 mklisp_addfloatparam)
    (picknoise "pickNoise:" 228 mklisp_addfloatparam)
    (pitchbend "pitchBend:" 4 mklisp_addintparam)
    (pitchbendsensitivity "pitchBendSensitivity:" 28 mklisp_addintparam)
    (portamento "portamento:" 33 mklisp_addfloatparam)
    (programchange "programChange:" 5 mklisp_addintparam)
    (relvelocity "relVelocity:" 17 mklisp_addintparam)
    (rvibamp "rvibAmp:" 70 mklisp_addfloatparam)
    (songposition "songPosition:" 7 mklisp_addintparam)
    (songselect "songSelect:" 8 mklisp_addintparam)
    (soundfile "soundFile:" 0 error)
    (sustain "sustain:" 68 mklisp_addfloatparam)
    (svibamp "svibAmp:" 68 mklisp_addfloatparam)
    (svibamp0 "svibAmp0:" 128 mklisp_addfloatparam)
    (svibamp1 "svibAmp1:" 68 mklisp_addfloatparam)
    (svibfreq "svibFreq:" 67 mklisp_addfloatparam)
    (svibfreq0 "svibFreq0:" 120 mklisp_addfloatparam)
    (svibfreq1 "svibFreq1:" 67 mklisp_addfloatparam)
    (sysexclusive "sysExclusive:" 10 mklisp_addstringparam)
    (sysrealtime "sysRealTime:" 12 mklisp_addintparam)
    (timecodeq "timeCodeQ:" 6 mklisp_addintparam)
    (timeOffset "timeOffset:" 0 error)
    (tunerequest "tuneRequest:" 9 mklisp_addintparam)
    (velocity "velocity:" 16 mklisp_addintparam)
    (velocitysensitivity "velocitySensitivity:" 19 mklisp_addintparam)
    (waveform "waveform:" 34 mklisp_addstringparam)
    (waveform0 "waveform0:" 92 mklisp_addstringparam)
    (waveform1 "waveform1:" 34 mklisp_addstringparam)
    (wavelen "waveLen:" 35 mklisp_addintparam)
    (waveformatt "waveformAtt:" 93 mklisp_addfloatparam)
    (waveformenv "waveformEnv:" 91 mklisp_addstringparam)
    (waveformrel "waveformRel:" 94 mklisp_addfloatparam)))
))

;;;
;;; this around method is invoked for all types of Music Kit notes.
;;; it handles note name,type,and tag printing and updates the
;;; scorefile's time if appropriate.  the main write-event methods
;;; are defined for each terminal music-kit-note class and simply
;;; handle parameter printing.
;;;

(defmethod write-event ((event music-kit-note)
                        (stream music-kit-event-file))
  (let ((file (slot-value stream 'stream))
        (part (if (slot-boundp event 'part) (slot-value event 'part)
                (find-part (class-name (class-of event)))))
        (last (slot-value stream 'file-time))
        (time (slot-value event 'time))
        (timescale (slot-value-or-default stream 'timescale))
        (*print-case* ':downcase))
    ;; print time
    (when (> time last)
      (write-string "t " file)
      (princ (if timescale (* time timescale) time) file)
      (write-char #\; file)
      (terpri file)
      (setf (slot-value stream 'file-time) time))
    ;; part name
    (princ (part-name part) file)
    ;; mark part as included in the current output file
    (setf (part-include part) t)
    (write-char #\space file)
    (write-char #\( file)
    ;; type and tag
    (let ((type (slot-value-or-default event 'type ':duration)))
      (ecase type
        (:duration
	  (princ (if timescale 
                     (* (slot-value-or-default event 'duration 
                                            (slot-value event 'rhythm))
                        timescale)
                   (slot-value-or-default event 'duration 
                                          (slot-value event 'rhythm)))
                 file))
        (:noteUpdate
         (if (slot-boundp event 'tag)
             (progn (write-string "noteUpdate," file)
                    (princ (slot-value event 'tag) file))
           (write-string "noteUpdate" file)))
        (:noteOn
         (write-string "noteOn," file)
         (princ (slot-value event 'tag) file))
        (:noteOff
         (write-string "noteOff," file)
         (princ (slot-value event 'tag) file))	
        (:mute
         (write-string "mute" file)))
      (write-string ") " file)
      (mk-print-parameters event file)
      (write-char #\; file)
      (terpri file))))   

#+MIDI
(defmethod write-event ((object midi-note)
                        (stream music-kit-event-file))
  (let ((file (slot-value stream 'stream))
        (part (elt *parts* (slot-value object 'channel)))
        (last (slot-value stream 'file-time))
        (time (slot-value object 'time))
        (timescale (slot-value-or-default stream 'timescale))
        (note (slot-value object 'note)))
    (when (> time last)
      (format file "t ~S;~%" (if timescale (* time timescale) time))
      (setf (slot-value stream 'file-time) time))
   (setf (part-include part) t)
   (format file "~(~A~) (~S) ~:[freq~;keynum~]:~(~A~) amp:~S;~%"
           (part-name part) 
           (if timescale (* (slot-value object 'duration) timescale)
             (slot-value object 'duration))
           (integerp note) note (slot-value object 'amplitude))))

#+CMN
(defmethod write-event ((object music-kit-note) (stream cmn-stream))
  (let ((staff (funcall (slot-value stream 'staffer) stream object))
        (timescale (slot-value-or-default stream 'timescale)))
    (unless staff
      (error "~S didn't return a staff when called with ~S and ~S."
            (slot-value stream 'staffer) stream object))        
    (cmn::add-note-to-staff stream staff
                            (if timescale 
                                (* (slot-value object 'time) timescale)
                              (slot-value object 'time))
                            (let ((dur (if (slot-boundp object 'duration)
                                           (slot-value object 'duration)
                                         (slot-value object 'rhythm))))
                              (if timescale (* dur timescale) dur))
                            (cmn-eval (if (slot-boundp object 'keynum)
                                          (scale-note (slot-value object 
                                                                  'keynum)
                                                    *standard-scale*)
                                        (slot-value object 'freq))))))

;;;
;;; parameter mixins
;;;

(defclass standard-music-kit-parameters ()
  ((freq :initarg freq :initarg :freq)
   (keyNum :initarg keyNum :initarg :keyNum)
   (pitchBend :initarg pitchBend :initarg :pitchBend)
   (pitchBendSensitivity :initarg pitchBendSensitivity 
			 :initarg :pitchBendSensitivity)
   (bearing :initarg bearing :initarg :bearing)
   (amp :initarg amp :initarg :amp)
   (velocity :initarg velocity)
   (velocitySensitivity :initarg velocitySensitivity 
			:initarg :velocitySensitivity)
   (controlChange :initarg controlChange :initarg :controlChange)
   (controlVal :initarg controlVal :initarg :controlVal)))

(defclass standard-oscillator-parameters ()
  ((ampEnv :initarg ampEnv :initarg :ampEnv)
   (amp0 :initarg amp0 :initarg :amp0)
   (amp1 :initarg amp1 :initarg :amp1)
   (ampAtt :initarg ampAtt :initarg :ampAtt)
   (ampRel :initarg ampRel :initarg :ampRel)
   (freqEnv :initarg freqEnv :initarg :freqEnv)
   (freq0 :initarg freq0 :initarg :freq0)
   (freq1 :initarg freq1 :initarg :freq1)
   (freqAtt :initarg freqAtt :initarg :freqAtt)
   (freqRel :initarg freqRel :initarg :freqRel)
   (portamento :initarg portamento :initarg :portamento)
   (waveform :initarg waveform :initarg :waveform)
   (waveLen :initarg wavelen :initarg :waveLen)
   (phase :initarg phase :initarg :phase)))

(defclass waveform0-parameters ()
  ((waveform0 :initarg waveform0 :initarg :waveform0)
   (waveform1 :initarg waveform1 :initarg :waveform1)))

(defclass waveform-envelope-parameters (waveform0-parameters)
  ((waveformAtt :initarg waveformAtt :initarg :waveformAtt)
   (waveformRel :initarg waveformRel :initarg :waveformRel)
   (waveformEnv :initarg waveformEnv :initarg :waveformEnv)))

(defclass vibrato-parameters ()
  ((sVibFreq :initarg svibfreq :initarg :svibFreq)
   (sVibAmp :initarg svibamp :initarg :svibAmp)
   (sVibFreq0 :initarg svibfreq0 :initarg :svibFreq0)
   (sVibFreq1 :initarg svibfreq1 :initarg :svibFreq1)
   (sVibAmp0 :initarg svibamp0 :initarg :svibAmp0)
   (sVibAmp1 :initarg svibamp1 :initarg :svibAmp1)
   (rVibAmp :initarg rvibamp :initarg :rvibAmp)))

(defclass noise-modulation-parameters ()
  ((noiseAmp :initarg noiseamp :initarg :noiseAmp)
   (noiseAmpEnv :initarg noiseampenv :initarg :noiseAmpEnv)
   (noiseAmp0 :initarg noiseamp0 :initarg :noiseAmp0)
   (noiseAmp1 :initarg noiseamp1 :initarg :noiseAmp1)
   (noiseAmpAtt :initarg noiseampatt :initarg :noiseAmpAtt)
   (noiseAmpRel :initarg noiseamprel :initarg :noiseAmpRel)
   (breathSensitivity :initarg breathSensitivity 
                      :initarg :breathSensitivity)))

(defclass midi-panning-parameters ()
  ((panSensitivity :initarg panSensitivity :initarg :panSensitivity)))

(defclass midi-balance-parameters ()
  ((balanceSensitivity :initarg balanceSensitivity 
                       :initarg :balanceSensitivity)))

(defclass fm-parameters ()
  ((cRatio :initarg cratio :initarg :cRatio)
   (m1Ratio :initarg m1ratio :initarg :m1Ratio)
   (m1Ind :initarg m1ind :initarg :m1Ind)
   (m1IndEnv :initarg m1indenv :initarg :m1IndEnv)
   (m1Ind0 :initarg m1ind0 :initarg :m1Ind0)
   (m1Ind1 :initarg m1ind1 :initarg :m1Ind1)
   (m1IndAtt :initarg m1indatt :initarg :m1IndAtt)
   (m1IndRel :initarg m1indrel :initarg :m1IndRel)
   (m1Waveform :initarg m1waveform :initarg :m1Waveform)
   (m1Phase :initarg m1phase :initarg :m1Phase)
   (bright :initarg bright :initarg :bright)
   (afterTouch :initarg afterTouch :initarg :afterTouch)
   (afterTouchSensitivity :initarg afterTouchSensitivity 
                          :initarg :afterTouchSensitivity)))

(defclass double-fm-parameters (fm-parameters)
  ((m2Ratio :initarg m2ratio :initarg :m2Ratio)
   (m2Ind :initarg m2ind :initarg :m2Ind)
   (m2IndEnv :initarg m2indenv :initarg :m2IndEnv)
   (m2Ind0 :initarg m2ind0 :initarg :m2Ind0)
   (m2Ind1 :initarg m2ind1 :initarg :m2Ind1)
   (m2IndAtt :initarg m2indatt :initarg :m2IndAtt)
   (m2IndRel :initarg m2indrel :initarg :m2IndRel)
   (m2Waveform :initarg m2waveform :initarg :m2Waveform)
   (m2Phase :initarg m2phase :initarg :m2Phase)))

(defclass midi-parameters ()
  ((midiChan :initarg midiChan :initarg nil :reader music-kit-midi-chan)
   (keyPressure :initarg keyPressure :initarg :keyPressure)
   (afterTouch :initarg afterTouch :initarg :afterTouch)
   (controlChange :initarg controlChange :initarg :controlChange)
   (pitchBend :initarg pitchBend :initarg :pitchBend)
   (programChange :initarg programChange :initarg :programChange)
   (timeCodeQ :initarg timeCodeQ :initarg :timeCodeQ)
   (songPosition :initarg songPosition :initarg :songPosition)
   (songSelect :initarg songSelect :initarg :songSelect)
   (tuneRequest :initarg tuneRequest :initarg :tuneRequest)
   (sysExclusive :initarg sysExclusive :initarg :sysExclusive)
   (chanMode :initarg chanMode :initarg :chanMode)
   (sysRealTime :initarg sysRealTime :initarg :sysRealTime)
   (basicChan :initarg basicChan :initarg :basicChan)
   (controlVal :initarg controlVal :initarg :controlVal)
   (monoChans :initarg monoChans :initarg :monoChans)
   (velocity :initarg velocity :initarg :velocity)
   (relVelocity :initarg relVelocity :initarg :relVelocity)
   (keyNum :initarg keyNum :initarg :keyNum)
   (freq :initarg freq :initarg :freq)))

(defclass pluck-parameters (standard-music-kit-parameters)
  ((lowestFreq :initarg lowestfreq :initarg :lowestFreq)
   (sustain :initarg sustain :initarg :sustain)
   (pickNoise :initarg picknoise :initarg :pickNoise)
   (decay :initarg decay :initarg :decay)   
   (bright :initarg bright :initarg :bright)
   (ampRel :initarg amprel :initarg :ampRel)))

;;;
;;; the Music Kit Note classes...
;;;

(defobject DBFm1vi (music-kit-note standard-music-kit-parameters 		   
		                   standard-oscillator-parameters waveform0-parameters
		                   midi-balance-parameters fm-parameters vibrato-parameters)
  ()
  (:parameters 
   freq keyNum pitchBend pitchBendSensitivity bearing amp velocity 
   velocitySensitivity controlChange controlVal ampEnv amp0 amp1 ampatt
   amprel freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen
   phase waveform0 waveform1 balancesensitivity cratio m1ratio m1ind
   m1indenv m1ind0 m1ind1 m1indatt m1indrel m1waveform m1phase bright 
   aftertouch aftertouchsensitivity svibfreq svibamp svibfreq0 svibfreq1
   svibamp0 svibamp1 rvibamp))

(defmethod mk-print-parameters ((obj DBFm1vi) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
     freq keyNum pitchBend pitchBendSensitivity bearing amp velocity 
     velocitySensitivity controlChange controlVal ampEnv amp0 amp1 ampatt
     amprel freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen
     phase waveform0 waveform1 balancesensitivity cratio m1ratio m1ind
     m1indenv m1ind0 m1ind1 m1indatt m1indrel m1waveform m1phase bright 
     aftertouch aftertouchsensitivity svibfreq svibamp svibfreq0 svibfreq1
     svibamp0 svibamp1 rvibamp))

;;;
;;;
;;;
    
(defobject DBWave1vi (music-kit-note standard-music-kit-parameters 
                                     standard-oscillator-parameters vibrato-parameters
                                     midi-balance-parameters waveform0-parameters )
  ()
  (:parameters 
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
   velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt
   amprel freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen 
   phase svibfreq svibamp svibfreq0 svibfreq1 svibamp0 svibamp1 rvibamp
   balancesensitivity waveform0 waveform1))

(defmethod mk-print-parameters ((obj DBWave1vi) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt
    amprel freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen 
    phase svibfreq svibamp svibfreq0 svibfreq1 svibamp0 svibamp1 rvibamp
    balancesensitivity waveform0 waveform1))

;;;
;;;
;;;
    
(defobject DBWave2vi (music-kit-note standard-music-kit-parameters 
		                     standard-oscillator-parameters vibrato-parameters
		                     midi-panning-parameters waveform-envelope-parameters )
  ()
  (:parameters
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
   velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
   freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
   svibfreq svibamp svibfreq0 svibfreq1 svibamp0 svibamp1 rvibamp 
   pansensitivity waveform0 waveform1 waveformatt waveformrel waveformenv ))

(defmethod mk-print-parameters ((obj dbwave2vi) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
    freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
    svibfreq svibamp svibfreq0 svibfreq1 svibamp0 svibamp1 rvibamp 
    pansensitivity waveform0 waveform1 waveformatt waveformrel waveformenv))

;;;
;;;
;;;
    
(defobject Fm1i (music-kit-note standard-music-kit-parameters 
                                standard-oscillator-parameters fm-parameters )
  ()
  (:parameters 
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
   velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
   freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
   cratio m1ratio m1ind m1indenv m1ind0 m1ind1 m1indatt m1indrel m1waveform 
   m1phase bright aftertouch aftertouchsensitivity))

(defmethod mk-print-parameters ((obj fm1i) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
    freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
    cratio m1ratio m1ind m1indenv m1ind0 m1ind1 m1indatt m1indrel m1waveform 
    m1phase bright aftertouch aftertouchsensitivity))

;;;
;;;
;;;
    
(defobject Fm1vi (music-kit-note standard-music-kit-parameters 
                                 standard-oscillator-parameters fm-parameters
                                 vibrato-parameters )
  ()
  (:parameters 
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
   velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
   freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
   cratio m1ratio m1ind m1indenv m1ind0 m1ind1 m1indatt m1indrel m1waveform 
   m1phase bright aftertouch aftertouchsensitivity svibfreq svibamp svibfreq0 
   svibfreq1 svibamp0 svibamp1 rvibamp))

(defmethod mk-print-parameters ((obj fm1vi) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
    freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
    cratio m1ratio m1ind m1indenv m1ind0 m1ind1 m1indatt m1indrel m1waveform 
    m1phase bright aftertouch aftertouchsensitivity svibfreq svibamp svibfreq0 
    svibfreq1 svibamp0 svibamp1 rvibamp))

;;;
;;;
;;;
    
(defobject Fm2cnvi (music-kit-note standard-music-kit-parameters
                                   standard-oscillator-parameters noise-modulation-parameters
                                   midi-panning-parameters double-fm-parameters 
                                   vibrato-parameters )
  ()
  (:parameters 
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
   velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
   freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
   noiseamp noiseampenv noiseamp0 noiseamp1 noiseampatt noiseamprel 
   breathsensitivity pansensitivity cratio m1ratio m1ind m1indenv m1ind0 
   m1ind1 m1indatt m1indrel m1waveform m1phase bright aftertouch 
   aftertouchsensitivity m2ratio m2ind m2indenv m2ind0 m2ind1 m2indatt 
   m2indrel m2waveform m2phase svibfreq svibamp svibfreq0 svibfreq1 svibamp0 
   svibamp1 rvibamp))

(defmethod mk-print-parameters ((obj fm2cnvi) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
    freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
    noiseamp noiseampenv noiseamp0 noiseamp1 noiseampatt noiseamprel 
    breathsensitivity pansensitivity cratio m1ratio m1ind m1indenv m1ind0 
    m1ind1 m1indatt m1indrel m1waveform m1phase bright aftertouch 
    aftertouchsensitivity m2ratio m2ind m2indenv m2ind0 m2ind1 m2indatt 
    m2indrel m2waveform m2phase svibfreq svibamp svibfreq0 svibfreq1 svibamp0 
    svibamp1 rvibamp))

;;;
;;;
;;;
    
(defobject Fm2cvi (music-kit-note standard-music-kit-parameters
                                  standard-oscillator-parameters double-fm-parameters
                                  vibrato-parameters )
  ()
  (:parameters 
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
   velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
   freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
   cratio m1ratio m1ind m1indenv m1ind0 m1ind1 m1indatt m1indrel m1waveform 
   m1phase bright aftertouch aftertouchsensitivity m2ratio m2ind m2indenv 
   m2ind0 m2ind1 m2indatt m2indrel m2waveform m2phase svibfreq svibamp 
   svibfreq0 svibfreq1 svibamp0 svibamp1 rvibamp))

(defmethod mk-print-parameters ((obj fm2cvi) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
    freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
    cratio m1ratio m1ind m1indenv m1ind0 m1ind1 m1indatt m1indrel m1waveform 
    m1phase bright aftertouch aftertouchsensitivity m2ratio m2ind m2indenv 
    m2ind0 m2ind1 m2indatt m2indrel m2waveform m2phase svibfreq svibamp 
    svibfreq0 svibfreq1 svibamp0 svibamp1 rvibamp))

;;;
;;;
;;;
    
(defobject Fm2pnvi (music-kit-note standard-music-kit-parameters
                                   standard-oscillator-parameters noise-modulation-parameters
                                   midi-panning-parameters midi-balance-parameters
                                   double-fm-parameters vibrato-parameters )
  ()
  (:parameters 
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
   velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
   freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
   noiseamp noiseampenv noiseamp0 noiseamp1 noiseampatt noiseamprel 
   breathsensitivity pansensitivity balancesensitivity cratio m1ratio m1ind 
   m1indenv m1ind0 m1ind1 m1indatt m1indrel m1waveform m1phase bright 
   aftertouch aftertouchsensitivity m2ratio m2ind m2indenv m2ind0 m2ind1 
   m2indatt m2indrel m2waveform m2phase svibfreq svibamp svibfreq0 svibfreq1 
   svibamp0 svibamp1 rvibamp))

(defmethod mk-print-parameters ((obj fm2pnvi) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
    freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
    noiseamp noiseampenv noiseamp0 noiseamp1 noiseampatt noiseamprel 
    breathsensitivity pansensitivity balancesensitivity cratio m1ratio m1ind 
    m1indenv m1ind0 m1ind1 m1indatt m1indrel m1waveform m1phase bright 
    aftertouch aftertouchsensitivity m2ratio m2ind m2indenv m2ind0 m2ind1 
    m2indatt m2indrel m2waveform m2phase svibfreq svibamp svibfreq0 svibfreq1 
    svibamp0 svibamp1 rvibamp))

;;;
;;;
;;;
    
(defobject Fm2pvi (music-kit-note standard-music-kit-parameters 
                                  standard-oscillator-parameters midi-panning-parameters 
                                  midi-balance-parameters double-fm-parameters 
                                  vibrato-parameters )
  
  ()
  (:parameters 
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
   velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel
   freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
   pansensitivity balancesensitivity cratio m1ratio m1ind m1indenv m1ind0 
   m1ind1 m1indatt m1indrel m1waveform m1phase bright aftertouch 
   aftertouchsensitivity m2ratio m2ind m2indenv m2ind0 m2ind1 m2indatt 
   m2indrel m2waveform m2phase svibfreq svibamp svibfreq0 svibfreq1 svibamp0 
   svibamp1 rvibamp))

(defmethod mk-print-parameters ((obj fm2pvi) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel
    freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
    pansensitivity balancesensitivity cratio m1ratio m1ind m1indenv m1ind0 
    m1ind1 m1indatt m1indrel m1waveform m1phase bright aftertouch 
    aftertouchsensitivity m2ratio m2ind m2indenv m2ind0 m2ind1 m2indatt 
    m2indrel m2waveform m2phase svibfreq svibamp svibfreq0 svibfreq1 svibamp0 
    svibamp1 rvibamp))

;;;
;;;
;;;
    
(defobject mkMidi (music-kit-note midi-parameters)
  ()
  (:parameters 
   keypressure aftertouch controlchange pitchbend programchange timecodeq 
   songposition songselect tunerequest sysexclusive chanmode sysrealtime
   basicchan controlval monochans velocity relvelocity keynum freq))

(defmethod mk-print-parameters ((obj mkmidi) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    keypressure aftertouch controlchange pitchbend programchange timecodeq 
    songposition songselect tunerequest sysexclusive chanmode sysrealtime
    basicchan controlval monochans velocity relvelocity keynum freq))

;;;
;;;
;;;

(defobject Mixsounds (music-kit-note)
  ((amp :initarg amp :initarg :amp)
   (ampEnv :initarg ampEnv :initarg :ampEnv)
   (timeOffset :initarg timeOffset :initarg :timeOffset)
   (soundFile :initarg soundFile :initarg :soundFile)
   (bearing :initarg bearing :initarg :bearing)
   (freq0 :initarg freq0 :initarg :freq0)
   (freq1 :initarg freq1 :initarg :freq1)
   (ampEnvTimeScale :initarg ampEnvTimeScale :initarg :ampEnvTimeScale))
  (:parameters 
   amp timeoffset soundfile bearing freq0 freq1 ampenvtimescale ampenv))

(defmethod mk-print-parameters ((obj mixsounds) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    amp timeoffset soundfile bearing freq0 freq1 ampenvtimescale ampenv))
;;;
;;;
;;;

(defobject Pluck (music-kit-note pluck-parameters)
  ()
  (:parameters 
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
   velocitysensitivity controlchange controlval lowestfreq sustain picknoise
   decay bright amprel))

(defmethod mk-print-parameters ((obj pluck) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval lowestfreq sustain picknoise
    decay bright amprel))

;;;
;;;
;;;
    
(defobject Wave1i (music-kit-note standard-music-kit-parameters 
                                  standard-oscillator-parameters )
  ()
  (:parameters
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
   velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
   freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase))

(defmethod mk-print-parameters ((obj wave1i) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
    freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase))

;;;
;;;
;;;
    
(defobject Wave1vi (music-kit-note standard-music-kit-parameters
		   standard-oscillator-parameters vibrato-parameters)
  ()
  (:parameters 
   freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
    freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase))

(defmethod mk-print-parameters ((obj wave1vi) stream)
  (declare (optimize (speed 3) (safety 0)))
  (formatting-slots (obj stream :print-if :bound
                                :prefix mk-make-prefix)
    freq keynum pitchbend pitchbendsensitivity bearing amp velocity 
    velocitysensitivity controlchange controlval ampenv amp0 amp1 ampatt amprel 
    freqenv freq0 freq1 freqatt freqrel portamento waveform wavelen phase 
    svibfreq svibamp svibfreq0 svibfreq1 svibamp0 svibamp1 rvibamp))
