;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(defsyntax midi 
  :pathname (pathname cm:*default-midi-pathname*)
  :stream-types `((midi-file ,(pathname-type cm:*default-midi-pathname*))))

;;;
;;; midi streams. both midifile and a real time streams are supported
;;; 

(defclass midi-stream (event-stream)
  ((syntax :initform (find-syntax ':midi))))

(defobject midi-listener (midi-stream)
  ((direction :initform ':io)
   (port :initarg :port :initarg port))
  (:parameters start end timescale port ))

(defmethod print-object ((object midi-listener) stream)
  (format stream "#<Midi Listener (port: ~A)>"
          (slot-value-or-default object 'stream +slot-unset+)))

(defmethod open-event-stream ((stream midi-listener) &rest args)
  (if (getf args 'open t)
    (let ((open (cm:midi-open-p))
          port)
      (if (slot-boundp stream 'port)
        (unless (cm:midi-port-reference-p
                 (setf port (slot-value stream 'port)))
	  (error "~S is not a valid midi port reference." port))
        (setf port
              (and *command-prompting*
                   (ask-port :prompt "Open MIDI on port: " 
                             :type 'midi :check t))))
      (if (or (null port) (eq port ':aborted))
        (progn
          (unless port (error "Can't open MIDI: No port specifed."))
          nil)
        (progn
          (when (and open (not (eq open port)))
            (cm:midi-close)
            (setf open nil))
          (setf (slot-value stream 'port) port)
          (setf (slot-value stream 'stream) port)
          (unless open (cm:midi-open :port port))
          stream)))
    stream))

(defmethod initialize-stream-for-processing ((stream midi-listener))
  (cm:midi-set-time 0)
  t)

(defmethod close-event-stream ((stream midi-listener) &optional mode)
 (when (eq mode ':force)
   (slot-makunbound stream 'stream)
   (and (cm:midi-close) t)))

(defobject midi-file (midi-stream event-file)
  ((last-output-time :initform 0.0)
   (tempo :initform 120.0 :initarg :tempo :initarg tempo)
   (divisions-per-quarter :initform 96 :initarg divisions-per-quarter
   	                  :initarg :divisions-per-quarter)
   (key-signature :initform nil :initarg key-signature :initarg :key-signature)
   (time-signature :initform '(4 4) :initarg time-signature 
                   :initarg :time-signature)
   (time-scale :initform 192.0)
   (tracks :initform 1 :initarg tracks :initarg :tracks)
   (format :initform 0 :initarg format :initarg :format)
   (length :initform 0)
   (element-type :initform '(unsigned-byte 8)))
  (:parameters start end timescale key-signature time-signature))

(defmethod initialize-stream-for-processing ((stream midi-file))
  (setf (slot-value stream 'last-output-time) 0.0)
  (setf (slot-value stream 'length ) 0)
  (setf (slot-value stream 'time-scale)
	(/ (slot-value stream 'divisions-per-quarter)
	   (/ 60.0 (slot-value stream 'tempo))))
  (unless (= (slot-value stream 'format) 0)
      (warn "Ignoring ~s value for format. Only level 0 currently implemented."
          (slot-value stream 'format)))
  (unless (= (slot-value stream 'tracks) 1)
    (warn "Ignoring ~s value for tracks. Only 1 track currently implemented."
          (slot-value stream 'tracks)))
  ;; write midi header
  (let ((file (slot-value stream 'stream)))
    (cm:write-midi-file-header file (slot-value stream 'format)
			       (slot-value stream 'tracks)
			       (slot-value stream 'divisions-per-quarter))
    (cm::write-track-header file 0)
    ;; write tempo and time signature
    (multiple-value-bind (msg data)
        (cm:make-tempo-change (floor 60000000.0 (slot-value stream 'tempo)))
      (incf (slot-value stream 'length)
        (cm:write-message msg file 0 data)))
    (multiple-value-bind (msg data)
        (apply #'cm:make-time-signature (slot-value stream 'time-signature))
      (incf (slot-value stream 'length)
        (cm:write-message msg file 0 data)))
    t))

(defmethod deinitialize-stream-for-processing ((stream midi-file))
  (let ((file (slot-value stream 'stream)))
    ;; write end of track
    (incf (slot-value stream 'length) 
      (cm:write-message (cm:make-eot) file 0 nil))
    ;; update track length in header
    (file-position file 14)
    (cm::write-track-header file (slot-value stream 'length))
    #+mcl (ccl:set-mac-file-type file "Midi")
    t))

;;;
;;; midi listening/writing
;;;

(defmethod open-listener ((syntax midi) &rest args)
  (let ((stream (slot-value-or-default syntax 'listener)))
    (when (setf stream
            (apply #'open-event-stream (or stream 'midi-listener) args))
      (setf (slot-value syntax 'listener) stream))
    stream))

(defmethod close-listener ((object midi-listener) &optional (mode :force))
  (close-event-stream object mode))

(defmethod open-event-stream ((syntax midi) &rest args)
  (apply #'open-event-stream (find-class 'midi-file) args))

#+(or mcl aclpc (and next excl) (and |NeXT| akcl))
(defmethod play-using-syntax ((syntax midi) file &rest args)
  (unless (cm:midi-open-p) 
    (cm:midi-open :port (ask-port)))
  (apply #'cm:midifile-play file args))

#-(or mcl aclpc (and next excl) (and |NeXT| akcl))
(defmethod play-using-syntax ((syntax midi) file &rest args)
  (declare (ignore file args))
  (format t "~%Sorry, can't play midi files in this port of Common Music because there is no connection to a midi driver. Either use an external application, or else fix play-using-syntax to call an external program via a shell script, or implement the connection yourself.~%"))

(defmethod import-using-syntax ((syntax midi) file &rest args)
  (apply #'midifile-import file args))

(defun midifile-import (pathname &rest pairs)
  (let ((class (find-class 'thread))
        (events '())
        object channels name)
    (dopairs (n v pairs)
      (case n
        ((channels :channels) (setf channels v))
        ((object :object)
         (cond ((symbolp v) 
                (setf class (find-class v)))
               ((typep object 'standard-classs)
                (setf class object object nil))
               (t
                (setf object v))))
        ((name :name) (setf name v))))
    (midifile-parse pathname
      #'(lambda (chan beg rhy dur key amp)
          (declare (ignore beg))
          (push (make-instance 'midi-note :channel chan
                               :rhythm rhy :duration dur
                               :note (scale-note key *standard-scale*) 
                               :amplitude (/ amp 127.0))
                events))
      :channels channels)
    (if object
        (remove-all-objects object)
     (setf object
       (if name (make-instance class :id name)
         (make-instance class))))
    (add-objects (nreverse events) object)
    object))

;;;
;;; midi event classes
;;;

(defclass essential-midi (timed-object)
  ((message :initarg :message :initarg message :accessor midi-message)))

(defclass channel-mixin ()
  ((channel :initform 0 :initarg :channel :initarg channel
            :accessor midi-channel)))

(defclass midi-event (essential-midi rhythmic-element)
  ((rhythm :initarg rhythm :initarg :rhythm :accessor midi-rhythm)))

;;;
;;; the midi-release class implements "ephemeral" note-off events
;;; and are kept in a resource to avoid consing as much as possible.
;;;

(defclass midi-release (essential-midi)
  ((holder :accessor midi-release-holder)))

(defvar *midi-release-class* (find-class 'midi-release))

(defun make-release (resource)
  (declare (ignore resource))
  (allocate-instance *midi-release-class*))

(utils:defresource releases () 
  :size 64
  :initializer nil
  :deinitializer nil
  :reinitializer nil
  :matcher nil
  :finder #'utils:basic-resource-finder
  :constructor #'make-release)

(eval-when (load eval)
  (utils:initialize-resource 'releases 32 ))

(defvar *midi-release-resource* (get 'releases :resource))

(defmethod write-event :after ((object midi-release) (stream t))
  ;; score errors should release all the held ephemereal messages. i don't
  ;; think i currently do this.
  (let ((holder (slot-value object 'holder)))
    (utils:fast-deallocate-resource *midi-release-resource* holder)))

;;;
;;; if a stream has a start or end associated with it, write-event checks
;;; and scales each object before it is output. we must override the
;;; normal methods for midi-release.
;;;

(defmethod write-event? ((object midi-release) (stream event-stream))
  ;; midi-releases should always be output, even if > stream's end time.
  t)

(defmethod shift-time ((object midi-release) (stream event-stream))
  ;; we dont left shift time. i don't know why this is the case - i cant 
  ;; understand my scheduling code anymore!
  nil)

;;;
;;;
;;;

(defobject midi-message (midi-event)
  ((message :initform 0 :initarg message :initarg :message)
   (data :initform nil :initarg data :initarg :data))
  (:parameters rhythm message))

;(channel :initform 0 :initarg channel :initarg :channel 
;         :accessor midi-channel)

(defobject midi-note (midi-event channel-mixin)
  ((note :initarg note :initarg :note :accessor midi-note)
   (amplitude :initarg amplitude :initarg :amplitude :accessor midi-amplitude)
   (duration :initarg :duration :initarg duration :accessor midi-duration))
  (:parameters note rhythm duration amplitude channel))

(defmethod print-object ((object midi-note) stream)
  (let ((note (slot-value-or-default object 'note "-unset-"))
        (rhy (slot-value-or-default object 'rhythm "-unset-"))
        (dur (slot-value-or-default object 'duration "-unset-"))
        (amp (slot-value-or-default object 'amplitude "-unset-"))
        (chan (slot-value-or-default object 'channel "0")))
    ; format was "MIDI ~4:@A  ~6,3F  ~6,3F  ~6,3F  ~2@A " 
    ; compacted for default unix terminal display, hkt.
    (printing-random-thing (object stream)
      (format stream "MIDI ~4:@A ~6,3F ~6,3F ~6,3F ~2@A"
              note rhy dur amp chan))))

(defmethod setable-slots ((object midi))
  '(rhythm message data))

(defmethod setable-slots ((object midi-note))
  '(rhythm channel note duration amplitude))

;;;
;;;
;;;

(defmethod archive-form :before ((object midi-note))
  (slot-makunbound object 'message))						  

(defmethod write-event :before ((element essential-midi) (stream midi-file))
  (let ((time (slot-value element 'time))
	(last (slot-value stream 'last-output-time)))
    (if (> time last)
        (let ((value (round (if (slot-boundp stream 'timescale)
                                (* (- time last)
	                           (slot-value stream 'time-scale)
                                   (slot-value stream 'timescale))
                              (* (- time last)
                                 (slot-value stream 'time-scale))))))
          (incf (slot-value stream 'length)
            (cm::write-variable-quantity value (slot-value stream 'stream)))
          (setf (slot-value stream 'last-output-time) time))
      (progn
        (write-byte 0 (slot-value stream 'stream))
        (incf (slot-value stream 'length) 1)))
    element))
	   
(defmethod write-event :before ((element midi-note) (stream midi-stream))
  (let ((channel (slot-value element 'channel))
        (key (cm:scale-degree (slot-value element 'note)
                              cm:*standard-scale*))
        (velocity (floor (* (slot-value element 'amplitude) 127))))				 
    ;; compute midi noteOn for output
    (setf (slot-value element 'message) 
      (cm:make-note-on channel key velocity))
    ;; schedule midi noteOff with current data.
    (multiple-value-bind (release holder) 
	    (utils:basic-resource-finder *midi-release-resource*)
      (setf (midi-release-holder release) holder)    
      (setf (midi-message release) (cm:make-note-off channel key 127))
      (setf (object-time release)
        (+ (slot-value element 'time) (slot-value element 'duration)))
      (enqueue-element release *merge* t))
    element))

;;;
;;; main methods on write-event send messages to file or mididriver
;;;

(defmethod write-event ((element essential-midi) (stream midi-listener))
  (cm:midi-write-message (slot-value element 'message) 
                         (quanta-time (if (slot-boundp stream 'timescale)
                                          (* (slot-value stream 'timescale)
                                             (slot-value element 'time))
                                        (slot-value element 'time))))
  element)

(defmethod write-event ((element essential-midi) (stream midi-file))
  (let ((message (slot-value element 'message)))
    (incf (slot-value stream 'length)
      (cm:write-message message (slot-value stream 'stream) nil
			(if (cm::midi-channel-message-p message)
		            nil
			  (slot-value element 'data))))
    element))

(defmethod write-event ((element essential-midi) (stream midi-listener))
  (cm:midi-write-message (slot-value element 'message) 
                         (quanta-time (if (slot-boundp stream 'timescale)
                                          (* (slot-value stream 'timescale)
                                             (slot-value element 'time))
                                        (slot-value element 'time))))
  element)

