;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

;;;
;;; @, @+1 and % are provided for use in mapping clauses.  @ and @+1 return
;;; the position (zero and one based, respectively) of an object, or nil if 
;;; the object has less or more than one container. when an object has more
;;; than one container, the relevant container to return the position in is
;;; specified as the second argument. % returns the position of an object as
;;; a percentage. if the object has no position, 0.0 is returned.  ? 
;;; references slot values. if the slot exists and is bound in the object.
;;; the value is returned, otherwise, ? returns a default value or nil.
;;;

(defun @ (object &optional container)
  (unless container (setf container (the-container object)))
  (if (and container
           (setf object   
             (position object 
                       (bag-cache (slot-value container 'elements)))))
      (values object container)        ; object var holds position now
   (values nil nil)))

(defun @+1 (object &optional container)
  (multiple-value-bind (pos can) (@ object container)
    (if pos (values (1+ pos) can)
      (values nil nil))))

(defun % (object &optional container)
  (unless container (setf container (the-container object)))
  (if container
      (let ((cache (bag-cache (slot-value container 'elements))))
        (if cache 
            (* 100.0 (/ (or (position object cache) 0) (1- (length cache))))
          0.0))
    0.0))

(defun ? (object slot &optional value)
  (if (and (slot-exists-p object slot)
           (slot-boundp object slot))
      (slot-value object slot)
    (if (eq value ':no-value)
        (throw :no-value nil)
      value)))

(defun quantize (value step)
  (* (round value step) step))

(defun quantize! (value step)
  (expt step (round (log value step))))

(defun decimals (value decimals)
  (/ (round value (expt 10.0 (- decimals)))
     (expt 10.0 decimals)))  

(cm::defprop :set :map-syntax '(:repeat (:slot :expr)))
(cm::defprop :unset :map-syntax '(:repeat (:slot)))
(cm::defprop :scale :map-syntax '(:repeat (:slot :expr)))
(cm::defprop :increment :map-syntax '(:repeat (:slot :expr)))
(cm::defprop :transpose :map-syntax '(:repeat (:slot :expr)))
(cm::defprop :invert :map-syntax '(:repeat (:slot :expr)))
(cm::defprop :quantize :map-syntax '(:repeat (:slot :expr)))
(cm::defprop :quantize! :map-syntax '(:repeat (:slot :expr)))
(cm::defprop :round :map-syntax '(:repeat (:slot :expr)))

(cm::defprop :change :map-syntax '(:class :optional-slots))

(cm::defprop :collect :map-syntax '(:expr :into))
(cm::defprop :minimize :map-syntax '(:expr :into))
(cm::defprop :maximize :map-syntax '(:expr :into))
(cm::defprop :highest :map-syntax '(:expr :into))
(cm::defprop :lowest :map-syntax '(:expr :into))
(cm::defprop :sum :map-syntax '(:expr :into))
(cm::defprop :count :map-syntax '(:expr :into))
(cm::defprop :average :map-syntax '(:expr :into))
(cm::defprop :analyze :map-syntax '(:slot))

(cm::defprop :find :map-syntax '(:expr))
(cm::defprop :list :map-syntax '(:expr))
(cm::defprop :show :map-syntax '(:expr))
(cm::defprop :do :map-syntax '(:expr))
(cm::defprop :print :map-syntax '(:expr))
(cm::defprop :insert :map-syntax '(:expr))
(cm::defprop :append :map-syntax '(:expr))
(cm::defprop :insert-at :map-syntax '(:repeat (:expr :expr)))
(cm::defprop :append-at :map-syntax '(:repeat (:expr :expr)))
(cm::defprop :delete :map-syntax '(:expr))
(cm::defprop :undelete :map-syntax '(:expr))
(cm::defprop :hide :map-syntax '(:expr))
(cm::defprop :unhide :map-syntax '(:expr))

(cm::defprop :when :map-syntax '(:expr :command))
(cm::defprop :unless :map-syntax '(:expr :command))
(cm::defprop :while :map-syntax '(:expr :command))
(cm::defprop :until :map-syntax '(:expr :command))

(defvar $)		    ; current object (doesn't need to be global)
(defvar $$)		    ; current container
(defvar *report-width* nil) ; find op reports group position or groups width

#+MCL
(defvar ?)	         ; stop eval from complaining in mcl

(defparameter *map-operator-strings*
  (vector "SET" "SCALE" "INCREMENT" "TRANSPOSE" "INVERT" 
          "COLLECT" "MINIMIZE" "MAXIMIZE" "HIGHEST" "LOWEST" "SUM" "COUNT"
          "DO" "PRINT" "AVERAGE" "WHEN" "UNLESS" "AND" 
          "FIND" "INSERT" "APPEND" "INSERT-AT" "APPEND-AT" 
          "DELETE" "UNDELETE" "HIDE" "UNHIDE" "WHILE" "UNTIL" "UNSET"
          "CHANGE" "LIST" "QUANTIZE" "QUANTIZE!" "ROUND"  "ANALYZE"))

(defun next-token-is-operator (string start)
  (let ((p (tl:next-whitespace-start string :start start)))
    (find string *map-operator-strings*
          :test #'(lambda (i j) (string-equal i j :start1 start :end1 p)))))

;;;
;;; map needs common music's [] chord notation disabled
;;; so that its reader can see references like $NOTE[2].
;;;

(defvar readtable-without-chords (copy-readtable nil))

;;;
;;; read-map-form reads the next thing out of string and returns two
;;; values: the sub form read and the starting position of the next
;;; form in the string.
;;;

(defun read-map-form (string start length)
  (let ((*readtable* readtable-without-chords))
    (multiple-value-bind (? end)
        (read-from-string string nil :eof :start start :preserve-whitespace t)
      (values ? (or (tl:next-token-start string :start end) length)))))

;;;
;;;
;;;

(defstruct (clause (:type list)) op (count 0) forms cache external)

;;;
;;; make-cache returns a clause cache for a given operator. the cache 
;;; is usually a bookkeeping list in the form (fn value). the value can
;;; be updatated each new objects mapped. the operators :find :insert
;;; and :append use the cache to store positions, new objects etc. These
;;; are generally stored indirectly, as the value of a holder variable.
;;; the holder variable is either a gensymed symbol or else a varible
;;; specified by the via the optional "INTO" operator.
;;; 

(defmacro clause-cache-holder (c)
  `(cadr (clause-cache ,c)))

(defmacro clause-cache-value (c)
  `(symbol-value (clause-cache-holder ,c)))

(defun replace-cache-holder (clause var)
  ;; replace gensymed value holder with user supplied "into" variable
  (set var (clause-cache-value clause))
  (setf (clause-cache-holder clause) var)
  (values))

(defun make-cache (op)
  (flet ((cv (v) (let ((s (gensym))) (set s v) s)))
    (case op
      (:collect (list #'cons (cv nil)))
      ((:sum :average) (list #'+ (cv 0)))
      (:count (list #'(lambda (x n) (if x (1+ n) n)) (cv 0)))
      (:minimize (list #'min (cv most-positive-fixnum)))
      (:maximize (list #'max (cv most-negative-fixnum)))
      (:lowest (list #'(lambda (x y) (if (scale< x y) x y))
                     (cv most-positive-fixnum)))
      (:highest (list #'(lambda (x y) (if (scale> x y) x y))
                      (cv most-negative-fixnum)))
      (:analyze (make-hash-table))
      (t nil))))

(defun parse-map-input (string &key (start 0) command instance terminate
                                    only-one group-width)
  (let ((length (length string))
        (clauses ())            ; list of clauses parsed
        (clause ())             ; current clause
        (op nil)                ; current operator
        (repeat nil)            ; operator supports multiple forms
        state fields form next $maperr$)
    (declare (special $maperr$))
    (loop while (< start length)
          do
       (if repeat
           (unless command
             (when (next-token-is-operator string start)
               (setf op nil)))
         (unless command (setf op nil)))
       (when (null op)
         (when clause
           (setf clauses (nconc clauses (list clause))))
         (if command
             (setf op command)
           (let* ((str (next-token-is-operator string start)))
             (unless str
               (return-from parse-map-input
                 (values nil 
                   (format nil "Expected operator but got \"~A\" instead."
                           (subseq string start)))))
             ;; increment string if conjunction to continue or else
             ;; return if only one clause allowed
             (if (string-equal str "AND")
                 (multiple-value-setq (str start)
                   (read-map-form string start length))
               (when (and clause only-one)
                 (setf clause nil) (return)))             
             (multiple-value-setq (op start)
                (read-map-form string start length))
             (setf op (find-symbol (symbol-name op) :keyword))))
         (setf clause (make-clause :op op :cache (make-cache op)))
         (setf state (get op :map-syntax))
         (setf fields (or (position ':into state)(length state))))
       (when (eq (car state) ':repeat)   
         (setf repeat t state (cadr state)))
       (loop for s in state
             do
         (unless (or (member s '(:into :optional-slots)) (< start length))
           (return-from parse-map-input 
             (values nil
                     (format nil "Expected ~A but reached end of input." s))))
         (case s
           (:SLOT
            (multiple-value-setq (form next) 
              (read-map-form string start length))
            (unless (and (symbolp form)
                         (setf form (parse-object-reference (symbol-name form)
                                                            :slot-name
                                                            group-width)))
             (return-from parse-map-input
               (values nil
                       (or $maperr$ (format nil "~A cannot be a slot."
                                            form)))))
            (unless (or (null instance) 
                        (slot-exists-p instance 
                                       (if (consp form) (car form) form)))
              (return-from parse-map-input 
                (values nil 
                  (format nil "~A is not a slot~@[ for ~A objects~].~%" 
                          (if (consp form) (car form) form)
                          (and instance (class-name (class-of instance))))))))
           (:EXPR
            (let ((eval? nil))
              (when (char= (aref string start) #\,)
                (setf eval? t
                      start (or (tl:next-token-start string :start start)
                                length)))
              (multiple-value-setq (form next)
                (read-map-form string start length))
              (when eval? (setf form (eval form))))
              (setf form (walk-map-expr form op terminate group-width))
              (when $maperr$
                (return-from parse-map-input  (values nil $maperr$)))
              (setf terminate nil))
           (:COMMAND
             (multiple-value-setq (form next) 
               (parse-map-input string :start start :only-one t
                               :group-width group-width))
             (unless form 
               (return-from parse-map-input (values nil length))))
           (:INTO
            (let ((into nil))
              (when (string-equal string "INTO"
                                  :start1 start :end1 (min (+ start 4) length))
                (multiple-value-setq (into next) 
                  (read-map-form string (tl:next-token-start string 
                                          :start (+ start 4))
                                 length))
                (unless (and (symbolp into) (not (constantp into)))
                  (return-from parse-map-input 
                    (values nil
                      (format nil "INTO: ~S not a legal variable refererence." 
                              into))))
                (replace-cache-holder clause into))))
           (:CLASS
            (multiple-value-setq (form next) 
              (read-map-form string start length))
            (unless (and (symbolp form)
                         (setf form (find-class form)))
              (return-from parse-map-input 
                (values nil (format nil "Bad class specification.")))))
           (:OPTIONAL-SLOTS 
            (let (slot slots)
              (setf slots
                (loop until (or (>= start length)
                                (next-token-is-operator string start))
                      do (multiple-value-setq (slot next)
                           (read-map-form string start length))
                         (setf start next)
                      collect slot))
              (unless (or slots t)
                (return-from parse-map-input 
                  (values nil (format nil "Missing list of slots."))))
              (setf form slots))))
        (unless (eq s ':into)
          (setf (clause-forms clause) 
            (if (= fields 1) form
              (nconc (clause-forms clause) (list form))))
          (setf (clause-external clause) 
            (if (clause-external clause)
                (concatenate 'string (clause-external clause) 
                            (subseq string start next))
              (subseq string start next))))

        (setf start next)))
    (when clause (setf clauses (nconc clauses (list clause))))
    (values clauses start)))

;;;
;;; walk-map-expr is called on each mapping expression to perform whatever
;;; rewrites of the external form are necessary. the bulk of this work is
;;; converting $name or $name[n]  references to (slot-value? $ 'name) and
;;; (slot-value? (elt $ n) 'name) respectively. we also take care of
;;; evaluating item stream descriptions and converting them to a call
;;; on item with the stream embedded, ie. (item '#<stream>) 
;;; a number of operators like transpose, scale etc, assume that there
;;; is already a non-null slot value that we can perform the operation on.
;;; for convert these expressions to ones that first check the current
;;; value of the special varibale ? (which the set-slot function insures
;;; is holding the current value of the current slot of the current object)
;;; and if there isnt a value we abort the operation.
;;;

(defun walk-map-expr (external-form op throw-end-of-stream group-width)
  (let ((form
          (walk-form external-form #+clisp *toplevel-environment*
                                   #-clisp nil     
            #'(lambda (form context environment)
                (declare (ignore context environment))
                (cond ((constantp form) (values form t))
                      ;; if the form is a item stream constructor macro,
                      ;; we eval it to create the stream and and wrap the
                      ;; quoted result inside a call to item.
                      ((listp form)
                       (let ((car (car form)))
                         (if (and (symbolp car) (get car :item-expand))
                             (values `(item (quote ,(eval form))) t)
                           (values form nil))))
                      ;; we allow slot values to be referenced by $slot,
                      ;; where slot is the name of the slot. $slot is
                      ;; rewritten as a call to slot-value wrapped inside
                      ;; a bunch of accessing error checks.
                      ((symbolp form)
                       (if (boundp form)
                           (values form t)
                         (let* ((name (symbol-name form))
                                (char (elt name 0)))
                           (if (member char '(#\$ #\@ #\%) :test #'char=)
                               (values (parse-object-reference 
                                         name
                                         (if (char= char #\$)
                                             ':slot-expr ':variable)
                                         group-width)
                                        t)
                             (values `(quote ,form) t)))))
                      ((typep form 'cm::item-stream) 
                       `(item (quote ,form)))
                      (t (values form t)))))))
    ;; if we are setting an unknown number of elements (via the New command)
    ;; the expr for first slot determines when to stop creating and setting
    ;; new instances. the expr is wrapped in a call to item and the return 
    ;; status is checked for end-of-period.
    (when throw-end-of-stream
      (unless (and (consp form) (eq (car form) 'item))
        (setf form `(item ,form)))
      (setf form 
        `(locally (declare (special $done$))
           (multiple-value-bind ($datum$ $status$) ,form
             (and (eq $status$ cm::+end-of-stream-token+) 
                  (setq $done$ t))
             $datum$))))
    ;; increment, scale, transpose and invert must check current slot value.
    (case op
      (:increment (setf form `(if ? (+ ? ,form) (throw :no-value nil))))
      (:scale (setf form `(if ? (* ? ,form) (throw :no-value nil))))
      (:transpose (setf form `(if ? (transpose ? ,form) 
                                       (throw :no-value nil))))
      (:quantize (setf form `(if ? (quantize ? ,form) (throw :no-value nil))))
      (:quantize! (setf form `(if ? (quantize! ? ,form) 
                                (throw :no-value nil))))
      (:round (setf form `(if ? (decimals ? ,form) (throw :no-value nil))))

      (:invert (setf form `(if ? (invert ? ,form) (throw :no-value nil))))
      (:print (setf form `(print ,form)))
      (:delete (setf form `(delete-object ,form)))
      (:undelete (setf form `(undelete-object ,form)))
      (:hide (setf form `(hide-object ,form)))
      (:unhide (setf form `(unhide-object ,form))))
    form))

;;;
;;; parse-object-reference is responsible for rewriting slot and map variables
;;; ala $note, $note[2], @ and @[2] into legal lisp code. token is the token
;;; in question. mode is one of :slot-expr :slot-name or :variable. width is
;;; the mapping group width or nil if not mapping by groups. if the mode is
;;; :slot-expr the first character in the token is guaranteed to be $. if
;;; the mode is either :slot-name or :variable, then the whole symbol up
;;; to any index specification is taken to be then slot name or map variable
;;; respecively.  if width is a number then indicies following the symbol
;;; are legal and the "[n]" portion of the token is transformed into an 
;;; internal indexing form of some kind. for eample $note[2] would be written 
;;; (? (ELT $ 1) 'NOTE) if the mode were :slot-eval. similarly, note[2]
;;; would be written (NOTE . 1). if the mode were :variable, @[2] would be
;;; rewritten (@ (ELT $ 1))
;;;

(defun parse-object-reference (token mode width)
  (let ((end (length token))
        (beg 0)
        char left right num name symbol object)
    (flet ((err (&rest args)
             (declare (special $maperr$))
             (setf $maperr$ (apply #'format nil args))
             (return-from parse-object-reference nil)))
      ;; remove the dollar sign if parsing slot expr
      (if (eq mode ':slot-expr) 
          (incf beg)          
        (when (char= (elt token 0) #\$)
          (err "Bogus dollar notation in ~A." token)))
      (loop with pos = beg
            while (< pos end)
            do 
        (setf char (elt token pos))
        (cond ((char= char #\[) 
               (when left (err "Extraneous left bracket in ~A." token))
               (setf left pos))
              ((char= char #\]) 
               (unless (> (- pos left) 1)
                 (err "Empty brackets in ~A" token))
               (when right (err "Extraneous right bracket in ~A." token))
               (unless left (err "Right bracket without left in ~A." token))
               (setf right pos)))
        (incf pos))
      (if left
          (if width    
              (progn 
                (setf num (parse-integer token :start (1+ left) 
                                         :end right :junk-allowed t))    
                (unless (and num (<= 1 num width))
                  (err "Bogus index in ~A for grouping width ~A~%"
                       token width))
                (decf num))
            (err "~A group index but not mapping groups." token))
         (when width
           (warn "~A missing group index. Rewriting as ~A[1]" token token)
           (setf num 0)))
      (if num 
          (setf object `(elt $ ,num))
        (setf object '$))
      (if (string= (setf name (subseq token beg left)) "")
          (if (eq mode ':slot-expr)
              (setf symbol '$)
            (err "Missing slot name in ~A~%" token))
        (unless (setf symbol (find-symbol name))
          (err "~&~A cannot be a slot or variable~%" token)))
      (if (eq mode :slot-name)
          (if num (cons symbol num) symbol)
        (case symbol
          ($     object)
          (class `(class-name (class-of ,object)))
          (@     `(@ ,object $$))
          (@+1   `(@+1 ,object $$))
          (%     `(% ,object $$))
          (t     `(? ,object ',symbol ':no-value)))))))

(defun make-map-closure (clauses &optional (do-increment t))
  (if (= (length clauses) 1)
      (make-clause-closure (car clauses) do-increment)
    (let ((fns (loop for c in clauses 
                     collect (make-clause-closure c nil))))
       (if do-increment
           #'(lambda ($ &optional $$)
               (dolist (f fns) (funcall f $ $$)) ; do the clauses
               )
         #'(lambda ($ &optional $$) (dolist (f fns) (funcall f $ $$)))))))

;;;
;;; our main workhorse. returns a closure to implement each mapping clause
;;; the closures take a datum ($), and a container ($$). the container arg
;;; is &optinal because when the mapping mode is :container we pass containers
;;; as the first arg.
;;;

(defun make-clause-closure (clause &optional ignore)
  (declare (ignore ignore))
  (ecase (clause-op clause)
    ((:SET :INCREMENT :SCALE :TRANSPOSE :INVERT :QUANTIZE :QUANTIZE! :ROUND)
     #'(lambda ($ &optional $$)
         (set-slots $ (clause-forms clause))
         (incf (clause-count clause))))
    ((:UNSET)
     #'(lambda ($ &optional $$)
         (unset-slots $ (clause-forms clause))
         (incf (clause-count clause))))
     ((:SUM :COLLECT :MAXIMIZE :MINIMIZE :LOWEST :HIGHEST :AVERAGE :COUNT)
      #'(lambda ($ &optional $$) 
          (catch :no-value
            (setf (clause-cache-value clause)	; cache = (fn value)
              (funcall (car (clause-cache clause)) 
                       (eval (clause-forms clause))
                       (clause-cache-value clause))))
          (incf (clause-count clause))))       
     ((:FIND :LIST :SHOW)
      #'(lambda ($ &optional $$)
          (catch :no-value
            (when (eval (clause-forms clause))
              (let ((set (assoc $$ (clause-cache clause)))
                    (pos (object-position (if (consp $) (car $) $) $$)))
                (unless (numberp pos)
                  (error "Bad position ~S while mapping ~S in ~S." pos $ $$))
                (unless set
                  (setf set (list $$))
                  (push set (clause-cache clause)))
               (push pos (cdr set))   ; but display is 1 based...
               (incf (clause-count clause)))))))
      ((:INSERT :APPEND)
       #'(lambda ($ &optional $$)
           (let ((value (eval (clause-forms clause))))
             (when value
               (let ((set (assoc $$ (clause-cache clause)))
                     (pos (if (not (consp $))
                              (object-position $ $$)
                            (object-position (car (if (eq (clause-op clause)
                                                       ':insert)
                                                   $ (last $))) 
                                              $$))))
                 (unless (numberp pos)
                   (error "Bad position ~S while mapping ~S in ~S." pos $ $$))
                 (unless set
                   (setf set (list $$))
                   (push set (clause-cache clause)))
                 (push (cons (if (eq (clause-op clause) ':insert)
                                 pos (1+ pos)) 
                             value)
                       (cdr set))))
             (incf (clause-count clause)))))
      ((:INSERT-AT :APPEND-AT)
       #'(lambda ($ &optional $$)
           (let ((set (assoc $$ (clause-cache clause)))
                 pos new)
             (unless set
               (setf set (list $$))
               (push set (clause-cache clause)))
             (loop for (loc obj) on (clause-forms clause) by #'cddr
                   do 
               (unless (numberp (setf pos (eval loc)))
                 (error "Bad position ~S while mapping ~S in ~S." pos $ $$))
               (when (setf new (eval obj))
                 (push (cons (if (eq (clause-op clause) ':insert-at)
                                 pos (1+ pos)) 
                             new)
                       (cdr set))))
             (incf (clause-count clause)))))
      ((:CHANGE)
       ;; would allow obj specified as expr.
       ;#'(lambda ($ &optional $$)
       ;    (declare (ignore $ $$))
       ;    (apply #'change-object (eval (car (clause-forms clause))) 
       ;          (cdr (clause-forms clause)))
       ;    (incf (clause-count clause)))
       #'(lambda ($ &optional $$)
           (apply #'change-object $ (clause-forms clause))
           (incf (clause-count clause))))
     ((:DO :PRINT :HIDE :UNHIDE :DELETE :UNDELETE)
      #'(lambda ($ &optional $$) 
          (eval (clause-forms clause))
          (incf (clause-count clause))))
     ((:WHEN :UNLESS)
      (let ((test (car (clause-forms clause)))
            (action (make-map-closure (cadr (clause-forms clause)) nil)))
        (if (eq (clause-op clause) ':WHEN)
            #'(lambda ($ &optional $$) 
                (catch :no-value
                  (when (eval test) (funcall action $ $$))))
          #'(lambda ($ &optional $$) 
              (catch :no-value
                (unless (eval test) (funcall action $ $$)))))))
     ((:WHILE :UNTIL)
      (let ((test (car (clause-forms clause)))
            (action (make-map-closure (cadr (clause-forms clause)) nil)))
        (if (eq (clause-op clause) ':WHILE)
            #'(lambda ($ &optional $$) 
                (catch :no-value
                  (if (eval test)
                      (funcall action $ $$)
                    (throw :mapslots nil))))
          #'(lambda ($ &optional $$) 
              (catch :no-value
                (if (eval test)
                    (throw :mapslots nil) 
                  (funcall action $ $$)))))))
     ((:ANALYZE)
      ;; store positions of each descrete slot value in a hash table
      (let ((slot (clause-forms clause))
            (table (clause-cache clause)))
         #'(lambda ($ &optional $$)
             (catch :no-value
               (let ((value (? $ slot))
                     entry)
                 (if (setf entry (gethash value table))
                     (push (@ $ $$) (cdr entry))
                   (progn
                     (setf entry (list (@ $ $$)))
                     (setf (gethash value table) entry))))
               (incf (clause-count clause))))))
  ))

(defun set-slots ($ forms)
  (let ((obj $)
        (? nil)
        (new nil))
    (declare (special ?))
    (loop for (slot form) on forms by #'cddr
          do
      (when (consp slot)
        (setf obj (nth (cdr slot) $) slot (car slot)))
      (when (slot-exists-p obj slot)
        (catch ':no-value 
          (setf ? (slot-value-or-default obj slot))
          (setf new (eval form))
          (setf (slot-value obj slot) new))))
    $))

(defun unset-slots ($ slots)
  (let ((obj $))
    (loop for slot in slots
          do
      (when (consp slot)
        (setf obj (nth (cdr slot) $) slot (car slot)))
      (when (slot-exists-p obj slot)
        (slot-makunbound obj slot)))
    $))

(defun mapslots (selections clauses width)
  (let ((fun (make-map-closure clauses t))
        (lev (mapping-level))
        (mod (mapping-mode t)))
    (catch :mapslots
      (mapref #'(lambda (o c) (map-object-aux o fun c mod 0 lev))
              selections t t))
    (let ((*expr-length* 0)
          (*map-results* ()))
      (declare (special *expr-length* *map-results*))
      (post-process-clauses clauses width)
      (when *map-results*
        (let ((countpos (+ *expr-length* 10))
              (valuepos (+ *expr-length* 17)))
        (format t "~&CLAUSE       ~VTCOUNT  VALUE~%" countpos)
        (dolist (result (nreverse *map-results*))
          (apply (car result) countpos valuepos (cdr result))))))))

(defun print-result-1 (count-column value-column op expr count value)
  (declare (ignore value-column))
  (format t "~(~9A~)~A~VT~5D  ~S~%" op expr count-column count value))

(defun print-external-references (stream container list width)
  (let (low high flg len)
    (when (and width (> width 1))
      (decf width)
      (setf len (object-count container)))
    (if container
        (format stream "~A[" (object-namestring container))
      (format stream "[" ))
    (loop while list
          do
      (setf low (pop list))
      (if len
          (setf high (min (+ low width) len))
        (progn 
          (setf high low)
          (loop while list
                while (= (1+ high) (car list))
                do (incf high) 
                   (pop list))))
      (when flg (format stream ",") )
      (setf flg t)
      (if (> high low)
          (format stream "~D:~D" (1+ low) (1+ high))
        (format stream "~D" (1+ low))))
      (format stream "]"))) ;(format stream "]~%")

(defun print-find-result (count-column value-column op expr count cache width)
  (let ((first t))
    (dolist (sub cache)
      (if (not first)
          (format t "~VT" value-column)
        (progn (setf first nil)
               (format t "~(~9A~)~A~VT~5D  " op expr count-column count)))
      (if (cdr sub)
          (cond ((eq op :find) 
                 (print-external-references *standard-output*
                                            (car sub) (cdr sub) 
                                            (and *report-width* width))
                 (terpri))
                (t
                 (let ((refs (with-output-to-string (foo)
                               (print-external-references foo
                                                          (car sub) (cdr sub) 
                                                          (and *report-width*
                                                               width)))))
                   (write-line refs)
                   (if (eq op :list)
                       (list-cmd refs)
                    (show-cmd refs)))))
        (format t "None~%")))))

(defun post-process-clauses (clauses width)
  (declare (special *expr-length* *map-results*))
  (loop for clause in clauses
        for op = (car clause)
        do
    (case op
      ((:when :unless :while :until) 
       (post-process-clauses (cadr (clause-forms clause)) ; omit predicate
                             width)) 
      ((:sum :collect :minimize :maximize :lowest :highest :average :count)
       (let ((value (clause-cache-value clause))
             (count (clause-count clause))
             (expr (clause-external clause)))
         (setf *expr-length* (max *expr-length* (length expr)))
         (case op
           (:collect 
             (setf (clause-cache-value clause) (setf value (nreverse value))))
           (:average 
             (when (> value 0) 
               (setf value (/ value count))
               (setf (clause-cache-value clause) value))))
         (push (list #'print-result-1 op expr count value) *map-results*)))
      ((:find :insert :append :insert-at :append-at :list :show)
       (let ((cache (clause-cache clause))
             (count (clause-count clause))
             (expr (clause-external clause)))
         (setf *expr-length* (max *expr-length* (length expr)))
         (case op
           ((:find :list :show)
            (setf cache (nreverse cache))
            (dolist (s cache) (setf (cdr s) (nreverse (cdr s))))
            (push (list #'print-find-result op expr count cache width) 
                  *map-results*))
           ((:insert :append :insert-at :append-at)
            (loop with sort = (> (length (clause-forms clause)) 2)
                  and container 
                  for additions in (nreverse cache)
                  do
              (setf container (pop additions))
              ;; we process the (position . object) specs in additions 
              ;; in decending order.
              (when additions
                (when sort
                  (setf additions (sort additions #'> :key #'car)))
                (loop for spec in additions
                      do
                  (if (consp (cdr spec))
                      (add-objects (cdr spec) container (car spec))
                    (add-object (cdr spec) container (car spec))))
                (format t "~&New objects ")
                (let ((nums (loop with off = 0 for x in (nreverse additions)
                                  collect (+ (car x) off)
                                  do (incf off (if (consp (cdr x)) 
                                                   (length (cdr x)) 
                                                 1)))))
                  (print-external-references container nums 0 nil))))))))
      (:analyze
       (setf *expr-length* 
         (max *expr-length* (length (clause-external clause))))
       (push (list #'print-statistics clause) *map-results*))
      (t ))))

(defun print-statistics (count-column value-column clause)
  (declare (ignore value-column))
  (let ((table (clause-cache clause))
        (count (clause-count clause))
        (numeric? t)
        (min most-positive-fixnum)
        (max most-negative-fixnum)
        (sum 0)
        (num 0)
        (vals ())
        avr var dev)
    (maphash #'(lambda (val locs &aux len)
                 (incf num)        ; number of descrete values.
                 (setf len (length locs))
                 (when numeric?
                   (if (numberp val)
                       (progn 
                         (setf min (min val min))
                         (setf max (max val max))
                         (incf sum (* val len))
                         )
                     (setf numeric? nil)))
                 ; vals=list of (val n . locs)
                 (push (list* val len (sort locs #'<)) vals)
                 )
              table)
    (format t "analyze  ~A~VT~5D  ~%" (clause-external clause)
            count-column count)
    (format t "~%Unique:~VT~S" 13 num)
    (when (and numeric? (cdr vals))
      (format t "~%Minimum:~VT~,3F" 13 min)
      (format t "~%Maximum:~VT~,3F" 13 max)
      (setf avr (/ sum count))
      (setf var
        (/ (loop for v in vals
                 sum (loop with x = (- (first v) avr)
                           repeat (second v)
                           sum (* x x)))
                 (- count 1)))
      (setf dev (sqrt var))
      (format t "~%Mean:~VT~,3F" 13 avr)
      (format t "~%Variance:~VT~,3F" 13 var)
      (format t "~%Deviation:~VT~,3F" 13 dev)
      (setf vals (sort vals #'< :key #'first))
      )
    (format t "~%Breakdown:   N   %    Pos")
    (loop for entry in vals
          for val = (pop entry)
          for cnt = (pop entry)
          do
      (if (and (numberp val) (floatp val))
          (format t "~%~9,3F" val)
        (format t "~%~9@S" val))
      (format t "~5@A~6,1F  " cnt (* (/ cnt count) 100.0))
      (print-external-references t nil entry nil))
    (format t "~%~%")
  ))

#|
             
the square root of the mean of the squared differences
from the mean.  Which is to say, something like:

(sqrt (/ (- (* n (loop for i from 1 to n sum (sqr x[i])))
            (sqr (loop for i from 1 to n sum x[i])))
         (* n (- n 1))))
or
(sqrt (/ (loop for i from 1 to n sum (sqr (- x[i] xmean))) (- n 1)))

(defun sqr (x) (* x x))

(defun w1 (X &aux (n (length x)))
  (sqrt (/ (- (* n (loop for i from 1 to n sum (sqr (elt x (1- i)))))
              (sqr (loop for i from 1 to n sum (elt x (1- i)))))
           (* n (- n 1)))))

(defun w2 (X &aux (n (length x)) xmean)
  (setf xmean (/ (loop for i in x sum i) n))
  (sqrt (/ (loop for i from 1 to n sum (sqr (- (elt x (1- i)) xmean))) 
           (- n 1))))

;;;;;;;;;;

(thread foo ()
   (loop with stream = (numbers to 1 by .1 in random)
        repeat 50
         do
     (object midi-note note (note (between 60 80))
                       rhythm (item stream))))

map foo analyze rhythm
map foo analyze note


|#

                       
                                        