;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

;;;
;;; event reading and writing. this will be implemented in terms of
;;; standard lisp streams once enough lisp vendors support stream io
;;; via clos instances and generic functions.
;;;

(defgeneric open-event-stream (type &rest args))
(defgeneric event-stream-matches-p (stream args))
(defgeneric close-event-stream (stream &optional mode))
(defgeneric read-event (stream))
(defgeneric write-event (event stream))
(defgeneric write-event? (event stream))
(defgeneric scale-time (event stream))
(defgeneric peek-event (stream))
(defgeneric poke-event (stream))
(defgeneric event-file-position (stream))
(defgeneric event-file-length (stream))
(defgeneric initialize-stream-for-processing (stream))
(defgeneric deinitialize-stream-for-processing (stream))

(defmacro openp (stream) `(slot-boundp ,stream 'stream))

;;;
;;; root classes for event io.
;;;

(defclass event-stream (flags-mixin container-mixin)
  (stream
   (start :initarg start :initarg :start)
   (end :initarg end :initarg :end)
   (timescale :initarg timescale :initarg :timescale)
   (direction :initarg :direction :initarg direction :initform ':output)
;   (settings :initarg :settings :initarg settings :initform nil)
   (syntax :allocation :class)))


(defmethod initialize-instance :after ((stream event-stream) &rest args)
  (declare (ignore args))
  (add-object stream .io-streams.))

(defclass event-file (event-stream)
  ((pathname :initarg pathname :initarg :pathname)
   (element-type :initarg element-type :initarg :element-type 
                 :initform 
                 #+(and excl (not cltl2)) 'string-char
                 #-(and excl (not cltl2)) 'character)))

(defmethod print-object ((object event-file) stream)
  (format stream "#<File: ~S>" 
          (namestring (slot-value object 'pathname))))

(defclass header-mixin ()
  ((header :initarg :header :initarg header)))

(defmethod event-stream-matches-p ((object event-stream) args)
  (declare (ignore args))
  nil)

(defmethod event-stream-matches-p ((object event-file) args)
  (let* ((p1 (or (getf args ':pathname) (getf args 'pathname)))
         (p2 (slot-value object 'pathname))
         (d1 (pathname-directory p1))
         (d2 (pathname-directory p2))
         (n1 (pathname-name p1))
         (n2 (pathname-name p2))
         (t1 (pathname-type p1))
         (t2 (pathname-type p2)))
  (if (and d1 d2)
      (if (equal d1 d2)
          (if (and (equal n1 n2) (equal t1 t2))
              t
             nil)
        nil)
    (if (and (equal n1 n2) (equal t1 t2))
        t 
      nil))))

;;;
;;; find-stream returns the listener if passed a syntax, otherwise the file.
;;;

(defmethod find-stream ((ref string) &optional args)
  ;; ref is either a pathname or a syntax name. if a syntax, return its
  ;; current listener, if any. if a pathname, find its stream.
  (let ((syn (find-syntax ref nil)))
    (if syn (find-stream syn)
      (find-stream `(pathname , ref ,@ args)))))

(defmethod find-stream ((ref cons) &optional args)
  ;; the actual work gets done here. ref is the slot and value list
  ;; that event-stream-mathches-p checks to see if the current stream
  ;; matches the data.
  #-aclpc (declare (ignore args))
  (find-if #'(lambda (x) (event-stream-matches-p x ref))
           (bag-cache (slot-value .io-streams. 'elements))))

(defmethod find-stream ((ref t) &optional args)
  #-aclpc (declare (ignore args))
  (find ref (bag-cache (slot-value .io-streams. 'elements))))

;;;
;;; maybe-customize-stream allows a new stream to be customized when it is
;;; created. the default method does nothing, for files it asks the user.
;;; 

(defmethod maybe-customize-stream ((obj t) flag)
  (declare (ignore flag))
  (values))

(defmethod maybe-customize-stream ((obj event-file) flag)
  (when (eq flag ':ask)
    (setf flag
      (ask-y-or-n :prompt (format nil "Modify new stream ~A?: "
                                  (namestring (slot-value obj 'pathname)))
                  :default ':no :abort-ok nil)))
  (when flag
    (tl:edit-object obj))
  (values))

;;;
;;; basic methods. often overridden by subclasses.
;;;
  
(defmethod open-event-stream ((stream symbol) &rest args)
  (apply #'open-event-stream (find-class stream) args))

(defmethod open-event-stream ((class standard-class) &rest args)
  (let ((stream (find-stream args)))
    (unless stream
      (setf stream (apply #'make-instance class :allow-other-keys t 
                          args)) 
      (maybe-customize-stream stream
                              (and *command-prompting*
                                   (pair-value 'edit args ':ask))))
    (apply #'open-event-stream stream args)))

(defmethod open-event-stream :before ((stream event-stream) &rest args)
  (when args (apply #'set-object stream args)))

(defmethod open-event-stream ((stream event-file) &rest args)
  (when (pair-value 'open args t)
    (setf (slot-value stream 'stream)
      (open (slot-value stream 'pathname)
            :direction (slot-value stream 'direction)
            :element-type (slot-value stream 'element-type)
            :if-exists ':supersede
            :if-does-not-exist ':create)))
  stream)

(defmethod close-event-stream ((stream event-stream) &optional mode)
  #-aclpc (declare (ignore mode))
  (when (slot-boundp stream 'stream)
    (when (streamp (slot-value-or-default stream 'stream nil))
      (close (slot-value stream 'stream)))
    (slot-makunbound stream 'stream)))

;;;
;;; default methods for initialization and writing do nothing.
;;;

(defmethod initialize-stream-for-processing ((stream event-stream))
  (values))

(defmethod initialize-stream-for-processing :before ((stream event-stream))
  (values))

(defmethod deinitialize-stream-for-processing ((stream event-stream))
  (values))

(defmethod write-event ((element element) (stream event-stream))
  (values))

(defmethod post-process-stream ((stream event-stream) inits)
  (declare (ignore inits))
  nil)

;;;
;;; default postprocessing asks if the file should be played
;;;

(defparameter *post-processing* ':ask)

(defmethod post-process-stream ((stream event-file) inits)
  (let ((file (namestring (slot-value stream 'pathname)))
        (play (pair-value 'play inits *post-processing* )))
    (when (eq play ':ask)
      (setf play
            (if *command-prompting*
              (ask-y-or-n :prompt (format nil "Play file ~A? " file)
                          :default 'yes)
              nil)))
    (when play
      (play-using-syntax (slot-value stream 'syntax) file))))


;;;
;;; write-event is the main function for output processing.  this method on
;;; containers is our main processing loop.  it calls select-element to get
;;; the next element and then write-event to output the object. each type of 
;;; object must implement a method for the output stream(s) it wants to
;;; support. write-event performs two possible main loops, based on whether
;;; the stream has a start or end time associated with it. if it does,
;;; each selected object is checked to see if it should be output, and 
;;; then is "left timeshifted" by the start time amount. if no start or end
;;; is specified then we optimize out the additional checks and processing.
;;;

(defmethod write-event ((object container) (stream event-stream))
  (let ((success nil))
    (unwind-protect 
      (let (element time)
        (score-reset object)
        (setf time (slot-value object 'time))
        (if (or (slot-boundp stream 'start) (slot-boundp stream 'end))
            (loop while (setf element (score-select object time))
                  when (write-event? element stream)
                    do (shift-time element stream)
                       (write-event element stream)
                  do
                     (score-unset element))
          ;; optimize loop if user didn't specify a start or end.
          (loop while (setf element (score-select object time))
                do (write-event element stream)
                   (score-unset element)))
        (setf success t))

      (unless success 
        (warn-user "Processing of ~S unexpectedly terminated.~&~
                    Performing recursive unset..." object))
      (score-unset object (not success)))
    object))

#|
(defmethod write-event ((object container) (stream event-stream))
  (let ((success nil))
    (unwind-protect 
      (let (element)
        (score-reset object)
        (loop while (setf element
                      (score-select object (slot-value object 'time)))
              do (write-event element stream)
	         (score-unset element))
	(setf success t))
      (unless success 
        (format t "Processing of ~S unexpectedly terminated.~&~
                  Performing recursive unset..." object))
      (score-unset object (not success)))
    object))
|#

;;;
;;; :around method for merges supports dynamic queuing during element
;;; writing. this is used by midi-note elements to schedule a future
;;; noteOff each time a note is written to the stream.
;;;

(defmethod write-event :around ((object merge) (stream event-stream))
  (let ((*merge* object)) 
    (call-next-method)))



;;;
;;; write-event? will be called on an object just before it is to be output
;;; if the stream has a start or end time associated with it. 
;;; it should return t if the event should be written, otherwise nil.
;;;

(defmethod write-event? ((object timed-object) (stream event-stream))
  (if (slot-boundp stream 'start)
      (if (< (slot-value object 'time) ;0
             (slot-value stream 'start))  
         nil
        (if (slot-boundp stream 'end)
            (if (> (slot-value object 'time) (slot-value stream 'end))
                nil
              t)
          t))
    (if (slot-boundp stream 'end)
        (if (> (slot-value object 'time) (slot-value stream 'end))
            nil
          t)
      t)))

;;;
;;; shift object's time left just before output. some objects, like
;;; midi-releases, shouldn't be time shifted, hence the generic function.
;;;

(defmethod shift-time ((object timed-object) (stream event-stream))
  (when (slot-boundp stream 'start)
    (setf (slot-value object 'time)
      (- (slot-value object 'time) (slot-value stream 'start)))))

;;;
;;; streams and syntax
;;;

(defmethod find-stream ((ref syntax) &optional args)
  ;; return current listener, if any
  #-aclpc (declare (ignore args))
  (if (slot-boundp ref 'listener) 
      (slot-value ref 'listener)
    nil))

(defgeneric open-listener (syntax &rest args))
(defgeneric close-listener (syntax &optional mode))
(defgeneric syntax-default-pathname (syntax stream))
(defgeneric play-using-syntax (syntax file &rest args))
(defgeneric import-using-syntax (syntax file &rest args))
(defgeneric load-using-syntax (syntax file &rest args))

(defmethod syntax-default-pathname ((syntax syntax) (stream event-file))
  (slot-value syntax 'pathname))

(defmethod open-listener ((syntax syntax) &rest args)
  (declare (ignore args))
  nil)

(defmethod close-listener ((syntax syntax) &optional  mode)
  (declare (ignore mode))
  nil)

;;;
;;;
;;;

(defun keyword-prefix (slot &optional (space t))
  (if space (format nil ":~(~A~) " slot)
    (format nil ":~(~A~) " slot)))

(defun &rest-printer (values stream)
  ;; initial argument delimiter is already printed.
  (loop while values
        do (prin1 (pop values) stream)
           (and values (write-char #\space stream))))

(defmacro formatting-slots ((object stream &rest keys) &rest slots)
  (apply #'formatting-slots-internal object stream slots keys))

(defun formatting-slots-internal (object stream specs 
                              &key (printer 'princ) (print-if t)
                                   eol prefix suffix format filter
                                   (delimiter '#\space) preamble postamble
                                   constructor default)
  (let (($stream$ (gensym))
        ($object$ (gensym))
        (forms ()))
    (setf forms
      (loop with del = nil 
            for spec in specs 
            when (eq spec '&optional)
            do (setf prefix nil print-if :bound )
            else
            when (eq spec '&key)
            do (setf prefix 'keyword-prefix print-if :bound default nil)
            else
            when (eq spec '&rest)
            do (setf prefix nil printer '&rest-printer print-if :bound 
                     default nil)
            else
            collect (if (consp spec)
                        (apply #'(lambda (slot &key (printer printer)
                                                    (print-if print-if)
                                                    (eol nil)
                                                    (prefix prefix)
                                                    (delimiter delimiter dp)
                                                    (suffix suffix)
                                                    (filter filter)
                                                    (format format)
                                                    (constructor constructor)
                                                    (default default)
                                                    )
                                    (slot-format-form $object$ slot $stream$
                                                      print-if printer format
                                                      (if dp delimiter
                                                        (if del delimiter nil))                            
                                                      prefix suffix eol filter
                                                      constructor default))
                       
                               spec)
                      (slot-format-form $object$ spec $stream$
                                        print-if printer format 
                                        (if del delimiter nil)
                                        prefix suffix nil filter
                                        constructor default))
           and do (setf del t)))
    (when preamble
      (let ((printer (select-printer preamble)))
        (push `(,printer ,preamble ,$stream$) forms)))
    (when postamble
      (let ((printer (select-printer postamble)))
        (nconc forms (list `(,printer ,postamble ,$stream$)))))
    (when eol
      (nconc forms (list `(terpri ,$stream$))))
    `(let (,@ (if stream `((,$stream$ ,stream)) nil)
           ,@ (if object `((,$object$ ,object)) nil))
        ,.forms)))


(defun slot-format-form (object slot stream print-if printer 
                                format delimiter prefix suffix eol 
                                filter constructor default)
  
  (let (value check-value check-bound forms)
    (cond (default 
           (setf value `(if (slot-boundp ,object ',slot)
                            (slot-value ,object ',slot)
                          ,default)))
          ((member print-if '(:value :valued))
           (setf value (gensym) check-value t check-bound t))
          ((eq print-if ':bound)
           (setf value `(slot-value ,object ',slot) check-bound t))
          ((member print-if '(t :always))
           (setf value `(slot-value ,object ',slot)))
          (t
           (error "~S is not a legal :PRINT-IF value." print-if)))
    (setf forms
      (cond (constructor 
             (when (or printer format delimiter prefix suffix eol)
               (error ":constructor excludes other keywords."))
             (list (list (funcall constructor slot) value)))
            (printer
             (list `(,printer ,(if filter (list filter value) value) ,stream)))
            ))
    (when format
      (ecase format
        (:quoted (push `(write-char #\' ,stream) forms))
        (:careful (push (let ((v (gensym)))
                          `(let ((,v (slot-value ,object ',slot)))
                             (and (or (consp ,v) 
                                      (and (symbolp ,v) (not (boundp ,v))))
                                  (write-char #\' ,stream))))
                      forms))
        (:string (push `(write-char #\" ,stream) forms)
                 (nconc forms (list `(write-char #\" ,stream))))))

    (when delimiter  
     (when (and (symbolp delimiter) (fboundp delimiter))
       (setf delimiter (funcall delimiter slot))))

    (when prefix   ; prefix is char, string or fn to call on slot
      (when (or (and (symbolp prefix) (fboundp prefix))
                (and (consp prefix) (eq (first prefix) 'lambda)))
        (setf prefix (funcall prefix slot)))
        ;; combine delimiter and prefix if possible.
        (when delimiter
          (setf prefix (concatenate 'string (string delimiter) 
                                    (string prefix)))
          (setf delimiter nil))
       (let ((printer (select-printer prefix)))
        (push `(,printer ,prefix ,stream) forms)))
    (when suffix   ; suffix is char, string or fn to call on slot
      (when (and (symbolp suffix) (fboundp suffix))
        (setf suffix (funcall suffix slot)))
      (let ((printer (select-printer suffix)))
        (nconc forms (list `(,printer ,suffix ,stream)))))
   (when delimiter  
     (let ((printer (select-printer delimiter)))
       (push `(,printer ,delimiter ,stream) forms)))
   (when eol
      (nconc forms (list `(terpri ,stream))))
    (when check-value
      (setf forms
        `((let ((,value (slot-value ,object ',slot)))
            (when ,value ,. forms)))))
    (if check-bound
        `(when (slot-boundp ,object ',slot)
           ,. forms)
       `(progn ,.forms))))

(defun select-printer (thing)
  (cond ((stringp thing) 
         'write-string)
        ((characterp thing)
         'write-char)
        (t
         (error "~S is not a string or character." thing))))

(defmacro collecting-slots (object &body slots)
  (let ((vals (gensym))
        specs)
    (flet ((fn1 (s) s `(lambda (x) (push x ,vals)))
           (fn2 (s) 
             `(lambda (val) ; push key first because of nreverse!!!
                (push ,(intern (string s) :keyword) ,vals)
                (push val ,vals))))      
       (setf specs
            (loop with if = t and fn = #'fn1
                  for s in slots
                  when (eq s '&optional)
                  do (setf if ':bound)
                  else
                  when (eq s '&key)
                  do (setf if ':bound fn #'fn2)
                  else
                  collect (if (listp s) 
                            (append s (list :print-if if :constructor fn))
                            (list s :print-if if :constructor fn))))
      `(let ((,vals ()))
         ,(formatting-slots-internal object nil specs :printer nil 
                                     :delimiter nil)
         (nreverse ,vals)))))


#|
;;; tests
(pprint (macroexpand '(formatting-slots (obj stream :default 0 
                                         :delimiter #\space)
                        foo bar (baz :filter note))))

(pprint (macroexpand '(formatting-slots (obj stream :constructor mk-make-sender
                                         :print-if :bound :printer nil)
                        freq amp )))
(pprint (macroexpand '(collecting-slots obj
                        foo &key bar (baz :default 12))))
(pprint (macroexpand '(collecting-slots o rhythm &key duration freq amp)))

|#
