;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(defun fheader (file &rest strings)
  (open-file file :edit nil :open nil
             :header (apply #'header strings)))

(defun open-file (file &rest pairs)
  (when pairs (canonicalize-pairs pairs))
  (multiple-value-bind (syntax class)
                       (pathname-syntax file 
                                        (pair-value 'syntax pairs))
    (declare (ignore syntax))
    (unless class (setf class 'event-file))
    (apply #'open-event-stream class 'pathname file pairs)))

;;;
;;; script-file loads a script file.
;;;

(defmacro script-insure (form &rest fail)
  (when (and (= (length fail) 1) (stringp (car fail)))
    (setf fail `((write-line ,(car fail)))))
  `(unless ,form ,@ fail (throw :script-quit nil)))

(defun script-file (file &rest pairs)
  (declare (special *commands*))
  (let ((output *standard-output*)
        (commands *commands*)
        (echo t)
        (prompt "Script: "))
    (dopairs (s v pairs)
      (case s
        ((commands :commands) (setf commands v) )
        ((output :output) (setf output v) )
        ((echo :echo) (setf echo t))
        ((prompt :prompt) (setf prompt v))))
    (catch :script-quit
      (tl:eval-from-file file :commands commands :echo-input echo
                         :output output :prompt prompt))))

;;;
;;; play-file plays a file according to a syntax
;;;

(defun play-file (file &rest pairs)
  (when pairs (setf pairs (canonicalize-pairs pairs nil)))
  (let ((syntax (pair-value 'syntax pairs
                            (or (pathname-syntax file) *syntax*))))
    (unless syntax (error "Can't determine syntax for playing ~S." file))
    (setf file (merge-pathnames file (slot-value syntax 'pathname)))
    (if (probe-file file)
        (apply #'play-using-syntax syntax file pairs)
      (error "File \"~A\" not found.~&" (namestring file)))))

;;;
;;; load-file load a file accoring to a syntax. if lisp, a compile option
;;; is also avaiable.
;;;

(defun load-file (file &rest pairs)
  (let ((syntax (or (pair-value 'syntax pairs)
                    (pair-value ':syntax pairs)
                    (pathname-syntax file))))
    (if syntax
        (apply #'load-using-syntax syntax file pairs)
      (let ((type (or (pathname-type file) "")))
        (cond ((string-equal type "tl")
               (apply #'script-file file pairs))
              ((string-equal type +stella-type+)
               (apply #'cload file :compile nil pairs))
              (t
               (apply #'cload file pairs)))))))

(defvar *cload-source-types*
  (list +stella-type+ +source-type+ "tl" "ins" "cm" "clm" "cmn" ))

(defun cload (file &rest pairs)
  (setf file (namestring file))
  (let ((compile nil)
        (type (pathname-type file))
        (verbose t) source loadfile)
    (dopairs (n v pairs)
      (case n 
         ((compile :compile) (setf compile v))
         ((verbose :verbose) (setf verbose v))))
    (if type
        (progn (setf source (probe-file file))
               (when (and (eq compile ':check)
                          (string-equal type +binary-type+))
                 (setf compile nil)))
      (setf source
            (loop for type in *cload-source-types*
                  when (probe-file (make-pathname :defaults file :type type))
                  return it)))
    (unless source
      (warn-user "File \"~A\" not found." file)
      (return-from cload nil))
    (setf loadfile
      (if (not compile) source
        (make-pathname :defaults source :type +binary-type+)))
    (when compile
      (setf source (truename source))
      (when (or (not (probe-file loadfile))
                (or (eq compile t) (eq compile ':force))
                (< (file-write-date loadfile)
                   (file-write-date source)))
        #+(or cltl2 lispworks clisp)
	(compile-file source :output-file loadfile
		      :verbose verbose)
        #-(or cltl2 lispworks)
	(compile-file source :output-file loadfile)))
     (setf loadfile (truename loadfile))
     (load loadfile :verbose verbose)
     loadfile))

;;;
;;; import-file imports a file according to a syntax.
;;;

(defun import-file (file &rest pairs)
  (when pairs (setf pairs (canonicalize-pairs pairs)))
  (let ((syntax (pair-value 'syntax pairs
                            (or (pathname-syntax file) *syntax*))))
    (apply #'import-using-syntax syntax file 
           (list* 'name (gensym (format nil "~A-"(pathname-name file))))
                  pairs)))

