(in-package :stella)

;;;
;;; this files shows how to implement an object and output printing methods
;;; for csound score files.  the object defined here is for a ficticious
;;; instrument called i1. i1 inherits its rhythm and time slots from Note,
;;; so all we need to do is add duration, frequency and amp. we also add
;;; a name slot just for fun and  give it default value of "i1".
;;;

(defobject i1 (note) 
  ((name :initform "i1")
   dur
   freq
   amp))

;;;
;;; we first define a method on print-object to print useful slot information
;;; whenever the List command prints an i1 object to the terminal.  if a slot
;;; is unbound, we print the value of +slot-unset+, the system print string
;;; denoting unbound slot values. we wrap the whole printout inside the
;;; printing-random-thing macro so that our display looks like other types of
;;; data defined in the editor.  a typical i1 might therefore display as:
;;;  #<ii 10.5 1.0 440 .1 12323231>
;;;

(defmethod print-object ((object i1) stream)
  (printing-random-thing (object stream)
    ;; show rhythm and print all unbound slots as "-unset-"
    (formatting-slots (object stream :default +slot-unset+)
      name rhythm dur freq amp)))

;;;
;;; this method on write-event writes our object to the open csound score 
;;; file. we get the file associated with the output stream, write the 
;;; current parameter values to it, and finish with an end-of-line. note
;;; that in this case there is no default print value for unbound slots so
;;; we will get an error if we try to write an i1 object to a score file
;;; with incomplete parameter information. this is a feature a bug because
;;; it catches the error at the earliest possible time.  note also that
;;; this method writes the time value to the score file, not the rhythm.
;;;

(defmethod write-event ((object i1) (stream csound-score-file))
  (formatting-slots (object (slot-value stream 'stream) :eol t)
     name time dur freq amp))



;;; after compiling and loading this file, try pasting everything between
;;; #| and |# (including blank lines) to stella's main prompt:

#|
(thread foo ()
  (dotimes (i 20)
    (object i1 freq (between 220. 440.) rhythm .2 dur .1 amp .25)))
rehash
list foo
syntax csound
write foo
0


|#