;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92, 93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; send to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(defun rename-object (object name &optional oldname)
  (let ((x (find-object name nil)))
    (when x (error "Id: ~s already used by ~s." name x)))
  (unless oldname
    (if (typep object 'id-mixin)
        (setf oldname (slot-value object 'id))
      (maphash #'(lambda (key val)
                   (when (eq object val) (setf oldname key)))
               *object-table*)))  
  (unless (eq (gethash oldname *object-table*) object)
    (error "Can't find ~s under old name ~s" object oldname))
  (remhash oldname *object-table*)
  (setf (find-object name) object)
  object)

(defun subobjectsp (object)
  ;; t if object is container with elements.
  (and (typep object 'container)
       (bag-cache (slot-value object 'elements))
       t))

(defun nth-object (index obj)
  (nth index (bag-cache (slot-value obj 'elements))))

(defun object-position (element &optional container)
  (unless container
    (setf container (slot-value element 'container)))
  (position element (bag-cache (slot-value container 'elements))))

(defun object-count (element)
  (if (typep element 'container)
      (length (bag-cache (slot-value element 'elements)))
    nil))


;;;
;;; add-object adds a new element to a container at a specified position.
;;; if position is nil, add-object is optimized to add the element
;;; to the end of the collection, otherwise position should be an zero based
;;; index into the existing elements.  the new element will be inserted
;;; at this position, ie just before the element that currently occupies this
;;; index. its faster to add things at the end with a null position than it
;;; is to actually specify length+1 index.
;;;

(defgeneric add-object (object container &optional pos &rest args))

(defmethod add-object ((object t) (container t)
                        &optional pos &rest args)
  (declare (ignore #-aclpc pos args))
  nil)

(defmethod add-object ((object t) (container container)
                        &optional pos &rest args)
  (declare (ignore args))		
  (if pos 
      (insert-bag object (slot-value container 'elements) pos)
    (append-bag object (slot-value container 'elements)))
  (unless (logtest (slot-value container 'flags) +no-link+)
    (link-object object container))
  object)

;;;
;;; add-objects optimizes add-object for more than one element. it performs
;;; destructive operations on the list of elements passed in, so normally the
;;; :copy-first arg should be t. this function could possibly be another method
;;; on add-object that specializes on sequences but then lists could not be
;;; elements of containers...
;;;

(defmethod add-objects ((elements list) (container container) 
                         &optional pos &key (copy-first t))
  ;; copy list so users data doesnt get modified.
  (when copy-first (setf elements (copy-list elements)))
  ;; if not a system container, add backpointers
  (unless (logtest (slot-value container 'flags) +no-link+)
    (dolist (e elements) (link-object e container)))
  ;; splice the new elements into the collection at position
  (let* ((bag (slot-value container 'elements))
         (cache (bag-cache bag)))
    (if (null cache)
        (setf (bag-cache bag) elements
              (bag-tail bag) (last elements))
      (if pos
          (let ((tail (nthcdr pos cache)))
	    (if (null tail)
	        (progn
                  (rplacd (bag-tail bag) elements)
	          (setf (bag-tail bag) (last elements)))
              (progn 
	        (nconc elements tail)
   	        (if (= pos 0)
	            (setf (bag-cache bag) elements)
		  (progn
		    (setf tail (nthcdr (1- pos) cache))
                    (rplacd tail elements))))))
        (progn
          (rplacd (bag-tail bag) elements)
	  (setf (bag-tail bag) (last elements))))))
  elements)


;;;
;;; if adding elements by hand to generator, mark it frozen
;;;

(defun add-elements (&rest args)	; need for backward compatibility
  (apply #'add-objects args))

(defmethod add-objects :after ((elements list) (container generator) 
                         &optional pos &key copy-first)
  (declare (ignore #-aclpc pos copy-first))
  (setf (slot-value container 'flags)
     (logior (slot-value container 'flags) +frozen+))) 

;;;
;;; link-object adds a backpointer from an element to its container
;;; and unlink-object gets rid of it. both return the element's
;;; current links after the operation has been performed.
;;;

(defgeneric link-object (element container))

(defmethod link-object ((element t) container)
  (declare (ignore container))
  nil)
  
(defmethod link-object ((element container-mixin) container)
  (let ((links (slot-value-or-default element 'container)))
    (if (null links)
        (setf links container (slot-value element 'container) container)
        (if (consp links) 
            (or (find container links)
                (nconc (slot-value element 'container) (list container)))
          (or (eq container links) 
              (setf links (list links container)
                (slot-value element 'container) links))))
    links))	      

(defgeneric unlink-object (element container))

(defmethod unlink-object ((element t) container)
  (declare (ignore container))
  nil)

(defmethod unlink-object ((element container-mixin) container)
  (let ((links (slot-value-or-default element 'container)))
    (if (consp links)
        (setf links (delete container links :test #'eq))
       (when (eq links container)
         (setf links nil)))
    (if links
        (setf (slot-value element 'container) links) 
      (slot-makunbound element 'container))
    links))
      
;;;
;;; the-container returns a single container or nil + the number of containers
;;;

(defun the-container (element &optional (errorp t))
  (let ((can (careful-slot-value element 'container nil)))
    (if (consp can)
        (if errorp
	    (error "The element ~S is contained by more than one element: ~S."
	   	   element can)
	   (values nil (length can)))
      (values can (if can 1 0)))))

;;;
;;; object-containers returns a list of the element's containers 
;;;

(defun object-containers (element &optional ignore-system)
  (let ((cans (careful-slot-value element 'container)))
    (unless (listp cans)(setf cans (list cans)))
    (if ignore-system
        (loop for c in cans unless (systemp c) collect c)
      cans)))
    
;;;
;;; remove-object unlinks the element at a position or all instance of
;;; the object from the container.
;;;

(defmethod remove-object ((pos number) (container container))
  (let* ((bag (slot-value container 'elements))
	 (len (length (bag-cache bag)))
	 obj)
    (cond ((= len 1)
           (setf obj (car (bag-cache bag))
                 (bag-cache bag) nil 
                 (bag-tail bag) nil
                 (bag-elements bag) nil))
	  ((= pos 0)               ; element is head
           (setf obj (pop (bag-cache bag))))
          ((= pos (1- len))	; element is tail
           (setf (bag-cache bag) (nbutlast (bag-cache bag))
                 obj (car (bag-tail bag))
                 (bag-tail bag) (last (bag-cache bag))))
	  (t
           (let ((tail (nthcdr (1- pos) (bag-cache bag))))
             (setf obj (cadr tail) (cdr tail) (cddr tail)))))

    (when (and (not (logtest (slot-value container 'flags) +no-link+))
               (slot-boundp obj 'container)
               (not (find obj (bag-cache bag))))   ; added more than once
      (let ((cans (slot-value obj 'container)))
        (if (listp cans)
            (progn (setf cans (remove container cans))
                   (unless (cdr cans) (setf cans (car cans))))
          (setf cans nil))
        (if (null cans) 
            (slot-makunbound obj 'container)
          (setf (slot-value obj 'container) cans))))
    obj))

(defmethod remove-object ((object t) (container container))
  (loop for once = nil then t
        for pos = (object-position object container)
        while pos
        do (remove-object pos container)
        finally (return (if once object nil))))

;;;
;;; remove-all-objects unlinks all the elements of container
;;;

(defun remove-all-objects (container)
  (let* ((bag (slot-value container 'elements))
         (elements (bag-cache bag)))
    ;; if its not a system container we must unlink the elements from
    ;; this container, otherwise there are no backpointers and we
    ;; just flush the bag.
    (unless (logtest (slot-value container 'flags) +no-link+)
      (loop for element in elements
          do 
      (let ((p (slot-value element 'container)))
        (if (consp p)
            (setf (slot-value element 'container)
	      (delete container p))
          (slot-makunbound element 'container)))))
    (setf (bag-cache bag) nil (bag-tail bag) nil (bag-elements bag) nil)
    container))

;;;
;;; 
;;;

(defgeneric flag-test (element &rest flags))
(defgeneric flag-test-and (element &rest flags))
(defgeneric flag-set (element &rest flags))
(defgeneric flag-unset (element &rest flags))

(defmethod flag-test-and ((element flags-mixin) &rest flags)
  (let ((i (apply #'logior flags)))
    (= (logand (slot-value element 'flags) i) i)))

(defmethod flag-test ((element flags-mixin) &rest flags)
  (logtest (slot-value element 'flags) (apply #'logior flags)))

(defmethod flag-set ((element flags-mixin) &rest flags)
  (setf (slot-value element 'flags) 
        (apply #'logior (slot-value element 'flags) flags)))

(defmethod flag-unset ((element flags-mixin) &rest flags)
  (setf (slot-value element 'flags) 
        (logandc2 (slot-value element 'flags) (apply #'logior flags))))

;;;
;;; systemp
;;;

(defmethod systemp ((x t))
  nil)

(defmethod systemp ((x flags-mixin))
  (logtest (slot-value x 'flags) +system+))
	    
;;;
;;; deletedp, delete, undelete, expunge
;;;

(defgeneric deletedp (object))
(defgeneric delete-object (object &optional recursive))
(defgeneric undelete-object (object &optional recursive))

(defmethod deletedp ((element t))
  nil)

(defmethod deletedp ((element flags-mixin))
  (logtest +deleted+ (slot-value element 'flags)))

(defmethod delete-object ((object t) &optional recursive)
  #-aclpc (declare (ignore recursive))
  nil)

(defmethod delete-object :after ((object container) &optional recursive)
  (when recursive (map-subobjects #'(lambda (x) (delete-object x t)) object)))

(defmethod delete-object ((object flags-mixin) &optional recursive)
  #-aclpc (declare (ignore recursive))
  (setf (slot-value object 'flags)
        (logior (slot-value object 'flags) +deleted+ +hidden+))
  t)
  
(defmethod undelete-object ((object t) &optional recursive)
  #-aclpc (declare (ignore recursive))
  t)

(defmethod undelete-object ((object flags-mixin) &optional recursive)
  #-aclpc (declare (ignore recursive))
  (setf (slot-value object 'flags)
        (logandc2 (slot-value object 'flags) (logior +deleted+ +hidden+)))
  t)

(defmethod undelete-object :after ((object container) &optional recursive)
  (when recursive 
    (map-subobjects #'(lambda (x) (undelete-object x t)) object)))

;;;
;;; expunge-object - removes and frees all deleted elements.
;;;

(defmethod expunge-object ((object t))
  )
 
(defmethod expunge-object :around ((object element))
  (if (deletedp object) (call-next-method) nil))
    
(defmethod expunge-object :around ((object container))
  (map-subobjects #'expunge-object object)
  (if (deletedp object) (call-next-method) nil))

(defmethod expunge-object :before ((object container-mixin))
  (dolist (c (object-containers object))
    (remove-object object c)))

(defmethod expunge-object :after ((object id-mixin)) 
  (remhash (slot-value object 'id) *object-table*))

;;;
;;; object-makunbound - unbind object slots for gc
;;;

(defun object-makunbound (object)
  (object-makunbound-using-class (class-of object) object))
  
(defmethod object-makunbound-using-class ((class standard-class) instance)
  (loop for slot in (class-slots class)
        do (slot-makunbound-using-class class instance 
                                        #-allegro-v4.1 slot
                                        #+allegro-v4.1 
                                        (slot-definition-name slot)))
  (values))	

;;;
;;; frozenp, freeze-object, unfreeze-object
;;;

(defgeneric frozenp (element))
(defgeneric freeze-object (element))
(defgeneric unfreeze-object (element))
  
(defmethod frozenp ((element flags-mixin))
  (logtest +frozen+ (slot-value element 'flags)))
  
(defmethod freeze-object ((element t))
  nil)

(defmethod freeze-object ((element flags-mixin))
  (setf (slot-value element 'flags) 
	(logior (slot-value element 'flags) +frozen+))
  t)

(defmethod unfreeze-object ((element t))
  nil)

(defmethod unfreeze-object ((element flags-mixin))
  (setf (slot-value element 'flags) 
	(logandc2 (slot-value element 'flags) +frozen+))
  t)

  
;;;
;;; hiddenp, hide-object, unhide-object
;;;

(defgeneric hiddenp (element))
(defgeneric hide-object (element))
(defgeneric unhide-object (element))

(defmethod hiddenp ((element flags-mixin))
  (logtest +hidden+ (slot-value element 'flags)))
  
(defmethod hide-object ((element t))
  nil)

(defmethod hide-object ((element flags-mixin))
  (setf (slot-value element 'flags) 
	(logior (slot-value element 'flags) +hidden+))
  t)

(defmethod unhide-object ((element t))
  nil)

(defmethod unhide-object ((object flags-mixin))
  (if (logtest (slot-value object 'flags) +deleted+)
      nil
    (progn (setf (slot-value object 'flags) 
	      (logandc2 (slot-value object 'flags) +hidden+))
           t)))

;;;
;;; map-object maps a specified function over objects.  it has two optional
;;; control parameters. level controls the recursion depth, and defaults to
;;; the global variable *mapping-level*, which should be a number > 0 or T.
;;; the object passed is at level 1. mode controls what type of objects are
;;; mapped. if mode is :data, then basic data is mapped. if mode is
;;; :containers, then only containers are mapped. if mode is :both then 
;;; the mapping function is passed each datum along with its container.
;;;

(defparameter *mapping-level* 1 "Maximum mapping depth.")
(defparameter *mapping-mode* ':data "Map :Data or :Containers")

(defun mapping-level (&optional (level *mapping-level*))
  (cond ((eq level t) 
          most-positive-fixnum)
        ((and (numberp level) (integerp level) (> level 0))
         level)
        (t (error "Bad *MAPPING-LEVEL* value: ~S" level))))

(defun mapping-mode (&optional sys (mode *mapping-mode*))
  (cond ((eq mode ':data)
         (if sys ':both mode))
        ((find mode '(:containers :both))
         mode)
        (t (error "Bad *MAPPING-MODE* value: ~S" mode))))

(defun map-object (function object &key (level *mapping-level*) 
                                        (mode  *mapping-mode*))
  (let ((max (mapping-level level))
        (mod (mapping-mode nil mode)))
    (when (< 0 max)
      (map-object-aux object function NIL mod 0 max))
    (values)))
  
(defmethod map-object-aux ((object t) function container type lev max)
  (declare (ignore lev max) (optimize (speed 3) (safety 0)))
  (if (eq type ':both)
      (funcall function object container)
    (if (eq type ':data)
        (funcall function object)))
  (values))

(defmethod map-object-aux ((object container) function container type lev max)
  (declare (integer lev max) (ignore container)
           (optimize (speed 3) (safety 0)))
  (when (< lev max)
    (when (eq type ':containers)
      (funcall function object))
    (unless (> (incf lev) max)
      (mapc #'(lambda (o) (map-object-aux o function object type lev max))
            (bag-cache (slot-value object 'elements)))))
  (values))

(defmethod map-object-aux ((object algorithm) function container type lev max)
  (declare (ignore container) (integer lev max)
           (optimize (speed 3)(safety 0)))
  (when (and (< lev max) (eq type ':containers))
    (funcall function object))
  (values))

;;;
;;; map-subobjects maps a function across all the subobjects of a container.
;;; recursion is not automatically performed (the function passed in can do
;;; that itself).
;;;

(defgeneric map-subobjects (function object))

(defmethod map-subobjects (function (object container))
  (mapc function (bag-cache (slot-value object 'elements)))
  object)
  
;;;
;;; shuffle-object - randomly reorder the elements of a container
;;;

(defmethod shuffle-object ((object t)  &optional (lb 0) ub)
  #-aclpc (declare (ignore lb ub))
  object)

(defmethod shuffle-object ((object container) &optional (lb 0) ub)
  (let* ((bag (slot-value object 'elements))
         (cache (bag-cache bag)))
    (if ub (incf ub) (setf ub (length cache)))
    (loop with r = (- ub lb)
          for i from lb below ub
          do (rotatef (elt cache i)
                      (elt cache (+ lb (random r)))))
    (setf (bag-tail bag) (last cache))
    object))
  
;;;
;;; retrograde - reverse the order of elements in a container
;;;

(defmethod retrograde-object ((object t) &optional (lb 0) ub)
  #-aclpc (declare (ignore lb ub))
  object)
  
(defmethod retrograde-object ((object container) &optional (lb 0) ub)
  (let ((bag (slot-value object 'elements)))
    (if (and (= lb 0) (not lb))
      (setf (bag-cache bag) (nreverse (bag-cache bag)))
      (let ((tail (nthcdr lb (bag-cache bag)))
            rest)
        (when ub
          (setf tail
                (loop repeat (- (1+ ub) lb)
                      unless tail
                      do (error "~A out of bounds." ub) 
                      collect (pop tail)
                      do (setf rest tail))))
        (setf tail (nreverse tail))
        (when rest (nconc tail rest))
        (if (> lb 0) 
          ; hack around no (setf nthcdr) in Franz.
          (setf (cdr (nthcdr (1- lb) (bag-cache bag))) tail)
          (setf (bag-cache bag) tail))))
    (setf (bag-tail bag) (last (bag-cache bag)))
    object))

;;;
;;; setable-slots
;;;

(defmethod setable-slots ((object standard-object))
  (mapcar #'slot-definition-name (class-slots (class-of object))))

(defmethod setable-slots ((object container))
  nil)

(defmethod setable-slots ((object thread))
  '(start comment))

(defmethod setable-slots ((object algorithm))
  '(rhythm start length end comment))

(defmethod setable-slots ((object rhythmic-element))
  '(rhythm))

;;;
;;; change-object changes an object to a new class. slots in the
;;; newly redfined object may be initialized according to the
;;; slot list: (new1 old1 ...)
;;;

(defgeneric change-object (object new-class slots))

(defmethod change-object (object (new-class standard-class) slots)
  (when slots
    (setf slots
      (loop for (new old) on slots by #'cddr
            when (and (slot-exists-p object old)
                      (slot-boundp object old))
            collect new and collect (slot-value object old))))
  (change-class object new-class)
  (when slots     
    (loop for (new value) on slots by #'cddr
          when (slot-exists-p object new)
          do (setf (slot-value object new) value)))
  object)

;;;
;;;
;;;

(defun set-object (object &rest args)
  (let ((flag t))
    (loop for (slot value) on args by #'cddr
          do
      (if (slot-exists-p object slot)
          (if (find value +unbound-tokens+)
              (slot-makunbound object slot)
            (setf (slot-value object slot) value))
        (setf flag nil)))
     (if flag object nil)))
