;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92, 93 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; send to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

;;;
;;; main object processing functions
;;;

(defun insure-merge (objects offsets)
  ;; if objects is a single merge then use it, otherwise allocate a merge
  ;; and fill it with the objects. objects is treated as a reference
  (let (merge)
    (when (singleref? objects)
      (setf objects (refobject objects)))
    (cond ((typep objects 'merge)
           (setf merge objects objects nil))
          (t
           (remove-all-objects .scratch-score.)
           (setf merge .scratch-score.)))
    (flet ((setstart (obj off)
             (when (and (numberp off) (> off 0.0))
               (setf (slot-value obj 'start) off)
               (setf (slot-value obj 'flags) 
                 (logior (slot-value obj 'flags) +start-unset+)))))
      (if objects
          (progn
            (mapref #'(lambda (o) (add-object o merge)) objects)
            (map nil #'(lambda (o) 
                         (setstart o (if (consp offsets) (pop offsets)
                                         offsets)))
                 (container-objects merge)))
        (setstart merge (if (consp offsets) (pop offsets) offsets))))
    merge))

(defun insure-thread (objects repeat pause)
  ;; if objects is a single thread then use it, otherwise allocate a thread
  ;; and fill it with objects. if consp, objects is treated as a reference
  ;; so pauses are between each reference group, not each individual object.
  (unless repeat (setf repeat 1))
  (when pause
    (unless (consp objects) (setf objects (list objects)))
    (setf objects
          (loop with rest = (make-instance 'rest :rhythm pause)
                while objects
	        for obj = (pop objects)
                collect obj
                when objects
                collect rest)))
  (when (> repeat 1)
    (unless (consp objects) (setf objects (list objects)))
    (setf objects (loop repeat repeat append objects)))
  (when (singleref? objects)
    (setf objects (refobject objects)))
  (if (typep objects 'thread)
    objects
    (progn
      (remove-all-objects .scratch-thread.)
      (mapref #'(lambda (o) (add-object o .scratch-thread.)) objects)
      .scratch-thread.)))

(defun process-object (object inits syntax opener closer)
  (unless syntax (error "Can't process objects without a syntax."))
  (let (stream success?)
    (unwind-protect
      (if (setf stream (apply opener (find-syntax syntax) inits))
          (progn
            (initialize-stream-for-processing stream)
            (write-event object stream)
            (deinitialize-stream-for-processing stream)
            (setf success? t))
         nil)
      (when stream
        (funcall closer stream (if success? nil ':error))
      (when success?
        (post-process-stream stream inits))))))

(defun listen-object (objects &optional (offsets 0.0) &rest pairs)
  (when pairs (setf pairs (canonicalize-pairs pairs nil)))
  (let ((syntax (pair-value 'syntax pairs *syntax*))
        (merge (insure-merge objects offsets)))
    (process-object merge pairs syntax #'open-listener #'close-listener)))

(defun slisten-object (objects &optional (offsets 0.0) &rest pairs)
  (when pairs (setf pairs (canonicalize-pairs pairs nil)))
  (let ((syntax (pair-value 'syntax pairs *syntax*))
        thread merge start repeat pause)
    (if (consp offsets)
      (setf start (pop offsets) repeat (pop offsets) pause (pop offsets))
      (setf start offsets))
    ;; we always want to process in a merge to allow sprouting and
    ;; dynamic events, like midi note-offs, etc.
    (setf thread (insure-thread objects repeat pause))
    (setf merge (insure-merge thread start))
    (process-object merge pairs syntax #'open-listener #'close-listener)))

(defun write-object (objects file &optional (offsets 0.0) &rest pairs)
  (when pairs (setf pairs (canonicalize-pairs pairs nil)))
  (let ((syntax (pair-value 'syntax pairs *syntax*))
        (merge (insure-merge objects offsets)))
    (setf pairs (list* 'pathname file pairs))
    (process-object merge pairs syntax 
                    #'open-event-stream #'close-event-stream)))

(defun swrite-object (objects file &optional (offsets 0.0) &rest pairs)
  (when pairs (setf pairs (canonicalize-pairs pairs nil)))
  (let ((syntax (pair-value 'syntax pairs *syntax*))
        thread merge start repeat pause)
    (if (consp offsets)
      (setf start (pop offsets) repeat (pop offsets) pause (pop offsets))
      (setf start offsets))
    ;; we always want to process in a merge to allow sprouting and
    ;; dynamic events, like midi note-offs, etc.
    (setf pairs (list* 'pathname file pairs))
    (setf thread (insure-thread objects repeat pause))
    (setf merge (insure-merge thread start))
    (process-object merge pairs syntax
                    #'open-event-stream #'close-event-stream)))

(defun map-events (function object &optional (offsets 0.0) &rest pairs)
  (when pairs (setf pairs (canonicalize-pairs pairs nil)))
  (let ((merge (insure-merge object offsets))
        (start (pair-value 'start pairs))
        (end (pair-value 'end pairs))
        element)
    ;; optimize out start end checks where possible
    (unwind-protect
      (if start	
          (if end
              (loop while (and (setf element (score-select merge start))
                               (<= (object-time element) end))
                    unless (< (object-time element) start)
                    do (funcall function element))
            (loop while (setf element (score-select merge start))
                   unless (< (object-time element) start)
                   do (funcall function element)))
        (if end
            (loop while (and (setf element (score-select merge start))
                             (<= (object-time element) end))
                  do (funcall function element))
          (loop while (setf element (score-select merge start))
                do (funcall function element))))
      (score-unset object t))
    (values)))

(defun run-object (objects &optional (offsets 0.0) &rest pairs)
  (apply #'map-events #'(lambda (x) x) objects offsets pairs))

(defun list-object (ref &optional (stream *list-stream*)) 
  (let ((cnt 0)
        (max (or *listing-length* most-positive-fixnum)))
    (multiple-value-bind (con lb ub by wid)
                         (destructureref ref)
      (unless (container-objects con)
        (tell-user "~:(~A~) does not have any objects to list."
                   (object-name con))
        (return-from list-object nil))
      (unless (numberp ub) (setf ub (object-count con)))
      (unless by (setf by 1))
      (unless lb 
        (setf lb 0
              ref (makeref con 0 (1- (min ub max)))))
      (format stream "~:(~A~):~%" (object-name con))
      (block printblock
        (mapref 
         #'(lambda (o)
             (let ((f (or (careful-slot-value o 'flags 0) 0)))
               (if (eq o *focus*)
                 (write-string " > " stream)
                 (write-string "   " stream))
               (format stream "~3D. ~S ~@[(~A)~]~&" (+ 1 lb cnt)
                       o (if (/= f 0) (status-string f))) 
               (if wid
                 (progn (setf cnt (mod (1+ cnt) wid))
                        (when (= cnt 0) (incf lb by)))
                 (incf lb by))))
         ref))
      (when (< max ub)
        (format stream "More unlisted objects.")))))

(defun tlist-object (object &optional (offset 0.0) &rest pairs)
  (let (last-can last-time counts num)
    (flet ((tlistit (object)
           (let ((can (slot-value-or-default object 'container last-can))
                 (tim (object-time object)))
             (unless (eq can last-can)
               (setf last-can can 
		     num (or (assoc can counts)
		             (let ((y (cons can 0)))
                               (push y counts)
                                y)))
               (write-string (object-namestring can) *list-stream*)
               (terpri *list-stream*))
             (if (and (eq can last-can) (equal tim last-time))
                 (format *list-stream* "~8T  ~3D. ~S~%" 
                         (incf (cdr num)) object)
               (format *list-stream* "~8,2F  ~3D. ~S~%" 
                       tim (incf (cdr num)) object))
             (setf last-time tim))))
      (apply #'map-events #'tlistit object offset pairs))))

(defun show-object (object &optional all?)
  (format *show-stream* "~%~:[Object~;Focus~]:~10T ~:(~A~)~%" 
          (eq object *focus*) 
          (if (typep object 'id-mixin)
            (object-name object) object))
  (format *show-stream* "Type:~10T~:(~A~)~%" 
          (class-name (class-of object)))
  (when (typep object 'flags-mixin)
    (format *show-stream* "Status:~10T~A~%" 
            (status-string (slot-value object 'flags))))
  (mapref #'(lambda (c &aux p)
              (setf p (1+ (object-position object c)))
              (format *show-stream* "Position:~10T~A in ~:(~A~)~%"
                      p (object-name c)))
          (careful-slot-value object 'container))
  (let ((count (object-count object)))
    (when count
      (format *show-stream* "Objects:~10T~A~%" count)))
  (when (typep object 'container)
    (unless (systemp object)
      (format *show-stream* "Start:~10T~A~%" 
              (slot-value-or-default object 'start +slot-unset+))))
  (when all?
    (cond ((typep object 'algorithm)
           (format *show-stream* "Code:")
           (pprint (slot-value object 'external) *show-stream*)
           (terpri *show-stream*))
          ((not (typep object 'container))
           (let ((class (class-of object)))
             (format *show-stream* "Slots:~%")
             (loop for slot in (class-slots class)
	           for name = (slot-definition-name slot)
	           for label = (string-capitalize (symbol-name name))
                   maximize (length label) into tab
	           collect (cons name label) into showthem
	           finally
                   (dolist (s showthem)
                     (format *show-stream* "~10T~A~VT~:[~A~;~S~]~%" 
                             (cdr s) (1+ (+ tab 10))
                             (slot-boundp object (car s))
                             (slot-value-or-default object (car s) 
                                                    +slot-unset+)))))))))

(defun status-string (flags &optional (normal "Normal"))
  (cond ((= flags 0) normal)
        ((logtest flags +deleted+) "Deleted")
	(t
	 (cond ((and (logtest flags +hidden+) (logtest flags +frozen+))
  	        "Hidden, Frozen")
	       ((logtest flags +hidden+) "Hidden")
	       ((logtest flags +frozen+) "Frozen")
	       ((logtest flags +system+) "System")))))
;;;
;;;
;;;
