;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92, 93 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; send to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(defvar *container* nil)
(defvar *element* nil)
(defparameter cm::*part-instance-variable* '*element*)

;;;
;;;
;;;

(defun status? (status &optional (object *container*))
  (ecase status
    (:chording (logtest (slot-value object 'last-status) +chording+))
    (:resting (logtest (slot-value object 'status) +resting+))
    (:ending (logtest (slot-value object 'status) +ending+))
    (:killed (logtest (slot-value object 'status) +killed+))
    ((:noteon :noteoff :noteupdate)
     (eq (slot-value object 'type) status))
    (:unqueued (logtest (slot-value object 'status) +unqueued+))))

;;;
;;; make-object returns a new or redefined element. spec can be either
;;; a symbol, in which case is the class of the element to create, or
;;; else a list (class id), in which case if id names an already existing
;;; element then this element will be destructively modified according
;;; to the initialization list, otherwise a new instance is created with
;;; the specified id.
;;;

(defun make-object (spec &rest inits)
  (let (class id? id object)
    (if (consp spec)
        (progn 
          (setf id? (consp (cdr spec))) 
          (setf class (find-class (car spec)) id (cadr spec))
          (when id (setf object (find-object id nil))))
      (setf class (find-class spec)))
    (if object
        (progn (unless (eq class (class-of object))
                 (change-class object class))
               (apply #'redefine-object object :id id inits))
      (if id?			; id could be explicitly nil
          (setf object 
            (let ((new (apply #'make-instance class :id id inits))) 
              (when (and id (not *merge*))
                (update-top-level :add new))
              new))
        (setf object (apply #'make-instance class inits))))
    (when *container* (add-object object *container*))
    object))

;;;
;;; redefine object "remakes" an object, ie after the redefinition the
;;; object should behave as if it were actually a new instance, but it
;;; is still eq to its old form.
;;;

(defparameter *warn-if-redefine-object* t)
(defparameter *break-if-no-obvious-termination*	nil)

(defmethod redefine-object ((instance standard-object) &rest inits)
  (when *warn-if-redefine-object* (warn "Redefining ~S." instance))
  (object-makunbound instance)
  (apply #'initialize-instance instance inits)
  instance)
  
(defmethod redefine-object :before ((object container) &rest initargs)
  (declare (ignore initargs))
  (remove-all-objects object))

;;; when we redefine an object we still want it to contain the same
;;; backpointers to its containers as it had before the redefinition.

(defmethod redefine-object :around ((object container-mixin) &rest initargs)
  #-aclpc (declare (ignore initargs))
  (let ((container (slot-value-or-default object 'container))
        (redefined (call-next-method)))
    (when container
	  (setf (slot-value redefined 'container) container))
    redefined))	  
;;;
;;;
;;;

(defmacro element (type &rest initargs)
  (warn "Note: the macro ELEMENT continues to work but has been renamed to OBJECT to avoid confusion in the documentation. Please use OBJECT.")
  `(object ,type ,@initargs))

(defmacro object (type &rest initargs)
  (let ((class (find-class type nil)))
    #+aclpc (unless (class-finalized-p class) (finalize-inheritance class))
    (unless (and class (typep (class-prototype class) 'element))
      (error "~A is not an element class." type))
    `(make-object ',type ., (expand-initargs class initargs))))  

;;;
;;;
;;;
  
(defun expand-container (type name inits body)
  (unless (consp name) (setf name `(quote ,name)))
  `((lambda (*container*) ,@body *container*)
    (make-object (list ',type ,name)
	            .,(expand-initargs (find-class type) inits))))
  
(defmacro thread (name inits &body body &environment environment)
  (declare (ignore environment))
  (expand-container 'thread name inits body)) 

(defmacro heap (name inits &body body &environment environment)
  (declare (ignore environment))
  (expand-container 'heap name inits body)) 

(defmacro merge (name inits &body body &environment environment)
  (declare (ignore environment))
  (expand-container 'merge name inits body))

(defun print-warnings (type unknown-setfs unknown-evals terminates)
  (flet ((notify (type mode unknowns)
         (let ((num (if (= (length unknowns) 1) 0 1)))
           (warn  "A ~(~a~) definition ~(~a~) the variable~[~;s~]~{ ~S,~} which ~[is~;are~] not local to the ~(~a~)'s code or globally declared.~@[~%Archiving this object will likely cause problems upon reloading.~]~%" type mode num unknowns num type
                             (eq mode 'references )))))
    (when unknown-setfs
      (notify type 'sets unknown-setfs))
    (when unknown-evals
      (notify type 'references unknown-evals)))
  (unless terminates
    (let ((str (format nil "A nonterminating ~(~A~) is probably being defined. You may have forgotten to specify a kill statement or to provide initial values for the end or length slots. Unless you have specified some stopping condition, this ~(~A~) will produce ~A notes." 
                      type type most-positive-fixnum)))
      (if *break-if-no-obvious-termination* 
          (error str)
        (warn str)))))

#-mcl
(defun check-for-unknown-variables (form context environment)
  (declare (special unknown-setfs unknown-evals))
  (when (and (symbolp form)
             (not (or (constantp form)
                      (variable-lexical-p form environment)
                      (variable-special-p form environment)
                      (variable-globally-special-p form))))
    (if (eq context ':set)
        (pushnew form unknown-setfs)
      (if (eq context ':eval)
        (pushnew form unknown-evals)))))

#+mcl ; disabled in mcl until i figure out why it warns about evaling slots
      ; names inside with-slots! i think variable-lexical-p is failing...
(defun check-for-unknown-variables (form context environment)
  (declare (special form unknown-setfs unknown-evals)
           (ignore  context environment))
  (values))

(defun expand-algorithm (type name note-class inits body whole environment)
  (unless note-class (setf note-class 'rhythmic-element))
  (let ((iclass (find-class note-class nil))
        (gclass (find-class type))
        (saved inits)
        (terminates? nil)
        ginitargs iinitargs form)
    #+aclpc (unless (class-finalized-p iclass) (finalize-inheritance iclass))
    #+aclpc (unless (class-finalized-p gclass) (finalize-inheritance gclass))
    (unless (and iclass (typep (class-prototype iclass) 'element))
      (error "~S is not a legal element class." note-class))
    (loop with gslots = (class-slots gclass)
          and  islots = (class-slots iclass)
          and arg and init and value
          while inits
          do (setf init (pop inits))
             (setf arg (if (constantp init) init `(quote ,init)))
             (if inits 
                 (setf value (pop inits))
               (error "Uneven initialization list: ~S" saved))
             (unless terminates?
               (setf terminates? 
                 (and (member init '(length events end :length :events :end) 
                              :test #'eq)
                      t)))
          if (find init gslots :key #'slot-definition-initargs :test #'member)
          nconc (list arg value) into ginits
          else
          if (find init islots :key #'slot-definition-initargs :test #'member)    
          nconc (list arg value) into iinits
          else 
          do (error "~S is not a legal initialization argument for ~A or ~A."
                        init (class-name iclass) (class-name gclass))
          finally (setf ginitargs ginits iinitargs iinits))

    ;; variable references to count, length, status, and last-status in the
    ;; body of the algorithm function are resolved by with-accessors forms
    ;; in the algorithm object.  any other slot references are resolved by
    ;; with-slots in the note-class instance.

    (let* ((gen '*container*)                        ;(gentemp "G")
           (cm::*part-instance-variable* '*element*) ;(gentemp "O")
           (cm::*part-status-variable* `(algorithm-status ,gen))       
           (cm::*part-status-accessor* `(algorithm-status ,gen))          
           (cm::*part-last-status-accessor* `(algorithm-last-status ,gen))
           (cm::*let-bindings* '())
           (cm::*kill-flag* terminates?)
           (container-slot-names (mapcar #'slot-definition-name
                                        (class-slots gclass)))
           (object-slot-names (mapcar #'slot-definition-name
                                        (class-slots iclass)))
;           (unknown-setfs ())
;           (unknown-evals ())
           fun)
      (declare
        (special cm::*part-walking* cm::*let-bindings* 
                 cm::*part-status-accessor* cm::*part-status-variable*
                 cm::*part-last-status-accessor* cm::*kill-flag*
;                 unknown-setfs unknown-evals known-globals
                 ))
      (flet ((parse-vars (vars)
               (declare (special cm::*let-bindings*))
               (dolist (spec (cdr vars)) ; car = VARS
                 (if (consp spec)
                     (let ((old spec)
                           (var (pop spec)) 
                           (val (pop spec)))
                       (unless (and (symbolp var) (not (constantp var)))
                         (error "~S cannot be a variable." var))
                       (when spec
                         (error "~S is not a variable binding." old))
                       (push (list var (macroexpand val)) cm::*let-bindings*))
                   (progn 
                     (unless (and (symbolp spec) (not (constantp spec)))
                         (error "~S cannot be a variable." spec))
                     (push spec cm::*let-bindings*)))))
             (walk-algorithm (form context environment)
               (if (and (listp form)
                        (eq (car form) 'item)
                        (eq context :eval))
                   (progn 
;                     (check-for-unknown-variables (cadr form)
;                                                  context environment)
                      (cm::walk-item form environment)

                     )
                 (progn 
;                   (check-for-unknown-variables form context environment)
                   (values form nil))))
             #+mcl
             (slot-references (form env object-slots container-slots)
                (let ((objp nil)
                      (conp nil))
                  (walk-form form env
                    #'(lambda (form context env &aux x)
                        (when (and form 
                                   (symbolp form)
                                   (or (eq context :set) 
                                       (eq context :eval))
                                   (not (variable-lexical-p form env)))
                          (unless objp
                            (setf objp 
                                  (setf x (find form object-slots))))
                          (unless (or conp x)
                            (setf conp 
                                  (find form container-slots))))
                        (values form nil)))
                  (if (and objp conp) #b11
                      (if conp #b01 
                          (if objp #b10 #b00))))))

        ;; allow first form in body to be a declare statement. we only
        ;; support a LOCAL clause, which adds user specified variables
        ;; to the binding list.
        (when (and body (consp (car body)) (eq (caar body) 'vars))
          (parse-vars (pop body)))

        ;; compute a runtime function by walking the body of the algorithm.
        ;; rewrite variable references in the body as slot accessing forms,
        ;; and item stream references as references to lexical variables
        ;; that are bound to the stream at initialization time. make sure
        ;; that algorithm slots "shadow" instance slots
 
        (let ((bits #-mcl #b11
                    #+mcl (slot-references `(progn ,@body)
                                           environment
                                           object-slot-names
                                           container-slot-names)))
          ;; bits=#b11, body uses object and container
          ;; bits=#b01, body uses container
          ;; bits=#b10, body uses object

          (when (logtest bits #b01)
            (setf body `((with-slots ,container-slot-names 
                                     ,gen 
                           ,@ body))))
          
          (when (logtest bits #b10)
            (setf body `((with-slots ,object-slot-names
                                     ,cm::*part-instance-variable* 
                           ,@ body))))
          ;; wrap everything in a lambda and walk it...
          (setf fun
            (walk-form `(lambda (,gen ,cm::*part-instance-variable*)
                          ,@body)
                       environment #'walk-algorithm))))
      (unless (consp name) (setf name `(quote ,name)))
      (setf form
       `(make-object (list ',type ,name)
                     :class (find-class ', note-class)
                     :instance-init #'(lambda (*element*)
                                        (initialize-instance *element*
                                                             .,iinitargs)
                                        *element*)
                     :function-init #'(lambda ()
                                        ,(if cm::*let-bindings* 
                                             `(let* ,(nreverse 
                                                       cm::*let-bindings*)
                                                (function ,fun))
                                           `(function ,fun)))
                     :external ',whole
                     ., ginitargs))
;      (if (or unknown-setfs unknown-evals (not cm::*kill-flag*))
;          `(progn 
;             (print-warnings ',type ',unknown-setfs
;                             ',unknown-evals ',cm::*kill-flag*)
;             ,form)
;         form)

      (unless cm::*kill-flag* (print-warnings type nil nil cm::*kill-flag*))
      form))) 

;;;
;;; algorithm, mute and generator
;;;

(defmacro algorithm (&whole whole name note-class inits &body body
                     &environment environment)
  (expand-algorithm 'algorithm name note-class inits body whole environment))

(defmacro mute (&whole whole name inits &body body
                &environment environment)
  (expand-algorithm 'algorithm name nil inits body whole environment))

(defmacro generator (&whole whole name note-class inits &body body
                     &environment environment)
  (expand-algorithm 'generator name note-class inits body whole environment))

;;;
;;; layout
;;;

(defmacro layout (name directive)
  `(parse-layout-directive ',name ',directive))

(defun parse-layout-directive (id ref &optional superior-type num-expected)
  ;; if recursing, superior-type is class we are parsing for, otherwise nil 
  (labels ((parsestart (x)
             (if (numberp x)
               (if (not (< x 0))
                 x
                 (error "negative start time: ~S" x))
               (error "~S is not a start time." x)))
           (parsenum (s p)
             (parsestart (read-from-string (subseq s p)))))
    (cond
     ((listp ref)
      (when num-expected (error "bogus @ in ~S" ref))
      (cond ((member (car ref) '(mix seq))
             (let* ((typ (if (eq (car ref) 'mix) 'merge 'thread))
                    sec refs)
               (setf refs
                 (loop with tail = (cdr ref)
                       with last = nil 
                       and flag = nil                   ; expect num?
                       while tail
                       for form = (pop tail)
                       for this = (parse-layout-directive nil form typ flag) 
                       when (null this)                 ; @
                       do (if (eq flag t) (error "bogus @ in ~S" ref)
                            (setf flag t))               
                       else when (numberp this)         ; 5
                       do (progn 
                            (unless last 
                              (error "start time without object@ in: ~S" ref)) 
                            (setf (cdr last) (parsestart this))
                            (setf flag nil))
                       else when (consp this)
                       collect
                       (progn 
                         (if (eq flag t) (error "bogus @ in ~S" ref) 
                           (setf flag nil))
                         (when (eq (cdr this) :time)    ; foo@
                           (setf (cdr this)
                             (if (numberp (car tail))
                                 (parsestart (pop tail))
                               (error "@ without start time in ~S" ref))))
                         this)
                       and do (setf last this)))
               (setf sec 
                 (make-object (if id (list typ id) typ)
                              :flags (logior +no-link+ +start-unset+)))
               (loop for r in refs
                     collect (car r) into subs 
                     when (numberp (cdr r)) collect r into starts
                     finally (when subs
                               (add-objects subs sec)
                               (when starts 
                                 (setf (slot-value sec 'starts) starts))))
               (if (null superior-type) sec (list sec))))
            (t (error "expected MIX or SEQ but got: ~S" ref))))
     ((symbolp ref)
      (when num-expected (error "bogus @ in ~S" ref))
      (cond ((eq ref '@)                                ; @
          (unless (eq superior-type 'merge)
               (error "@ outside of mix: ~S" ref))
             nil)
            ((char= (elt (symbol-name ref) 0) #\@)      ; @5
             (unless (eq superior-type 'merge)
               (error "@ outside of mix: ~S" ref))
             (parsenum (subseq (symbol-name ref) 1) 0) )
            ((char= (elt (symbol-name ref)              ; foo@
                         (1- (length (symbol-name ref)))) #\@)
             (let ((s (read-from-string (symbol-name ref) nil nil
                                        :start 0 :end (1- (length ref)))))
               (cons (find-object s) :time)))
            (t
             (let* ((s (symbol-name ref))               ; foo or foo@5
                    (p (position #\@ s)))
               (if p 
                 (if (eq superior-type 'merge)
                   (let ((n (read-from-string s nil nil :end p)))
                     (setf ref (cons (find-object n)
                                     (parsenum s (1+ p)))))
                   (error "@ outside of mix: ~S" ref))
                 (setf ref (list (find-object ref))))
               ref))))
     ((numberp ref) 
      ref)
     (t 
      (error "~S is not a layout directive" ref)))))

;;;
;;; sprout
;;;

(defgeneric enqueue-p (object queue))
(defgeneric sprout (object &optional queue))

(defmethod enqueue-p ((object container) queue)
  (<= (object-time queue) (slot-value object 'start)))

(defmethod enqueue-p ((object algorithm) queue)
  (and (<= (object-time queue)
           (slot-value object 'start)
           (slot-value object 'end))
       (< (slot-value object 'count) (slot-value object 'length))
       (not (logtest (slot-value object 'status) +killed+))))

(defmethod enqueue-p ((object timed-object) queue)
  (declare (ignore queue))
  t)

(defmethod sprout (object &optional (queue *merge*))
  (score-reset object 0)    ; i dont understand why this has to be zero
  (if (enqueue-p object queue)
      (progn 
         (enqueue-element object queue t)
        object)
    (progn 
      (format t "Ignoring queue of ~S at start time=~S, queue time=~S~%"
              object (slot-value object 'time) (slot-value queue 'time))
      nil)))

(defmethod sprout :before ((object startable-element) &optional
                             (queue *merge*))
  (if (slot-boundp object 'start)
      (setf (slot-value object 'start) 
            (+ (slot-value object 'start)
	       (slot-value queue 'offset)))
    (error "No start time specified for scheduling ~S" object)))

