;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

;;;
;;; compiling this file without cmn loaded will fail.
;;; 

(defsyntax common-music-notation 
          :nicknames '(cmn)
          :stream-types '((cmn-stream "cmn" "eps"))
          :pathname (pathname cmn::*cmn-output-pathname*))

(defparameter *cmn-previewer* #+(or next |NeXT|) "Preview" 
                              #-(or next |NeXT|) nil)

(defvar *cmn-clef-names*
   '(:french-violin :treble :soprano :mezzo-soprano
     :alto :tenor :baritone :baritone-C :baritone-F
     :bass :sub-bass :percussion :both :none))

;;;
;;; since cmn has no external symbols, and we dont want to try to figure out
;;; which internal ones a user might need for any particular situation,
;;; this hairball insures that we evaluate a cmn directive (a variable or 
;;; function call) in the cmn package, regardless of what package the symbol(s)
;;; in the directive are defined in. so we can type (meter 2 4) as a piece
;;; of cmn object data in the stella package (or any other package), and the
;;; form will be evaluated as if it were typed in the cmn package.
;;;

(defun cmn-eval (form)
  (flet ((cmnfun (sym)
         (let ((fun (find-symbol (string sym) :cmn)))
           (unless (and fun (fboundp fun))
             (error "~S is not a cmn function!" sym))
           fun))
         (cmnvar (sym)
           (let ((var (find-symbol (string sym) :cmn)))
             (unless (and var (boundp var))
               (error "~S is not a cmn variable." sym))
             (symbol-value var))))
   (cond ((consp form) 
          (apply (cmnfun (car form)) (mapcar #'cmn-eval (cdr form))))
         ((constantp form) 
          form)
         (form 
          (cmnvar form))
         (t 
          nil))))

;;;
;;; cmn output streams support .eps or .cmn output depending on the file name
;;; extension of the output file.  all of the user settable slots except Staves
;;; are inherited from the cmn score object. Staves (as opposed to cmn::staves)
;;; holds a list of staff specifications for manuscript layout.  the value of
;;; this slot is specified when the stream is created or modified using 
;;; Staff command under Edit.
;;;

(defobject cmn-stream (event-file cmn::score)
  ((syntax :initform (find-syntax ':cmn))
   (staves :initform nil :initarg staves :initarg :staves) ; not cmn::staves
   (staffer :initarg :staffer :initarg staffer :initform 'container-staff)
   (marker :initarg :marker :initarg marker)
   (meter :initarg :meter :initarg meter)
   (metronome :initform *standard-tempo* :initarg metronome
              :initarg :metronome)
   (size :initform 24 :initarg size :initarg :size))
  (:parameters start end timescale 
               metronome size meter staves staffer marker))

(defmethod syntax-default-pathname ((syntax common-music-notation) 
                                    (stream cmn-stream)) 
  cmn::*cmn-output-pathname*)

(defmethod open-event-stream ((syntax common-music-notation) &rest args)
  (apply #'open-event-stream (find-class 'cmn-stream) args))

;;;
;;; we override the main open/close methods because clm manages all the
;;; actual file io itself via init-with-sound and finish-with-sound.
;;;

(defmethod close-event-stream ((stream cmn-stream) &optional mode)
  #-aclpc (declare (ignore mode))
  t)

(defmethod open-event-stream ((stream cmn-stream) &rest args)
  (declare (ignore args))
  stream)

#-(and clisp (not clos))
(defmethod tl:edit-object :around ((object cmn-stream) &rest args)
  (declare (ignore args))
  (call-next-method object :slots t
                   :ignore-slots '(flags direction stream element-type 
                                   if-does-not-exist if-exists syntax
                                   cmn::staves cmn::pages cmn::lines
                                   cmn::line-data cmn::time-lines 
                                   cmn::time-line cmn::systems cmn::x cmn::x0
                                   cmn::x1 cmn::y cmn::y0 cmn::y1)
                   :slot-commands '((staves :command-name "STAVES" ))))

#+(and clisp (not clos))
(defmethod tl:edit-object ((object cmn-stream) &rest args)
  ;; this is probably most horrible hack i have ever written. it saves
  ;; me performaing major surgery on code that should work in any clos
  ;; implementation worth a damn. unfortunately, clisp's isn't, hence
  ;; this end run around its call-next-method limitation.
  (declare (ignore args))
  (let ((method (find-method #'tl:edit-object nil 
                             (list (find-class 'standard-object)))))
    ;; lifted from clos.lsp
    (when (null (clos::std-method-function method))
      (let ((h (funcall (clos::std-method-initfunction method) method)))
        (setf (clos::std-method-function method) (car h))
        (when (car (cdr h)) ; konnte die Variable ",cont" wegoptimiert werden?
          (setf (clos::std-method-wants-next-method-p method) nil))))
    ;; funcall the methods function object.
    (funcall (clos::std-method-function method) #'tl:edit-object
             object :slots t
            :ignore-slots '(flags direction stream element-type 
                            if-does-not-exist if-exists syntax
                            cmn::staves cmn::pages cmn::lines
                            cmn::line-data cmn::time-lines 
                            cmn::time-line cmn::systems cmn::x cmn::x0
                            cmn::x1 cmn::y cmn::y0 cmn::y1)
            :slot-commands '((staves :command-name "STAVES")))))

;;;
;;; staff descriptors
;;;

(defstruct (stfd (:type list)) label objects clefs meter)

(defun print-stfd (stfd &optional (stream t))
  (format stream "#<Staff ~A ~S clefs=~A meter=~A>"
          (stfd-label stfd) (stfd-objects stfd) 
          (stfd-clefs stfd)
          (stfd-meter stfd)))

(defun staff (objects &key clef name meter recursive)
  (unless (listp objects)
    (setf objects (list objects)))
  (unless name
    (setf name
      (if (= (length objects) 1)
          (cond ((typep (car objects) 'id-mixin)
                 (object-namestring (car objects)))
                ((or (stringp (car objects)) 
                     (symbolp (car objects)))
                 (string (car objects)))
                (t (prin1-to-string (car objects))))
         (string (gentemp "STF-")))))

  ;; coerce container names to the actual objects
  (when (typep (car objects) 'symbol)
    (setf objects (mapcar #'find-object objects)))

  ;; if recursive, add all subcontainers as well
  (when (and recursive (typep (car objects) 'id-mixin))
    (let ((all '()))
      (dolist (o objects) 
        (map-object #'(lambda (c) (push c all))
                    o :mode ':containers :level t))
      (setf objects (nreverse all))))
  (setf clef
    (loop for c in (if (listp clef) clef (list clef)) 
          collect
       (find c *cmn-clef-names* :test #'string-equal :key #'string)))
  (make-stfd :label name :objects objects :clefs clef :meter meter))

(defmacro staves (&rest args)
 `(list ,@ (loop for c in args
                 collect (if (listp c) 
                             (cons 'staff (mapcar #'cm::quote-if-necessary c))
                           (list 'staff (cm::quote-if-necessary c))))))

 
;;;
;;; cmn-stream protocol
;;;

(defvar *cmn-staves* (make-hash-table :test #'eq)
  "Links containers to cmn staves.")
  
(defmethod initialize-stream-for-processing ((stream cmn-stream))
  (let ((score (cmn::init-clm-input stream)))
    (setf (cmn::output-file score) 
      (namestring (slot-value stream 'pathname)))
    (clrhash *cmn-staves*)
    ;; set up any user specifed staves
    (loop for staff in (slot-value stream 'staves)
          do (let ((label (stfd-label staff)))
               (dolist (x (stfd-objects staff))
                 (setf (gethash x *cmn-staves*)
                       (cmn::add-staff stream label nil)))))
    (tell-user "~%Creating cmn score...")))

(defun set-staff-meter (staff meter)  
  (let ((stfd (find staff cmn::staff-descriptors :key #'cmn::stfdat-staff)))
    (setf (cmn::staff-data (cmn::stfdat-staff stfd))
      (append (cmn::staff-data (cmn::stfdat-staff stfd))
        (list (cmn-eval meter))))))

(defun set-meter (meter) 
  (loop for stf in cmn::staff-descriptors do
    (setf (cmn::staff-data (cmn::stfdat-staff stf))
      (append (cmn::staff-data (cmn::stfdat-staff stf))
              (list (cmn-eval meter))))))

(defmethod deinitialize-stream-for-processing ((stream cmn-stream))
  (let* ((path (slot-value stream 'pathname))
         (type (if (string-equal (pathname-type path) "CMN") :CMN NIL))
         (gmeter (slot-value-or-default stream 'meter))
         (staves ()))
    (if (eq type ':cmn)
        (tell-user "~&Saving ~A~&" (namestring path))
      (tell-user "~&Manuscripting ~A..." (namestring path)))
    (setf staves
      (loop for staff in (slot-value stream 'staves) 
            for name = (stfd-label staff)
            for clefs = (stfd-clefs staff)
            for meter = (or (stfd-meter staff) gmeter)
            collect `(cmn::set-staff-number ,name , (if (member ':both clefs)
                                                        2 1))
            if clefs collect `(cmn::set-staff-clef ,name ,@clefs)
            if meter collect `(set-staff-meter ,name ,meter)))
    (apply #'cmn::finish-clm-input stream type nil staves)
    (format t "~&Done!")))

(defmethod post-process-stream ((stream cmn-stream) inits)
  (declare (ignore inits))
  (let ((p (slot-value stream 'pathname)))
    (unless (string-equal (pathname-type p) "CMN")
      (when *cmn-previewer*
        (let ((f (namestring p)))
          (when (ask-y-or-n :prompt (format nil "Display file ~A? " f)
                            :default 'yes)))))))

(defmethod post-process-stream ((stream cmn-stream) inits)
  (let* ((file (namestring (slot-value stream 'pathname)))
         (load (string-equal (pathname-type file) "CMN"))
         (doit (pair-value (if load 'load 'display)
                           inits *post-processing*)))
    (when (eq doit ':ask)
      (setf doit
            (if *command-prompting*
              (ask-y-or-n :prompt (format nil "~:[Display~;Load~] file ~A? "
                                          load file)
                          :default 'yes)
              nil)))
    (when doit
      (if load 
        (load-using-syntax  (slot-value stream 'syntax) file)
        (cm::shell (format nil "open -a ~A ~A" *cmn-previewer* file))))))

(defmethod load-using-syntax ((syntax common-music-notation) file &rest pairs)
  (when pairs (setf pairs (canonicalize-pairs pairs nil t ':keyword)))
  (apply #'load file pairs))

(defun link-container-to-staff (score container &optional name (recurse t))
  (unless name (setf name (object-namestring container)))
  (let ((staff (or (cmn::find-staff score name)
                   (cmn::add-staff score name nil))))
    ;; link this container and all inferiors to the staff
    (if recurse
        (map-object #'(lambda (c)
                        (let ((o (gethash c *cmn-staves*)))
                          (unless o (setf (gethash c *cmn-staves*) staff))))
                    container :mode ':containers :level t)
      (unless (gethash container *cmn-staves*) 
        (setf (gethash container *cmn-staves*) staff)))     
    staff))

(defun container-staff (score object &optional staff-name)
  (let* ((container (slot-value object 'container))  ; assumes only one!
         (staff (gethash container *cmn-staves*)))
    (or staff
        (progn
          (unless staff-name
            (setf staff-name (object-namestring container)))
          (setf staff (or (cmn::find-staff score staff-name)
                          (cmn::add-staff score staff-name nil)))
          (setf (gethash container *cmn-staves*) staff))     
          staff)))

(defun channel-staff (score object &optional staff-name)
  (let* ((channel (slot-value object 'channel)) 
         (staff (gethash channel *cmn-staves*)))
    (or staff            ; staff already exists
        (progn
          (unless staff-name     ; autocreate staff name
            (setf staff-name (prin1-to-string channel)))
          (setf staff (or (cmn::find-staff score staff-name)
                          (cmn::add-staff score staff-name nil)))
          (setf (gethash channel *cmn-staves*) staff)     
          staff))))

;;;
;;; default method for cmn-stream does nothing
;;;

(defmethod write-event ((object t) (stream cmn-stream))
  nil)

;;;
;;; allow marks and other data after each note
;;;

(defmethod write-event :after ((object timed-object) (stream cmn-stream))
  (when (slot-boundp stream 'marker)
    (funcall (slot-value stream 'marker) stream object)))

;;;
;;; the cmn object holds cmn directives other than note, rhythm and amplitude 
;;; values, which are normally part of the stella objects we want to
;;; manuscript. it has a single slot, Data, which holds a LIST of forms 
;;; (cmn variables and function calls) that we want evaluated as cmn 
;;; directives. as a convient shorthand for the most common case, the value of 
;;; the data slot may also be a just single cmn varible name.
;;;

(defobject cmn (element  #+MIDI channel-mixin) 
  ((data :initarg data :initarg :data :initform nil))
  (:parameters data #+MIDI channel))

(defmethod print-object ((object cmn) stream)
  (printing-random-thing (object stream)
    (format stream "CMN ~(~A~)" (slot-value object 'data))))

(defmethod write-event ((object cmn) (stream t)) )

(defmethod write-event ((object cmn) (stream cmn-stream))
  (let* ((staff (funcall (slot-value stream 'staffer) stream object))
         (data (slot-value object 'data)))
      (if (listp data)
          (dolist (form data)
            (cmn::add-data-1 stream staff (cmn-eval form)))
        (cmn::add-data-1 stream staff (cmn-eval data)))))

(defclass tempo (cmn)
  ((pulse :initarg pulse :initarg :pulse)
   (draw? :initarg draw? :initarg :draw? :initform t)))

(defmethod print-object ((object tempo) stream)
  (printing-random-thing (object stream)
    (format stream "TEMPO ~A" 
           (slot-value-or-default object 'data +slot-unset+))))

(defmethod write-event ((object tempo) (stream cmn-stream))
  (cmn::set-tempo stream 
                  (funcall (slot-value stream 'staffer) stream object)
                  (slot-value object 'time)
                  (slot-value object 'data)
                  (slot-value-or-default object 'pulse nil)
                  (slot-value object 'draw?)))

#+MIDI
(defmethod write-event ((object midi-note) (stream cmn-stream))
  (let ((staff (funcall (slot-value stream 'staffer) stream object))
        (timescale (slot-value-or-default stream 'timescale)))
    (unless staff
      (error "~S didn't return a staff when called with ~S and ~S."
              (slot-value stream 'staffer) stream object))
    (cmn::add-note-to-staff stream staff
                            (if timescale 
                                (* (slot-value object 'time) timescale)
                              (slot-value object 'time))
                            (if timescale 
                                (* (slot-value object 'duration) timescale)
                              (slot-value object 'duration))
                            (let ((nte (slot-value object 'note)))
                              (cmn-eval (if (integerp nte) (note nte) nte))))))

