;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(defsyntax common-lisp-music 
   :pathname (pathname "test.clm")
   :stream-types '((clm-sound-file "snd") (clm-event-file "clm"))
   :nicknames '(clm))

(defclass snd-mixin ()
  (;(output :initarg output :initarg :output :initarg sndfile :initarg :sndfile 
   ;        :initform clm:default-sound-file-name)
   (channels :initarg channels :initarg :channels :initform 1)
   (srate :initarg srate :initarg :srate :initarg sampling-rate
          :initarg :sampling-rate :initform clm:sampling-rate)
   (continue-old-file :initarg continue-old-file :initarg :continue-old-file)
   (reverb :initarg reverb :initarg :reverb)
   (reverb-data :initarg reverb-data :initarg :reverb-data)
   (reverb-channels :initarg reverb-channels :initarg :reverb-channels
                    :initform 1)
   (revfile :initarg revfile :initarg :revfile) 
   (play :initarg play :initarg :play :initform ':ask)
   (play-options :initarg play-options :initarg :play-options)
   (cleanup-first :initarg cleanup-first :initarg :cleanup-first :initform t)
   (wait :initarg wait :initarg :wait)
   (notehook :initarg notehook :initarg :notehook)
   (statistics :initarg statistics :initarg :statistics)
   (decay-time :initarg decay-time :initarg :decay-time :initform 1.0)
   (output-buffer-size :initarg output-buffer-size 
                       :initarg :output-buffer-size)
   (info :initarg info :initarg :info)
   (type :initarg type :initarg :type)
   (force-recomputation :initarg force-recomputation 
                        :initarg :force-recomputation)
   (verbose :initarg verbose :initarg :verbose :initform clm:*clm-verbose*)
   (comment :initarg comment :initarg :comment
            :initarg commentary :initarg commentary)))

(defobject clm-event-file (event-file header-mixin snd-mixin)
  ((syntax :initform (find-syntax ':clm)))
  (:parameters start end timescale))
   
(defobject clm-sound-file (event-file snd-mixin)
  ((syntax :initform (find-syntax ':clm))
   (element-type :initform '(unsigned-byte 32)))
  (:parameters start end timescale
               srate channels reverb reverb-data 
               decay-time ))

(defmethod initialize-instance :after ((stream clm-sound-file) &rest args)
  (declare (ignore args))
  (setf (slot-value stream 'pathname)
    (if (slot-boundp stream 'pathname)
        (merge-pathnames (slot-value stream 'pathname)
                         default-sound-file-name))))

;;;
;;; we override the main open/close methods because clm manages all the
;;; actual file io itself via init-with-sound and finish-with-sound.
;;;

(defun tell-snd (file channels srate reverb decay-time reverb-data)
  (tell-user "~&File: ~A~%~
              Channels: ~D~%~
              Srate: ~F~%~
              Reverb: ~A~@[, decay time: ~S ~]~@[, reverb data: ~S ~]~%"
	      file
	      (floor channels)
	      srate
	      (or reverb "None") 
              (and reverb decay-time) 
              (and reverb reverb-data)))

(defmethod open-event-stream ((stream clm-sound-file) &rest args)
  ;; dont open file if just initializing slots
  (unless (pair-value 'open args t)
    (return-from open-event-stream stream))
  (let ((args (collecting-slots stream
                 &key channels srate continue-old-file 
                      (reverb :print-if :value) reverb-data
                      reverb-channels revfile play play-options cleanup-first
                      wait notehook statistics decay-time output-buffer-size
                      info type force-recomputation verbose comment)))
    (setf (slot-value stream 'stream)
      (apply #'clm:init-with-sound :output (slot-value stream 'pathname)
             args)))
  (let ((reverb (slot-value-or-default stream 'reverb)))
    (tell-snd (namestring (slot-value stream 'pathname))
              (floor (slot-value stream 'channels))
              (slot-value stream 'srate)
              reverb 
              (slot-value-or-default stream 'decay-time) 
              (slot-value-or-default stream 'reverb-data)))
  stream)
  
(defmethod close-event-stream ((stream clm-sound-file) &optional mode)
  ;; don't play if closing with error or force
  (let ((wsd (slot-value stream 'stream)))
    (when mode (setf (clm::wsdat-play wsd) nil))
    (clm:finish-with-sound wsd))
  (slot-makunbound stream 'stream)
  t)
   
(defmethod initialize-stream-for-processing :after ((stream clm-event-file))
  (let ((s (slot-value stream 'stream)))
    (format s ";;; Common Music output: ~A~&" (cm::date-string))
    (when (slot-boundp stream 'header)
      (write-line ";;; Header" s)
      (format s "~A" (slot-value stream 'header))
      (format s "~%;;; End of Header~&"))))

(defmethod post-process-stream ((stream clm-sound-file) inits)
  ;; clm:finish-with-sound does the sound file postprocessing for us.
  (declare (ignore inits))
  nil)

;;;
;;; clm listening/writing
;;;

(defmethod open-listener ((syntax common-lisp-music) &rest args)
  (if (slot-boundp syntax 'listener)
      (apply #'open-event-stream (slot-value syntax 'listener) args)
    (let ((stream (apply #'open-event-stream 
                         (find-class 'clm-sound-file)
                         'pathname default-sound-file-name
                         'edit nil 'open nil
                         args)))
      (when stream
        (setf (slot-value syntax 'listener) stream))
      stream)))

(defmethod close-listener ((stream clm-sound-file) &optional mode)
  (close-event-stream stream mode))

(defmethod syntax-default-pathname ((syntax common-lisp-music)
                                    (stream clm-sound-file))
  clm:default-sound-file-name) 

(defmethod open-event-stream ((syntax common-lisp-music) &rest args)
  (let ((type (let ((path (or (getf args ':pathname) (getf args 'pathname))))
                (and path (pathname-type path))))
        class)
    (if (and type (string= type "snd"))
        (setf class (find-class 'clm-sound-file))
      (setf class (find-class 'clm-event-file)))
    (apply #'open-event-stream class args)))

(defmethod play-using-syntax ((syntax common-lisp-music)
                              file &rest args)
  (declare (ignore args))
  (when (string-equal (pathname-type file) "SND")
    (dac file)))

(defmethod load-using-syntax ((syntax common-lisp-music) file &rest pairs)
  (when pairs (setf pairs (canonicalize-pairs pairs nil t ':keyword)))
  (tell-snd (namestring (truename file))
            (pair-value :channels pairs 1)
            (pair-value ':srate pairs 22050)
            (pair-value ':reverb pairs)
            (pair-value ':decay-time pairs 1.0)
            (pair-value ':reverb-data pairs))
  (apply #'clm-load file pairs))
