;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(defclass timed-object ()
  ((time :accessor object-time)))

(defclass startable-element (timed-object)
  ((start :initarg :start :initarg start
          :documentation "Local start time.")
   (offset :accessor container-offset
           :documentation "Start time offset.")
   (selected :initform nil :documentation "The currently selected element.")
   (initializer :initarg initializer :initarg :initializer)
   (finalizer :initarg finalizer :initarg :finalizer)))
      
(defclass flags-mixin ()
  ((flags :initform 0 :initarg :flags :initarg flags
          :accessor element-flags :documentation "Status flags.")))

(defclass container-mixin ()
 ((container :documentation "Containing object."
             :accessor object-container)))

(defclass id-mixin ()
  ((id :initarg :id :initarg id :accessor object-name :accessor object-id)))

(defmethod print-object ((object id-mixin) stream)
  (if (slot-value object 'id)
      (format stream "#<~A: ~:(~A~)>" 
              (class-name (class-of object)) (slot-value object 'id))
    (printing-random-thing (object stream)
      (format stream "~A" (class-name (class-of object))))))

(defmethod initialize-instance :after ((object id-mixin) &rest args)
  (declare (ignore args))
  ;; gentemp a new id if unbound. install in namespace if not null
  (unless (slot-boundp object 'id)
    (setf (slot-value object 'id)
      (gentemp (format nil "~A-" (class-name (class-of object))))))
  (when (slot-value object 'id)
    (setf (gethash (slot-value object 'id) *object-table*) object)))

;;;
;;; subclasses of container are used to hold objects.
;;;

(defclass container (startable-element flags-mixin id-mixin)
  ((elements :initform (make-bag) :accessor container-bag
             :documentation "Contained elements.")
   (comment :initarg :comment :initarg comment
            :documentation "Object comment.")))

;;;
;;; note: container-objects is not a slot accessor. it accesses
;;; an internal structure called a bag (utils.lisp)
;;;

(defmethod container-objects ((object container))
  (bag-cache (slot-value object 'elements)))

;;;
;;; merges are containers that treat their elements as parallel streams 
;;; to be merged at selection time.
;;; 

(defvar *merge* nil)	; supports dynamic queuing

(defobject merge (container container-mixin)
  ((starts ))
  (:parameters start comment))


; :initarg :starts :initarg starts
  
;;;
;;; scores are the "top level" containers of musical data.
;;;

(defclass score (merge)
  ())

;;;
;;; threads are containers for any type of score data, ie notes,
;;; rests, threads or merges.  thread data are
;;; sequentially accessed from a structure called a "bag", which
;;; optimizes element insertion. actually, there is nothing preventing 
;;; patterns other than sequence being implemented; potentially 
;;; threads are just subtypes of item streams.
;;;

(defobject thread (container container-mixin)
  ()
  (:parameters start comment))

;;;
;;; a heap is a thread that reshuffles itself every time it is reset.
;;;

(defobject heap (thread)
  ()
  (:parameters start comment))
	
;;;
;;; a generator is a thread that, if it has no elements or is
;;; not "frozen", computes and caches a new element each time it is
;;; requested. if a generator has elements or is frozen then it
;;; behaves exactly like its superclass.
;;;

(defobject algorithm (startable-element flags-mixin id-mixin container-mixin)
  ((count :initform 0 :accessor algorithm-count
          :documentation "Current position in generation.")
   (length :initform most-positive-fixnum :initarg :length :initarg length
           :initarg :events :initarg events  :accessor algorithm-length
           :documentation "Length of generation.")
   (end :initform most-positive-fixnum :initarg :end :initarg end
        :accessor algorithm-end :documentation "End time of generation.")
   (status :initform 0 :accessor algorithm-status
           :documentation "Current state of generator.")
   (last-status :initform 0 :accessor algorithm-last-status
                :documentation "Last state of generator.")
   (class :initarg :class :documentation "Element class to generate.")
   (instance :documentation "Instance template.")
   (function :documentation "Function to create new instances.")
   (function-init :initarg :function-init
                  :documentation "Funcalled to create generator.")
   (instance-init :initarg :instance-init
                  :documentation "Funcalled to create instance.")
   (external :initarg :external :initform nil
             :documentation "External source code."))
  (:parameters start length end ))

(defparameter *compile-algorithms* #+clisp t #-clisp nil)

#-clisp
(defun compileable? (fn) 
  ;; only compile if code resulted in normal lambda expression
  (and (consp fn) (eq (car fn) 'lambda)))

#+clisp
(defun compileable? (fn)
  ;; clisps byte compiler apparently eats anything
  (declare (ignore fn))
  t)

(defmethod initialize-instance :after ((object algorithm) &rest args)
  (declare (ignore args))
  ;; if class is nil we are creating a mute, so we use the basic
  ;; element class for our instance. 
  (let ((class (slot-value-or-default object 'class 'rhythmic-element)))
    (when (symbolp class)
      (setf class (find-class class) (slot-value object 'class) class))
    (setf (slot-value object 'instance)
      (allocate-instance (slot-value object 'class))))
  ;; try to compile simple lambda functions. 
  (when (or *compile-algorithms*
            (logtest (slot-value object 'flags) +compile+))
    (when (compileable? (slot-value object 'instance-init))
      (setf (slot-value object 'instance-init)
        (compile nil (slot-value object 'instance-init))))
    (when (compileable? (slot-value object 'function-init))
      (setf (slot-value object 'function-init)
        (compile nil (slot-value object 'function-init)))))
  ;; if the name of this object is a function call in the code, ie
  ;; (algorithm (name foo) ... ) then we repace the function call form
  ;; with the acual name of the object. 
  (let ((code (slot-value object 'external)))
    (when (consp (second code))
      (setf code (copy-list code)) ;; make setf on macro source code safe !!!
      (setf (second code) (slot-value object 'id))
      (setf (slot-value object 'external) code))))

(defobject generator (algorithm thread)
  ()
  (:parameters start length end comment))
     
;;;
;;; elements are non-aggregate objects.  they cannot "contain" any
;;; sub-objects and they must be contained by at least one object.
;;;

(defclass element (timed-object flags-mixin container-mixin)
  ())




(defclass rhythmic-element (element)
  ((rhythm :accessor object-rhythm  :initarg :rhythm :initarg rhythm
           :documentation "Time increment to next event.")))

;;;
;;; note is the basic superclass for all parameterized note data.
;;;

(defclass note (rhythmic-element)
  ()) 

;;;
;;; rests may be explicitly inserted in threads.
;;;

(defobject rest (rhythmic-element) 
  ()
  (:parameters rhythm))

(defmethod print-object ((object rest) stream)
  (let ((name (class-name (class-of object)))
        (rhy (slot-value-or-default object 'rhythm nil)))
    (printing-random-thing (object stream)
      (if rhy
          (format stream "~A ~A" name rhy)
        (format stream "~A ~A" name 
               (slot-value-or-default object 'time "unset"))))))


