;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92, 93 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; send to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

;;;
;;; Wrapper functions for common cases of tl:ask-user. 
;;;

(defun ask-file (&key (prompt "File: ") (input "") default pathname-defaults
                      (error "~S is a bogus file name.")
                      error-return (direction ':read) (abort-ok t) help
                      (stream *standard-input*))
  (let (file flag)
    (setf file
      (ask-user :prompt prompt :input input :stream stream
                :default (and default (namestring default))
                :null-ok (and default :ask) :dont-read t
                :check-default default :error-return error-return
                :error (and error
                            #'(lambda (p s)
                                (declare (ignore s)) 
                                (warn-user error (namestring p))))
                :predicate
                #'(lambda (p) 
                    (let ((d (or default pathname-defaults)))
                      (when d
                        (setf p (merge-pathnames p d))))
                    (if (probe-file p)
                      (setf file p flag t)
                      (if (eq direction ':read)
                        (setf file nil flag nil)
                        #+aclpc (setf file p flag nil)
                        #-aclpc
                        (let ((d (pathname-directory p)))
                          (if (not d)
                            (setf file p flag nil)
                            (if #-clisp 
                                (probe-file (make-pathname :directory d))
                                #+clisp
                                (directory (make-pathname :directory d))
                              (setf file p flag nil)
                              (setf file nil flag nil))))))
                      (and file (namestring file)))
                :predicate-is-filter t :abort-ok abort-ok :help help))
    (values file flag)))

(defun ask-number (&key (prompt "Number: ") (input "") default
                        (abort-ok t) help (stream *standard-input*)
                        (type 'number) error-return 
                        (error "~S is not a number.")
                        (min most-negative-fixnum)
                        (max most-positive-fixnum))
  (ask-user :prompt prompt :input input :default default :stream stream
            :default-prompt (and default 
                                 (format nil "(<cr>=~:(~A~))" default))
            :error-return error-return
            :error (and error
                        #'(lambda (p s)
                            (declare (ignore s)) 
                            (warn-user error p)))
            :predicate #'(lambda (x)
                           (or (equal x default)
                               (and (typep x type)(>= x min)(<= x max) x)))
            :null-ok (and default ':ask) :abort-ok abort-ok :help help))

(defun ask-function (&key (prompt "Function: ") (input "") (stream *standard-input*) default
                          (abort-ok t) help (null-ok nil no) (compiled t)
                          (error "~S is not a function.") error-return)
  (ask-user :prompt prompt :default default :input input :stream stream
            :null-ok (if no null-ok (and default ':ask))
            :error-return error-return
            :error (and error
                        #'(lambda (p s)
                            (declare (ignore s)) 
                            (warn-user error p)))
            :abort-ok abort-ok :predicate-is-filter t :check-default t
            :help (or help "Specify the name of a function.~&")
            :predicate #'(lambda (x)
                           (cond ((consp x) NIL)
                                 ((symbolp x)
                                  (and (fboundp x)
                                       (setf x (symbol-function x))
                                       (or (not compiled) 
                                           (compiled-function-p x))
                                       x))))))
			                     

(defun ask-member (list &key (prompt "Pick one: ") (only-one t)
                           abort-ok (null-ok nil nop) default input
			   (test #'eql) key (stream *standard-input*))
  (tl:ask-user :prompt prompt :input input :default default :stream stream
               :default-prompt (and default (format nil "(<cr>=~:(~A~))" 
                                                    default))
               :null-ok (if nop null-ok (and default :ask))
               :abort-ok abort-ok :dont-read (not only-one)
               :predicate-is-filter t
               :predicate #'(lambda (x)
                              (if only-one 
                                  (car (member x list :test test :key key))
                                (loop for tok in (tl:string-forms x)
                                      for obj = (car (member tok list 
                                                     :test test :key key))
                                      unless obj return nil else collect obj)))
               :help (format nil "Select from: ~S~&" 
                       (if key (mapcar key list) list))))

(defun ask-type-or-token (&key prompt default default-prompt input type token
                               (token-value token) (null-ok nil nop) help
			       check-default (stream *standard-input*)(abort-ok t))
  (when token (setf type `(or ,type (member ,token))))			   
  (when (and default (not default-prompt))
    (setf default-prompt (format nil "(<cr>=~:(~A~))" default)))
  (unless help (setf help (format nil "Input must be of type: ~S~&" type)))    
  (let ((ans (tl:ask-user :prompt prompt :input input :stream stream 
                          :default default :default-prompt default-prompt  
                          :check-default check-default :help help
                          :null-ok (if nop null-ok (and default :ask))
                          :predicate-is-filter t :abort-ok abort-ok
                          :predicate #'(lambda (x) (and (typep x type) x)))))
    (if (and token (eql token ans))
        token-value
       ans)))

(defun match-yes-or-no (token yes-ok no-ok)
  (let* ((string (string token))
         (length (length string)))
    (if (and yes-ok (string-equal string "YES" :end2 length))
        ':yes
      (if (and no-ok (string-equal string "NO" :end2 length))
          ':no
         nil))))

(defun yes-type (token)
  (let ((string (string token)))
    (and (string-equal string "YES" :end2 (length string))
         ':yes)))

(defun no-type (token)
  (let ((string (string token)))
    (and (string-equal string "NO" :end2 (length string))
         ':no)))

(defun yes-or-no-type (token)
  (let* ((string (string token))
         (length (length string)))
    (if (string-equal string "YES" :end2 length)
        ':yes
      (if (string-equal string "NO" :end2 length)
          ':no
        nil))))
	
(defun ask-y-or-n (&rest args)
  (let ((? (apply #'ask-type-or-token :type '(satisfies yes-or-no-type) args)))
    (let ((n (symbol-name ?)))
      (if (member n '("Y" "YE" "YES") :test #'string=)
          t
        (if (member n '("N" "NO") :test #'string=)
	    nil
	  ?)))))

(defun y-or-n-or-member (&key prompt default input list (yes-ok t)(no-ok t)
                              (null-ok nil nop)(stream *standard-input*)(test #'eql)
                              value-in-list key (abort-ok t) )
  (let ((ans (tl:ask-user :prompt prompt :input input :default default
                          :default-prompt (and default 
		                            (format nil "(<cr>=~:(~A~))" 
	                                            default)) 
                          :stream stream :check-default t
                          :null-ok (if nop null-ok (and default :ask))
                          :predicate-is-filter t :abort-ok abort-ok
                          :predicate #'(lambda (x &aux v)
                                         (setf v 
                                           (if key (find x list :test test
                                                         :key key)
                                             (find x list :test test)))
                                         (if v
                                             (if value-in-list v x)
                                           (and (symbolp x)
                                                (match-yes-or-no x yes-ok
                                                                 no-ok))))
                          :help (format nil 
                                  "Type~:[~; Yes,~]~:[~; No,~] or one of: ~S~&" 
                                  yes-ok no-ok 
                                 (if key (mapcar key list) list)))))
    (cond ((eq ans ':yes) t)
          ((eq ans ':no) nil)									     
          (t ans))))

(defun y-or-n-or-token (&key prompt default input predicate (yes-ok t)
                             (no-ok t)(null-ok nil nop)(stream *standard-input*)(abort-ok t))
  (let ((ans (tl:ask-user :prompt prompt :input input :default default
                          :stream stream 
                          :default-prompt (and default 
		                            (format nil "(<cr>=~:(~A~))" 
	                                            default)) 
                          :null-ok (if nop null-ok (and default :ask))
                          :predicate-is-filter t :abort-ok abort-ok
                          :predicate #'(lambda (x)
                                         (if (and predicate 
                                                  (funcall predicate x))
                                             x
                                           (if (symbolp x)
                                               (or (match-yes-or-no
                                                     x yes-ok no-ok)
                                                   (if predicate nil x))
                                             nil)))
                          :help (format nil 
                                  "Type~:[~; Yes,~]~:[~; No,~] or token:&" 
                                  yes-ok no-ok ))))
    (cond ((or (eq ans ':yes) (eq ans 'yes)) t)
          ((or (eq ans ':no) (eq ans 'no)) nil)									     
          (t ans))))

(defun ask-syntax (&key (prompt "Select output syntax: ") (input "")
                        (stream *standard-input*)
                        default default-prompt (abort-ok t)
                        none-ok error-return
			(error "~S is not a syntax.")
                        (null-ok nil no))
  (when default
    (setf default (slot-value (find-syntax default) 'name))
    (unless default-prompt 
      (setf default-prompt (format nil "(<cr>=~:(~A~))" default))))
  (ask-user :stream stream :prompt prompt :default default
            :default-prompt default-prompt :abort-ok abort-ok 
            :error-return error-return
            :error (and error
                        #'(lambda (p s)
                            (declare (ignore s)) 
                            (warn-user error p)))
            :null-ok (if no null-ok (and default ':ask))
            :input input :predicate-is-filter t :check-default t
            :predicate #'(lambda (x) 
                           (if (and (eq x 'none) none-ok) 
                             (return-from ask-syntax nil)
                             (find-syntax x nil)))))

(defun ask-name (&key (prompt "Name: ") (input "") default (abort-ok t)
                      (stream *standard-input*) (null-ok nil nop) (junk-allowed nil)
                      (error "~S already names an object.")
                      error-return)
  (let ((dprompt (and default (format nil "(<cr>=~:(~A~))" default))))
    (tl:ask-user :prompt prompt :input input :default default :stream stream 
                 :junk-allowed junk-allowed :abort-ok abort-ok 
                 :default-prompt dprompt :error-return error-return
                 :null-ok (if nop null-ok (and default :ask))
                 :error (and error
                             #'(lambda (p s)
                                 (declare (ignore s)) 
                                 (warn-user error p)))
                 :predicate #'(lambda (x)
                                (or (eq x default) 
                                    (not (find-object x nil))))
                 :help "Type a unique identifier for the new object.")))

(defun ask-class (&key (prompt "Class: ") (input "") default none-ok help
                       error-return (error "~S is not an object class.")
                       (abort-ok t) (type nil) (null-ok nil nop) (stream *standard-input*))
  (ask-user :prompt prompt :input input :default default :stream stream
            :default-prompt (and default
                                 (format nil "(<cr>=~:(~A~))" default))
            :predicate #'(lambda (x) 
                           (and (or (and none-ok (eq x 'none))
                                    (and (find-class x nil)
                                         (or (null type) (typep x type))))
                                x))
            :error-return error-return
            :error (and error
                        #'(lambda (p s)
                            (declare (ignore s)) 
                            (warn-user error p)))
            :null-ok (if nop null-ok (and default :ask))
            :predicate-is-filter t :abort-ok abort-ok
            :help (or help "No help available.~%")
            :error "~S is not a legal class specifiation.~%"))
		
;;;
;;;
;;;

(defun ask-slot (class &key (prompt "Slot: ") (abort-ok t) (none-ok nil)
                            (null-ok nil nop) default input slot
                            (stream *standard-input*))
  (let ((slots (class-slots class)))
    (y-or-n-or-member :list slots :prompt prompt :input input :default default
                      :stream stream :abort-ok abort-ok :value-in-list slot
                      :null-ok (if nop null-ok (and default :ask))
                      :no-ok none-ok :yes-ok nil :key #'slot-definition-name)))

;;;
;;; ask-objects returns a user specified set of objects.  the value returned
;;; depends on whether the user specified a single element, a range of
;;; elements or a list of the above.  normally the results of ask-objects
;;; are returned in a list, even if only one element was specified.  this
;;; can be overridden by specifying :only-one t, in which case only one 
;;; element is allowed to be selected and it is returned as a non list value.  
;;; multiple elements are always returned in a list, but the type of each 
;;; element in the list depends on how its reference was specified by the 
;;; user.  references to single elements return the element, references to
;;; ranges ie, 1:5 or foobar[4:6] are returned as ((lb . ub) . container) 
;;; where lb and ub are both atomic elements that delimit all the elements
;;; beween them (inclusive) in the container.  references to lists of elements
;;; are returned as normal lists, so to distinguish between lists and ranges
;;; examine the cdr of the list - if its atomic the list is a range. 
;;; 

(defun ask-objects (&key (input "") (prompt "Object: ")
                         (stream *standard-input*) (focus nil) only-one top-ok 
                         (abort-ok t) (help "No help available.") predicate
                         (any-level t) null-ok default typep error-return
                         (error nil ep) (predicate-error nil pp)
                         (include-containers t icp) (mapper #'mapref) )
  (declare (special .top-level.))
  (let ((done nil)
        (root (if top-ok .top-level.))
        line refs dprompt prederr)
    (when (and (not icp) only-one) (setf include-containers nil))
    (when default 
      (cond ((symbolp default) (setf default (symbol-name default)))
            ((typep default 'id-mixin)
             (setf default (object-namestring default))))
      (setf dprompt (format nil "(<cr>=~A)"  default)) )
    (when typep 
      (setf predicate #'(lambda (o) (typep o typep)))
      (unless pp
        (setf predicate-error 
              (format nil "~~S references one or more non-~A objects."
                      typep))))
    (unless ep
      (setf error "~S contains one or more bad references."))
    (loop until done
          do
      (setf line 
        (tl:ask-user :prompt prompt :input input :stream stream
                     :null-ok (or null-ok (and default :ask))
                     :abort-ok abort-ok :dont-read t :default default
                     :help help :default-prompt (and default dprompt)))
      (if (eq line ':aborted)
          (setf done t refs line line nil)
        (progn 
          (setf refs 
            (parse-references line focus :only-one only-one
                              :include-containers include-containers									                             	                                                                                                                        
                              :any-level any-level :root root))
          (cond ((null refs)
                 (unless null-ok
                   (setf refs ':error)))
                (predicate
                 (block predcheck
                   (funcall mapper 
                            #'(lambda (x) 
                                (unless (funcall predicate x )
                                  (setf refs ':error 
                                        prederr predicate-error)
                                  (return-from predcheck nil)))
                            refs))))
          (cond ((eq refs ':error)
                 (when error 
                   (warn-user (or prederr error) line))
                 (if error-return
                   (setf done t) 
                   (setf input "" prederr nil)))
                (t (setf done t))))))
    (when (and only-one (consp refs))
      (setf refs (car refs)))
    (values refs line)))

(defun ask-position (&key (prompt "Container position: ") (input "") 
                          (only-one t) (abort-ok t) null-ok focus default 
                          (include-containers t) 
                          (error "~S is a bad position.") error-return)
  (multiple-value-bind (v s)
      (ask-objects :prompt prompt :input input :only-one only-one :focus focus
                   :default default :error-return error-return :error error
                   :null-ok null-ok :abort-ok abort-ok
                   :include-containers include-containers)
    (values v s)))

(defun ask-clause (&key prompt input op terminate error-return (abort-ok t) 
                        class error width only-one)
  (declare (ignore error))
  (let (form err)
    (multiple-value-setq (form input)
      (ask-user :prompt (or prompt
                            (case op
                              (:change "New class and slots: ")
                              (:unset  "Unset slots: ")
                              (t "Slots and values: ")))
                :input input  :error-return error-return
                :dont-read t :null-ok (if (eq op ':new) ':ask)
                :predicate-is-filter t :abort-ok abort-ok
                :predicate #'(lambda (x) 
                               (multiple-value-setq (x err)
                                 (parse-map-input 
                                  x :command (if (eq op ':new) ':set op)
                                  :terminate terminate
                                  :group-width width))
                               x)
                :error #'(lambda (x s)
                           (declare (ignore x s))
                           (warn-user err))
                :help #'(lambda (stream)
                          (declare (ignore stream))
                          (when class
                            (tell-user "~A slots: ~{~A ~}~%"
                                       (class-name class)
                                       (nreverse 
                                        (mapcar #'slot-definition-name 
                                                (class-slots class))))))))
    (when (and (consp form)
               only-one)
      (if (cdr form) 
        (progn (warn-user "More than one clause in ~S." input)
               (setf form ':error))
        (setf form (car form))))
    (values form input)))

