;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :stella)

(defparameter *pprint-archive* #+mcl t #-mcl nil)

(defun archive-object (objects pathname &key (verbose t))
  (unless (listp objects) (setf objects (list objects)))
  (setf pathname 
    (merge-pathnames pathname 
                     (make-pathname :name #-dos "test" #+dos "TEST"
                                    :type +stella-type+)))
  (flet ((archive-it (o file)
           (if (systemp o)
                 (tell-user "Ignoring system element ~S in archive." o)
	       (if *pprint-archive* 
                   (pprint  (archive-form o) file) 
                 (progn (write (archive-form o) :stream file)
                        (terpri file)))))) 
    (let ((error t)
          (file (open pathname :direction :output :if-exists :supersede
                               :if-does-not-exist :create)))
      (unwind-protect 
        (progn
          (when verbose 
            (tell-user "Archiving ~A."
                       (namestring #-aclpc (truename pathname)
                                   #+aclpc pathname)))
          ;; mark all sub containers
          (dolist (o objects)
            (map-object #'(lambda (c) 
                            (setf (slot-value c 'flags)
                              (logior (slot-value c 'flags) +marked+)))
                        o :mode ':containers :level t))
          (dolist (o objects)
            (archive-it o file))
          (setf error nil))
        (when file (close file))
        (when error
          (tell-user "Resetting after aborted archive.")
          (dolist (o objects)
            (map-object #'(lambda (c) 
                            (setf (slot-value c 'flags) 
                              (logandc2 (slot-value c 'flags) +marked+)))
                        o :mode ':containers :level t)))))
    (truename pathname)))

;;;
;;; archive-form is the main workhorse.
;;;

(defgeneric archive-form (object))

;;;
;;; in the perfect world, this default method for archive-form would
;;; just signal an error because we would be defining methods for all supported
;;; types of lisp objects. however, pcl doesn't support method dispatch on
;;; a number of built in classes, such as structures and functions, so we
;;; must check for these explicitly in this default method.
;;;

(defmethod archive-form ((object t))
  (cond ((functionp object)
         (multiple-value-bind (lambda specials) (lambda-archive-form object)
           (setf object
             (if (lexical-closure-p lambda)
                 lambda
               `(function ,lambda)))
           (if specials
               `(progn ,@specials ,object)
             object)))
        (t 
         (error "Don't know how to compute archive form for: ~s ~
                  of type ~s" object (type-of object)))))

(defun lambda-archive-form (spec)
  (let ((specials ())        
        lambda)
    (labels ((intern-lambda (x)
              (cond ((null x) nil)
                    ((consp x) (cons (intern-lambda (car x))
                                     (intern-lambda (cdr x))))
                    ((symbolp x)
                     (if (symbol-package x) x (intern (symbol-name x))))
                    (t x))))
      (setf spec (intern-lambda spec)))
    (setf lambda
      (walk-form spec #+clisp *toplevel-environment*
                      #-clisp nil 
         #'(lambda (form context env)
             (when (symbolp form)
               (when (eq context ':eval)
                   (let ((type (if (or (variable-globally-special-p form)
                                       (variable-special-p form env))
                                   'defparameter
                                 (if (not (variable-lexical-p form env))
                                     'setq nil))))
                     (when (and type (not (constantp form)))
                       (let ((sav (list type form
                                        `',(and (boundp form) (eval form)))))
                         (warn "~A is being treated as a global variable in a generator. It's current value will be archived with the generator using: ~S." form sav)
                         (push sav specials))))))
             (values form nil))))
    (values lambda specials)))

(defun lexical-closure-p (x)
  #+excl (excl::.lexical-closure.-p x)
  #+lispworks (system::%lexical-closure%-p x)
  #+kcl  (eq (car x) 'lisp:lambda-closure)
  #-(or excl kcl lispworks)
  (progn x (error "LEXICAL-CLOSURE-P not implemented in this lisp!")))

;;;
;;; alorithms just archive the user's external program, but we run a few
;;; checks using the internal program.
;;;

(defmethod archive-form ((object algorithm))
  (let ((globals ())
        (code (slot-value object 'external)))
    ;; walk the body of the external code and report any global vars
    (walk-form `(progn ,@ (cddddr code))
                #+clisp *toplevel-environment*
                #-clisp nil
              #'(lambda (form context env)
                  (declare (ignore env))
                  (and (symbolp form)
                       (eq context :eval)
                       (not (constantp form))
                       (variable-globally-special-p form)
                       (pushnew form globals))
                  form))
    (when globals
      (let ((num (if (= (length globals) 1) 0 1))
            (id (slot-value object 'id)))
        (warn "The archive code for ~(~A~) ~:(~A~) contains one or more references to the global variable~[~;s~]~{ ~S,~} which must exist when ~:(~A~) is loaded and rerun.~%"
              (class-name (class-of object)) id num globals id)))
    code))

(defmethod archive-form ((object timed-object))
  (instance-archive-form object))

(defmethod archive-form ((object number))
  object)
  
(defmethod archive-form ((object symbol))
  (if (constantp object)
      object
    `(quote ,object)))

(defmethod archive-form ((object string))
  object)    

(defmethod archive-form ((object cons))
   ;; this functionp check is unfortunate but pcl has no function type...
   (if (functionp object)
       ;; recons the lambda form, interning any uninterned symbols.
       ;; this is necessary because the lambda likely contains macroexpanded
       ;; forms with uninterned symbols that will read as non-eq symbols
       ;; when the form is reloaded! 
       (call-next-method)
    `(cons ,(archive-form (car object))
           ,(archive-form (cdr object)))))

;;;
;;; we insist that all class objects that are archived actually exist 
;;; before the loading process begins by archiving a form that finds
;;; the class rather than creates it.
;;;

(defmethod archive-form ((object standard-class))
  `(find-class ',(class-name object)))

;;;
;;; score-element-archive-form takes advantage of :initarg declarations to
;;; produce an archive form for each value of each slot that has an initarg
;;; associated with it.  this means that slots that have no initargs are 
;;; considered to be "i_only", ie the initialize-instance method must
;;; initializes these values because their values are never archived.
;;; this function should only be called on instances whose slot definition
;;; :initargs can be used to decide which slots are passed initial values
;;; via the make-instance form and which are internally initialized by
;;; initialize-instance itself.
;;;

(defun instance-archive-form (object)
  (let ((class (class-of object)))
    `(make-object ,(if (typep object 'id-mixin) 
                       `'(,(class-name class) ,(slot-value object 'id))
                     `',(class-name class))
       .,(loop with value 
       	       for slot in (class-slots class)
	       when (and (slot-boundp-using-class class object slot)
	  	         (not (equal (setf value (slot-value-using-class 
			 			   class object slot))
				     (slot-definition-initform slot)))
		         (slot-definition-initargs slot)) 
	       nconc (list (archive-form 
	  		     (car (slot-definition-initargs slot)))
	  	           (archive-form value))))))

(defmethod archive-form ((object syntax))
  `(in-syntax ',(slot-value object 'name) t))

(defmethod archive-form :around ((object container))
  (let ((seed (call-next-method)))
    (if (or (null seed) (eq (car seed) 'find-object))
        seed
      (let ((subs (loop for o in (bag-cache (slot-value object 'elements))
                        for f = (archive-form o)
                        when f collect f)))
        (if (not subs)
            seed
          `((lambda (%container%) 
              (add-objects (list ., subs) %container% nil :copy-first nil)
              %container%)
             ,seed))))))

(defmethod archive-form :around ((object startable-element))
  (let ((marked? (logtest (slot-value object 'flags) +marked+)))
    (when marked? 
      (setf (slot-value object 'flags) 
        (logandc2 (slot-value object 'flags) +marked+)))
    (if (logtest (slot-value object 'flags) +deleted+)
     	(values nil)
      (if marked?
          (call-next-method)
       `(find-object ',(slot-value object 'id))))))

(defmethod archive-form :around ((object element))
  (if (logtest (slot-value object 'flags) +deleted+)
      (values nil)
    (call-next-method)))
