;;; -*- Mode: Lisp; Syntax: Common-lisp; Package: common-music; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

;; set by with-part.lisp or stella/constructors.lisp
(defparameter *part-instance-variable*  nil)  

(defun get-part-status-form (&optional force-accessor-form)
  (declare (special *part-status-variable* *part-status-accessor*))
  (if (and (boundp '*part-status-variable*)
	   (not force-accessor-form))
      *part-status-variable* 
    *part-status-accessor*))

(defun get-part-last-status-form ()
  (declare (special *part-last-status-accessor*))
  (if (boundp '*part-last-status-accessor*)
     *part-last-status-accessor*
    `(slot-value ,*part-instance-variable* 'last-status)))

(defmacro unless-chording (&body body)
  `(unless (logtest ,(get-part-last-status-form) +chording+)
     ,@body))

(defmacro when-chording (&body body)
  `(when (logtest ,(get-part-last-status-form) +chording+)
     ,@body))

(defmacro unless-ending (&body body)
  `(unless (logtest ,(get-part-status-form) +ending+)
     ,@body))

(defmacro when-ending (&body body)
  `(when (logtest ,(get-part-status-form) +ending+)
     ,@body))

(defmacro unless-resting (&body body)
  `(unless (logtest ,(get-part-status-form) +resting+)
     ,@body))

(defmacro when-resting (&body body)
  `(when (logtest ,(get-part-status-form) +resting+)
      ,@body))

;;;
;;; wrap-item-stream computes a form that (possibly) links the state of a score
;;; part to the state produced by reading the current item from an item stream.
;;; Some additional hair is added in order to attempt to optimize slot access.
;;;

(defun add-binding (form)
  (declare (special *let-bindings*))
  (push form *let-bindings*))
  
(defun wrap-item-stream (stream-form status-form &key check-for-ending
						      check-for-chording
						      check-for-resting)
  (let ((v1 (gentemp "V"))
 	(v2 (gentemp "V"))
	(body '())
 	real-status status-slot-update-form)
    ;; if status-form is an access form like "(event-status *part*)" we can
    ;; optimize redundant slot access from each check by substituting a local
    ;; variable. if we are calling this function from with-part then we have 
    ;; already done the optimization and are passing in the status variable.
    (if (consp status-form)
	(setf real-status status-form
	      status-form (gentemp "S"))
      (setf status-slot-update-form t))
    ;; since :end-of-period and :chording are disjoint states we try to
    ;; combine the checks into a single IF statement. If check-for-ending
    ;; is an integer the ending check includes the additional check for
    ;; however many periods has been indicated.
    (when check-for-ending
      (let ((end-form `(setf ,status-form (logior ,status-form +ending+))))
	(when (and (numberp check-for-ending)
		   (> check-for-ending 1))
	  (let ((counter (gentemp "K")))
	    (add-binding `(,counter 1))
	    (setf end-form `(and (> (setq ,counter (1+ ,counter)) 
				    ,(coerce check-for-ending 'integer))
				 ,end-form))))
	(if (eq check-for-ending t)
          nil
          (when (symbolp check-for-ending)
	    (let ((counter (gentemp "K")))
	      (add-binding `(,counter 1))
	      (setf end-form `(and (> (setq ,counter (1+ ,counter)) 
				      ,check-for-ending)
				   ,end-form)))))
	(setf body `((if (eq ,v2 +end-of-stream-token+)
                       ,end-form)))))
    (when check-for-chording
      (setf body `((if (eq ,v2 ':chording)
		       (setf ,status-form (logior ,status-form +chording+))
		       ,.body))))
    ;; resting and end-of-period are compatible states so we must keep
    ;; the status checks separate.
    (when check-for-resting 
      (setf body `((if (restp ,v1) 
		       (setf ,status-form (logior ,status-form +resting+))) 
		   ,.body)))
    ;; if we have rewritten slot accessors, we need to initialize the
    ;; local variable with the actual status form and then do a setf of
    ;; that form at the end.
    ;; we only make a wrapper if we are actually doing status checks.
    (if body
	(progn
	  (if real-status
	      (setf body `((let ((,status-form ,real-status))
				,.body
				(setf ,real-status ,status-form))))
	    (nconc body 
	       (list `(setf ,(get-part-status-form t) ,status-form))))
	  (walk-form `(multiple-value-bind (,v1 ,v2) (item ,stream-form)
					   ,.body
					   (values ,v1 ,v2))))
      `(item ,stream-form))))

;;;
;;; the varible *default-alloc-mode* controls whether or not with-part will
;;; allocate a variable for item streams when a user does not specify an 
;;; :alloc value to the special form of item that with-part supports.
;;;

(defvar *default-alloc-mode* t
  "The default item stream allocation mode for WITH-PART.")

;;;
;;; the code walker for item streams inside with-part
;;;

(defprop make-chord-stream :chord-check t)
(defprop make-note-stream :chord-check t)
(defprop make-pitch-stream :chord-check t)
(defprop make-degree-stream :chord-check t)
(defprop make-interval-stream :chord-check t)
(defprop make-step-stream :chord-check t)
(defprop make-voicing-stream :chord-check t)

(defprop make-note-stream :rest-check t)
(defprop make-pitch-stream :rest-check t)
(defprop make-degree-stream :rest-check t)

(defun walk-item (form environment)
  ;(declare (ignore environment) )
  (declare (special *let-bindings* *kill-flag*))
  (apply #'(lambda (stream &key (alloc *default-alloc-mode*) kill
				(rest t restp) (chord t chordp))
             (declare (IGNORE CHORDP RESTP))				
	     (let (var)
	       (when alloc
		 (setf var (gentemp "I")))
	       (when kill
                 (setf *kill-flag* t)
	         (when (numberp kill)
		   (unless (and (integerp kill)
			        (> kill 0))
		     (error
		       ":kill must be t, nil or an integer greater than 0."))))
	       (cond ((or (constantp stream)
			  (symbolp stream))
		      nil)
		     ((consp stream)
		      (cond ((null alloc)
			     (setf stream `(quote ,(eval stream))))
			    ((eq alloc ':dynamic)
			     (push var *let-bindings*)
			     (setf stream `(or ,var (setq ,var ,stream))))
			    (t
			     (push (list var 
                                         (if (symbolp (car stream))
                                             (if (get (car stream)
                                                      ':item-expand)
                                                 (macroexpand stream 
                                                              environment)
                                               (if (fboundp (car stream))
                                                   stream
                                                 (error "~S (inside ITEM) is not a function or macro." (car stream))))
                                           stream))
                                   *let-bindings*)
			     (setf stream var))))
		     ((typep stream 'item-stream)
		      (setf stream `(quote ,stream)))
		     (t nil))
	       (let ((x (wrap-item-stream 
			   stream 
			   (get-part-status-form)
			   :check-for-ending kill
			   :check-for-chording chord
			   :check-for-resting rest)))
		 (values x t))))
	 (cdr form)))

