;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(defclass essential-scale ()
  ((lowest :accessor scale-lowest :initarg :lowest)
   (name :accessor scale-name :initarg :name)
   (divisions-per-octave :accessor scale-divisions-per-octave
                        :initarg :divisions-per-octave)
   (number-of-octaves :accessor scale-number-of-octaves
                      :initarg :number-of-octaves)
   (number-of-entries :accessor scale-number-of-entries
                      :initarg :number-of-entries)
   (entries :accessor scale-entries 
            :initform (make-hash-table :size 512))
   (tokens :accessor scale-tokens :initarg :tokens :initform nil)))

(defmethod print-object ((object essential-scale) stream)
  (format stream "#<~A ~:(~A~)>" (class-name (class-of object))
                                 (scale-name object)))
(defclass general-scale (essential-scale)
  ((pitches :accessor scale-pitches :initarg :pitches)))

(defclass octave-mixin ()
  ((octave-ratio :accessor scale-octave-ratio :initarg :octave-ratio)))
   
(defclass gapped-scale (essential-scale octave-mixin)
  ((gaps :accessor scale-gaps :initarg :gaps))) 

(defclass equal-tempered-scale (essential-scale octave-mixin)
  ((interval-ratio :accessor scale-interval-ratio 
                   :initarg :interval-ratio)))

(defvar *scales* (make-hash-table)
  "Table of musical scales.")

(defun list-all-scales ()
  (let ((scales ()))
    (maphash #'(lambda (k v) k (push v scales)) *scales*)
    (nreverse scales)))

(defun find-scale (name &optional (mode :error))
  (or (gethash name *scales*)
      (ecase mode
	(:error (error "No scale named ~S." name))
	(:ask nil))))

(defun cents (cents)
  (expt 10 (* cents (/ (log 2 10) 1200))))

(defun centify (scaler)
  (* (/ 1200 (log 2 10)) (log scaler 10)))

(defconstant *rest-name* 'r)        ; these should be scale specific
(defconstant *rest-degree* most-negative-fixnum)
(defconstant *rest-pitch* 0.0)

(defun restp (x)
  (cond ((eq x *rest-name*) t)
        ((integerp x) (= x *rest-degree*))
        ((floatp x) (= x *rest-pitch*))
        (t nil)))

;;;
;;;
;;;

(defun make-scale (&key name class entries (number-of-octaves nil nop)
                        lowest-pitch octave-ratio interval-ratio
                        first-octave-number divisions-per-octave
                        suboctave-enharmonic 
                        superoctave-enharmonic)
  ;; normalize entries: ((&rest notes) &key pitch scaler ratio)
  (setf entries
       (loop for spec in entries
             collect
             (cond ((consp spec)
                    (if (consp (car spec)) 
                        spec
                      (if (find (cadr spec) '(:pitch :scaler :ratio))
                          (cons (list (car spec)) (cdr spec))
                        (list spec))))
                   (t (list (list spec))))))
  (let ((state 0) 
        octave-suffixes scale)
    (when (null nop) 
      (setf number-of-octaves 9))
    (if (null divisions-per-octave)
        (setf divisions-per-octave (length entries))
     (unless number-of-octaves
       (setf number-of-octaves 
             (/ (length entries) divisions-per-octave))))
    (if interval-ratio
        (if octave-ratio 
            (error "Found exclusive keywords :octave-ratio, ~
                    :interval-ratio.")
          (setf octave-ratio (expt interval-ratio divisions-per-octave)))
      (when octave-ratio
        (setf interval-ratio 
              (expt octave-ratio (/ 1 divisions-per-octave)))))
    (cond ((and (or (stringp first-octave-number) 
                    (symbolp first-octave-number))
                (string= (string first-octave-number) "00"))
           (setf octave-suffixes
             (cons '|00|
                (loop for i from 0 below (1- number-of-octaves)
                      collect i))))
          ((integerp first-octave-number)
           (setf octave-suffixes
             (loop repeat number-of-octaves
                   collect first-octave-number
                   do (incf first-octave-number))))
          ((null first-octave-number)
           (setf octave-suffixes
             (loop for i below number-of-octaves collect i)))
          (t (error "Bogus first-octave-number ~S."
                    first-octave-number)))
    ;; parse entries, determining a scaler and frequency value for each
    ;; entry. the state variable keeps track of which keywords are used
    ;; in the specs; more than one state is an error.
    (setf entries
      (loop with accumlated-scaler = 1.0 and entry
            for spec in entries
            for position from 0
            do (multiple-value-setq (entry lowest-pitch accumlated-scaler
                                     state)
                 (parse-entry spec position lowest-pitch interval-ratio
                              accumlated-scaler state))
            collect entry
            finally (unless (member state '(0 1 2 4))
                      (error "Can't mix :scaler, :pitch and :ratio."))))
    ;(pprint entries)
    (unless class
      (setf class
        (if (= state 0) 'equal-tempered-scale
          (if octave-ratio 'gapped-scale 'general-scale))))
    (setf scale
      (make-instance class :name (or name (gentemp class))
                     :lowest lowest-pitch 
                     :number-of-octaves number-of-octaves
                     :divisions-per-octave divisions-per-octave))
    (fill-scale scale entries number-of-octaves
                octave-ratio interval-ratio octave-suffixes 
                suboctave-enharmonic superoctave-enharmonic)
    (setf (gethash name *scales*) scale)
    scale))

(defun parse-entry (spec position lowest-pitch interval-ratio
                         accumlated-scaler state)
  ;; parse spec, check keyword consistency across all the specs.
  (let ((notes (pop spec))
        pitch scaler)
    (unless (or (null spec) (= (length spec) 2))
      (error "Malformed keyword list: ~S" spec))
    (cond ((eq (car spec) ':pitch)
           (setf pitch (cadr spec) state (logior state #b001))
           (when (= position 0) (setf lowest-pitch pitch)))
          ((eq (car spec) ':scaler)
           (setf scaler (cadr spec) state (logior state #b010))
           (setf pitch (* lowest-pitch scaler)))
          ((eq (car spec) ':ratio)
           (setf scaler accumlated-scaler state (logior state #b100)
                 accumlated-scaler (* accumlated-scaler (cadr spec))))
          (spec
           (error "~S not a legal keyword specification." spec))
          (t 
           (when (logtest state #b001)
             (error "Missing :pitch value for ~S" spec))
           (setf scaler (expt interval-ratio position))))
    (when (= position 0)
      (unless (= scaler 1.0)
        (error "Scaler for first degree not 1.0: ~S" spec)))
    (unless lowest-pitch (error "Missing :lowest-pitch value."))

    (values (list notes position
                  (or pitch (* lowest-pitch scaler))
                  scaler) 
            lowest-pitch accumlated-scaler state)))

(defmethod fill-scale ((scale essential-scale) #-aclpc &optional
                        entries number-of-octaves
                        octave-ratio interval-ratio octave-suffixes 
                        suboctave-enharmonic superoctave-enharmonic)
  (declare (ignore octave-ratio interval-ratio
                   suboctave-enharmonic superoctave-enharmonic))
  ;; each entry is a list: (names position pitch scaler)
  ;; make a token table to hold both note names and octave suffixes 
  ;; for resolving note names without an octave number. because of the 
  ;; octave "number" 00,  which the lisp reader rightly confuses with 0, 
  ;; we must hash octave suffixes to actual octave numbers ...Sooo, the
  ;; octave suffix is either a number or a symbol. its hash values are 
  ;; the starting step position in the scale. we also reverse hash the
  ;; octave's step position to return the octave token for fast lookup of
  ;; a note's octave designator. the step postions are all offset by 1024
  ;; to avoid collision with numerical octave hash keys.
  (when (> number-of-octaves 0)
    (let ((tokens (make-hash-table 
                    :size (+ (* number-of-octaves 2)
                                (loop for e in entries 
                                      sum (length (first e)))))))
      ;; link note names to position in octave
      (loop for e in entries 
           do (loop for n in (first e) 
                    do (setf (gethash n tokens) (second e))))
      ;; reverse link octaves to octave numbers
      (loop for s in octave-suffixes   
            for o below number-of-octaves
            do  (setf (gethash s tokens) o)
                (setf (gethash (+ o 1024) tokens) s))
      (setf (slot-value scale 'tokens) tokens)))
  scale)

(defmethod fill-scale :after ((scale octave-mixin) #-aclpc &optional
                              entries number-of-octaves
                              octave-ratio interval-ratio octave-suffixes 
                              suboctave-enharmonic superoctave-enharmonic)
  (declare (ignore interval-ratio))
  ;; each entry is a list: (names position pitch scaler)
  (let ((divisions (slot-value scale 'divisions-per-octave))
        (table (slot-value scale 'entries)))
    ;; now we are ready to fill the actual entry hashtable.   
    ;; iterate over the octaves creating the actual note names in the
    ;; scale out of the note descritions and the octave suffixes.
    (loop with last = (1- (* number-of-octaves divisions))
          for octave-number below number-of-octaves
          for octave-scaler = (expt octave-ratio octave-number)
          for octave-step = (* octave-number divisions)
          do (loop for entry in entries
                   for step  = (+ (second entry) octave-step)
                   for pitch = (float (* (third entry) octave-scaler))
                   for notes = (make-note-names (first entry) 
                                                octave-number 
                                                octave-suffixes
                                                superoctave-enharmonic
                                                suboctave-enharmonic
                                                (= step 0) (= step last))
                   do ;(print (list notes pitch))
                             (add-scale-entries table step pitch notes))
          ;; add rest values to the scale.  these should really be scale
          ;; specific but currently they aren't...
          finally (add-scale-entries table *rest-degree* *rest-pitch* 
                                   (list *rest-name*)))
    (setf (slot-value scale 'octave-ratio) octave-ratio)
    (setf (slot-value scale 'number-of-entries)
      (* number-of-octaves divisions))
    scale))

(defmethod fill-scale :after ((scale equal-tempered-scale) #-aclpc &optional
                              entries number-of-octaves
                              octave-ratio interval-ratio octave-suffixes 
                              suboctave-enharmonic superoctave-enharmonic)
  (declare (ignore entries number-of-octaves octave-ratio octave-suffixes 
                   suboctave-enharmonic superoctave-enharmonic))
  (setf (slot-value scale 'interval-ratio) interval-ratio))


(defmethod fill-scale :after ((scale gapped-scale)  #-aclpc &optional
                              entries number-of-octaves
                              octave-ratio interval-ratio octave-suffixes 
                              suboctave-enharmonic superoctave-enharmonic)
  (declare (ignore number-of-octaves
                   octave-ratio interval-ratio octave-suffixes 
                   suboctave-enharmonic superoctave-enharmonic))
  (let ((gaps (make-array (slot-value scale 'divisions-per-octave))))
    (loop for i from 0 as e in entries 
          do (setf (svref gaps i) (fourth e)))
    (setf (slot-value scale 'gaps) gaps)))


(defmethod fill-scale :after ((scale general-scale) #-aclpc &optional
                              entries number-of-octaves
                              octave-ratio interval-ratio octave-suffixes
                              suboctave-enharmonic superoctave-enharmonic)
  (declare (ignore interval-ratio octave-ratio suboctave-enharmonic 
                   number-of-octaves
                   superoctave-enharmonic))
  (let ((divisions (slot-value scale 'divisions-per-octave))
        length table pitches)
    (setf length (length entries))
    (setf table (slot-value scale 'entries))
    (setf pitches (make-array length :element-type 'float))
    (setf (slot-value scale 'number-of-entries) length)
    (loop with last = (1- length)
          for entry in entries 
          for step from 0
          for octave-number = (floor step divisions)
          for pitch = (float (or (third entry)
                                 (error "Missing :pitch information for ~
                                         general-scale degree ~S." entry)))
          for notes = (make-note-names (first entry) 
                                       octave-number octave-suffixes
                                       nil nil (= step 0) (= step last))
          do (add-scale-entries table step pitch notes)
             (setf (svref pitches step) pitch)
          finally (add-scale-entries table *rest-degree* *rest-pitch* 
                                     (list *rest-name*)))
    (setf (slot-value scale 'pitches) pitches)
    scale))
    
    
(defun add-scale-entries (table degree pitch names)
  ;; create note and degree hash entries in scale. hash value for degree
  ;; is the cons (pitch . names).  hash value for name is (pitch . degree)
  ;; floating point pitch is a not hashed, each class of scale
  ;; implements its own methods to map frequency to note or degree.
  (setf (gethash degree table) (cons pitch names))
  (let ((entry (cons pitch degree))) 
    (dolist (name names) (setf (gethash name table) entry)))
  (values))

(defun make-note-names (names octave-number octave-suffixes
                        superoctave-enharmonic suboctave-enharmonic 
                        first? last?)
  (cond ((and first? suboctave-enharmonic)
         (setf names (remove suboctave-enharmonic names)))
        ((and last? superoctave-enharmonic)
         (setf names (remove superoctave-enharmonic names))))
  (loop for name in names
        for suffix = (nth (if (eq name superoctave-enharmonic) 
                              (1+ octave-number)
                            (if (eq name suboctave-enharmonic)
                                (1- octave-number)
                              octave-number))
                          octave-suffixes)
        collect (intern (concatenate 'string
                                     (string name) 
		                     (if (numberp suffix)
			                 (prin1-to-string suffix)
			               (string suffix)))) ))

;;;
;;; defscale and the standard chromatic scale definition
;;;

(defmacro defscale (name args &body body)
  (let (variable form)
    (if (find ':name args)
        (setf variable name)
      (setf args (list* ':name `(quote ,name) args) variable nil))
    (setf form    
      `(make-scale :entries
                   (list
                     ,@ (loop for form in body
                              if (and (consp form)
                                      (member (second form) 
                                              '(:scaler :pitch :ratio)))
                              collect `(list (quote ,(car form)) 
                                             ,@(cdr form))
                              else collect `(quote ,form)))
                   ,@args))
    (if variable
        `(defparameter ,variable ,form)
      form)))

(defparameter *standard-scale* nil)

(eval-when (load eval)
(defscale standard-chromatic-scale 
          (:lowest-pitch 8.175798 
           :octave-ratio (cents 1200)
           :suboctave-enharmonic 'bs
           :superoctave-enharmonic 'cf 
           :first-octave-number '|00|)
  (C CN BS)
  (CS DF)
  (D DN)
  (DS EF)
  (E EN FF)
  (F FN ES)
  (FS GF)
  (G GN)
  (GS AF)
  (A AN)
  (AS BF)
  (B BN CF))
(setf *standard-scale* (find-scale 'standard-chromatic-scale)))

;;;
;;; scale referencing. scale-pitch returns the floating point frequency
;;; of a scale degree reference. scale-note returns the symbolic name of
;;; a scale degree reference. scale-degree returns the ordinal position
;;; of a scale degree reference. octave-and-interval returns the octave
;;; and interval position in the octave of a scale degree reference.
;;;

(defmethod scale-pitch ((ref symbol) (scale essential-scale))
  (car (gethash ref (slot-value scale 'entries))))

(defmethod scale-pitch ((ref integer) (scale essential-scale))
  (car (gethash ref (slot-value scale 'entries))))

(defmethod scale-pitch ((ref float) (scale essential-scale))
  ref)

;;;
;;;
;;;

(defmethod scale-note ((ref symbol) (scale essential-scale))
  ref)
  
(defmethod scale-note ((ref integer) (scale essential-scale))
  (cadr (gethash ref (slot-value scale 'entries))))

(defmethod scale-note ((ref float) (scale equal-tempered-scale))
  (let ((degree (round (log (/ ref (slot-value scale 'lowest))
			    (slot-value scale 'interval-ratio)))))
    (cadr (gethash degree (slot-value scale 'entries)))))

(defmethod scale-note ((ref float) (scale gapped-scale))
  (multiple-value-bind (oct int) (octave-and-interval ref scale)
    (cadr (gethash (+ (* oct (slot-value scale 'divisions-per-octave))
                      int)
                   (slot-value scale 'entries)))))

(defmethod scale-note ((ref float) (scale general-scale))
  (cadr (gethash (scale-degree ref scale)
                 (slot-value scale 'entries))))
;;;
;;;
;;;

(defmethod scale-degree ((ref symbol) (scale essential-scale))
  (cdr (gethash ref (slot-value scale 'entries))))

(defmethod scale-degree ((ref integer) (scale essential-scale))
  ref)

(defmethod scale-degree ((ref float) (scale equal-tempered-scale))
  (values (round (log (/ ref (slot-value scale 'lowest))
		      (slot-value scale 'interval-ratio)))))

(defmethod scale-degree ((ref float) (scale gapped-scale))
  (multiple-value-bind (oct int) (octave-and-interval ref scale)
    (+ (* oct (slot-value scale 'divisions-per-octave) )
       int)))

(defmethod scale-degree ((ref float) (scale general-scale))
  (loop with pitches = (slot-value scale 'pitches)
        for i from 0 below (slot-value scale 'number-of-entries)
        while (<= (svref pitches i) ref) 
        finally (return (1- i))))

;;;
;;;
;;;

(defmethod octave-and-interval ((ref integer) (scale essential-scale))
  (floor ref (slot-value scale 'divisions-per-octave)))

(defmethod octave-and-interval ((ref symbol) (scale essential-scale))
  (let ((degree (cdr (gethash ref (slot-value scale 'entries)))))
    (when degree
      (floor degree (slot-value scale 'divisions-per-octave)))))

(defmethod octave-and-interval ((ref float) (scale equal-tempered-scale))
  (let ((degree (round (log (/ ref (slot-value scale 'lowest))
			    (slot-value scale 'interval-ratio)))))
    (floor degree (slot-value scale 'divisions-per-octave))))

(defmethod octave-and-interval ((ref float) (scale gapped-scale))
  (let* ((lowest (slot-value scale 'lowest))
         (ratio (slot-value scale 'octave-ratio))
         (octnum (floor (log (/ ref lowest) ratio)))
         (octmul (expt ratio octnum)))
    ;; this test is a house of cards, at least for acl/next. the 
    ;; calculation of (* gap lowest octmul) below MUST be computed in
    ;; exactly the manner and order that fill-scale created the scale 
    ;; entry. lowest*octmul is actually a constant for the loop, but we
    ;; set octfrq=lowest*octmul and then use octfrq*gap in the loop, the
    ;; value arrived at is sometimes very slightly off and causes an
    ;; interval to be miscounted.
    (values octnum 
           (loop with gaps = (slot-value scale 'gaps)
                 for i below (slot-value scale 'divisions-per-octave)
                 while (<= (* (svref gaps i) lowest octmul) ref) 
                 finally (return (1- i))))))

(defmethod octave-and-interval ((ref float) (scale general-scale))
  (let ((step (scale-degree ref scale)))
    (if (> (slot-value scale 'number-of-octaves) 0)
        (floor step (slot-value scale 'divisions-per-octave))
     (values nil step))))

;;;
;;; use this function to check a scale to make sure that it can
;;; map pitches to degrees and notes.
;;;

(defun test-scale (scale &optional only-errors (fudge 0.0))
  (loop for step from 0
        for note = (scale-note step scale)
        while note
        for pitch = (+ (scale-pitch note scale) fudge)
        for check1 = (scale-note pitch scale)
        for check2 = (scale-degree pitch scale)
        for check3 = (and (eq check1 note) (eq check2 step))
        when (or (not only-errors) (not check3))
        do (format t "~&~A	~A	~12,6F ~:[!~;=~] ~A ~A"
                   step note pitch check3 check1 check2)))

;;;
;;; user functions all use *standard-scale*.
;;;

(defun in-scale (scale)
  (setf *standard-scale* (find-scale scale)))

(defmethod note ((note t) &optional (scale *standard-scale*))
  (scale-note note scale))

(defmethod note ((note item-stream) &optional scale)
  #-aclpc (declare (ignore scale))
  note)

(defmethod pitch ((note t) &optional (scale *standard-scale*))
  (scale-pitch note scale))

(defmethod pitch ((note item-stream) &optional scale)
  #-aclpc (declare (ignore scale))
  note)

(defmethod degree ((note t) &optional (scale *standard-scale*))
  (scale-degree note scale))

(defmethod degree ((note item-stream) &optional scale)
  #-aclpc (declare (ignore scale))
  note)

;;;
;;; predicates for comparing scale references. only need to use 
;;; in a scale if types are different or note names are used.
;;; methods are binary, the >, < and = forms are n-ary.
;;;

(defun scale<= (&rest args)
  (loop for (x y) on args by #'cdr while y 
        always (scale-leq x y)) )

(defmethod scale-leq ((x t) (y t) &optional (scale *standard-scale*))
  (<= (scale-degree x scale) (scale-degree y scale)))

(defmethod scale-leq ((x integer) (y integer) &optional scale)
  #-aclpc (declare (ignore scale))
  (<= x y))

(defmethod scale-leq ((x float) (y float) &optional scale)
  #-aclpc (declare (ignore scale))
  (<= x y))

(defun scale>= (&rest args) 
  (loop for (x y) on args by #'cdr while y 
        always (scale-geq x y)) )

(defmethod scale-geq ((x t) (y t) &optional (scale *standard-scale*))
  (>= (scale-degree x scale) (scale-degree y scale)))

(defmethod scale-geq ((x integer) (y integer) &optional scale)
  #-aclpc (declare (ignore scale))
  (>= x y))

(defmethod scale-geq ((x float) (y float) &optional scale)
  #-aclpc (declare (ignore scale))
  (>= x y))

(defun scale/= (&rest args) 
  (loop for (x y) on args by #'cdr while y 
        always (scale-neq x y)) )

(defmethod scale-neq ((x t) (y t) &optional (scale *standard-scale*))
  (/= (scale-degree x scale) (scale-degree y scale)))

(defmethod scale-neq ((x integer) (y integer) &optional scale)
  #-aclpc (declare (ignore scale))
  (/= x y))

(defmethod scale-neq ((x float) (y float) &optional scale)
  #-aclpc (declare (ignore scale))
  (/= x y))

(defun scale> (&rest args) 
  (loop for (x y) on args by #'cdr while y 
        always (scale-greaterp x y)))

(defmethod scale-greaterp ((x t) (y t) &optional (scale *standard-scale*))
  (> (scale-degree x scale) (scale-degree y scale)))

(defmethod scale-greaterp ((x integer) (y integer) &optional scale)
  #-aclpc (declare (ignore scale))
  (> x y))

(defmethod scale-greaterp ((x float) (y float) &optional scale)
  #-aclpc (declare (ignore scale))
  (> x y))

(defun scale< (&rest args)
  (loop for (x y) on args by #'cdr while y 
        always (scale-lessp x y)) )

(defmethod scale-lessp ((x t) (y t) &optional (scale *standard-scale*))
  (< (scale-degree x scale) (scale-degree y scale)))

(defmethod scale-lessp ((x integer) (y integer) &optional scale)
  #-aclpc (declare (ignore scale))
  (< x y))

(defmethod scale-lessp ((x float) (y float) &optional scale)
  #-aclpc (declare (ignore scale))
  (< x y))

(defun scale= (&rest args) 
  (loop for (x y) on args by #'cdr while y 
        always (scale-equalp x y)) )

(defmethod scale-equalp ((x t) (y t) &optional (scale *standard-scale*))
  (= (scale-degree x scale) (scale-degree y scale)))

(defmethod scale-equalp ((x integer) (y integer) &optional scale)
  #-aclpc (declare (ignore scale))
  (= x y))

(defmethod scale-equalp ((x float) (y float) &optional scale)
  #-aclpc (declare (ignore scale))
  (= x y))

(defun scale- (x y &optional (scale *standard-scale*))
  (- (scale-degree x scale) (scale-degree y scale)))
