;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(defvar *standard-tempo* 60 "The default tempo factor.")

(defclass rhythm-stream (item-stream)
  ((tempo :initform 1.0 :initarg :tempo :initarg :scaler)))

(defmethod initialize-instance :after ((object rhythm-stream) &rest args)
  ;(format t "~%initialize-instance :after rhythm-stream")
  (when (and (numberp (slot-value object 'tempo))
             (find ':tempo args))
    (setf (slot-value object 'tempo)
      (/ 60.0 (slot-value object 'tempo)))))

(defmethod datum-parser ((object rhythm-stream))
  #'parse-rhythm)

;;;
;;; rhythm should only be scaled one time so that substreams can return
;;; rhythms in their own local tempi. this around method scales a value
;;; only if it is eq with the stream's actual datum. otherwise the datum
;;; is a substream and the scaling is skipped.  since the datum slot is
;;; accessed, the class precedence list must insure that this around method
;;; is called after the main around method has set a valid next datum.
;;;

(defmethod item :after ((object rhythm-stream))
  ;(declare (optimize (speed 3) (safety 0)))
  (let ((value (slot-value object 'value)))
    (when (eq value (slot-value object 'datum))
      (setf (slot-value object 'value)
        (* value (read-tempo (slot-value object 'tempo)
                             value))))))

;;;
;;; tempo functions support dynamic tempo change. the tempo scaler is updated
;;; either before or after the calculation, depending on whether the change
;;; should start on, or or end in, the final tempo.
;;;

(defclass tempo-function ()
  ((coords :initarg :coords)
   (beat   :initform 'q :initarg :beat)
   (time   :initform 0.0)
   (mode   :initform ':before :initarg :mode)
   (length :initarg :length :initform nil)))

(defmethod initialize-instance :after ((object tempo-function) &rest args)
  (declare (ignore args))
  (macrolet ((coord-x (l) `(car ,l))
	     (coord-y (l) `(car (cdr ,l))))
    ;; convert tempi to scalers and beat length to time.
    (setf (slot-value object 'beat)
          (rhythm (slot-value object 'beat) 60))
     (loop with last for tail on (slot-value object 'coords) by #'cddr
          do (setf (coord-y tail) 
	           (/ 60.0 (coord-y tail))
		   last tail)  
	  finally (let ((length (slot-value object 'length)))
		    (setf (slot-value object 'length)
			  (if length
                            (* length (slot-value object 'beat))
			    (* (coord-x last) 
                               (slot-value object 'beat))))))))
;;;
;;; read-tempo returns a tempo value.  tempo may be a number, item
;;; stream, envelope or function.
;;;

(defmethod read-tempo ((tempo t) rhythm)
  (declare (ignore rhythm))
  (if (functionp tempo)
      (funcall tempo)
    (error "Bad tempo: ~S." tempo)))

(defmethod read-tempo ((tempo number) rhythm)
  (declare (ignore rhythm))
  tempo)

(defmethod read-tempo ((tempo item-stream) rhythm)
  (declare (ignore rhythm))
  (item tempo))

(defmethod read-tempo ((object tempo-function) rhythm)
  (declare (ignore stream))
  (if (eq (slot-value object 'mode) ':before)
      (interp (/ (setf (slot-value object 'time)
                       (+ (mod (slot-value object 'time) 
                          (slot-value object 'length))
                          rhythm))
                 (slot-value object 'beat))
             (slot-value object 'coords))
    (let ((val (interp (/ (mod (slot-value object 'time)
                               (slot-value object 'length))
                          (slot-value object 'beat))
                       (slot-value object 'coords))))
    (incf (slot-value object 'time) rhythm)
    val)))

;;;
;;; in-tempo sets the global tempo to a specified value.
;;;

(defun in-tempo (tempo &optional (pulse 'q))
  (unless (and (numberp tempo) (> tempo 0))
    (error "Tempo ~A is not greater than 0." tempo))
  (setf *standard-tempo* (* tempo (rhythm pulse 60))))

(defmethod rhythm ((rhythm t) &optional (tempo *standard-tempo*))
   (* (parse-rhythm rhythm) (/ 60.0 tempo)))

(defmethod rhythm ((rhythm rhythm-stream) &optional tempo)
  #-aclpc (declare (ignore tempo))
  rhythm)

;;;
;;; 
;;;

(defmethod parse-rhythm ((rhythm t))
  (error "Illegal rhythm: ~s" rhythm))

(defmethod parse-rhythm ((rhythm number))
  (if (= rhythm 0) 0 (/ 4.0 rhythm)))

(defmethod parse-rhythm ((rhythm item-stream))
  rhythm)

(defmethod parse-rhythm ((rhythm symbol))
  (parse-rhythm-string (symbol-name rhythm)))

(defun parse-rhythm-string (rhythm)
  (flet ((parse-rhythm-token (str)
	   (let* ((end (length str))
		  (beg (if (char= (elt str 0) #\T) 1 0))
		  (trp (= beg 1))
		  (chr (char str beg))
		  (pos (position chr '(#\S #\E #\Q #\H #\W #\L #\D)))
		  rhy)
	     (cond (pos
		    (incf beg)
		    (setf rhy (expt 2.0 (- pos 2))))
		   (t
		    (setf pos beg)
		    (loop with flg = nil
			  while (< beg end)
			  do (setf chr (elt str beg))
			  if (or (digit-char-p chr)
				 (and (char= chr #\.)
				      (not flg)
				      (< beg (1- end))
				      (digit-char-p (elt str (1+ beg)))
				      (setf flg t)))
			  do (incf beg)
			  else do (return))
		    (if (> beg pos)
		        (setf rhy (/ 4.0 (read-from-string (subseq str pos
                                                                   beg))))
		      (error "Can't parse ~A as a rhythm." str)) 
		    (when (and (<= (setf pos (+ beg 2)) end)
			       (or (string= str "TH" :start1 beg :end1 pos)
				   (string= str "ND" :start1 beg :end1 pos)
				   (string= str "ST" :start1 beg :end1 pos)))
		      (setf beg pos))))
	     (loop while (and (< beg end)
			      (char= (elt str beg) #\.))
		   count (incf beg) into dot
		 finally (unless (= dot 0)
			   (setf rhy (- (* 2 rhy) (* rhy (expt 2 (- dot)))))))
	     (when trp
	       (setf rhy (* rhy 2/3)))
	     (if (= beg end)
		 rhy
	       (error "Can't parse ~A as a rhythm." str))))
	 (next-token-position (string lb len)
	   (loop with chr for i from lb below len
             do (setf chr (elt string i))
             until (find chr '(#\+ #\- #\* #\/) :test #'char=)
             finally (return i))))
    ;; parse rhythmic expression. operator precedence is not supported. 
    (let* ((len (length rhythm))
           (lb 0)
           (ub (next-token-position rhythm lb len)))
      (unless (< lb ub) (error "Can't parse ~A as a rhythm." rhythm))
      (loop with sum = (parse-rhythm-token (subseq rhythm lb ub))
            and val and op
            while (< ub len)
            do
        (setf op (elt rhythm ub))
        (setf lb (1+ ub))
        (setf ub (next-token-position rhythm lb len))
        (unless (< lb ub)
          (error "Can't parse ~A as a rhythm." rhythm))
        (cond ((char= op #\+)
               (setf val (parse-rhythm-token (subseq rhythm lb ub)))
              (incf sum val))
              ((char= op #\-)
               (setf val (parse-rhythm-token (subseq rhythm lb ub)))
			   (decf sum val))
              ((char= op #\*)
               (setf val (read-from-string (subseq rhythm lb ub)))
			   (setf sum (* sum val)))
              ((char= op #\/)
               (setf val (read-from-string (subseq rhythm lb ub)))
			   (setf sum (/ sum val))))
            finally (return sum)))))

;;;
;;; rhythm stream classes
;;;

(defclass cyclic-rhythm-stream (rhythm-stream cycle-pattern) ())
(defclass palindromic-rhythm-stream (rhythm-stream palindrome-pattern) ())
(defclass sequential-rhythm-stream (rhythm-stream sequence-pattern) ())
(defclass accumulating-rhythm-stream (rhythm-stream accumulation-pattern) ())
(defclass heap-rhythm-stream (rhythm-stream heap-pattern) ())
(defclass random-rhythm-stream (rhythm-stream random-pattern) ())
(defclass graph-rhythm-stream (rhythm-stream graph-pattern) ())
(defclass functional-rhythm-stream (rhythm-stream function-pattern) ())
(defclass rotational-rhythm-stream (rhythm-stream rotation-pattern) ())


