(in-package :common-music)

(defparameter *part-instance-variable* '*PART*)

;;;
;;; the following macros support conditionalization according to the current
;;; state of a part without having to know any of 
;;; the gory details about part states.
;;;

(defun status? (status &optional (part *part*))
  (ecase status
    (:chording (logtest (event-last-status part) +chording+))
    (:resting (logtest (event-status part) +resting+))
    (:ending (logtest (event-status part) +ending+))
    (:killed (logtest (event-status part) +killed+))
    ((:noteon :noteoff :noteupdate)
     (unless (typep part 'music-kit-mono-part)
       (error "Status ~s only applies to Music Kit mono parts." status))
     (eq (slot-value part 'notetype) status))
    (:unqueued
     (logtest (event-status part) +unqueued+))))

(defun set-status (status &optional (part *part*))
  (let ((value (event-status part)))
    (ecase status
      (:chording
       (setf (event-status part) (logior value +chording+)))
      (:resting
       (setf (event-status part) (logior value +resting+)))
      (:ending
       (setf (event-status part) (logior value +ending+)))
      (:killed
       (setf (event-status part) (logior value +killed+)))
      ((:noteon :noteoff :noteupdate)
       (error "set-status: ~S not supported." status))
	 (:unqueued
       (setf (event-status part) (logior value +unqueued+))))
  (event-status part)))

(defmacro if-status (status then &optional else)
  (let (predicate-form)
    (ecase status
      (:chording
       (setf predicate-form `(logtest (event-last-status
				       ,*part-instance-variable*) +chording+)))
      (:resting
       (setf predicate-form `(logtest ,(get-part-status-form) +resting+)))
      (:ending
       (setf predicate-form `(logtest ,(get-part-status-form) +ending+)))
      (:killed
       (setf predicate-form `(logtest ,(get-part-status-form) +killed+)))
      ((:noteon :noteoff :noteupdate)
       (setf predicate-form `(eq (slot-value ,*part-instance-variable*
                                             'notetype)
				',status))))
    `(if ,predicate-form ,then ,else)))

;;;
;;; with-part is the standard interface for scheduling score parts.  it does
;;; three friendly things. first, it allows slots to be referenced by name,
;;; as if they were lexical variables within the scope of the macro. 
;;; the second thing it does is to check for sloppy variable usage. the
;;; user is warned if a variable is set but not lexical or special in
;;; the current environment. the third thing with-part does is treat the item
;;; function as a special form that supports a few keyword arguements related
;;; to using item streams in parts.  Inside with-part item behaves as
;;; if it were defined: (item stream &key alloc kill rests chords)
;;; if kill is t, the part will terminate at the end of its period.
;;; if kill is a number, the part will terminate after that many periods.
;;; if alloc is t a lexical variable will be allocated to contain the stream.
;;; the composer function will become a lexical closure around that stream.
;;; if alloc is nil, the stream will be evaled and compiled as a constant.
;;; if alloc is unsupplied it defaults to the value of *default-alloc-mode*.
;;; rest and chord are normally not used.  they allow explicit control over
;;; whether or not the macro checks for chords and rests.  the default values
;;; are t if the stream is some type of note stream, otherwise nil.
;;;

(defparameter *with-part-break-if-no-obvious-termination* nil)

(defmacro with-part (class initoptions &body body &environment environment)
  (let ((found (find-class class)))
    (unless (typep (class-prototype found) 'PART)
      (error "~a is not a class of score parts." class))
    (setf class found))
  (let ((slots (class-slots class)) ;(slot-and-accessor-pairs class)
        (*part-walking* t)
        (*let-bindings* '())
        (*part-instance-variable* (class-name class))
        (*part-status-accessor* `(event-status ,(class-name class)))
        (*part-last-status-accessor* `(event-last-status ,(class-name class)))
        (*part-status-variable* (gentemp "PS"))
        (*kill-flag* nil)
        inits)
    (declare (special *part-walking* *let-bindings*
                      *part-last-status-accessor*
                      *part-status-variable* 
                      *part-status-accessor* *kill-flag*))
    (let ((fn (walk-form 
                `(lambda (,*part-instance-variable*)
                   (with-slots ,(mapcar #'slot-definition-name slots)
				               ,*part-instance-variable*
                     (let ((,*part-status-variable*
                       (event-status ,*part-instance-variable*)))
                       ,@body)))
                environment
                #'(lambda (form context environment)
                    (if (and (listp form) (eq (car form) 'item)
                             (eq context :eval))
                        (walk-item form environment)
                      (progn 
                        (when (and (eq context :set) (symbolp form)
                                   (not 
                                     (or (variable-lexical-p form environment)
                                         (variable-special-p form environment)
                                         (variable-globally-special-p form))))
                          (warn "Setting symbol ~S, which is not a lexical, ~
                                 special or global variable." form))
                        (values form nil)))))))
      (setf inits
        (loop with saved = initoptions 
              and valid = (apply #'append 
                                (mapcar #'slot-definition-initargs slots))
              while initoptions
              for prop = (pop initoptions)
              for val = (if initoptions
                            (pop initoptions)
                          (error "Malformed (uneven) initargs list: ~s"
                                 saved))
              do (unless (member prop valid :test #'eq)
                   (error "~s is not a valid initarg for part class ~s."
                          prop (class-name class)))       
                 (unless *kill-flag*
                   (setf *kill-flag* 
                     (member prop '(events :events end :end))))
              collect (quote-if-necessary prop)
              collect val))
      (unless *kill-flag*
        (let ((*break-on-warnings* 
                 *with-part-break-if-no-obvious-termination*))
          (warn "A ~A part is being created without any obvious means of terminating. You may have forgotten to specify a kill statement or to provide initial values for the end or events slots. Unless you have somehow specified an ending condition, this part will produce ~A notes." 
              (class-name class) most-positive-fixnum)))
      `(make-score-event ',(class-name class)
                         :composer ,(if *let-bindings*
                                        `(let ,*let-bindings* (function ,fn))
                                      `(function ,fn))
                         ,.inits))))


