;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(defun select-list-element (list &optional (stream t) prompt &rest args)
  (flet ((print-list (list stream prompt args)
	   (when prompt
	     (apply #'format stream prompt args))
	   (loop for e in list
	    for i from 1
	    do
	     (format stream "~&~d~4t~s~%" i e))))
    (let ((length (length list)))
      (loop with answer
       do (print-list list stream prompt args)
	  (format stream "Select number: ")
       when (and (integerp (setf answer (read stream)))
		 (< 0 answer (1+ length)))
       return (nth (1- answer) list)))))

(defvar *common-music-syntaxes* '(:CLM :MusicKit :MIDI :Csound))
(defvar *syntax* nil)              ; no default syntax
(defvar *common-music-output* nil) 	        ; scorefile output stream
(defvar *score* nil)             	        ; current score
(defvar *scheduling* nil)        	        ; t when scheduling

(eval-when (compile load eval)
(defparameter *default-scorefile-pathname* ':system-default)
(defparameter *default-scorefile-after* ':system-default)
(defparameter *default-scorefile-header* nil)
(defparameter *last-scorefile-written* nil)
)
;;;
;;; in-syntax can be used to change syntax. it can be placed at the
;;; beginning of a lisp file to assure the proper scorefile syntax.
;;;

(defun get-syntax (syntax)
  (let ((sym (find syntax *common-music-syntaxes* 
		   :test #'(lambda (x y) 
			     (string= (symbol-name x) (symbol-name y))))))
    (and sym (values (find-symbol (symbol-name sym) ':keyword)))))   			  

(defun in-syntax (syntax)
  (let ((sym (get-syntax syntax)))
    (if sym (progn (unless (eq sym *syntax*)
		     (setf *last-scorefile-written* nil))
		   (setf *syntax* sym))
      (error "~S is not one of the Common Music syntaxes: ~S" 
	     syntax *common-music-syntaxes*))))

#+ExCl
(top-level:alias ("syntax" 2) (&optional name)
  (unless name
    (setf name (or *syntax*
		   (find-syntax nil "No current syntax defined.
Select a syntax from the following list:"))))
  (let ((syn (get-syntax name)))
    (if syn
      (progn (in-syntax syn)
	     (format t "The ~S syntax is current.~&" syn))
    (format t "~S is not one of the Common Music syntaxes: ~S" 
	    name *common-music-syntaxes*))))

;;;
;;;
;;;

(defun pathname-syntax (pathname)
  (let ((type (pathname-type pathname)))
    (cond ((string-equal type "CLM")
           (find-syntax ':clm))
          ((string-equal type "MIDI")
	   (find-syntax ':midi))
          ((string-equal type "SCORE")
	   (find-syntax ':musickit))
          ((string-equal type "SCO")
	   (find-syntax ':Csound))
          ((string-equal type "SND")
	   (find-syntax ':clm))
	  (t nil))))

#+excl
(top-level:alias ("play" 2 :case-sensitive)
    (&optional (pathname *last-scorefile-written*))
  (play pathname))

(defun play (&optional (pathname *last-scorefile-written*) &rest args)
  (let ((save pathname))
    (when (or (not pathname)
              (not (setf pathname (probe-file (pathname pathname)))))
      (if save
          (format t "~&File ~A missing or not found.~&" (namestring save))
        (format t "~&No scorefile specified.~%"))
      (return-from play nil)))
  (let ((syntax (pathname-syntax pathname)))
    (unless syntax
      (progn (format t "~&Don't know how to play ~A.~%" pathname)
             (return-from play nil)))
    (play-using-syntax syntax pathname args)))

(defun play-using-syntax (syntax pathname args)
  (cond ((eq syntax (find-syntax ':midi))
         (apply 'midifile-play pathname args))
        ((eq syntax (find-syntax ':musicKit))
         (apply 'play-score-file pathname args))
        ((eq syntax (find-syntax ':clm))
         (if (string-equal (pathname-type pathname) "SND")
	     (dac pathname)
           (apply 'clm-load pathname args)))
        (t
         (format t "~&Don't know how to play ~A using syntax ~A~%" 
                (namestring pathname) syntax)))
  (namestring pathname))
    
;;;
;;; since the system supports multiple syntaxes, we need a way for general
;;; purpose macros like defscorefile, defsoundfile and defsequence to
;;; determine at run time what the appropriate class of score to use is. 
;;; to find this class these macros use the function find-score-class which
;;; first calls find-syntax to return or select the current syntax and then
;;; returns the score class that stored as its property on the syntax symbol.
;;;

(defun find-syntax (&optional type prompt)
  (if type (let ((syn (get-syntax type)))
             (if syn (setf *syntax* syn)
               (error "~s is not one of: ~s" type *common-music-syntaxes*)))
    (or *syntax*
        (and (= (length *common-music-syntaxes*) 1)
             (setf *common-music-syntaxes*  (car *common-music-syntaxes*)))
        (setf *syntax*
          (select-list-element *common-music-syntaxes* t prompt)))))


(defun find-score-class (type syntax)
  (let ((syn (find-syntax syntax "No current score syntax defined. Select
the default syntax from the following list:")))
    (or (get syn type)
	(error "The syntax ~S does not support ~A definition." syn type))))  

(defun default-scorefile-name (syntax)
 (let (type)
    (ecase syntax
      ((clm :clm) (setf type "clm"))
      ((musicKit :musickit) (setf type "score"))
      ((midi :midi) (setf type "midi"))
      ((csound :csound) (setf type "sco")))
     (namestring (make-pathname :defaults (user-homedir-pathname)
     				:name "test" :type type))))
				
;;;
;;; to minimize consing we explicitly allocate and deallocate scores
;;; from a score resource.
;;;

(utils:defresource scores (class)
  :matcher #'instance-resource-matcher
  :constructor #'instance-resource-constructor
  :deinitializer #'instance-resource-deinitializer
  :size 8)

;;;
;;; the basic score class.
;;;

(defclass score () 
  ((schedule :accessor score-schedule :initform nil
	     :documentation "The event scheduling queue.")
   (events :accessor score-events :initform nil
	   :documentation "The set of score events.")
   (time :accessor score-time :initform 0
	 :documentation "The current time of the score.")
   (initializer :accessor score-initializer :initform nil
		:initarg :initializer :initarg initializer
		:documentation "Function to apply when score is created.")
   (finalizer :accessor score-finalizer :initform nil
	      :initarg :finalizer :initarg finalizer
	      :documentation "Function to apply when score is terminated.")
   (after :accessor score-after :initarg after :initarg :after :initform nil
	  :documentation "A command to execute after score is completed.")
   (default-after :documentation "A default after command.")
   (return-value :initform nil
		 :documentation "The value to return by realize-score.")))

;;;
;;; realize score is the main function for processing scores.
;;; 

(defmethod realize-score ((score score))
  ;; apply optional initialization function
  (let ((function (slot-value score 'initializer)))
    (when function
      (funcall function score)))
  ;; call the scheduler to process the score parts
  (schedule-score-events score)
  ;; process any after commands
  (let ((after (slot-value score 'after))
	cmd args)
    (when (eq after ':system-default)
      (setf after (slot-value score 'default-after)))
    (when after
      (if (consp after)
	  (setf cmd (car after) args (cdr after))
	(setf cmd after args nil))
      (score-after-command score cmd args)))
  ;; apply optional clean up function
  (let ((function (slot-value score 'finalizer)))
    (when function
      (funcall function score)))
  ;; return value if any
  (slot-value score 'return-value))

;;;
;;; the basic method to process after options to scores. Since pcl does
;;; not support "and" method combination, subclasses of score  should
;;; override this main method, exeute any command that is appropriate to
;;; their class but explicitly call-next-method on command that they don't
;;; recognize. this will insure that this method is invoked
;;;

(defmethod score-after-command ((score score) command args)
  (case command
    (eval (apply #'eval args))
    (call (apply #'funcall args))
    (shell (apply #'shell args))
    (t (warn "~s is not an appropriate command for this score." command))))

;;;
;;; the sequencer score class captures the output of score parts as lists
;;; in a vector.
;;;

(defclass sequencer (score)
  ((name :initarg name :initarg :name :initform nil)
   (sequence :initarg :sequence :initarg sequence :initform nil 
	     :documentation "The event sequence.")
   (start :initarg start :initarg :start :initform 0
	  :documentation "The staring position for new events.")
   (length :initarg length :initarg :length :initform 0
	   :documentation "The initial length of the sequence.")))

;;;
;;; find-sequence is used by the various sequencing functions to resolve
;;; sequence "references".  a sequence reference is either a sequence or 
;;; the name of a sequence. find-sequence is setf-able.
;;;

(defvar *sequence-table* (make-hash-table :test #'eq))

(defun find-sequence (name &optional (error t))
  (if (vectorp name)
      name
    (or (gethash name *sequence-table*)
	(if error (error "~A does not name a sequence." name)
	  nil))))

(defsetf find-sequence (name &optional (error t)) (sequence)
  (declare (ignore error))
  `(setf-find-sequence ,name ,sequence))

(defun setf-find-sequence (name sequence)
  (setf (gethash name *sequence-table*) sequence))

;;;
;;;
;;;

(defun make-event-sequence (&key name (length 0) (start 0))
  (let ((array (make-array length :adjustable t :fill-pointer t)))
    (when start (setf (fill-pointer array) start))
    (when name (setf (find-sequence name) array))
    array))

(defmethod initialize-instance :after ((score sequencer) &rest args)
  (declare (ignore args))
  (let* ((name (slot-value score 'name))
	 (sequence (slot-value score 'sequence)))
    (if name
	(if sequence
	    (setf (find-sequence name nil) sequence)
	  (setf sequence
	    (setf (slot-value score 'sequence)
	      (or (find-sequence name nil)
		  (make-event-sequence :name name
				       :length (slot-value score 'length)
				       :start nil)))))
      (unless (vectorp sequence)
	(error
	 "You must supply a name for the sequence or the sequence itself.")))
    (if (array-has-fill-pointer-p sequence)
	(setf (fill-pointer sequence) (slot-value score 'start))
      (error
       "Event sequence for sequence does not contain a fill pointer."))
    (setf (slot-value score 'return-value) sequence)))

;;;
;;; for now there is only one class of sequencer so we can go ahead
;;; and allocate one in the score resource.
;;;

(eval-when (load eval)
  (utils:initialize-resource 'scores 1 'sequencer))

;;;
;;; basic scorefile classes
;;;


(defclass scorefile (score)
  ((pathname :accessor scorefile-pathname :initarg :pathname
	     :initarg pathname :initform nil
	     :documentation "The output file name.")
   (element-type :accessor scorefile-element-type)
   (after :initform *default-scorefile-after* )
   (default-path :accessor scorefile-default-path)
   (save-state :initform nil :initarg save-state :initarg :save-state)
   (preload :initform nil :initarg preload :initarg :preload)))
   	 
(defclass ascii-scorefile (scorefile)
  ((header :accessor scorefile-header :initarg :header
	   :initarg header :initform *default-scorefile-header*
	   :documentation "A header string for the output file.")
   (element-type :initform #+(and excl (not cltl2)) 'string-char
                           #-(and excl (not cltl2)) 'character)))

(defmethod initialize-instance :after ((score scorefile) &rest initargs)
  (declare (ignore initargs))
  (let ((pathname (slot-value score 'pathname))
	(default (if (eq *default-scorefile-pathname* ':system-default)
		     (slot-value score 'default-path)
		   *default-scorefile-pathname*)))
    (if pathname
	(unless (eq pathname t)
	  (setf (slot-value score 'pathname) 
	    (merge-pathnames pathname default)))
      (setf (slot-value score 'pathname) 
	(pathname default)))
	
    (let ((preload (slot-value score 'preload)))
      (when preload
	(format t "~&;;; Preloading: ~a~%" (namestring preload))	
	(load preload :verbose nil)))

    (let ((save (slot-value score 'save-state)))
      (when save
      	(when (eq save t) 
	  (setf save (format nil "~a-preload.lisp"
	    		     (pathname-name (slot-value score 'pathname)))))
        (setf save (merge-pathnames save (slot-value score 'pathname)))
        (with-open-file (f save :direction :output :if-exists :supersede)
	  (format t "~&;;; Saving *cm-state* in: ~a~%" (namestring save))
          (write `(setq *cm-state* ',*cm-state*) :stream f))))))

;;;
;;; an :around method on schedule-score-events for scorefiles wraps the
;;; scheduler in a with-open-file to insure proper error recovery. the
;;; main method on this function is defined in parts.lisp
;;;

(defmethod schedule-score-events :around ((score scorefile))
  (let ((pathname (slot-value score 'pathname)))
    (if (eq pathname t)			; allow output to terminal
	(let ((*common-music-output* *standard-output*))
	  (call-next-method))
        (with-open-file (*common-music-output* pathname 
			 :direction :output :if-exists :supersede
			 :element-type (slot-value score 'element-type))
	  (format t "~&;;; Writing: ~a~%"
		  (namestring (truename *common-music-output*)))
	  (call-next-method)))
    (setf *last-scorefile-written* (slot-value score 'pathname))
    (setf (slot-value score 'return-value) *last-scorefile-written*)))

;;;
;;; compute-score-expansion computes the macro expansions of defscorefile,
;;; defsoundfile and defsequence.
;;;

(defvar *warning* "Please note:
The type option for scores is no longer necessary.  All that is necessary
is for you to set the global score syntax you would like to work with.  
The possible syntax choices are ~S.
The syntax is currently set to ~S. To set the syntax use the top-level
lisp command :syntax, or the lisp function in-syntax. If no syntax exists 
when a score is created you will automatically be asked to select one.
This syntax will remain in effect until you change it.
")

(defun compute-score-expansion (scoretype options body environment)
  (let (syntax scoreargs)
    (loop with saved = options and option 
          while options
          do
      (setf option (pop options))
      (unless options
	(error "Malformed (uneven) score options list: ~s." saved))
      (if (or (eq option ':type) (eq option 'type))
	  (progn 
	    (warn *warning* *common-music-syntaxes* *syntax*)
	    (pop options))
	(progn
	  (push (pop options) scoreargs)
	  (push (if (constantp option) option `(quote ,option)) scoreargs))))
    (let ((*readtable* *common-music-readtable*)
	  (holder (gensym)))
      (walk-form
       `(multiple-value-bind (*score* ,holder)
	    (utils:allocate-resource 'scores
				     (find-score-class ',scoretype ',syntax))
	  (unwind-protect
	      (progn
	      	(initialize-instance *score* .,scoreargs)
		(let ((*scheduling* nil)
		      (*standard-tempo* *standard-tempo*)
		      (*standard-scale* *standard-scale*)
		      (*readtable* *common-music-readtable*)) 
		  ,@body
		  (realize-score *score*)))
	    (utils:deallocate-resource 'scores *score* ,holder)))
       environment))))

(defmacro defscorefile (options &body body &environment environment)
  (compute-score-expansion :scorefile options body environment))

(defmacro defsoundfile (options &body body &environment environment)
  (compute-score-expansion :soundfile options body environment))

(defmacro defsequence (options &body body &environment environment)
  (compute-score-expansion :sequence options body environment))
