;;; -*- Mode: LISP; Syntax: Common-lisp; Package: COMMON-MUSIC; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92, 93 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(defvar *standard-tempo* 60 "The default tempo factor.")

(defclass rhythm-stream-mixin ()
  ((tempo :accessor rhythm-stream-tempo :initform 1.0))) 

(defclass sequential-rhythm-stream (rhythm-stream-mixin sequential-item-stream)
  ())

(defclass cyclic-rhythm-stream (rhythm-stream-mixin cyclic-item-stream)
  ())

(defclass accumulating-rhythm-stream (rhythm-stream-mixin 
                                      accumulating-item-stream )
  ())

(defclass heap-rhythm-stream (rhythm-stream-mixin heap-item-stream)
  ())

(defclass random-rhythm-stream (rhythm-stream-mixin random-item-stream)
  ())

(defclass graph-rhythm-stream (rhythm-stream-mixin graph-item-stream)
  ())

(defclass functional-rhythm-stream (rhythm-stream-mixin functional-item-stream)
  ())

;;;
;;; Tempo functions support dynamic tempo changes in rhythm stream.  The
;;; subclasses determine if the tempo is updated before or after the rhythm,
;;; ie whether an accelerando starts in the original tempo or ends in the
;;; final tempo.
;;;

(defclass tempo-function ()
  ((coords :initarg :coords)
   (beat  :initarg :beat)
   (time  :initform 0.0)
   (length :initarg :length :initform nil)))

(defclass pre-incrementing-tempo-function (tempo-function)
  ())

(defclass post-incrementing-tempo-function (tempo-function)
  ())

(defmethod initialize-instance :after ((f tempo-function) &rest initargs)
  (declare (ignore initargs))
  (macrolet ((coord-x (l) `(car ,l))
	     (coord-y (l) `(car (cdr ,l))))
    ;; Convert tempi to scalers and beat length to time.
    (loop with last for tail on (slot-value f 'coords) by #'cddr
          do (setf (coord-y tail) 
	           (/ 60.0 (coord-y tail))
		   last tail)  
	  finally (let ((length (slot-value f 'length)))
		    (setf (slot-value f 'length)
			  (if length
			      (* length (slot-value f 'beat))
			    (* (coord-x last) (slot-value f 'beat))))))))

(defun make-tempo-function (coords &key (pulse 'q) (update ':before))
  (make-instance (ecase update
		   ((:before before)
		    'pre-incrementing-tempo-function)
		   ((:after after)
		    'post-incrementing-tempo-function))
                 :coords (apply #'interpolation coords)
		 :beat (parse-rhythm pulse)))

(defmacro tempo (&body body)
  (let ((len (length body)))
    (if (< len 3)
	    (if (= len 1)
            (car body)
          (* (car body) (rhythm (cadr body) 60)))			
    (let ((coords (loop for tail on body
                        while (numberp (car tail))
                        collect (pop body)))
          (args ())
          (old body)
          from to pulse in update)

      (loop with prop and value
            while body
            do
        (setf prop (pop body))
        (unless body
          (error "Malformed (uneven) options list: ~S." old))
        (setf value (pop body))
        (flet ((check-duplicate (var option)
               (when var (error "The option ~A was specified twice." option))))
          (ecase prop
            (FROM
              (check-duplicate from 'from)
              (setf from value))
            (TO
              (check-duplicate to 'to)
              (setf to value))
            (IN
              (check-duplicate in 'in)
              (setf in value))
            (UPDATE
              (check-duplicate update 'update)
              (setf update value)
              (push (quote-if-necessary update) args)
              (push ':update args))
            (PULSE
              (check-duplicate pulse 'pulse)
              (setf pulse value)
              (push (quote-if-necessary pulse) args)
              (push ':pulse args)))))
    (when (or from to in)
	  (when coords
	    (error "Both tempo coordinates and ranges specified."))
	  (unless from
	    (error "Missing starting tempo in ~s." old))
	  (unless to
	    (error "Missing ending tempo in ~s." old))
	  (unless in
	    (error "Missing number of beats in ~s." old))
	  (setf coords (if *coordinates-are-x-y-pairs*
			   (list 0 from in to)
			 (list from 0 to in))))		   

    `(make-tempo-function ',coords ,.args)))))

;;;
;;; Read-tempo returns a tempo factor for the current rhythm.  A tempo factor
;;; may be expressed as either a number, an item stream, or an envelope.
;;;

(defmethod read-tempo ((tempo t) rhythm stream)
  (declare (ignore rhythm stream))
  (if (functionp tempo)
      (funcall tempo)
	(error "Bogus tempo: ~S." tempo)))

(defmethod read-tempo ((tempo number) rhythm stream)
  (declare (ignore rhythm stream))
  tempo)

(defmethod read-tempo ((tempo item-stream) rhythm stream)
  (declare (ignore rhythm stream))
  (item tempo))

(defmethod read-tempo ((tempo cons) rhythm stream)
  (declare (ignore rhythm))
  (interp (slot-value stream 'count) tempo))

(defmethod read-tempo ((tempo pre-incrementing-tempo-function) rhythm stream)
  (declare (ignore stream))
  (interp (/ (setf (slot-value tempo 'time)
	       (+ (mod (slot-value tempo 'time) 
		       (slot-value tempo 'length))
		  rhythm))
	     (slot-value tempo 'beat))
	  (slot-value tempo 'coords)))

(defmethod read-tempo ((tempo post-incrementing-tempo-function) rhythm stream)
  (declare (ignore stream))
  (let ((val (interp (/ (mod (slot-value tempo 'time)
			       (slot-value tempo 'length))
			  (slot-value tempo 'beat))
		       (slot-value tempo 'coords))))
    (incf (slot-value tempo 'time) rhythm)
    val))

;;;
;;; The item method for all rhythm streams scales a logical rhythm
;;; by a tempo factor to determine the actual rhythm value returned.
;;;

(defmethod item :around ((stream rhythm-stream-mixin))
  (declare (optimize (speed 3) (safety 0)))
  (multiple-value-bind (rhythm state scaled?) (call-next-method)
    (let ((tempo (read-tempo (slot-value stream 'tempo) rhythm stream)))
      (if scaled?
          (values rhythm state t)
        (values (* rhythm tempo) state t)))))

;;;
;;; in-tempo sets the global tempo to a specified value.
;;;

(defun in-tempo (tempo &optional (pulse 'q))
  (unless (and (numberp tempo) (> tempo 0))
    (error "Tempo ~A is not greater than 0." tempo))
  (setf *standard-tempo* (* tempo (rhythm pulse 60))))

;;;
;;;
;;;

(defmethod rhythm ((rhythm t) &optional (tempo *standard-tempo*))
   (* (parse-rhythm rhythm) (/ 60.0 tempo)))

(defmethod rhythm ((rhythm rhythm-stream-mixin) &optional tempo)
  #-aclpc (declare (ignore tempo))
  rhythm)

;;;
;;; 
;;;

(defmethod parse-rhythm ((rhythm t))
  (error "Illegal rhythm: ~s" rhythm))

(defmethod parse-rhythm ((rhythm number))
  (if (= rhythm 0) 0 (/ 4.0 rhythm)))

;(defmethod parse-rhythm ((rhythm rhythm-stream-mixin))
;  ;; embedded subtreams always have tempo factor of 1.0.
;;  (when *system-parsing* (setf (slot-value rhythm 'tempo) 1.0))
;  rhythm)

(defmethod parse-rhythm ((rhythm item-stream))
  rhythm)

(defmethod parse-rhythm ((rhythm symbol))
  (parse-rhythm-string (symbol-name rhythm)))

(defun parse-rhythm-string (rhythm)
  (flet ((parse-rhythm-token (str)
	   (let* ((end (length str))
		  (beg (if (char= (elt str 0) #\T) 1 0))
		  (trp (= beg 1))
		  (chr (char str beg))
		  (pos (position chr '(#\S #\E #\Q #\H #\W #\L #\D)))
		  rhy)
	     (cond (pos
		    (incf beg)
		    (setf rhy (expt 2.0 (- pos 2))))
		   (t
		    (setf pos beg)
		    (loop with flg = nil
			  while (< beg end)
			  do (setf chr (elt str beg))
			  if (or (digit-char-p chr)
				 (and (char= chr #\.)
				      (not flg)
				      (< beg (1- end))
				      (digit-char-p (elt str (1+ beg)))
				      (setf flg t)))
			  do (incf beg)
			  else do (return))
		    (if (> beg pos)
		      (setf rhy (/ 4.0 (read-from-string (subseq str pos beg))))
		      (error "Can't parse ~A as a rhythm." str)) 
		    (when (and (<= (setf pos (+ beg 2)) end)
			       (or (string= str "TH" :start1 beg :end1 pos)
				   (string= str "ND" :start1 beg :end1 pos)
				   (string= str "ST" :start1 beg :end1 pos)))
		      (setf beg pos))))
	     (loop while (and (< beg end)
			      (char= (elt str beg) #\.))
		   count (incf beg) into dot
		 finally (unless (= dot 0)
			   (setf rhy (- (* 2 rhy) (* rhy (expt 2 (- dot)))))))
	     (when trp
	       (setf rhy (* rhy 2/3)))
	     (if (= beg end)
		 rhy
	       (error "Can't parse ~A as a rhythm." str))))
	 (next-token-position (string lb len)
	   (loop with chr for i from lb below len
             do (setf chr (elt string i))
             until (find chr '(#\+ #\- #\* #\/) :test #'char=)
             finally (return i))))
    ;; parse rhythmic expression. operator precedence is not supported. 
    (let* ((len (length rhythm))
           (lb 0)
           (ub (next-token-position rhythm lb len)))
      (unless (< lb ub) (error "Can't parse ~A as a rhythm." rhythm))
      (loop with sum = (parse-rhythm-token (subseq rhythm lb ub))
            and val and op
            while (< ub len)
            do
        (setf op (elt rhythm ub))
        (setf lb (1+ ub))
        (setf ub (next-token-position rhythm lb len))
        (unless (< lb ub)
          (error "Can't parse ~A as a rhythm." rhythm))
        (cond ((char= op #\+)
               (setf val (parse-rhythm-token (subseq rhythm lb ub)))
              (incf sum val))
              ((char= op #\-)
               (setf val (parse-rhythm-token (subseq rhythm lb ub)))
			   (decf sum val))
              ((char= op #\*)
               (setf val (read-from-string (subseq rhythm lb ub)))
			   (setf sum (* sum val)))
              ((char= op #\/)
               (setf val (read-from-string (subseq rhythm lb ub)))
			   (setf sum (/ sum val))))
            finally (return sum)))))
