;;; -*- Mode: Lisp; Syntax: Common-lisp; Package: common-music; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 89, 90, 91 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice. Suggestions, comments and bug reports are welcome.  Please 
;;; address electronic correspondence to: hkt@ccrma.stanford.edu
;;; **********************************************************************

(in-package :common-music)

;;;
;;; Allow dynamic access to the currently executing part.
;;;

(defvar *part* nil "The current scheduled part.")
(defvar *parts* nil "The list of loaded part classes.")

;;;
;;; score events and parts are generally allocated from a resource associated
;;; the class. see defpart.lisp and defresource.lisp for more details.
;;;

(defun score-event-constructor (resource class)
  (declare (ignore resource))
  (allocate-instance (find-class class)))

(defun score-event-deinitializer (resource object)
  (declare (ignore resource))
  (dolist (s (class-slots (class-of object)))
    (slot-makunbound object (slot-definition-name s))))

;;;
;;; allocate-score-resource may be used to preallocate score parts and events.
;;;

(defun allocate-score-resource (class &optional (number 1))
  (unless (find-class class)
    (error "Score part or event class ~A is not defined." class))
  (unless (get class ':resource)
    (error "No resource for score part or event class ~A." class))
  (utils:initialize-resource class number class))

;;;
;;; event is the basic class for score parts and events.
;;;

(defclass event ()
  ((time :initarg :time :initarg time :accessor event-time
	 :documentation "The current time of the event.")
   (holder :accessor event-holder :initform nil
	   :documentation "A holder for resourced events.")))

;;;
;;; Anonymous-Composer is the default composer function for score parts.
;;; this allows the :around method on score-event to avoid checking for
;;; a composer function each time a part is run.
;;;
(eval-when (compile load eval)
(defun Anonymous-Composer (part)
  "The default composer function for score parts."
  (declare (ignore part))
  (values))
)

;;;
;;; the basic class of score parts.
;;;

(defclass part (event)
  ((time :initform 0 :documentation "The current time of the event.")
   (name :initarg :name :initarg name :accessor event-name
	 :documentation "An identifier for the part.")
   (composer :initarg :composer :initarg composer 
	     :accessor event-composer :initform #'Anonymous-Composer
	     :documentation "Function applied to part each time it is run.")
   (initializer :initarg :initializer :initarg initializer 
		:accessor event-initializer :initform nil
		:documentation "Function to apply when part is created.")
   (finalizer :initarg :finalizer :initarg finalizer 
	      :accessor event-finalizer :initform nil
	      :documentation "Function to apply part when part is killed.")
   (events :initarg :events :initarg events :accessor event-events
	   :initform most-positive-fixnum
	   :documentation "The number of events to output.")
   (end :initarg :end :initarg end :accessor event-end
	:initform most-positive-fixnum
	:documentation "The last run time for the part.")
   (count :accessor event-count :initform 1)
   (rhythm :initarg :rhythm :initarg rhythm
	   :initform 0  :accessor event-rhythm
	   :documentation "The amount to increment time by.")
   (status :initform 0 :accessor event-status
	   :documentation "The current status of the part.")
   (last-status :initform 0 :accessor event-last-status)
   (syntax :allocation :class :initform nil)))

(eval-when (load eval)
  (pushnew 'part *parts*))

;;;
;;; make-score-event is the top level constructor for parts and events.
;;; do *not* use make-instance directly.  make-score-event returns an instance
;;; of the specified class of event or part and links it to the current
;;; score, if any.  if make-score-event is called after time 0, i.e. by a
;;; running part, make-score-event also enqueues the new instance into
;;; the score's run queue.
;;;

(defun make-score-event (class &rest args)
  (let ((resource (get class ':resource))
	event holder)
    (if resource			; if there's a resource use it
	(progn 
	  (multiple-value-setq (event holder) 
	    (utils::basic-resource-finder resource class))
	  (setf (event-holder event) holder)
	  (apply #'initialize-instance event :holder holder args))
      (setf event (apply #'make-instance class args)))
    (when *score*                       ; link event to current score.
      (push event (score-events *score*))
      (when *scheduling*
	(score-enqueue event *score*)))
    event))

#|
	  ;; added this check args to make sure that they are legal slot
	  ;; initargs.  this was added because pcl:initialize-instance 
	  ;; does not check initargs (i dont know why not), so resourced
	  ;; instances were not getting their initargs checked.
	  (flet ((check-initargs (class args)
		   (let ((inits (apply #'append 
		                  (mapcar #'slot-definition-initargs
					(class-slots class)))))
		     (loop for prop in args by #'cddr
		      unless (member prop inits :test #'eq)
		      do (error "~s is not a valid initarg for part class ~s."
				prop (class-name class))))))
	    (check-initargs (class-of event) args))
|#


;;;
;;; find-part returns true if the specified name references a class of part.
;;;

(defun find-part (name &optional (error t))
  (let ((x (find-class name error)))
    (and x (typep (class-prototype x) 'part) x)))

;;;
;;; enqueue-latest performs linear enqueueing on cons cells. an event is
;;; inserted into the queue at the last possible position, ie after all the
;;; queued events with the same begin time. this preserves the same order in
;;; the output that the parts were declared in. enqueue-latest is a macro
;;; instead of a function because it allows methods to optimize slot-value.
;;;

(defmacro enqueue-latest (queue event)
  `(let ((schedule (slot-value ,queue 'schedule)))
     (let ((time (event-time ,event))
	   (tail schedule)
	   (last nil))
       (prog nil
	 (go next-loop)
	loop-body (setq last tail tail (cdr tail))
	next-loop (and tail (<= (event-time (car tail)) time) 
		       (go loop-body)))
       (if last
	   (rplacd last (cons ,event (cdr last)))
	 (if tail
	     (rplaca (rplacd tail (cons (car tail) (cdr tail)))
		     ,event)
	   (setf (slot-value  ,queue 'schedule) (list ,event)))))))

;;;
;;; the main method on schedule-score-events is the event scheduler. before
;;; the scheduling loop begins, any events that were created before the 
;;; function was invoked, ie all the "top level" parts defined in the
;;; defscorefile,  are inserted into the queue in the same order in which
;;; they were created. once this is done a *scheduling* flag is set to T
;;; and the scheduling loop is entered. thereafter, any event allocated by
;;; make-score-event during the span of the loop will be automatically 
;;; inserted into the scheulding queue.  during the scheduling process
;;; parts and events are evaluated using the generic function score-event.
;;; this function is responsible for performing whatever actions are 
;;; appropriate to the particular class of score and event being evauluated.
;;; score-event must return nil if the part or event is to be terminated, or
;;; else the object is reinserted into the queue at its new time.  scheduling
;;; terminates when there are no more objects in the queue.
;;;

(defmethod schedule-score-events ((score score))
  (declare (optimize (speed 3) (safety 0)))
  ;; enqueue pushed events in score order.
  (let ((events (slot-value score 'events)))
    (loop for i from (1- (length events)) downto 0 
          do (score-enqueue (elt events i) score)))
  ;; the scheduling loop 
  (let ((this-event nil)
	(next-event nil)
	(*scheduling* t))
    (loop while (setf this-event (pop (slot-value score 'schedule)))
          if (setf next-event (score-event this-event score))
          do (enqueue-latest score next-event)
          else do (score-dequeue this-event score)))
  (values))

;;;
;;; the main method on score-enqueue inserts an event into the score's queue.
;;;

(defmethod score-enqueue ((event event) (score score))
  (declare (optimize (speed 3)(safety 0)))
  (enqueue-latest score event))

;;;
;;; an :around method on score-enqueue for events checks the time value
;;; against the current score time to see if an enqueue is appropriate.
;;;

(defmethod score-enqueue :around ((event event) (score score))
  (declare (optimize (speed 3)(safety 0)))
  (if (> (score-time score)
	 (slot-value event 'time))
      (warn "Skipping enqueue of ~S: Queue time ~S > Event time ~S"
	    event (score-time score) (slot-value event 'time))
    (call-next-method)))

;;;
;;; an :around method on score-enqueue for parts funcalls an optional
;;; initialization function and checks the status to see if an enqueue is
;;; appropriate.
;;;

(defmethod score-enqueue :around ((part part) (score score))
  (declare (optimize (speed 3) (safety 0)))
  (let ((syn (slot-value part 'syntax)))
    (and syn (not (eq syn *syntax*))
        (error
"Syntax mismatch: Part ~S uses syntax ~s but current syntax is ~s."
		part syn *syntax*)))
  (let ((fn (slot-value part 'initializer)))
    (when fn
      (let ((*part* part))
	(funcall fn part))))
  (unless (or (logtest (slot-value part 'status) +killed+)
	      (> (slot-value part 'count) 
		 (slot-value part 'events))
	      (> (slot-value part 'time) 
		 (slot-value part 'end)))
    (call-next-method)))

;;;
;;; the main method on score-dequeue puts back resourced objects
;;;

(defmethod score-dequeue ((event event) (score score))
  (declare (optimize (speed 3) (safety 0)))
  (let ((holder (slot-value event 'holder)))
    (when holder
      (utils:deallocate-resource (caddr holder) (car holder) holder)))
  (values))

;;;
;;; an :around method on score-dequeue for parts checks for the dequeue
;;; status. if its found nothing happens, otherwise the part is terminated
;;; by calling an optional finializer function and continuing the method.
;;;

(defmethod score-dequeue :around ((part part) (score score))
  (declare (optimize (speed 3)(safety 0)))
  (unless (logtest (slot-value part 'status) +unqueued+)
    (let ((fn (slot-value part 'finalizer)))
      (when fn
	(let ((*part* part))
	  (funcall fn part))))
    (call-next-method))
  (values))

;;;
;;; the default method on score-rest does nothing. the music kit specializes
;;; methods for this function. i really wish this were not necessary, but
;;; in the music kit a rest is not necessarily the absence of an event from
;;; a scorefile.
;;;

(defmethod score-rest ((part part)(score score))
  (declare (optimize (speed 3)(safety 0)))
  (values))

;;;
;;; a method on score-event for the basic part class is provided so that
;;; instances of the basic part class can be scheduled in scores.  since this
;;; class declares no output parameters it may serve as a PLA mute.
;;;

(defmethod score-event ((part part) (score score))
  (declare (optimize (speed 3)(safety 0)))
  (values))

;;;
;;; the :around method on score-event for parts invokes a composer function
;;; while the global var *part* is bound to the currently executing part.
;;; after the composer function returns, the current status of the part is
;;; checked see if the method should be continues or the part be rescheduled.
;;; if it is to be rescheduled, time is updated to the next run time and the
;;; part is returned as the value of the method.
;;;

(defmethod score-event :around ((part part) (score score))
  (declare (optimize (speed 3)(safety 0)))
  (let ((*part* part))
    (funcall (slot-value part 'composer) part))
  (let ((new-status (slot-value part 'status))
        (next nil))
    (unless (logtest new-status +removed+)
      (setf (slot-value part 'last-status) new-status)
      (if (logtest new-status +resting+)
	  (score-rest part score)	; sigh, let MusicKitMono do its thing.
        (call-next-method))
      (if (logtest new-status +chording+)
        (setf next part)
        (progn
          (when (logtest new-status +ending+) 
            (setf new-status +killed+))
          (unless (or (= new-status +killed+)
		      (> (incf (slot-value part 'count))
			 (slot-value part 'events))
		      (> (incf (slot-value part 'time) 
			       (slot-value part 'rhythm))
			 (slot-value part 'end)))
            (setf next part))))
      (setf (slot-value part 'status) 
            (logandc2 new-status +normal-mask+)))
    next))

;;;
;;; require-part insures that a part is defined before it is used.
;;; 

(defvar *require-part-search-list* 
	(list "~/" (string-append *common-music-directory* "clm/")
		   (string-append *common-music-directory* "clm/")
		   (string-append *common-music-directory* "mk/")
		   (string-append *common-music-directory* "csound/")))
	
(defun require-part (name &optional allocate pathname)
  (flet ((part-class-name-p (name)
	   (let ((class (find-class name nil)))
	     (and class (typep (class-prototype class) 'part) class)))
	 (find-part-file (part directory)
	   (or (probe-file (make-pathname :directory directory :name part
					  :type "fasl"))
	       (let ((p (probe-file (make-pathname :directory directory
						   :name part
						   :type "lisp"))))
		 (when p
		   (warn "Loading uncompiled part file. ~A"
			 (namestring p)))
		 p))))
    (let ((class (part-class-name-p name)))
      (unless class
	(unless pathname
	  (let ((n (string-downcase (string name))))
	    (setf pathname (or (loop for dir in *require-part-search-list*
				when (find-part-file n dir) return it)
			       (error "Can't find part file for part ~A."
				      name)))))
	(load pathname)
	(or (setf class (part-class-name-p name))
	    (error "File ~A did not contain definition for part ~A."
		   pathname name)))
      (when allocate
	(let ((r (get name ':resource)))
	  (unless r
	    (error "Can't allocate ~A: no resource defined for this part."
		   name))
	  (if (eq allocate t)
	      (setf allocate 1)
	    (unless (>= allocate 0)
	      (error "Allocate value ~S is not a postive number.")))
	  (when (< (utils::resource-free r) allocate)
	    (allocate-score-resource name allocate))
	  class)))))

#| 

;;;
;;; Potentially useful classes of events, currently not used.
;;;


;;;
;;; The formatter part class may be used to schedule arbitrary
;;; output to the score.
;;;

(defclass formatter (part)
  ((format :initarg format :initarg :format :initform "~A")
   (form :initarg form :initarg :form :initform nil)))

(defmethod score-event ((part formatter) (score score))
  (let ((form (slot-value part 'form)))
    (when form
      (format *common-music-output* (slot-value part 'format) form))))



(defclass lambda-event (event)
  ((function :accessor lambda-event-function 
	     :initarg :function :initarg function)))

(defmethod score-event ((event lambda-event))
  (declare (optimize (speed 3)(safety 0)))
  (funcall (slot-value event 'function)))
|#