;;; -*- Mode: Lisp; Syntax: Common-lisp; Package: common-music; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 89, 90, 91 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice. Suggestions, comments and bug reports are welcome.  Please 
;;; address electronic correspondence to: hkt@ccrma.stanford.edu 
;;; **********************************************************************

(in-package :common-music)

(eval-when (load eval)
  (setf (get ':musicKit :scorefile) 'music-kit-scorefile)
  (setf (get ':musicKit :sequencer) 'sequencer))

;;;
;;; the Music Kit scorefile
;;;

(defclass music-kit-scorefile (ascii-scorefile)
  ((gentag :initform 0 :accessor music-kit-scorefile-gentag)
   (part-file :initform nil :accessor music-kit-scorefile-part-file
	      :initarg part-file :initarg :part-file)
   (dynamic-parts :initform nil :accessor music-kit-scorefile-dynamic-parts)
   (default-after :initform 'play)
   (default-path :initform (default-scorefile-name :musicKit))))

(eval-when (load eval)
  (utils:initialize-resource 'scores 1 'music-kit-scorefile))

(defun print-part-info (part stream)
  (let ((name (slot-value part 'name))
  	l1 l2)
    (setf l1 (format stream "part ~A;~%" name))
    (if (typep part 'synth-patch-mixin)
	(setf l2 (format stream "~A synthPatch:~S ~@[synthPatchCount:~D~];~%"
			 name (slot-value part 'Patch)
			(slot-value part 'SynthPatchCount)))
      (if (typep part 'music-kit-midi-mixin)
	  (setf l2 (format stream "~A synthPatch: ~S ~@[midiChan:~D~];~%"
			   name (slot-value part 'Patch)
			   (slot-value part 'MidiChan)))))
    (if stream nil (concatenate 'string l1 l2))))


(defmethod schedule-score-events :before ((score music-kit-scorefile))
  (format *common-music-output* "/* Common Music output ~A */~%~%" 
	  (date-string))
  (let ((header (slot-value score 'header)))	   
    (when header
      (format *common-music-output* "~&~a~%" header)))
  (loop with p for x in (slot-value score 'events)
	do
    (setf p (if (consp x) (car x) x))
    (when (typep p 'music-kit-part)
      (print-part-info p *common-music-output*)))
  ;; add an include statement for dynamic part file if 
  ;; we are not writing the score to the tty.
  (let ((this-file (slot-value score 'pathname))
	(part-file (slot-value score 'part-file)))
    (when part-file
      (if (eq this-file t)
	  (setf (slot-value score 'part-file) nil)
	(let ((path (make-pathname :type "hdr" :defaults this-file)))
	  (unless (eq part-file t)
	    (setf path (merge-pathnames part-file path)))
	  (format *common-music-output* "~%include \"~A\";~%"
		  (namestring path))
	  (setf (slot-value score 'part-file) path)))))
  (format *common-music-output* "~&BEGIN;~%"))


(defmethod schedule-score-events :after ((score music-kit-scorefile))
  (format *common-music-output* "END;~%")
  (let ((path (slot-value score 'part-file)))
    (when path
      (with-open-file (f path :direction ':output :if-exists ':supersede)
	(dolist (p (nreverse (slot-value score 'dynamic-parts)))
	  (write-string p f))))))

;;;
;;; Music Kit score after commands:
;;;
;;; (mixsounds &optional soundfile scorefile)
;;; (play &key repeat file write debug fast quiet player)
;;;

(defmethod score-after-command ((s music-kit-scorefile) command argl)
  (let ((fn (get command ':music-kit-after)))
    (if fn 
        (funcall fn s argl)
      (call-next-method))))

(defprop play :music-kit-after 'music-kit-score-after-play)
(defprop mixsounds :music-kit-after 'music-kit-score-after-mixsounds)

(defparameter *.score-player* nil)

(defun play-score-file (&optional (file *LAST-SCOREFILE-WRITTEN*)
			    &key (player *.score-player*) args background)
  (unless file (error "No file specified."))
  (setf file (merge-pathnames file "~/test.score")) 
  (unless player			; defer to runtime
    (setf *.score-player*
    	  (concatenate 'string (namestring *common-music-directory*)
	               "oldsys/mk/playscore")
	  player *.score-player*))
  (let ((s (format nil "~a~@[ -~a~] ~a~@[ &~]"
                            player
			    args
                            (namestring file)
                            background)))
    (format t ";;; Playing: ~a~%" s)
    (shell s))
   (values file))

(defun music-kit-score-after-play (s argl)
  (let ((saved argl)
        (file nil)
	(args '())
	(options '()))
    (loop with option and value
       while argl
       do
        (setf option (pop argl))
        (unless argl
          (error "Malformed (uneven) options list to after command: ~S" saved))
        (setf value (pop argl))
        (ecase option
          ((repeat :repeat) (push "q" options))
          ((file :file) (setf file value))
          ((write :write) (setf options (list (format nil "w ~a" value))))
          ((debug :debug) (push "d" options))
          ((fast :fast) (push "f" options))    
          ((quiet :quiet) (push "q" options))
          ((player :player) (push value args) (push ':player args))
          ((& :&) (push t args) (push ':background args))))
    (unless file
      (setf file (namestring (truename (merge-pathnames
                                        (or (slot-value s 'pathname)
                                             *last-scorefile-written*)
                                        (slot-value s 'default-path))))))
    (when options
      (push (apply #'string-append options) args)
      (push :args args))      
    (apply #'play-score-file file args)))
  
(defun music-kit-score-after-mixsounds (s argl)
  (let (sndfile scorefile string)
    (setf sndfile (make-pathname :defaults (or (pop argl)
		                               (slot-value s 'pathname))
			         :type "snd"))
    (setf scorefile (make-pathname :defaults (or (pop argl)
			                         (slot-value s 'pathname))
			           :type "score"))
    (setf string (format nil "~A ~A ~A~%"
                         (concatenate 'string
			              (namestring *common-music-directory*)
	 	                      "mk/mixsounds")
			 (namestring scorefile)
			 (namestring sndfile)))
    (format t ";;; Mixing: ~A" string)
    (shell string)
    (setf string (format nil "~A ~A~%" "sndplay" (namestring sndfile)))
    (format t ";;; Playing: ~A" string)
    (shell string)))

;;;
;;;
;;;

(defun music-kit-notetag ()
  (if *score* (incf (music-kit-scorefile-gentag *score*))
    nil))

(defmethod initialize-instance :after ((score music-kit-scorefile) &rest args)
  (declare (ignore args))
  (setf (slot-value score 'time) -1))

;;;
;;; note off events are automagically generated by some musickit parts.
;;;

(defclass MK_NoteOff (event)
  ((part-name :initarg :part-name)
   (part-notetag :initarg :part-notetag)))

;;;
;;; the basic music kit part classes
;;;

(defclass music-kit-part (part) 
  ((syntax :initform :musickit)
   (print-once :initarg print-once :initform nil
	       :accessor music-kit-print-once)
   (part-tag :initarg part-tag :initarg :part-tag)))


(defclass music-kit-poly-part (music-kit-part) 
  ((duration :accessor part-duration :initarg :duration 
	     :initarg duration :initform nil)))

(defclass music-kit-mono-part (music-kit-part) 
  ((noteType :accessor part-notetype :initform ':noteOn :initarg noteType)
   (noteTag :accessor part-notetag :initform (music-kit-notetag)
	    :initarg noteTag)
   (dont-optimize :accessor part-dont-optimize :initform nil
		  :initarg dont-optimize)
   (cache :accessor music-kit-mono-part-cache)))

(defvar *music-kit-unique-part-tag* 0)

(defmethod initialize-instance :after ((part music-kit-part) &rest args)
  (declare (ignore args))
  (unless (slot-boundp part 'part-tag)
    (setf (slot-value part 'part-tag) 
      (prog1 *music-kit-unique-part-tag* (incf *music-kit-unique-part-tag*))))
  (unless (slot-boundp part 'name)
    (setf (slot-value part 'name)
      (objc-string (string-append
		    (class-name (class-of part))
		    (prin1-to-string (slot-value part 'part-tag)))))))

(defmethod initialize-instance :after ((part music-kit-mono-part) &rest args)
  (declare (ignore args))
  (let ((type (slot-value part 'notetype)))
    (unless (or (eq type ':noteOn)
		(eq type ':noteUpdate))
      (error "noteType ~s is not :noteUpdate or :noteOn." type)))
  ;; create the output value cache and initialize
  ;; the entries for output optimizing.
  (let ((cache (make-hash-table :test #'eq 
 	                        :size (+ (length (event-parameters 
						  (class-of part)))
					 2))))
    (setf (slot-value part 'cache) cache)))

;;;
;;; an :after method on score-enqueue outputs the initial parameter values
;;; as a note update. The primary score-event method  flushes the parameter
;;; values after performing a note update.
;;;

(defmethod score-enqueue :after ((part music-kit-part)
				 (score music-kit-scorefile))
  (declare (optimize (speed 3)(safety 0)))
  (let ((table (if (typep part 'music-kit-mono-part)
		   (music-kit-mono-part-cache part)))
	(*print-case* ':downcase)
	(count ':first))
    ;; cache names of dynamic parts.
    (when (and *scheduling*
	       (slot-value score 'part-file))
      (let ((parts (slot-value score 'dynamic-parts)))
	(setf (slot-value score 'dynamic-parts)
	  (cons (print-part-info part nil) parts))))
    (let ((pars (event-parameters (class-of part)))
	  par)
      (dolist (p (slot-value part 'print-once))
	(unless (setf par (find p pars :test #'eq :key #'car))
	  (error "print-once value ~A is not a parameter in ~S." p part))
	(let* ((name (pinfo-name par))
	       (value (slot-value part name)))
	  (when value
	    (when (eq count ':first)
	      (princ (slot-value part 'name) *common-music-output*)
	      (write-string " (noteUpdate)" *common-music-output*)
	      (setf count ':last))
	    (write-char #\Space *common-music-output*)
	    (write-string (or (pinfo-message par)
			      (error "Can't find message for parameter ~A."
				     name))
			  *common-music-output*)
	    ;; should use parameter's formatter...
	    (princ value *common-music-output*)
	    (when table
	      (setf (gethash name table) value))))))
    (when (eq count ':last)
      (write-char #\; *common-music-output*)
      (terpri *common-music-output*))))

;;;
;;; scorefile-dequeue for mono parts schedules a noteOff event based on the
;;; current time and rhythm of the part.
;;;

(defmethod score-dequeue :before ((part music-kit-mono-part) 
				  (score music-kit-scorefile))
  (declare (optimize (speed 3)(safety 0)))
  (make-score-event 'MK_NoteOff
		    :part-name (slot-value part 'name)
		    :part-notetag (slot-value part 'noteTag)
		    :time (+ (slot-value part 'time)
			     (slot-value part 'rhythm))))

;;;
;;; This method on scorefile-rest for mono parts outputs a noteOff and
;;; clears the output hashtable.
;;;

(defmethod score-rest ((part music-kit-mono-part) 
		       (score music-kit-scorefile))
  (declare (optimize (speed 3)(safety 0)))
  (let ((time (slot-value part 'time)))
    (when (> time (slot-value score 'time))
      (write-string "t " *common-music-output*)
      (princ time *common-music-output*)
      (write-char #\; *common-music-output*)
      (terpri *common-music-output*)
      (setf (slot-value score 'time) time))
    ;; all this hair to avoid format! it prints "part1 (noteOff,23);"
    (princ (slot-value part 'name) *common-music-output*)
    (write-string " (noteOff," *common-music-output*)
    (princ (slot-value part 'noteTag) *common-music-output*) 
    (write-char #\) *common-music-output*)
    (write-char #\; *common-music-output*)
    (terpri *common-music-output*)
    ;; clear the hashtable because we are at the end of a phrase.
    (clrhash (slot-value part 'cache))))


(defmethod score-event ((event mk_noteoff) (score music-kit-scorefile))
  (declare (optimize (speed 3)(safety 0)))
  (let ((time (slot-value event 'time)))
    (when (> time (slot-value score 'time))
      (write-string "t "  *common-music-output*)
      (princ time *common-music-output*)
      (write-char #\; *common-music-output*)
      (terpri *common-music-output*)
      (setf (slot-value score 'time) time))
    (format *common-music-output*  "~A (noteOff,~S);~%"
	    (slot-value event 'part-name) (slot-value event 'part-notetag))))

;;;
;;;
;;;

(defmethod make-part-methods ((part music-kit-part) new-part-name score-classes
			      parameters)
  (unless (listp score-classes)
    (setf score-classes 
      (list (find-class 'music-kit-scorefile))))
  (loop for class in score-classes
        collect (make-score-event-method (class-prototype class)
                                         part new-part-name parameters)))

(defmethod make-score-event-method ((score music-kit-scorefile)
				    (part music-kit-poly-part)
				    class-name parameters)
  `(defmethod score-event ((part ,class-name) (score music-kit-scorefile))
     (declare (optimize (speed 3)(safety 0)))
     (let ((*print-case* ':downcase))
       ;; N.B. the vars part, and count are constants
       ;; in forms built by parameter-printing-forms.
       (let ((time (slot-value part 'time)))
	 (when  (> time (slot-value score 'time))
	   (write-string "t " *common-music-output*)
	   (princ time *common-music-output*)
	   (write-char #\; *common-music-output*)
	   (terpri *common-music-output*)
	   (setf (slot-value score 'time) time)))
       ;; print the name and duration
       (princ (slot-value part 'name) *common-music-output*)
       (write-char #\space *common-music-output*) 
       (write-char #\( *common-music-output*)
       (prin1 (or (slot-value part 'duration)
		  (slot-value part 'rhythm))
	      *common-music-output*)
       (write-char #\) *common-music-output*)
       ;; Print each parameter
       (let ((print-once (slot-value part 'print-once)))
	,.(loop with d = '((write-char #\space *common-music-output*))
	        for par in parameters
	        append (parameter-printing-forms part par 0 d)))
       (write-char #\; *common-music-output*)
       (terpri *common-music-output*))))

(defmethod make-score-event-method ((score music-kit-scorefile)
				    (part music-kit-mono-part)
				    class-name parameters)
  `(defmethod score-event ((part ,class-name) (score music-kit-scorefile))
     (declare (optimize (speed 3)(safety 0)))
     (block $method-body-block$
      (when (logtest +chording+ (slot-value part 'status))
	(error "Chords are not allowed in music-kit-mono-parts."))
      (let ((time (slot-value part 'time))
	    (cache (slot-value part 'cache))
	    (dont-optimize (slot-value part 'dont-optimize))
	    (*print-case* ':downcase))
	 ;; N.B. the vars part, cache, always, doall and count
	 ;; are constants in the forms built by parameter-printing-forms.
	(when (> time (slot-value score 'time))
	  (write-string "t " *common-music-output*)
	  (princ time *common-music-output*)
	  (write-char #\; *common-music-output*)
	  (terpri *common-music-output*)
	  (setf (slot-value score 'time) time))
	 ;; all this hair to avoid the format statement...
	(princ (slot-value part 'name) *common-music-output*)
	(ecase (slot-value part 'notetype)
	  (:noteOn
	   (write-string " (noteOn," *common-music-output*) )
	  (:noteUpdate
	   (write-string " (noteUpdate," *common-music-output*))) 
	(princ (slot-value part 'notetag) *common-music-output*)
	(write-char #\) *common-music-output*)
	;; print the parameters
	(let ((print-once (slot-value part 'print-once)))
	  ,.(loop with d = '((write-char #\space *common-music-output*))
	     for par in parameters
	     append (parameter-printing-forms part par 0 d)))
	(write-char #\; *common-music-output*)
	(terpri *common-music-output*)))))

(defmethod default-message-string ((part music-kit-part) pinfo)
  (string-append (objc-string (pinfo-name pinfo)) ":"))

(defmethod parameter-printing-forms ((part music-kit-poly-part)
				     pinfo position delimiters)
  (declare (ignore position))
  (unless (eq (pinfo-type pinfo) ':message)
    (error "MusicKit only supports &message parameters."))
  (let ((var (gentemp "V")))
    `((let ((,var (slot-value part ',(pinfo-name pinfo))))
        (when (and ,var
		   (not (member ',(pinfo-name pinfo)
				print-once :test #'eq)))
	  ,@delimiters
	  (write-string ,(or (pinfo-message pinfo)
			     (default-message-string part pinfo))
			*common-music-output*)
	  ,(value-print-form var pinfo))))))
					    

(defmethod parameter-printing-forms ((part music-kit-mono-part)
				     pinfo position delimiters)
  (declare (ignore position))
  (unless (eq (pinfo-type pinfo) ':message)
    (error "MusicKit only supports &message parameters."))
  (let ((name (pinfo-name pinfo))
	(var (gentemp "V")))
    `((let ((,var (slot-value part ',name)))
        (when (and ,var
		   (not (member ',name print-once :test #'EQ))
		   (or (not (equal ,var (gethash ',name cache)))
		       (member ',name dont-optimize :test #'eq)))
	  ,@delimiters
	  (write-string ,(or (pinfo-message pinfo)
			     (default-message-string part pinfo))
			*common-music-output*)
	  ,(value-print-form var pinfo)
	  (setf (gethash ',name cache) ,var))))))

