(in-package :common-music)                 ; This should begin every file.

(in-syntax :musicKit)

(defscorefile (after NIL pathname "Pluck-1") 
  (with-part PluckPoly (name "guitar" duration .25)
    (setf rhythm (item (rhythms q e q tempo 120)))
    (setf freq (item (notes d3 e f g a in heap) :kill 5))))


;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

;;; In this example a few more elements are added:
;;; * A more complex pitch pattern is used. 
;;; * Durations are randomly set. This means that sometimes overlaps occur and 
;;;     sometimes they dont.
;;; * Brightness, an expression parameter, is set to vary in a pattern
;;;     described by a Common Music interpolation function.
;;; * The entire passage is played at a tempo of q = 128. (Note, however,
;;;     that the brightness function is looked up AFTER the tempo is applied.)

(in-package :common-music)		       ; This must begin every file.

(in-syntax :musicKit)

(defscorefile (after NIL pathname "Pluck-2")
  (in-tempo 128)
  (let ((brightness (interpolation 0 .1 3 .8 6 .2 10 .9 12 .1 15 1.0)))
    ;; Part begins at time 0, generates 50 notes, then stops.
    (with-part PluckPoly (name "guitar" time 0 events 50)
      (setf rhythm (item (rhythms e s e)))    ; Rhythm is 8th, 16th, 8th    
      (setf freq (item (items (notes c2 d e)  ; Randomly choose between 4 lists
			      (notes c3 e g)
			      (notes e4 d fs)
			      (notes e5 b4 c5)
			      in heap))) 
      (setf bright (function-value time brightness))
      (setf duration (+ .05 (random 1.5))))))

;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

;;; This example demonstrates the Wave1vi SynthPatch in Poly mode. It 
;;; illustrates the creation of wave tables and envelopse.

(in-package :common-music)

(in-syntax :musicKit)

;;; Music Kit envelopes and such are specified here. This text goes directly
;;; to the header of the scorefile.  We include one envelope for the amplitude.
;;; This envelope has three break-points in the attack segment and one in the 
;;; release segment.  There is no limit to how many breakpoints you can 
;;; have in a Music Kit envelope. We progressively increase the attack time 
;;; by changing the ampAtt parameter.
;;; We also include two wave tables. One is only odd harmonics and thus 
;;; resembles a clarinet. The other has two formants and suggests, 
;;; possibly, an oboe.

(defparameter *header* 
    (header "waveTable clarinetWave = [{1,1}{3,.5}{5,.25}{7,.125}{9,.1}
                                      {11,.05}{13,.025}];"
	    "waveTable oboeWave = [{1,.1}{2,.2}{3,.3}{4,.4}{5,.3}{6,.2}{7,.1}
                                  {8,.2}{9,.25}{10,.2}{11,.1}];"
	    "envelope ampFn  = [(0,0)(.2,1)|(.35,0.0)];"))

;;;
;;; Write the score to the file "Wave1i-1.score". The scorefile header will
;;; be the value of the *header* variable.
;;;

(defscorefile (after NIL pathname "Wave1i-1" header *header*)

  ;; Use the Wave1iPoly part. Name it "waveDemo" and play 15 note at a constant
  ;; rhythm of .25 seconds. Initialize the amplitude to .1, the amplitude 
  ;; envelope to "ampFn", the amplitude release to .5 and the amplitude attack
  ;; to .01. Since the values of these last 3 slots don't change, we print
  ;; them once only, using a noteUpdate at the start of the scorefile.

  (with-part Wave1iPoly (name "waveDemo" rhythm .25 events 15 amp .1 
			 ampEnv "ampFn" ampRel .5 ampAtt .01
 			 print-once '(ampEnv amp ampRel))
    (setf freq (item (pitches d4 e fs)))
    ;; Alternate waveTables
    (setf waveform (item (items "clarinetWave" "oboeWave"))) 
    ;; Pan back and forth
    (setf bearing (item (items -45 -27 0 27 45 in palindrome))) 
    ;;  AmpAtt gets ampAtt multiplied by 1.5
    (setf ampAtt (* ampAtt 1.5))))



;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

;;; This example illustrates vibrato. First 3 notes are played, each with 
;;; more sinusoidal vibrato. Then 3 notes are played, each with more
;;; random vibrato. Then 3 notes are played with both types of vibrato
;;; and the speed is progressively increased.

(in-package :common-music)

(in-syntax :musicKit)

(setf *header*
    (header "envelope ampFn   = [(0,0)(.2,1)|(.35,0.0)];"
	    "waveTable aWave = [{1,.4}{2,.2}{3,.3}{4,.4}{5,.3}{6,.2}{7,.1}
                                {8,.2}{9,.25}{10,.2}{11,.1}];"))
;;;
;;; Write the score to the file "Wave1i-1.score". The scorefile header will
;;; be the value of the *header* variable. Play the score after writing it.
;;;

(defscorefile (pathname "Wave1vi-1" header *header* after NIL )

  ;; Use the Wave1viPoly part and name it "vibDemo". Start the part 
  ;; at time 0 and play 15 notes. initialize the amplitude to .1,
  ;; the amplitude envelope to "ampFn", the waveform to "aWave", 
  ;; the percentage of sinusoidal vibrato to .009, the speed of
  ;; sinusoidal vibrato to 4 and the percentage of random vibrato 
  ;; to .005. Print the amp, AmpEnv, and waveform parameter values
  ;; one time only in a noteUpdate at the beginning of the scorefile.

  (with-part Wave1viPoly (name "vibDemo" events 15 time 0
			  amp .1  ampEnv "ampFn" waveform "aWave"
			  rhythm 1.25 duration 1.0 sVibAmp .009
			  sVibFreq 4 rVibAmp .005
			  print-once '(amp ampEnv waveform))
    (setf freq1 (item (pitches d4 e fs r)))
    ;; Evaluate only when not resting
    (unless-resting                        
     (setf sVibAmp (item (items (items 0 .0075 .015)    ; increasing sVib
				(items 0 for 3)         ; then no sVib
				(items .01 for 3))      ; then constant sVib
			  :kill t)
	   rVibAmp (item  (items (items 0 for 3)        ; no rVib
				 (items 0 .0075 .015)   ; then increasing rVib
				 (items .0075 for 3)))  ; then constant rVib
	   sVibFreq (item (items (items 4.5 for 6)      ; constant sVibFreq
				 (items 3.4 4.5 5.25))) ; then increasing 
	   ))))


;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

(in-package :common-music)
(in-syntax :musicKit)

;;; This example uses the NeXT Timbre Data Base in a monophonic context.
;;;      Notice that the perceived timbre remains relatively constant,
;;;      despite the pitch changes. Compare this to DBWave1vi-2.lisp, where 
;;;      the same wave table is used on every note. 
;;;      Also compare this to DBWave2vi-1.lisp which uses true
;;;      DSP wavetable interpolation and thus avoids the little clicks that
;;;      appear in this example.

(setf *header*
    (header "envelope ampEnvelope = [(0,0)(.2,1)(.3 ,.5)|(.4,0.0)];"
	    "envelope freqEnvelope   = [(0,.98)(.2,1.0)|(.4,.99)];"))

(defscorefile (after NIL pathname "DBWave1vi-1" header *header*)
  (with-part DBWave1viMono (name "bass"
			    ampEnv "ampEnvelope" 
			    sVibAmp .01
			    freqEnv "freqEnvelope" 
			    freqAtt .25

			    ;; the value of waveform1 must appear as a
			    ;; quoted string in the scorefile

			    waveform1 (quotify "BA") 
			    rvibAmp .008 

			    ;; Dont-optimize is a special Mono part
			    ;; initializer. It takes a list of  parameters
			    ;; to ALWAYS print, whether or not their value
			    ;; has changed. We need to ALWAYS print the
			    ;; waveform so the Music Kit Timbre Data 
			    ;; Base will select a new timbre on each note. 

			    dont-optimize '(waveform1)   
			    print-once '(ampEnv sVibAmp freqEnv freqAtt 
					 rvibAmp)
			    freq 'd2)
    (setf freq0 freq 
	  freq  (item (pitches d2 e f g a bf d3 e) :kill t)
	  rhythm (item (rhythms e s e. e))
	  svibFreq (+ 4.5 (random 1.0)))));;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

(in-package :common-music)
(in-syntax :musicKit)

(setf *header* 
    (header "envelope ampEnvelope = [(0,0)(.2,1)(.3 ,.5)|(.4,0.0)];"
	    "envelope freqEnvelope = [(0,.98)(.2,1.0)|(.4,.99)];"))

;;; This example uses the NeXT Timbre Data Base in a monophonic context.
;;; Notice that the perceived timbre gets "munchkinized". Compare
;;; this to DBWave1vi-1.lisp where the perceived timbre remains relatively
;;; constant. The only difference is that we do not include the 
;;; DONT-OPTIMIZE initialization that was present in DBWave1vi-1. 

(defscorefile (after NIL pathname "DBWave1vi-2" header *header*)
  (write-line "Here comes the munchkin!")
  (with-part DBWave1viMono (ampEnv "ampEnvelope" 
			    sVibAmp .01
			    freqEnv "freqEnvelope" 
			    freqAtt .25
			    waveform1 (quotify "BA")
			    rvibAmp .008
			    freq 'd2
			    print-once '(ampEnv sVibAmp freqEnv freqAtt 
					 waveform1 rvibAmp))
    (setf freq0 freq 
	  freq  (item (pitches d2 e f g a bf d3 e) :kill t)
	  rhythm (item (rhythms e s e. e))
	  svibFreq (+ 4.5 (random 1.0)))))

;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

;;; This example illustrates a simple canonic melody using DBWave1vi 

(in-package :common-music)
(in-syntax :musicKit)


(setf *header* 
    (header "envelope marcato = [(0,0)(.1,1,2)(.2,.1)|(.3,0.0)];"
	    "envelope smooth   = [(0,0)(.1,1)|(.3,0.0)];"))

(defscorefile (after NIL pathname "DBWave1vi-3" header *header*)
  (loop with ampf = (interpolation 0 .1 5 .2 6 .2 10 .1)	
        and attf = (interpolation 0 .5 5 .05 10 .5)
        and relf = (interpolation 0 .1 5 1 10 .1)
        for part-name in '("part1" "part2" "part3")
        for timbre in '("SA" "SI" "SU")
        as i from 0
        do
    (let ((start-time i))
      (with-part DBWave1viPoly
	  (name part-name 
	   time start-time 
	   end (+ 10 start-time) 
	   duration .25 
	   sVibAmp .01 
	   bearing (- (* i 45) 45)
	   waveform (quotify timbre)
	   rVibAmp .006
	   print-once '(bearing waveform sVibAmp rVibAmp))
	(setf rhythm (item (items .5 .25 .75 .5))
	      freq (item (notes d4 e f g a bf d5 e))
	      ampEnv (if (> rhythm .3) "marcato" "smooth")
	      svibFreq (+ 4 (random 1.0)))
	(let ((x (- time start-time)))
	  (setf amp (function-value x ampf)
		ampAtt (function-value x attf)
		ampRel (function-value x relf)))))))

;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

;;; In this example, we carefully interpolate between two wavetables to
;;; give perfectly smooth transitions. (Compare with DBWave1vi-1.lisp.)
;;; See the documentation on the NeXT Timbre Data Base for details.
;;; Notice the clever way we alternate between wave tables.  This insures 
;;; that no "bump" or "click" is heard when wavetables are changed, because
;;; we always change a wavetable when its amplitude is 0. 

(in-package :common-music)
(in-syntax :musicKit)


(setf *header*
    (header "envelope ampEnvelope = [(0,0)(.2,1)(.3 ,.5)|(.4,0.0)];"
	    "envelope ramp01 =  [(0,0)(.3,1)];"
	    "envelope ramp10 =  [(0,1)(.3,0)];"))

(defscorefile (after NIL pathname "DBWave2vi-1" header *header*)
  (with-part DBWave2viMono (name "bass"
			    ampEnv "ampEnvelope" 
			    sVibAmp .01
			    freqEnv "ramp01" 
			    freqAtt .25
			    rvibAmp .008 
			    dont-optimize '(waveform1 waveform0)
			    print-once '(ampEnv sVibAmp freqEnv freqAtt 
					 rvibAmp)
			    freq 'd2)
    (setf freq0 freq 
	  freq (item (pitches d2 e f g a bf d3 e) :kill t)
	  rhythm (item (rhythms e s e. e))
	  waveformAtt rhythm
	  waveform0 (quotify (item (items "0BA" "BA")))
	  waveform1 (quotify (item (items "BA" "0BA")))
	  waveformEnv (item (items "ramp01" "ramp10"))
	  svibFreq (+ 4.5 (random 1.0)))))

;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

;;; This example illustrates continuous transition between a variety of 
;;; timbres. We use the same "trick" we used in DBWave2vi-1.lisp.

(in-package :common-music)
(in-syntax :musicKit)


(setf *header*
    (header "envelope ampEnvelope = [(0,0)(.2,1)(.3 ,.5)|(.4,0.0)];"
	    "envelope ramp01 =  [(0,0)(.3,1)];"
	    "envelope ramp10 =  [(0,1)(.3,0)];"))

(defscorefile (after NIL pathname "DBWave2vi-2" header *header*)
  (with-part DBWave2viMono (name "bass"
			    ampEnv "ampEnvelope" 
			    sVibAmp .01
			    freqEnv "ramp01" 
			    freqAtt .25
			    rvibAmp .008 
			    dont-optimize '(waveform1 waveform0)
			    print-once '(ampEnv sVibAmp freqEnv freqAtt 
					 rvibAmp)
			    freq 'd2)
    (setf freq0 freq 
	  freq  (item (pitches d2 e f g a bf d3 e) :kill t)
	  rhythm (item (rhythms e s e. e))
	  waveformAtt rhythm
	  waveform0 (quotify (item (items "0BE" "BU" "0BU" "BE" "0BE" "BE")))
	  waveform1 (quotify (item (items "BA" "0BA" "BO" "0BO" "BA" "0BA")))
	  waveformEnv (item (items "ramp01" "ramp10"))
	  svibFreq (+ 4.5 (random 1.0)))))

;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

;;; This example illustrates using the LISP loop macro to make a round
;;; See DBWave2vi-1.lisp and DBWave2vi-2.lisp for more comments.

(in-package :common-music)
(in-syntax :musicKit)

(setf *header* 
    (header "info tempo:42;"
	    "envelope smooth  = [(0,0)(.3 ,1.0)|(1.4,0.0)];"
	    "envelope ramp01 =  [(0,0)(.3,1)];"
	    "envelope ramp10 =  [(0,1)(.3,0)];"))

(defscorefile (after NIL pathname "DBWave2vi-3" header *header*)
  (loop with ampf = (interpolation 0 .1 5 .2)
        for part-name in '("bass" "tenor" "alto" "soprano1" "soprano2")
        for start-time in '(0 .5 1.0 1.5 1.5)
        for start-pitch in '(d2 f3 d4 g5 e5)
        for pitch-stream in (list (pitches d2 e f g a bf d3 e a bf a)
				    (pitches f3 e a g c4 d f g a)
				    (pitches d4 c e d f a bf a)
				    (pitches g5 a f e c d e a4)
				    (pitches e5 f d c a4 bf g a))
        for rhythm-stream in (list (rhythms e s e. e)
				   (rhythms q s e s)
				   (rhythms e. s e e)
				   (rhythms e s e e.)
				   (rhythms e s e e.))
        for waveform0-stream in (list (items "0BE" "BU" "0BU" "BE" "0BE" "BE")
				      (items "0TE" "TU" "0TU" "TE" "0TE" "TE")
				      (items "0SE" "SU" "0SU" "SE" "0SE" "SE")
				      (items "0SE" "SU" "0SU" "SE" "0SE" "SE")
				      (items "0SE" "SU" "0SU" "SE" "0SE" "SE"))
        for waveform1-stream in (list (items "BA" "0BA" "BO" "0BO" "BA" "0BA")
				      (items "TA" "0TA" "TO" "0TO" "TA" "0TA")
				      (items "SA" "0SA" "SO" "0SO" "SA" "0SA")
				      (items "SA" "0SA" "SO" "0SO" "SA" "0SA")
				      (items "SA" "0SA" "SO" "0SO" "SA" "0SA"))
        for amp-offset in '(0 0 0.1 0 0)
        for bearing-value in '(0 -45 45 -33 33)
        do

    ;;  establish new variable bindings each time through the loop

    (let ((start-time start-time) 
	  (start-pitch start-pitch)
	  (pitch-stream pitch-stream)
	  (rhythm-stream rhythm-stream)
	  (waveform1-stream waveform1-stream)
	  (waveform0-stream waveform0-stream)
	  (amp-offset amp-offset))
 
     (with-part DBWave2viMono (time start-time 
				ampEnv "smooth" 
				sVibAmp .01
				freqEnv "ramp01" 
				bearing bearing-value 
				rVibAmp .008 
				dont-optimize '(waveform1 waveform0)  
				freq start-pitch		           
				print-once '(ampEnv sVibAmp freqEnv bearing 
					     rVibAmp))
	(setf freq0 freq 
	      freq  (item pitch-stream :kill t)
	      rhythm (if (= count 1) 1.0 (item rhythm-stream))
	      waveformAtt rhythm
	      ampAtt (if (= count 1) 1.0 .3)
	      amp (+ amp-offset (function-value (- time start-time) ampf))
	      waveform0 (quotify (item waveform0-stream))
	      waveform1 (quotify (item waveform1-stream))
	      waveformEnv (item (items "ramp01" "ramp10"))
	      svibFreq (+ 4.5 (random 1.0))
	      freqAtt (/ rhythm 2.0))))))
;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

;;; This example plays a simple melody with the Fm1vi SynthPatch 
;;; in Poly mode. 

(in-package :common-music)

(in-syntax :musicKit)

;;; Music Kit envelopes and such are specified here. This text goes directly
;;; to the header of the scorefile. 

(setf *header* 
    (header "envelope ampFn   = [(0,0)(.2,1)(.3,.7)|(.35,0.0)];"
	    "envelope indFn   = [(0,0)(.1,1)(.2,.7)|(.3,.7)];"))

;;;
;;; Write score to file "Fm1vi-1.score".  The header to the score file will 
;;; be the value of the *header* variable.  Play score file after writing it.
;;;

(defscorefile (after NIL pathname "Fm1vi-1" header *header*)
  (with-part Fm1viPoly (name "clarinet"  ; Name of the part
			time 0           ; Part begins at time 0
			amp .1           ; Amplitude initialized to .1
			ampEnv "ampFn"   ; Envelope initialized to "ampFn"
			ampAtt .3        ; Amplitude envelope attack time
			ampRel .2        ; Amplitude envelope release time
			m1IndAtt .1      ; Modulator index attack time
			m1IndRel .1      ; Modulator index release time
			m1Ratio 2.001    ; Modulator portion of c:m ratio
			cRatio 1.0       ; Carrier portion of c:m ratio
			m1Ind1 1         ; Modulation index when envelope is 1
			m1Ind0 0         ; Modulation index when envelope is 0
			m1IndEnv "indFn" ; Modulation index envelope
			sVibAmp .009     ; Percentage of sinusoidal vibrato
			sVibFreq 4       ; Speed of sinusoidal vibrato
			rVibAmp .005     ; Percentage of random vibrato
			bearing 0        ; Left-right panning.
                        print-once '(amp ampEnv ampAtt ampRel m1IndAtt m1IndRel
				     m1Ratio cRatio m1Ind1 m1Ind0 m1IndEnv
				     sVibAmp sVibFreq rVibAmp bearing))
    (setf freq (item (pitches d4 e f f f e g f e d f e f g a)
		     :kill t)          
          rhythm (item (rhythms e e e e q e e e e q e e q q h))
	  duration (- rhythm .1))))      ; Release will cause slight overlaps
;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

;;; This is like Examp5, but a Mono part is used to allow for glissandi
;;; between notes and a greater ease of legato.  Also, the amplitude envelope
;;; is changed from time to time to give some variety of articulation.
;;; There a number of advantages to using Mono parts. In addition to the
;;; extra expression control they provide, Mono parts also prevent overlaps
;;; that needlessly sap your DSP resources.

(in-package :common-music)

(in-syntax :musicKit)

(setf *header* 
    (header "envelope legato   = [(0,0)(.2,1)|(.5,0.0)];"
	    "envelope marcato  = [(0,0)(.2,1)(.3,.1)|(.5,0.0)];"
	    "envelope freqFn  = [(0,.9)(.1,1)|(.2,.99)];"))

(defscorefile (after NIL pathname "Fm1vi-2" header *header*)
  (with-part Fm1viMono (name "clarinet"  ; Name of the part
			time 0           ; Part begins at time 0
			amp .1           ; Amplitude initialized to .1
			ampAtt .3        ; Amplitude envelope attack time
			ampRel .2        ; Amplitude envelope release time
			freqEnv "freqFn" ; Frequency envelope
			m1IndAtt .1      ; Modulator index attack time
			m1IndRel .1      ; Modulator index release time
			m1Ratio 2.001    ; Modulation ratio
			m1Ind1 1.3       ; Modulation index when envelope is 1
			m1Ind0 .1        ; Modulation index when envelope is 0
			sVibAmp .009     ; Percentage of sinusoidal vibrato
			sVibFreq 4       ; Speed of sinusoidal vibrato
			rVibAmp .005     ; Percentage of random vibrato
			bearing 0        ; Left-right panning.
			print-once '(amp ampAtt ampRel freqEnv m1IndAtt
				     m1IndRel m1Ratio m1Ind1 m1Ind0 sVibAmp
				     sVibFreq rVibAmp bearing))
    (setf freq (item (pitches d4 e f f f e g f e d f e f g a)
		     :kill t)   
          rhythm (item (rhythms e e e e q e e e e q e e q q h))
	  ;; We vary the envelope to add variety
	  ampEnv (item (items (items "legato" for 2) 
			      (items "marcato" for 2)
			      (items "legato" for 3)
			      "marcato"
			      (items "legato" for 2)
			      (items "marcato" for 2)
			      (items "legato" for 3)))
	  m1IndEnv ampEnv
	  ;; Portamento applies to the "arrival" note
	  portamento (if (= count 15) .5 .1))))



;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-

(in-package :common-music)

(in-syntax :musicKit)

(defscorefile (after NIL pathname "moving-sound")
  (with-part PluckMono (name "guitar" 
			bearing -48    ; First bearing will be -45 (see below)
			noteType :noteOn)
    (setf noteType (item (items :noteOn (items :noteUpdate for 10)) :kill 3)) 
    (if (eq noteType :noteOn)
	(setf freq (item (notes c4 c3 c4)))) 
    (setf rhythm .1 
	  bearing (+ bearing 3))))

#|

(time (load "/dist/cm/mk/examples/all"))

;;;======================="5/1/90  May Day PCL (REV 4b)" 
cpu time (non-gc) 52367 msec user, 27650 msec system
cpu time (gc)     1833 msec user, 983 msec system
cpu time (total)  54200 msec user, 28633 msec system
real time  153876 msec

cpu time (non-gc) 24767 msec user, 9084 msec system
cpu time (gc)     317 msec user, 83 msec system
cpu time (total)  25084 msec user, 9167 msec system
real time  54420 msec

cpu time (non-gc) 22267 msec user, 7983 msec system
cpu time (gc)     316 msec user, 50 msec system
cpu time (total)  22583 msec user, 8033 msec system
real time  49239 msec
;;;======================="March 92 PCL (1)" 
cpu time (non-gc) 23417 msec user, 11451 msec system
cpu time (gc)     716 msec user, 733 msec system
cpu time (total)  24133 msec user, 12184 msec system
real time  78829 msec

cpu time (non-gc) 16216 msec user, 5734 msec system
cpu time (gc)     434 msec user, 83 msec system
cpu time (total)  16650 msec user, 5817 msec system
real time  36472 msec

cpu time (non-gc) 16100 msec user, 5766 msec system
cpu time (gc)     300 msec user, 84 msec system
cpu time (total)  16400 msec user, 5850 msec system
real time  36435 msec
|#
 
