;;; -*- Mode: Lisp; Syntax: Common-lisp; Package: common-music; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

;;;
;;; midifile scores (level 0)
;;;

(eval-when (load eval)
  (setf (get ':midi :scorefile) 'midifile)
  )

(defclass midifile (scorefile)
  ((element-type :initform '(unsigned-byte 8))
   (default-path :initform (default-scorefile-name :midi))
   (default-after :initform 'play)
   (length :initform 0)
   (format :initform 0 :initarg :format :initarg format)
   (tracks :initform 1 :initarg :tracks :initarg tracks)
   (time-scale)
   (time-signature :initform '(4 4) :initarg :time-signature
		   :initarg time-signature)
   (key-signature :initform nil :initarg :key-signature
		  :initarg key-signature)
   (divisions-per-quarter :initform 96 :initarg :divisions-per-quarter
			  :initarg divisions-per-quarter :type integer)
   (tempo :initform 120.0 :initarg :tempo :initarg tempo)))

;;;
;;; add a midi score to the score resource
;;;

(eval-when (load eval)
  (utils:initialize-resource 'scores 1 'midifile))

;;;
;;;
;;;

(defmethod initialize-instance :after ((score midifile) &rest args)
  (declare (ignore args))
  (setf (slot-value score 'time-scale)
	(/ (slot-value score 'divisions-per-quarter)
	   (/ 60.0 (slot-value score 'tempo))))
  (unless (= (slot-value score 'format) 0)
      (warn "Ignoring ~s value for format. Only level 0 currently implemented."
          (slot-value score 'format)))
  (unless (= (slot-value score 'tracks) 1)
    (warn "Ignoring ~s value for tracks. Only 1 track currently implemented."
          (slot-value score 'tracks))))

;;;
;;; methods on schedule-score-events 
;;;

(defmethod schedule-score-events :before ((score midifile))
  ;; output midifile preamble
  (write-midi-file-header *common-music-output*
			  (slot-value score 'format)
			  (slot-value score 'tracks)
			  (slot-value score 'divisions-per-quarter))
  (write-track-header *common-music-output* 0)
  ;; write tempo message
  (multiple-value-bind (msg data)
      (make-tempo-change (floor 60000000.0 (slot-value score 'tempo)))
    (incf (slot-value score 'length)
      (write-message msg *common-music-output* 0 data)))
  ;; write time signature
  (multiple-value-bind (msg data)
      (apply #'make-time-signature (slot-value score 'time-signature))
    (incf (slot-value score 'length)
      (write-message msg *common-music-output* 0 data)))
  (values))

(defmethod schedule-score-events :after ((score midifile))
  ;; write midi eot
  (incf (slot-value score 'length) 
    (write-message (make-eot) *common-music-output* 0 nil))
  ;; update track length
  (file-position *common-music-output* 14)
  (write-track-header *common-music-output* (slot-value score 'length))
  #+mcl (ccl:set-mac-file-type *common-music-output* "Midi")
  )

;;;
;;; the play after command for midi scores
;;;

(defmethod score-after-command ((s midifile) command cmdargs)
  (if (not (eq command 'play))
      (call-next-method)
    (let ((options cmdargs)
    	  (args '())
    	  file)
      (loop with option and value while options
       do
	(setf option (pop options))
	(unless (setf value (pop options))
	  (error "Malformed (uneven) options list to PLAY: ~s" cmdargs))
	(ecase option
	  ((port :port)
	    (push value args)
	    (push ':port args))
	  ((player :player)
	   (push value args)
	   (push ':player args))))
      (unless file (setf file (truename
				(merge-pathnames
				  (or (slot-value s 'pathname)
				       *last-scorefile-written*)
				       (slot-value s 'default-path)))))
      (apply #'midifile-play file args))))

;;;
;;; midi events
;;;

#-PCL
(eval-when (compile load eval)
(defclass midi-event (event)
  ((message :initform 0 :initarg message :initarg :message
            :accessor event-message)
   (data :initform nil :initarg :data :initarg data :accessor event-data)))
)   

#+PCL
(defclass midi-event (event)
  ((message :initform 0 :initarg message :initarg :message
            :accessor event-message)
   (data :initform nil :initarg :data :initarg data :accessor event-data)))

;;;
;;; the before method for all midi events outputs the 
;;; current time in variable length format
;;;

(defmethod score-event :before ((event midi-event) (score midifile))
  (let ((time (slot-value event 'time))
	(last (slot-value score 'time)))
    (cond ((> time last)
	   (let ((value (round (* (- time last)
				  (slot-value score 'time-scale)))))
	
	     (incf (slot-value score 'length)
		   (write-variable-quantity value *common-music-output*)))
	   (setf (slot-value score 'time) time))
	  (t
	   (write-byte 0 *common-music-output*)
	   (incf (slot-value score 'length) 1))))
  (values))

;;;
;;; the main method writes the message to the scorefile. if the
;;; event message is a system or meta message the midi data
;;; associated with the message is also written.
;;;

(defmethod score-event ((event midi-event) (score midifile))
  (let* ((message (slot-value event 'message)))
    (incf (slot-value score 'length)
       (write-message message *common-music-output* nil
                      (if (midi-channel-message-p message)
        		  nil
			(slot-value event 'data)))))
  (values))

  
;;;
;;; Midi parts.  No paramaters are declared because we roll our
;;; own score-event methods directly.
;;;

(defpart midi (midi-event part)
	(message data)
  ((syntax :initform :midi))
  :define-event-method nil
  :define-resource t)

;;;
;;; midi-note allows us to write midi files without having to worry 
;;; about converting parameter data to midi format or  breaking events
;;; into note-on/note-off pairs. all this is done automatically ny
;;; a :before method to score-event.
;;;

(defpart midi-note (midi)
         (channel note amplitude)
  ((syntax :initform :midi)
   (channel :initform 0 :initarg channel :initarg :channel
  	    :accessor event-channel)
   (note :initarg note :initarg :note
	 :accessor event-note)
   (amplitude :initarg amplitude :initarg :amplitude
   	      :accessor event-amplitude)
   (duration :initarg :duration :initarg duration
	     :accessor event-duration )
   (release :initform nil :initarg :release :initarg release 
	    :accessor event-release)
   (coerce :initform t :initarg :coerce :initarg coerce 
	   :accessor event-coerce))
  :define-event-method nil
  :define-resource t)

;;;
;;;
;;;

(defmethod score-event :before ((part midi-note) (score midifile))
  (let ((channel (slot-value part 'channel))
  	(key (slot-value part 'note))
	(velocity (slot-value part 'amplitude))
	(release (slot-value part 'release))
	(coerce (slot-value part 'coerce)))
    (when coerce
      (when (or (eq coerce t) (member 'note coerce))
	(setf key (scale-degree key *standard-scale*)))
      (when (or (eq coerce t) (member 'amplitude coerce))
	(setf velocity (floor (* velocity 127))))
      (when (or (eq coerce t) (member 'release coerce))
	(and release (setf release (floor (* release 127))))))
    (setf (slot-value part 'message)
      (make-note-on channel key velocity))
    (make-score-event 'midi-event 
		      :time (+ (slot-value part 'time)
			       (slot-value part 'duration))
		      :message (make-note-off channel key (or release 127))))
  (values))

