;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

;;;
;;; +end-of-stream-token+ is returned as the second value of the item
;;; function if the first value is the last element in the stream's
;;; current period.
;;;

(defconstant +end-of-stream-token+ :end-of-period)

(defvar *system-parsing* nil)

;;;
;;; the basic class of item stream and definitions of the essential methods 
;;; in the item stream protocol: initialize-instance, reinitialize-instance,
;;; increment-item-stream, and item.
;;;

(defclass item-stream () 
  ((items :accessor item-stream-items :initarg :items)
   (count :accessor item-stream-count :initform 0)
   (limit :accessor item-stream-limit :initform 0)
   (cache :accessor item-stream-cache :initarg :limit :initform 1)
   (value)))

;;;
;;; reitintialize-instance is called by the generic function item
;;; at the beginning of each new period of an item stream.  the
;;; basic method on reinitialize-instance selects the length of the 
;;; next period by reading a value from the period length cache, which 
;;; is either a constant number or an item stream of numbers.
;;;

(defmethod reinitialize-instance ((stream item-stream) &rest initargs)
  (declare (ignore initargs) (optimize (speed 3) (safety 0)))
  ;; period length can vary each cycle.
;  (setf (slot-value stream 'limit) (item (slot-value stream 'cache)))
  (setf (slot-value stream 'limit) 
    (read-limit-cache (slot-value stream 'cache)))
  (setf (slot-value stream 'count) 0)
  (values))

(defmethod read-limit-cache ((limit number))
  limit)

(defmethod read-limit-cache ((limit item-stream))
  (item limit))

(defmethod read-limit-cache ((limit t))
  (if (functionp limit)
      (funcall limit)
    (error "Expect function but got ~S instead." limit)))
      
;;;
;;; the basic method for increment-item-stream increments the period count.
;;; subclasses of item-stream have :after methods defined on this function
;;; that also increment item elements.
;;;

(defmethod increment-item-stream ((stream item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (incf (slot-value stream 'count) 1)
  (values))

;;;
;;; The generic function item is used to read the next element from an item 
;;; stream.  The default method returns the object it was given and an
;;; end-of-period token.
;;;

(defmethod item ((x t))
  (declare (optimize (speed 3) (safety 0)))
  (values x +end-of-stream-token+))

;;;
;;; the :around method on item for all item streams impelements the basic item
;;; stream protocol. If the stream supplied to the function is at the end of
;;; its current period, it is reinitialized before elements are accessed.  
;;; call-next-method is then invoked to return the next item. The method that
;;; is called is usually the primary item method for whatever type of stream
;;; we have, but some subclasses have more complicated dispatching.  Note that
;;; this :around method does not care how the item was generated or what its
;;; datatype is.  Once an item is returned by call-next-method, the :around 
;;; method checks to see if the item returned end-of-period. If it did, the 
;;; method can increment the current stream, otherwise if the stream is at the
;;; end of its current period it returns end-of-period, otherwise it returns
;;; the state it got from call-next-method, if it wasn't end-of-period.
;;;

(defmethod item :around ((stream item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (let (values status)
    (declare (list values))
    (when (end-of-period-p stream)
      (reinitialize-instance stream))
    (setf values (multiple-value-list (call-next-method)))
	(setf (slot-value stream 'value) (car values)) 
	(setf status (cadr values))
    (when (eq status +end-of-stream-token+)
      (increment-item-stream stream))
    (if (end-of-period-p stream)
        (setf (cadr values) +end-of-stream-token+)
      (when (eq status +end-of-stream-token+)
        (setf (cadr values) nil)))
    (values-list values)))

;;;
;;; the generic function end-of-period returns true if the specified stream is 
;;; at the end of its current period.
;;;

(defmethod end-of-period-p ((stream item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (= (slot-value stream 'count)
     (slot-value stream 'limit)))

;(defmethod end-of-period-p ((stream t))
;  (declare (optimize (speed 3) (safety 0)))
;  t)

;;;
;;; the generic function period-length returns the length of the
;;; item stream's current period.
;;;

(defmethod period-length ((stream item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (slot-value stream 'limit))

(defmethod period-length ((stream t))
  (declare (optimize (speed 3) (safety 0)))
  1)

;;;
;;; return the last selected item, if any.
;;;

(defun item-stream-value (stream &optional default)
  (let ((flg (slot-boundp stream 'value)))
    (if flg
	    (values (slot-value stream 'value) t)
      (values default nil))))
 
;;;
;;; sequential item streams.
;;;

(defclass sequential-item-stream (item-stream) 
  ())

(defmethod item ((stream sequential-item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (item (first (slot-value stream 'items))))

(defmethod increment-item-stream :after ((stream sequential-item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (let ((tail (cdr (slot-value stream 'items))))
    (when tail
      (setf (slot-value stream 'items) tail))))

;;;
;;; cyclic item streams.
;;;

(defclass cyclic-item-stream (item-stream)
  ((item-top :accessor cyclic-item-stream-item-top :initarg :items)))

(defmethod initialize-instance :after ((stream cyclic-item-stream)
				       &rest initargs)
  (declare (ignore initargs) (optimize (speed 3) (safety 0)))
  (setf (slot-value stream 'item-top) (slot-value stream 'items)))

(defmethod item ((stream cyclic-item-stream))
  (declare (optimize (speed 3)(safety 0)))
  (item (car (slot-value stream 'items))))

(defmethod increment-item-stream :after ((stream cyclic-item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (let ((tail (cdr (slot-value stream 'items))))
    (if tail (setf (slot-value stream 'items) tail)
      (setf (slot-value stream 'items) (slot-value stream 'item-top)))))

;;;
;;; accumulating item streams.
;;;

(defclass accumulating-item-stream (item-stream)
  ((minor :initform 0)
   (major :initform 0)
   (length :initarg :length :initform nil)
   (top :initarg :items)))

(defmethod initialize-instance :after ((stream accumulating-item-stream) 
				       &rest initargs)
  (declare (ignore initargs))
  (unless (slot-value stream 'length)
    (setf (slot-value stream 'length)
      (length (slot-value stream 'items)))))
	
(defmethod item ((stream accumulating-item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (item (car (slot-value stream 'items))))

(defmethod increment-item-stream :after ((stream accumulating-item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (if (= (slot-value stream 'minor)
	 (slot-value stream 'major))
      (setf (slot-value stream 'minor) 0
	    (slot-value stream 'items) (slot-value stream 'top)
	    (slot-value stream 'major) (1+ (slot-value stream 'major)))
      (setf (slot-value stream 'items)
	    (cdr (slot-value stream 'items))
	    (slot-value stream 'minor) (1+ (slot-value stream 'minor))))
  (when (= (slot-value stream 'major)
	   (slot-value stream 'length))
    (setf (slot-value stream 'major) 0)))

;;;
;;; heap item streams.
;;;

(defclass heap-item-stream (item-stream)
  ((heap :accessor heap-item-stream-heap :initarg :items)
   (first :accessor heap-item-stream-first :initarg :first :initform nil)
   (state :accessor heap-item-stream-state :initarg :state
	  :initform *cm-state*)
   (length :accessor heap-item-stream-length :initarg :length)))

(defun shuffle-heap (heap length state first)
  (loop for i below length
        do (rotatef (elt heap i) (elt heap (random length state))))
  (when first
    (let ((pos (position first heap)))
      (rotatef (elt heap 0) (elt heap pos))))
  heap)

(defmethod initialize-instance :after ((stream heap-item-stream)
				       &rest initargs)
  (declare (ignore initargs) (optimize (speed 3) (safety 0)))
  (let ((first (slot-value stream 'first)))
    (setf (slot-value stream 'heap)
      (shuffle-heap (slot-value stream 'items)
		    (slot-value stream 'length) 
		    (slot-value stream 'state)
		    (cdr first)))
    (when (and first (eq (car first) ':initially))
      (setf (slot-value stream 'first) nil))))

(defmethod item ((stream heap-item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (item (car (slot-value stream 'heap))))

(defmethod increment-item-stream :after ((stream heap-item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (let ((tail (cdr (slot-value stream 'heap))))
    (if tail 
	(setf (slot-value stream 'heap) tail)
      (let ((items (slot-value stream 'items)))
	(shuffle-heap items (slot-value stream 'length)
		      (slot-value stream 'state)
		      (cdr (slot-value stream 'first)))
	(setf (slot-value stream 'heap) items)))))

;;;
;;; random item streams.
;;;

(defstruct (random-item (:type list))		
  datum index (weight 1.0) (min 1) max (count 0) id)

(defclass random-item-stream (item-stream)
  ((item :accessor random-item-stream-item :initform nil)
   (state :accessor random-item-stream-state :initarg :state
	  :initform *cm-state*)
   (first :accessor random-item-stream-first :initarg :first :initform nil)
   (range :accessor random-item-stream-range)))

;;;
;;; reinitialize-instance resets item probabilities and the
;;; initial element of the next period.
;;;

(defmethod reinitialize-instance :after ((stream random-item-stream)
					 &rest initargs)
  (declare (ignore initargs)(optimize (speed 3)(safety 0)))
  ;; reset-random-indicies recalculates the probabilities of the random items.
  ;; item weights are converted into monotonically increasing indicies over
  ;; the range (sum) of item weights.  set-random-indicies is called at the 
  ;; beginning of each new period. note that the item function is called on
  ;; the weight of each item.  this allows the probability distribution to
  ;; change over time.
  (macrolet ((reset-random-indicies (stream)
	       `(loop with range = 0 
		 for item in (slot-value ,stream 'items)
		 do (incf range (item (random-item-weight item)))
		    (setf (random-item-index item) range)
		 finally (setf (slot-value ,stream 'range) range)))
	     (reset-random-counts (stream)
	       `(loop for i in (slot-value ,stream 'items)
		 do (setf (random-item-count i) 0))))
    (reset-random-indicies stream)
    (let ((first (slot-value stream 'first)))
      (if first
	  (progn 
	    (setf (slot-value stream 'item) (cdr first))
	    (if (eq (car first) ':reinitially)
	        ;; if reselecting a "first" item each new period
	        ;; then we need to clear the current counts
	        (reset-random-counts stream)
	      ;; if there isn't a reselected first item then 
	      ;; we just set the first item and clear the cache.
	      ;; we don't clear the counts so that min/max constraints
	      ;; are applicable on item selection across period bounds.
	      (setf (slot-value stream 'first) nil)))
        (select-next-item stream)))))

;;;
;;; select-next-item selects a new item based on probability and optional
;;; constraints on minimum and maximum numbers of direct repetition.
;;;

(defmethod select-next-item ((stream random-item-stream))
  (macrolet ((pick-random-item (stream)
	       `(loop with index = (random (slot-value ,stream 'range)
					   (slot-value ,stream 'state))
		 for item in (slot-value ,stream 'items)
		 when (< index (random-item-index item))
		 return item))
	     (reset-random-counts (stream)
	       `(loop for i in (slot-value ,stream 'items)
		 do (setf (random-item-count i) 0))))
    (let ((last (slot-value stream 'item)))
      (when (or (null last)
		(>= (incf (random-item-count last) 1)
		    (random-item-min last)))
	(let ((next (loop with item
		     do (setf item (pick-random-item stream))
		     unless (and (random-item-max item)
				 (= (random-item-count item)
				    (random-item-max item)))
		     return item)))
	  (unless (eq next last)
	    (reset-random-counts stream))
	  (setf (slot-value stream 'item) next))))))


;;;
;;; random-item-streams operate "one ahead", ie the the current item is
;;; selected before the item is accessed, not afterward. this means that
;;; increment-item-stream must not increment the item if we are at the end of
;;; the period because reinitialize-instance selects the initial element for
;;; each period according to the existence of an element tagged as being the
;;; first item to return.
;;;

(defmethod increment-item-stream :after ((stream random-item-stream))
  (if (< (slot-value stream 'count) (slot-value stream 'limit))
      (select-next-item stream)
    (let ((first (slot-value stream 'first)))
      (if first
	  (progn 
	    (setf (slot-value stream 'item) (cdr first))
	    ;; reset counts
	    (loop for i in (slot-value stream 'items)
		  do (setf (random-item-count i) 0)))))))

(defmethod item ((stream random-item-stream))
  (declare (optimize (speed 3) (safety 0)))
  (item (random-item-datum (slot-value stream 'item))))

;;;
;;; graph item stream.
;;;

(defstruct (graph-node (:type list))		
  id datum to props)

(defun default-graph-node-select (stream node &rest lastids)
  (let ((to (graph-node-to node)))
    (if (consp to)
        (if (eq (car to) ':idsel)
            (markov-select stream node to lastids)
          (nth (random (length to) *cm-state*) to))
      (item to))))
;;;
;;; for markov processes
;;;

(defun match-ids (user last)
  ;; match the user's ids with the past choices. * is a wildcard. matching
  ;; could get really fancy if we wanted it to.
  (cond ((null user) t)
        ((null last) nil)
        ((consp user)
         (and (match-ids (car user) (car last))
              (match-ids (cdr user) (cdr last))))
        ((eq user last) t)
        ((eq user '*) t)
        ((eq last '*) t)
        (t nil)))

(defun markov-select (stream node table lastids)
  (declare (ignore stream))
  ;;  table is a list (:idsel <id1> <stream1> ...)
  (let ((prob (loop for (i s) on (cdr table) by #'cddr
                    when (match-ids i lastids)
                    return s)))
    (unless prob
      (error "Node ~S has no entry for previous ids ~S in ~S."
             (graph-node-id node) lastids table))
    (item prob)))

		 
(defclass graph-item-stream (item-stream) 
  ((item :initarg :first :accessor graph-item-stream-item :initform nil)
   (selectfn :initarg :selectfn :accessor graph-item-stream-selectfn)
   (last :initarg :last :initform nil :accessor graph-item-stream-last)
   (props :initarg :props :accessor graph-item-stream-props)))

(defmethod reinitialize-instance :after ((stream graph-item-stream)
					 &rest initargs)
  (declare (ignore initargs)(optimize (speed 3)(safety 0)))
  (unless (slot-value stream 'item)
    (setf (slot-value stream 'item)
          (car (slot-value stream 'items)))))

(defmethod item ((stream graph-item-stream))
  (declare (optimize (speed 3)(safety 0)))
  (item (graph-node-datum (slot-value stream 'item))))

(defmethod increment-item-stream :after ((stream graph-item-stream))
  ;  (declare (optimize (speed 3) (safety 0)))
  (let ((last (slot-value stream 'last))
        (this (slot-value stream 'item))
        next node)
    (if (setf next (apply (slot-value stream 'selectfn)
                          stream this (cdr last)))
      (if (setf node (find next (slot-value stream 'items) 
                           :key #'graph-node-id)) 
        (progn
          ;; next item becomes selected node
          ;; car of last is the number of choices to remember.
          ;; push old selection onto the list and flush the
          ;; the oldest element. since we cant setf nthcdr
          ;; in some lisps, we setf the cdr of nthcdr-1...
          (setf (slot-value stream 'item) node)
          (when last
            (push (graph-node-id this) (cdr last))
            (setf (cdr (nthcdr (1- (car last)) (cdr last))) nil)))
        (error "Missing node for id ~S." next))
      (error "No next id returned from node ~A and function ~S"
             (graph-node-id node) (slot-value stream 'selectfn)))))
    
;;;
;;; functional item streams.
;;;

(defclass functional-item-stream (item-stream)
  ((generator :accessor functional-item-stream-generator :initarg :generator)
   (funargs :accessor functional-item-stream-funargs :initarg :funargs 
	    :initform nil)
   (states :initform nil)))

(defmethod item ((stream functional-item-stream))
  (declare (optimize (speed 3)(safety 0)))
  (values (pop (slot-value stream 'items))
          (if (slot-value stream 'states)
              (pop (slot-value stream 'states)) 
            (if (slot-value stream 'items) nil +end-of-stream-token+))))

(defmethod reinitialize-instance :after ((stream functional-item-stream)
					 &rest initargs)
  (declare (ignore initargs)(optimize (speed 3) (safety 0)))
  (multiple-value-bind (items states)
                       (apply (slot-value stream 'generator) 
                              (slot-value stream 'funargs))
    ;; allow the user function to return the degenerate case of a
    ;; single item instead of the proper one element list.  
    ;; a null item list is still an error, however.
    (if (null items)
      (error "No item list returned to functional item stream.")
      (unless (listp items) items
              (setf items (list items))))
    (setf (slot-value stream 'items) items
          (slot-value stream 'limit) 1   ; could be :initform
          (slot-value stream 'states) states)))
#|
(setf x (repeat (items a b c in random) 2))
(read-items x t t) (read-items x t t) (read-items x t t)
(read-items x t t) (read-items x t t) (read-items x t t)

(setf x (mirror (chord a b c)))
(read-items x t t)
(read-items x t t)

(setf x (items in function with #'(lambda () (list 'a 'b 'c))))
(read-items x t t)
(read-items x t t)
|#

;;;
;;; read-items returns a list of items read from a stream. If number-of-times
;;; is not supplied the stream will be read for one of its periods.
;;;

(defun read-items (stream &optional (length t) states?)
  (if (numberp length)
      (loop with x and y repeat length
            do (multiple-value-setq (x y) (item stream))
            collect x into items
            if states? collect y into states
            finally (return (if states (values items states) items)))
    (loop with x and y
          do (multiple-value-setq (x y) (item stream))
          collect x into items
          if states? collect y into states
          until (eq y +end-of-stream-token+)
          finally (return (if states (values items states) items)))))


;;;
;;; the generic function map-items invokes a user specified function on each 
;;; element of an item stream.  The supplied function may return zero, one or
;;; two values. if the second value is non nil, map-items will replace the
;;; current element in the stream with the first value.
;;;

(defmethod map-items (function (stream sequential-item-stream))
  (map-items-1 function (slot-value stream 'items)))

(defmethod map-items (function (stream cyclic-item-stream))
  (map-items-1 function (slot-value stream 'item-top)))

(defmethod map-items (function (stream accumulating-item-stream))
  (map-items-1 function (slot-value stream 'top)))

(defmethod map-items (function (stream heap-item-stream))
  (map-items-1 function (slot-value stream 'items) 
	       (slot-value stream 'length)))

(defmethod map-items (function (stream random-item-stream))
  (map-items-2 function (slot-value stream 'items)))

(defmethod map-items (function (stream graph-item-stream))
  (map-items-2 function (slot-value stream 'items)))

(defmethod map-items (function (stream functional-item-stream))
  (declare (ignore function))
  (error "Attempt to call MAP-ITEMS on a FUNCTIONAL-ITEM-STREAM."))

;;;
;;; map-items-1 and map-items-2 both work on sequences, not just lists.
;;; map-item-1 assumes that each element of the sequence is an item to be
;;; mapped. map-item-2 assumes that the element to be mapped is in the
;;; car of each sequence element.
;;;

(defun map-items-1 (function sequence &optional length)
  (loop for i below (or length (length sequence))
        do
    (multiple-value-bind (value flag)
	(funcall function (elt sequence i))
      (when flag 
	(setf (elt sequence i) value))))
  (values))

(defun map-items-2 (function sequence &optional length)
  (loop with holder for i below (or length (length sequence))
       do
    (setf holder (elt sequence i))
    (multiple-value-bind (value flag)
	(funcall function (car holder))
      (when flag 
	(setf (car holder) value))))
  (values))

;;;
;;; every item stream constructor allows a stream to be "named". a named 
;;; stream can then be referenced via the read macro #@
;;;

(defvar *named-streams* (make-hash-table))

(defmacro cache-stream (name stream)
  `(setf (gethash ,name *named-streams*)
      ,stream))

(defun find-stream (name &optional (mode :error))
  (or (gethash name *named-streams*)
      (ecase mode
	(:error (error "~A is not a named item stream." name))
	(:ask nil))))

(defmacro motive (name)
  `(find-stream ',name))

;;;
;;;
;;;

(defmacro doitems ((var stream &key periods length) &body body)
  (declare (ignore length))
  (let ((streamvar (gensym))
        (periodvar (gensym))
        (countvar (gensym))
        itemvar statusvar)	
    (if (consp var)
        (setf itemvar (car var) statusvar (cadr var))
      (setf itemvar var statusvar (gensym)))	
    `(let ((,streamvar ,stream)
           (,periodvar (or ,periods 1))
           ,itemvar ,statusvar)
       (loop do 
             (multiple-value-setq (,itemvar ,statusvar)
 	                          (item ,streamvar))
             (progn ,@body)
             count (eq ,statusvar ':end-of-period) into ,countvar
             until (= ,countvar ,periodvar)))))

