;;; -*- Mode: Lisp; Syntax: Common-lisp; Package: common-music; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address electronic correspondence to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

;;;
;;; Interval, step and voicing definitions. Interval streams add the selected
;;; interval to an offset.  Step stream return the offset and then increment
;;; it by the selected interval. Voicing streams are like interval streams, 
;;; except that the offset it selected each time an interval is selected. 
;;; Inteval and step streams reselect the offset once each period.
;;;

(defclass essential-interval ()
  ((offset :accessor interval-stream-offset)
   (offset-cache :accessor interval-stream-offset-cache :initarg :offset)))

(defmethod reinitialize-instance :after ((stream essential-interval)
                                         &rest initargs)
  (declare (ignore initargs))
  (setf (slot-value stream 'offset)
        (read-offset (slot-value stream 'offset-cache) stream)))

(defclass interval-mixin (essential-interval)
     ())

(defclass step-mixin (essential-interval)
     ())

(defclass voicing-mixin (essential-interval)
     ())

(defmethod increment-item-stream :after ((stream voicing-mixin))
  (unless (= (slot-value stream 'count)
	     (slot-value stream 'limit))
    (setf (slot-value stream 'offset)
	  (read-offset (slot-value stream 'offset-cache) stream))))

;;;
;;; The item methods for the different interval types. Interval and step
;;; increment their offset once per period. Voicing stream increment the
;;; offset after each read.
;;;

(defmethod item :around ((stream interval-mixin))
  (declare (optimize (speed 3)(safety 0)))
  (multiple-value-bind (interval state) 
      (call-next-method)
    (let ((offset (slot-value stream 'offset)))
      (if (or (eq interval *rest-degree*)
              (restp offset))
          (values *rest-degree* state)
        (values (+ (the fixnum interval) (the fixnum offset))
                state)))))

(defmethod item :around ((stream step-mixin))
  (declare (optimize (speed 3)(safety 0)))
  (multiple-value-bind (interval state) 
      (call-next-method)	
    (let ((offset (slot-value stream 'offset)))
      (unless (or (eq interval *rest-degree*)
                  (restp offset))
        (incf (the fixnum (slot-value stream 'offset))
              (the fixnum interval)))
      (values offset state))))

(defmethod item :around ((stream voicing-mixin))
  (declare (optimize (speed 3)(safety 0)))
  (multiple-value-bind (interval state)
      (call-next-method)
    (let ((offset (slot-value stream 'offset)))
      (if (or (eq interval *rest-degree*)
              (restp offset))
          (values *rest-degree* state)
        (values (+ (the fixnum interval) (the fixnum offset))
                state)))))

;;;
;;;  Read-offset converts an offset to a transposition level.
;;;

(defmethod read-offset ((offset integer) stream)
  (declare (ignore stream))
  offset)

(defmethod read-offset ((offset float) stream)
  (declare (ignore stream))
  (scale-degree offset *standard-scale*))

(defmethod read-offset ((offset symbol) stream)
  (declare (ignore stream))
  (scale-degree offset *standard-scale*))

(defmethod read-offset ((offset note-stream-mixin) stream)
  (declare (ignore stream))
  (scale-degree (item offset) (note-stream-scale offset)))

(defmethod read-offset ((offset step-mixin) stream)
  (declare (ignore stream))
  (item offset))

(defmethod read-offset ((offset item-stream) stream)
  (declare (ignore stream))
  ;(item offset)
  (scale-degree (item offset) *standard-scale*)
   )

(defmethod read-offset ((offset cons) stream)
  (cond ((eq (car offset) ':initially-from)
         (current-scale-degree stream (cadr offset)))
        ((eq (car offset) ':linked-to)
         (current-scale-degree (find-stream (cadr offset)) (caddr offset)))
        (t (call-next-method)))) 

(defun current-scale-degree (stream default)
  (let ((slot (if (typep stream 'step-mixin) 'offset 'value))
        (scale (if (typep stream 'note-stream-mixin)
                   (slot-value stream 'scale)
                 *standard-scale*)))
    (if (slot-boundp stream slot)
        (scale-degree (slot-value stream slot) scale)
      (if default 
          (scale-degree default scale)
        (error "~S has no current scale degee." stream))))) 

(defmethod read-offset ((offset t) stream)  
  (if (functionp offset)
      (let ((*stream* stream))
        (declare (special *stream*))
        (funcall offset))
    (error "Expected functionp offset but got ~S instead." offset)))

;;;
;;; Interval stream definitions.
;;;

(defclass cyclic-interval-stream 
  (cyclic-item-stream interval-mixin) ())
(defclass cyclic-interval-note-stream 
  (cyclic-item-stream note-mixin interval-mixin) ())
(defclass cyclic-interval-pitch-stream 
  (cyclic-item-stream pitch-mixin interval-mixin) ())

(defclass cyclic-step-stream 
  (cyclic-item-stream step-mixin) ())
(defclass cyclic-step-note-stream 
  (cyclic-item-stream note-mixin step-mixin) ())
(defclass cyclic-step-pitch-stream 
  (cyclic-item-stream pitch-mixin step-mixin) ())

(defclass cyclic-voicing-stream 
  (cyclic-item-stream voicing-mixin) ())
(defclass cyclic-voicing-note-stream 
  (cyclic-item-stream note-mixin voicing-mixin) ())
(defclass cyclic-voicing-pitch-stream 
  (cyclic-item-stream pitch-mixin voicing-mixin) ())

;;;
;;;
;;;

(defclass sequential-interval-stream 
  (sequential-item-stream interval-mixin) ())
(defclass sequential-interval-note-stream 
  (sequential-item-stream note-mixin interval-mixin) ())
(defclass sequential-interval-pitch-stream 
  (sequential-item-stream pitch-mixin interval-mixin) ())

(defclass sequential-step-stream 
  (sequential-item-stream step-mixin) ())
(defclass sequential-step-note-stream 
  (sequential-item-stream note-mixin step-mixin) ())
(defclass sequential-step-pitch-stream 
  (sequential-item-stream pitch-mixin step-mixin) ())


(defclass sequential-voicing-stream 
  (sequential-item-stream voicing-mixin) ())
(defclass sequential-voicing-note-stream 
  (sequential-item-stream note-mixin voicing-mixin) ())
(defclass sequential-voicing-pitch-stream 
  (sequential-item-stream pitch-mixin voicing-mixin) ())

;;;
;;;
;;;

(defclass accumulating-interval-stream 
  (accumulating-item-stream interval-mixin) ())
(defclass accumulating-interval-note-stream 
  (accumulating-item-stream note-mixin interval-mixin) ())
(defclass accumulating-interval-pitch-stream 
  (accumulating-item-stream pitch-mixin interval-mixin) ())

(defclass accumulating-step-stream 
  (accumulating-item-stream step-mixin) ())
(defclass accumulating-step-note-stream 
  (accumulating-item-stream note-mixin step-mixin) ())
(defclass accumulating-step-pitch-stream 
  (accumulating-item-stream pitch-mixin step-mixin) ())


(defclass accumulating-voicing-stream 
  (accumulating-item-stream voicing-mixin) ())
(defclass accumulating-voicing-note-stream 
  (accumulating-item-stream note-mixin voicing-mixin) ())
(defclass accumulating-voicing-pitch-stream 
  (accumulating-item-stream pitch-mixin voicing-mixin) ())

;;;
;;;
;;;

(defclass random-interval-stream 
  (random-item-stream interval-mixin) ())
(defclass random-interval-note-stream 
  (random-item-stream note-mixin interval-mixin) ())
(defclass random-interval-pitch-stream 
  (random-item-stream pitch-mixin interval-mixin) ())

(defclass random-step-stream 
  (random-item-stream step-mixin) ())
(defclass random-step-note-stream 
  (random-item-stream note-mixin step-mixin) ())
(defclass random-step-pitch-stream 
  (random-item-stream pitch-mixin step-mixin) ())

(defclass random-voicing-stream 
  (random-item-stream voicing-mixin) ())
(defclass random-voicing-note-stream 
  (random-item-stream note-mixin voicing-mixin) ())
(defclass random-voicing-pitch-stream 
  (random-item-stream pitch-mixin voicing-mixin) ())

;;;
;;;
;;;

(defclass heap-interval-stream
  (heap-item-stream interval-mixin) ())
(defclass heap-interval-note-stream 
  (heap-item-stream note-mixin interval-mixin) ())
(defclass heap-interval-pitch-stream 
  (heap-item-stream pitch-mixin interval-mixin) ())

(defclass heap-step-stream 
  (heap-item-stream step-mixin) ())
(defclass heap-step-note-stream 
  (heap-item-stream note-mixin step-mixin) ())
(defclass heap-step-pitch-stream 
  (heap-item-stream pitch-mixin step-mixin) ())

(defclass heap-voicing-stream 
  (heap-item-stream voicing-mixin) ())
(defclass heap-voicing-note-stream 
  (heap-item-stream note-mixin voicing-mixin) ())
(defclass heap-voicing-pitch-stream 
  (heap-item-stream pitch-mixin voicing-mixin) ())

;;;
;;;
;;;

(defclass graph-interval-stream 
  (graph-item-stream interval-mixin) ())
(defclass graph-interval-note-stream 
  (graph-item-stream note-mixin interval-mixin) ())
(defclass graph-interval-pitch-stream 
  (graph-item-stream pitch-mixin interval-mixin) ())

(defclass graph-step-stream 
  (graph-item-stream step-mixin) ())
(defclass graph-step-note-stream 
  (graph-item-stream note-mixin step-mixin) ())
(defclass graph-step-pitch-stream 
  (graph-item-stream pitch-mixin step-mixin) ())

(defclass graph-voicing-stream 
  (graph-item-stream voicing-mixin) ())
(defclass graph-voicing-note-stream 
  (graph-item-stream note-mixin voicing-mixin) ())
(defclass graph-voicing-pitch-stream 
  (graph-item-stream pitch-mixin voicing-mixin) ())

;;;;;
;;;;;
;;;;;

(defclass functional-interval-stream 
  (functional-item-stream interval-mixin) ())
(defclass functional-interval-note-stream 
  (functional-item-stream note-mixin interval-mixin) ())
(defclass functional-interval-pitch-stream 
  (functional-item-stream pitch-mixin interval-mixin) ())

(defclass functional-step-stream 
  (functional-item-stream step-mixin) ())
(defclass functional-step-note-stream 
  (functional-item-stream note-mixin step-mixin) ())
(defclass functional-step-pitch-stream 
  (functional-item-stream pitch-mixin step-mixin) ())

(defclass functional-voicing-stream 
  (functional-item-stream voicing-mixin) ())
(defclass functional-voicing-note-stream 
  (functional-item-stream note-mixin voicing-mixin) ())
(defclass functional-voicing-pitch-stream 
  (functional-item-stream pitch-mixin voicing-mixin) ())

;;;
;;; intervals are parsed as intergers.
;;;

(defmethod parse-interval ((interval item-stream))
  interval)

(defmethod parse-interval ((interval integer))
  interval)

(defmethod parse-interval ((interval t))
  (if (or (eq interval *rest-name*) (eq interval *rest-pitch*))
      most-negative-fixnum
  (error "~&can't parse ~s as an interval." interval)))


;;;
;;; the series stream  is a cyclic interval stream that supports 
;;; most types of serial (twelve-tone) manipulations.  the intervals
;;; slot holds the series. the form (p i r ri), offset, multiple and
;;; modulus of the series is selected each time the series is
;;; incremented  to its initial interval.
;;;

(defclass series-stream (cyclic-item-stream)
  ((intervals :accessor series-intervals)
   (form :accessor series-form :initform 'p :initarg :form)
   (modulus :accessor series-modulus :initform most-positive-fixnum
            :initarg :modulus)
   (multiple :accessor series-multiple :initarg :multiple :initform 1)
   (offset :accessor series-offset :initform 0 :initarg :offset)))

(defclass series-note-stream (series-stream note-mixin)
  ())

(defclass series-pitch-stream (series-stream pitch-mixin)
  ())


(defmethod initialize-instance :after ((stream series-stream) &rest args)
  (declare (ignore args))
  (setf (series-intervals stream)
	(copy-tree (item-stream-items stream)))
  (set-current-series stream))


(defmethod increment-item-stream :after ((stream series-stream))
  (when (eq (slot-value stream 'item-top)
            (slot-value stream 'items))
    (set-current-series stream)))

(defun set-current-series (stream)
;  (declare (optimize (speed 3)(safety 0)))
  (let ((intervals (series-intervals stream))
	(form (item (series-form stream)))
	(multiple (item (series-multiple stream)))
	(modulus (item (series-modulus stream)))
	(offset (read-offset (series-offset stream) stream))
	(inversion 1))
    (ecase form
      ((:p p :prime prime))
      ((:r r :retrograde retrograde)
       (setf intervals (reverse intervals)))
      ((:i i :inversion inversion)
       (setf inversion -1))
      ((:ri ri :retrograde-inversion retrograde-inversion)
       (setf inversion -1 intervals (reverse intervals))))

    (loop for tail on (item-stream-items stream)
	  for interval in intervals
	  do
      (setf (car tail) (+ offset (* (mod (* interval multiple)
					 modulus)
				    inversion))))))
