;;; -*- Mode: LISP; Syntax: Common-lisp; Package: common-music; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

;;;
;;; class-with-parameters is the meta class for events and parts. we need
;;; class-with-parameters in order to define a "parameters" slot for all part
;;; and event classes.  this slot will hold a list of information about each
;;; slot in the class that has been declared to be an output parameter. each 
;;; element in this list is a structure (well, list...) called a "pinfo" 
;;; which stores information about a particular parameter slot. this
;;; information is used by various routines in the system, particularily
;;; the methods that generate score-event methods. class-with-parameters also
;;; declares an additional slot called  "accessors" which is used to store 
;;; a list of (slot-name slot-accessor) pairs.  this list is used by with-part
;;; to map variable names to accessor functions, ie, a form like
;;; (setf rhythm 1.0) is rewritten as (setf (event-rhythm #:a001) 1.0)).  
;;; the reason we need this accessor hair (rather than simply using clos's
;;; with-slots) is because the runtime evaluation environment is a simple
;;; function, not a method, so the compiler can't optimize slot-value.  in
;;; other words, accessors are a lot faster than slot-value in regular
;;; functions...sigh.
;;;

(defclass class-with-parameters (standard-class)
  ((parameters :accessor class-parameters :initarg :parameters :initform nil)
;   (accessors :accessor class-accessors :initform nil)
   ))

(defmethod initialize-instance :after ((class class-with-parameters)
                                       &rest initargs)
  (initialize-class-with-parameters class (getf initargs ':parameters)))

(defmethod reinitialize-instance :after ((class class-with-parameters)
                                         &rest initargs)
  (initialize-class-with-parameters class (getf initargs ':parameters)))

(defun initialize-class-with-parameters (class local-pars)
  ;; if the parameters were not locally declared, set them to the
  ;; union of all parameters of the direct superclasses.
  (if local-pars
      #+MCL (setf (slot-value class 'parameters ) local-pars)
      #-MCL NIL
    (setf (slot-value class 'parameters)
      (loop for super in (class-direct-superclasses class)
	        when (typep super 'class-with-parameters)
	        append (event-parameters super))))
;  ;; if this is a type of score part, compute and cache the slot and
;  ;; accessor pairs for use by with-part to avoid needless consing.
;  #+pcl
;  (when (typep (class-prototype class) 'part)
;    (setf (slot-value class 'accessors)
;      (compute-slot-and-accessor-pairs class)))
   )
   
;#-pcl
;(defmethod finalize-inheritance :after ((class class-with-parameters))
;  (when (typep (class-prototype class) 'part)
;    (setf (slot-value class 'accessors)
;      (compute-slot-and-accessor-pairs class))))


(defmethod validate-superclass ((subclass class-with-parameters)
                                (superclass standard-class))
  't)

;;;
;;; event-parameters returns the pinfos for a given class of event or part.
;;;

(defmethod event-parameters ((class class-with-parameters))
  (class-parameters class))

(defmethod event-parameters ((instance event))
  (event-parameters (class-of instance)))

;;;
;;; slot-and-accessor-pairs computes a list of slot and accessor pairs for
;;; a specified class. it is used by the code walker in with-part to resolve
;;; slots treated as variables. I really wish clos had something like a
;;; (declare (foo class)) declaration because it would allow compilers to
;;; optimize slot-value inside regular functions, not just generic functions.
;;;

;(defmethod slot-and-accessor-pairs ((class class-with-parameters))
;  (class-accessors class))

;(defmethod slot-and-accessor-pairs ((class standard-class))
;  (compute-slot-and-accessor-pairs class))

;;;
;;; pinfo's store information about slots that have been declared to be event
;;; parameters and are stored in the parameters slot of the class object.  at
;;; this point i don't remember why i just didn't use (defstruct (:type list)!
;;;

(defmacro pinfo-name (pinfo)
  `(car ,pinfo))

(defmacro pinfo-type (pinfo)
  `(getf (cdr ,pinfo) ':type))

(defmacro pinfo-printer (pinfo)
  `(getf (cdr ,pinfo) ':printer))

(defmacro pinfo-message (pinfo)
  `(getf (cdr ,pinfo) ':message))

(defmacro pinfo-value-type (pinfo)
  `(getf (cdr ,pinfo) ':value-type))

(defmacro pinfo-keyword (pinfo)
  `(getf (cdr ,pinfo) ':keyword))

(defun make-pinfo (&key name type printer value-type message default)
  (let ((pinfo (list (or name (error "Missing parameter name.")))))
    (when (or printer (setf printer (pinfo-printer default)))
      (setf (pinfo-printer pinfo) printer))
    (when (or value-type (setf value-type (pinfo-value-type default)))
      (setf (pinfo-value-type pinfo) value-type))
    (when (or message (setf message (pinfo-message default)))
      (unless (or (eq type ':message)
		  (eq (pinfo-type default) ':message))
	(error "Slot ~A contains a non local message declaration." name))
      (setf (pinfo-message pinfo) message))
    ;; all message parameters have a keyword name for event sequences.
    (when (eq type ':message)
      ;; easy if user didn't specify a message because the message is the
      ;; made up of the slot's name.  
      (if (not (pinfo-message pinfo)) 
	  (setf (pinfo-keyword pinfo)
	    (intern (string (pinfo-name pinfo)) ':keyword))
	;; if we have to use a user specified a message string, we must strip
	;; non-keywordy chars from the message and hope for the best...
	(let ((str (string-upcase (string-trim '(#\space #\: #\,) 
					       (pinfo-message pinfo)))))
	  (setf (pinfo-keyword pinfo) (intern str :keyword)))))
    (setf (pinfo-type pinfo)
      (if (eq type ':forward)
	  (or (pinfo-type default)
	      (error "Slot ~A never declared a parameter." name))
	(or type (pinfo-type default)
	    (error "Can't find a parameter type for ~A." name))))
    pinfo))

;;;
;;; parse-parameter-declaration used by defpart to parse its "lambda list"
;;; style parameter declaration.  &message replaces &key because this is what
;;; these things were called in sambox and are still called by the music kit...
;;;

(defconstant *parameter-types* '(&required &message &optional &rest))

(defun parse-parameter-declaration (decl)
  (let ((type (if (member (car decl) *parameter-types*)
		  (pop decl) '&REQUIRED))
	required optional message rest)
    (loop while decl
	  do
      (ecase type
	(&required
	  (when (or required message optional rest)
	    (error "Illegal &required found."))
	  (setf required (loop for s = (car decl)
			       while s until (member s *parameter-types* 
						     :test #'eq)
			       collect (pop decl))))
	(&optional
	  (when (or message optional rest)
	    (error "Illegal &optional found."))
	  (setf optional (loop for s = (car decl)
			       while s until (member s *parameter-types* 
						     :test #'eq)
			       collect (pop decl))))
	(&message
	  (when (or message rest)
	    (error "Illegal &message found."))
	  (setf message (loop for s = (car decl)
			      while s until (member s *parameter-types* 
						    :test #'eq)
			      collect (pop decl))))
	(&rest
	  (when rest
	    (error "Illegal &rest found."))
	  (setf rest (loop for s = (car decl)
			       while s until (member s *parameter-types* 
						     :test #'eq)
			       collect (pop decl)))
	  (when (> (length rest) 1)
	    (error "Found multiple &rest parameter ~s." rest))))
      (when (setf type (pop decl))
	(unless decl
	  (error "Found dangling ~A parameter qualifier." type))))
    (let ((dupes (or (intersection required message :test #'eq)
		     (intersection required optional :test #'eq))))
      (when dupes
	(error "Found duplicate parameter declarations ~s." dupes)))
    (values required optional message rest)))

;;;
;;; slots that have been declared to be parameters (either locally in the
;;; defpart declaration or inherited from some super class) support a
;;; few additional slot specifications to those normally supplied by
;;; defclass. a :format slot option allows a system supplied format keyword
;;; or a user specified print function to be associated with outputting the
;;; the slot value. a :message option allows the system supplied message
;;; string to be overridden for message parameters. if a :message string is
;;; supplied it must contain any trailing "white space" delimiter, if
;;; appropriate.
;;;

(defun parse-event-slot-spec (spec type &optional pinfo slot)
  (let ((name (if (listp spec) (pop spec)
		  (prog1 spec (setf spec nil))))
	(parsp type)
	(slotargs ())
	(printer ())
	(value-type ())
	(message ()))
    (loop for (prop value) on spec by #'cddr
	  do
      (case prop
	(:printer
	  (unless printer
	    (setf printer value))
	  (setf parsp t))
	(:value-type
	  (ecase value
	    (:float )
	    (:quoted )
	    (:quoted-string )
	    (:integer ))
	  (unless value-type
	    (setf value-type value))
	  (setf parsp t))
	(:message
	  (unless message
	    (setf message value))
	  (setf parsp t))
	(t
	  (setf slotargs (nconc slotargs (list prop value))))))
    (if parsp
	(progn
	  (unless (or (eq type ':required)
		      (getf slotargs ':initform))
	    (when (or (null slot)
;	              (eq (slot-definition-initform slot) pcl::*slot-unbound*)
;		      (not (slot-boundp slot 'initform))
		      (null (slot-definition-initform slot))
		      )
	      (setf (getf slotargs ':initform) 'nil)))
	  (unless (getf slotargs ':accessor)
	    (setf (getf slotargs ':accessor)
	      (or (and slot (loop with writers = (slot-definition-writers slot)
			          for r in (slot-definition-readers slot)
		                  when (find r writers :key #'cadr)
		                  return r))
		  (intern (format nil "EVENT-~A" name)))))
	  (unless (getf slotargs ':initarg)
	    (setf slotargs (nconc slotargs (list ':initarg name ':initarg 
						 (intern (string name)
						        "KEYWORD")))))
	  (values (list* name slotargs)
		  (make-pinfo :name name
			      :type (or type ':forward)
			      :value-type value-type
			      :printer printer
			      :message message
			      :default pinfo)))
	(values (if slotargs 
		    (list* name slotargs)
		    name)
		nil))))

;;;
;;; defpart is the top level macro for declaring new classes of score parts.
;;; it is very much like defclass, except that it supports a lambda-list
;;; parameter specification and a few additional slot options.  the keyword
;;; arguments after the slot list allows absolute control over what defpart
;;; actually does.  :define-event-method controls what score-event methods
;;; are genereated by the macro call.  if none is supplied, methods for
;;; for all types of scores are generated.
;;;

(defmacro defpart (part-name super-classes parameters slots
		    &key (define-event-method nil method-p)
		    (define-resource t))
    (let ((options '((:metaclass class-with-parameters)))
	part-class class-slots class-pars parameter-specs
	standard-specs pars)
    ;; Collect the slots and parameters from super classes.
    (loop for s in super-classes
          for c = (find-class s)
          do
;           (unless (class-finalized-p c) (finalize-inheritance c))
           (when (typep (class-prototype c) 'part)
            (when (and part-class (not (eq c part-class)))
              ;; does this really matter? it shouldn't...
              (error "Attempt to inherit from more than one event class."))
            (setf part-class c))
          append (class-slots c) into cs
          when (typep c 'class-with-parameters)
          append (event-parameters c) into cp
          finally (setf class-slots cs class-pars cp))
    ;; Parse the local parameter declaration, if any.
    (multiple-value-bind (required optional message rest)
	(parse-parameter-declaration parameters)
      (flet ((get-type (name)
	       ;; Parse the local slot list for parameter information.
	       (cond ((member name required :test #'eq) ':required)
		     ((member name message :test #'eq) ':message)
		     ((member name optional :test #'eq) ':optional)
		     ((member name rest :test #'eq) ':rest)
		     (t 'nil))))
	(loop with name and slotspec and parspec
	      for spec in slots 
	      do
	  (setf name (if (listp spec) (car spec) spec))
	  (multiple-value-setq (slotspec parspec)
	    (parse-event-slot-spec spec 
				  (get-type name)
				  (find name class-pars :key #'car)
				  (find name class-slots 
					:key #'slot-definition-name)))
	      collect slotspec into slotlist
	      when parspec
	      collect parspec into parlist
	      finally (setf parameter-specs parlist standard-specs slotlist)))
      (if parameters
	  ;; If there is a local parameter declaration then we must insure 
	  ;; that any parameters which were NOT declared in the local slot
	  ;; specification list actually refer to inherited slots.
	  (flet ((add-par (parname type)
		   ;; Find or make parname's pinfo.
		   (let ((par (or (find parname class-pars :key #'car)
				  (if (find parname class-slots 
					    :key #'slot-definition-name 
					    :test #'eq)
				      (make-pinfo :name parname :type type
						  :printer 'princ)
				    (error "Missing slot for parameter ~a."
					   parname)))))
		     (push par parameter-specs)))
		 (local-par? (parname)
		   ;; Return parname if locally defined.
		   (find parname parameter-specs :test #'eq :key #'car))
		 (collect-pars (parnames)
		   (loop for tail on parnames
		    do (setf (car tail)
			 (or (find (car tail) parameter-specs
				   :test #'eq :key #'car)
			     (error "All screwed up and don't know why."))))
		   parnames))
	    (when required
	      (dolist (parname required)
		(unless (local-par? parname)
		  (add-par parname ':required)))
	      (setf pars (nconc pars (collect-pars required))))
	    (when optional
	      (dolist (parname optional)
		(unless (local-par? parname)
		  (add-par parname ':optional)))
	      (setf pars (nconc pars (collect-pars optional))))
	    (when message
	      (dolist (parname message)
		(unless (local-par? parname)
		  (add-par parname ':message)))
	      (setf pars (nconc pars (collect-pars message))))
	    (when rest
	      (dolist (parname rest)
		(unless (local-par? parname)
		  (add-par parname ':rest)))
	      (setf pars (nconc pars (collect-pars rest)))))
	;; if there are parameter-specs without a parameter declaration the
	;; user has redeclared parameter attributes inside the slot
	;; specification list and we need to merge the local changes into the
	;; inherited class parameters.
	(when parameter-specs
	  (setf pars (loop for par in class-pars 
		      collect (or (find (pinfo-name par) parameter-specs
					:test #'eq :key #'car)
				  par))))))
    #-mcl (when pars (setf options (append options `((:parameters ,@pars)))))
    ;; define-event-method can be nil t or a list of score classes.
    (setf define-event-method
      (if method-p
          (progn
             ;; sigh, check for quoted value to macro keyword!
             (when (and (listp define-event-method)
		       (eq (car define-event-method) 'quote))
		       (setf define-event-method (cadr define-event-method)))
             (if (listp define-event-method)
                 (loop for class in define-event-method
                       collect (class-prototype (find-class class)))
               t))
	(if pars t nil)))
    `(progn
       #-pcl
       (eval-when (compile load eval)
         (defclass ,part-name ,super-classes (,@standard-specs) ,@options))
       #+pcl
       (defclass ,part-name ,super-classes (,@standard-specs) ,@options)
       #+mcl
       (initialize-class-with-parameters (find-class ',part-name) ',pars)	   
       (pushnew ',part-name *parts*)
       ,@(if define-event-method
	     (make-part-methods (class-prototype part-class) part-name
				define-event-method (or pars class-pars)))
         ,@(if define-resource
	     `((eval-when (compile load eval)
		 (utils:defresource ,part-name (class)
		   :size 16
		   :constructor #'score-event-constructor
		   :finder #'utils:basic-resource-finder
		   :deinitializer #'score-event-deinitializer)))
	   nil))))

;;;
;;; make-score-event-method returns a method on score-event for a new class
;;; of score part.  we define this method here because it is available to
;;; to all types of parts.
;;;

(defmethod make-score-event-method ((score sequencer) (part part)
				    new-part-name parameters)
  `(defmethod score-event ((part ,new-part-name) (score sequencer))
     (let ((event '()))
       (block $parameter-block$
       ,.(loop for p in parameters when (eq (pinfo-type p) :required)
	  collect `(push (slot-value part ',(pinfo-name p)) event))
       ,.(loop for p in parameters when (eq (pinfo-type p) :optional)
	  collect `(let ((value (slot-value part ',(pinfo-name p))))
		     (if value (push value event)
		       (return-from $parameter-block$))))
       ,.(loop for p in parameters when (eq (pinfo-type p) ':rest)
	  return `((setf event (append (reverse
					(slot-value part ',(pinfo-name p)))
				      event))))
       ,.(loop for p in parameters when (eq (pinfo-type p) ':message)
	  collect `(let ((value (slot-value part ',(pinfo-name p))))
		     (when value
		       (push ',(pinfo-keyword p) event)
		       (push value event)))))
       (vector-push-extend (nreverse event) (slot-value score 'sequence)))))


;;;
;;; make-parse-to-part-method returns a method on parse-to-part that implements
;;; the parsing of an event list into the slots of the supplied part instance.
;;; parsing is available to all parts so we define it in this file.
;;;

(defmethod make-parse-to-part-method ((part part) new-part-name parameters)
  `(defmethod parse-to-part ((part ,new-part-name) (args list))
     ;; parse positional arguments first
     ,@(loop for p in parameters 
	when (member (pinfo-type p) '(:required :optional))
	collect `(setf (slot-value part ',(pinfo-name p))
		   (pop args)))

     ;; &rest parameter gets rest of args.
     ,@(let ((rest (loop for p in parameters 
		    when (eq (pinfo-type p) ':rest) 
		    return p)))
	 (when rest
	   `((setf (slot-value part ',(pinfo-name rest)) args))))
     ;; &message parameters are searched for because they need
     ;; to be reset to nil anyway if no value in args is found.
     ,@(loop for p in parameters
	when (eq (pinfo-type p) ':message)
	collect `(setf (slot-value part ',(pinfo-name p))
		   (getf args ',(pinfo-keyword p))))
     part))

;;;
;;; value-print-form computes the actual form for outputting a parameter's
;;; value based on its various format properties.
;;;

(defun value-print-form (value pinfo)
  (let ((value-type (pinfo-value-type pinfo))
	(printer (or (pinfo-printer pinfo)
		     'princ)))
    (if value-type
	(ecase value-type
	  (:float
	    `(,printer (/ (fround ,value 1000) 1000)
		       *common-music-output*))
	  (:quoted
	    `(progn
	       (write-char #\' *common-music-output*)
	       (,printer ,value *common-music-output*)))
	  (:quoted-string
	    `(progn
	       (write-char #\" *common-music-output*)
               (,printer ,value *common-music-output*)
	       (write-char #\" *common-music-output*)))
	  (:integer
	    `(,printer (floor ,value) *common-music-output*)))
	`(,printer ,value *common-music-output*))))
  
;;;
;;;
;;;

(defun describe-part (&optional part (stream *standard-output*))
  (if part
    (flet ((describe-part-1 (part instance stream)
	     (let ((pars (and (typep part 'class-with-parameters)
			      (event-parameters part)))
		   (maxlen 0)
		   required optional message rest slots)
	       (loop for slot in (class-slots part)
		for name = (slot-definition-name slot)
		for type = (pinfo-type (find name pars :key #'car))
		maximize (length (symbol-name name)) into max
		if (eq type ':required)
		collect name into l1
		else if (eq type ':optional)
		collect name into l2
		else if (eq type ':message)
		collect name into l3
		else if (eq type ':rest)
		collect name into l4
		else collect name into l5
		finally (setf required l1 optional l2 message l3 
			      rest l4 slots l5 maxlen (1+ max)))
	       (format stream "~%~%Part: ~A" (class-name part))
	       (progn
		 (format stream "~%~%Class precedence list: ")
		 (pprint (mapcar #'(lambda (x)
				     (if (typep x 'standard-class)
					 (class-name x) 
				       x))
				 (class-precedence-list part))
			 stream))
	       (flet ((describe-slot (instance slotname stream tab)
			(let ((value (if (slot-boundp instance slotname)
					 (slot-value instance slotname)
				       "unbound")))
			  (format stream "~%~a~vt~s" slotname tab value))))
		 (when slots
		   (format stream "~%~%Slots:")
		   (dolist (s slots)
		     (describe-slot instance s stream maxlen)))
		 (when required
		   (format stream "~%~%Required parameters:")
		   (dolist (s required)
		     (describe-slot instance s stream maxlen)))
		 (when optional
		   (format stream "~%~%Optional parameters:")
		   (dolist (s optional)
		     (describe-slot instance s stream maxlen)))
		 (when message
		   (format stream "~%~%Message parameters:")
		   (dolist (s message)
		     (describe-slot instance s stream maxlen)))
		 (when rest
		   (format stream "~%~%Rest parameters:")
		   (dolist (s rest)
		     (describe-slot instance s stream maxlen))))
	       (values))))
      (when (symbolp part)
	(let ((save part))
	  (setf part (find-class part nil))
	  (unless part
	    (format stream "~%~A is not a part." save))))
      (let ((part-name (class-name part)))
	(if (get part-name ':resource)
	    (utils:using-resource (instance part-name part-name)
	      (initialize-instance instance)
	      (describe-part-1 part instance stream))
	  (describe-part-1 part (make-instance part) stream))))
    (format stream "~%Loaded parts:~{ ~s ~}" *parts*)))

;;;
;;;
;;;
      
#+ExCl (top-level:alias ("part" 3) (&optional name)
	 (describe-part name)
	 (values))

