;;; -*- Mode: Lisp; Syntax: Common-lisp; Package: common-music; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice. Suggestions, comments and bug reports are welcome.  Please 
;;; address electronic correspondence to: hkt@ccrma.stanford.edu
;;; **********************************************************************

(in-package :common-music)

(eval-when (load eval)
  (setf (get ':csound :scorefile) 'csound-scorefile)
  (setf (get ':csound :sequence) 'sequencer))

(defparameter *default-csound-play-options* '()
  "Default play options for csound: &key orchestra soundfile args")

(defclass csound-scorefile (ascii-scorefile)
  ((default-after :initform `(play ,@ *default-csound-play-options*))
   (orchestra :initform nil)
   (soundfile :initform nil)
   (default-path :initform (default-scorefile-name :csound))))

(defmethod schedule-score-events :before ((score csound-scorefile))
  (format *common-music-output* "; Written by Common Music ~A~%~%"
	  (date-string))
  (let ((header (slot-value score 'header)))	   
    (when header
      (format *common-music-output* "~&~a~%" header)))
  (format *common-music-output* "~&s~%"))

(defmethod schedule-score-events :after ((score csound-scorefile))
  (format *common-music-output* "~&e~%"))

(eval-when (load eval)
  (utils:initialize-resource 'scores 1 'csound-scorefile))

;;;
;;; play command
;;;

(defparameter *.sco-player* nil)

(defun play-csound-scorefile (file &key orchestra (soundfile nil sndp)
				   (player *.sco-player*) args)
  (unless player		; defer player pathname till runtime
    (setf *.sco-player*
          (concatenate 'string (namestring *common-music-directory*)
	               "oldsys/csound/playscorefile")    
           player  *.sco-player*))
  (let ((base (make-pathname :directory (pathname-directory file)
			     :name (pathname-name file)))
	perf)
    (unless orchestra
      (setf orchestra (merge-pathnames base "test.orc")))
    (when (or soundfile
	      (setf soundfile (and (not sndp)
				   (merge-pathnames base "test.snd"))))
      (setf args (format nil "~@[~a ~]-o ~a" args (namestring soundfile)))
      )
    (setf perf (format nil "~a ~@[~a ~]~a ~a" player args
		       (namestring orchestra) (namestring file)))
    (format t "~&;;; Performing: ~a~%" perf)
    (shell perf)
     file))

;;;
;;; the "play" after command for csound defscorefiles has the full form:
;;;               (play &key orchestra soundfile args)
;;; the command is merged with *default-csound-play-options* to produce
;;; the fully specified after command.  the keywords are:
;;;   :orchestra      the name of the orchestra file. defaults to the same
;;;                   name as the scorefile with an .orc extension added.
;;;   :soundfile      the name of the soundfile to produce. defaults to the
;;;                   same name as the scorefile with an .snd extension added.
;;;                   if soundfile is nil then the default csound soundfile
;;;                   is written and sndplay is not called.
;;;   :args           additional args passed to csound command
;;;

(defmethod score-after-command ((score csound-scorefile) command args)
  (if (not (eq command 'play))
      (call-next-method)
    (let ((file (merge-pathnames (or (slot-value score 'pathname)
				     *last-scorefile-written*)
				 *default-scorefile-pathname*)))
      (apply #'play-csound-scorefile file args))))

;;;
;;; csound parts
;;;

(defclass csound-part (part)
  ((syntax :initform :csound)
   (name :initform nil)))

(defmethod initialize-instance :after ((part csound-part) &rest initargs)
  (declare (ignore initargs))
  (let ((name (or (slot-value part 'name)
		  (class-name (class-of part)))))
    (if (integerp name)
	(setf name (format nil "i~d" name))
      (if (symbolp name)
	  (setf name (string-downcase (string name)))))
    (unless (and (stringp name)
		 (char= #\i (elt name 0))
		 (every #'digit-char-p (subseq name 1)))
      (warn "~a is not a proper csound instrument name." name))
    (setf (slot-value part 'name) name)))

(defmethod make-part-methods ((part csound-part) new-part-name score-classes
			      parameters)
  (unless (listp score-classes)
    (setf score-classes 
      (list (class-prototype (find-class 'sequencer))
	    (class-prototype (find-class 'csound-scorefile)))))
  (loop for score in score-classes
   collect (make-score-event-method score part new-part-name parameters)))


(defmethod make-score-event-method ((score csound-scorefile)(part csound-part)
				    new-part-name parameters)
  `(defmethod score-event ((part ,new-part-name) (score csound-scorefile))
     (block $parameter-block$
       ,@(loop with delimiters = '((write-char #\space *common-music-output*))
	  for par in parameters
	  for num from 0
	  append (parameter-printing-forms part par num delimiters)))
     (terpri *common-music-output*)))

(defmethod parameter-printing-forms ((part csound-part)
				     pinfo position delimiters)
  (unless (eq (pinfo-type pinfo) ':required)
    (error "CSound only supports &required parameters."))
  (let ((access `(slot-value part ',(pinfo-name pinfo))))
    `(,@(if (> position 0) delimiters nil) 
	,(value-print-form access pinfo))))

