;;; -*- Mode: Lisp; Syntax: Common-lisp; Package: common-music; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(eval-when (compile load eval)
  ;; Common Music uses the following CLM symbols:
  ;; clm::with-output-to-scorefile clm::default-sound-file-name
  ;; clm:*reverb* c56:c56-reset-cleanup clm:clm-cleanup clm:set-srate 
  ;; clu::last-dac-file-name clm:clm-initialize-statistics
  ;; clm:clm-initialize-notehook clu:reopen-output clu:open-output
  ;; clm:clm-get-default-header clm:make-header clu:close-output
  ;; clm:clm-print-statistics clm:clm-cleanup clu:dac-n clm:wait-for-dac
  ;; c56:c56-reset-cleanup clm:clm-end-run
  (if (find-package :clm)
      (progn (import (find-symbol "LAST-DAC-FILE-NAME" :clu)))
    (progn
      (warn "Building CLM syntax without CLM loaded. 
Be sure to use \"after nil\" in defscorefile options.
")
      (defvar default-sound-file-name "/zap/test.snd"))))

(defparameter *default-clm-load-options* ()
  "Default keyword options passed to clm-load.")	     

;;;
;;; set up the :CLM syntax properties
;;;

(eval-when (load eval)
  (setf (get ':clm :scorefile) 'clm-scorefile)
  (setf (get ':clm :soundfile) 'clm-soundfile)
  (setf (get ':clm :sequence) 'sequencer))

;;;
;;; the common lisp music scorefile. initialize-resource is called
;;; after the class definition to allocate an instance of
;;; clm-scorefile in the score resource.
;;;

(defclass clm-scorefile (ascii-scorefile)
  ((default-after :initform  `(load ,@ *default-clm-load-options*))
   (default-path :initform (default-scorefile-name :clm))))

(eval-when (load eval)
  (utils:initialize-resource 'scores 1 'clm-scorefile))

;;;
;;; a :before method on realize-score outputs header info.
;;;

(defmethod schedule-score-events :before ((score clm-scorefile))
  (format *common-music-output*
       ";;; -*- Mode: Lisp; Syntax: Common-lisp; Package: ~A; Base: 10 -*-~%~%"
       (package-name *package*))
  (format *common-music-output* ";;; Written by Common Music ~A~%~%"
	  (date-string))
  (let ((header (slot-value score 'header)))	   
    (when header
      (format *common-music-output* "~a~%~%" header))))

;;;
;;; clm scorefiles recognize a load command.
;;;

(defmethod score-after-command ((score clm-scorefile) command args)
  (if (eq command 'load)
      (let ((scorefile (slot-value score 'pathname))
	    (*load-verbose* nil))
	(print-clm-load-statistics scorefile args)
	(apply #'clm-load scorefile args))
    (call-next-method)))

(defun print-clm-load-statistics (scorefile args)
  (let ((rate sampling-rate)
	(reverb nil)
	(time 1.0)
	(data ())
	(play 1)
	(channels 1)
	(options args)
	(file default-sound-file-name))
    (loop while options
          for key = (pop options)
          for val = (pop options)
     do (case key 
	  (:reverb (setf reverb val))
	  (:sndfile (setf file val))
	  (:play (setf play val))
	  (:srate (setf rate val))
	  (:channels (setf channels val))
	  (:reverb-data (setf data val))
	  (:decay-time (setf time val))))
    (format t "~&;;; Loading: ~a
;;;  Sound file: ~A
;;;  Channels: ~D
;;;  Srate: ~F
;;;  Reverb: ~A ~@[ Decay time: ~S ~]~@[ Reverb data: ~S ~]~&~
~@[;;;  Times to play: ~S~]~%"
	    (namestring (truename scorefile))
	    (namestring (merge-pathnames file default-sound-file-name))
	    (floor channels)
	    rate
	    (or reverb "None") (and reverb time) (and reverb data)
	    (and play (if (numberp play) play 1)))))

;;;
;;; clm-soundfile allows parts to directly output to the soundfile, rather
;;; than first writing a scorefile and then automatically loading it.
;;;

(defclass clm-soundfile (score)
  ((sndfile :initform default-sound-file-name 
	    :initarg sndfile :initarg :sndfile)
   (channels :initform 1 :initarg channels :initarg :channels)
   (srate :initform 22050 :initarg srate :initarg :srate)
   (reverb :initform nil :initarg reverb :initarg :reverb)
   (reverb-data :initform nil :initarg reverb-data :initarg :reverb-data)
   (play :initarg play :initarg :play :initform t) 
   (wait :initform nil :initarg wait :initarg :wait)
   (comment :initform nil :initarg comment :initarg :comment)
   (statistics :initform nil :initarg statistics :initarg :statistics)
   (notehook :initform nil :initarg notehook :initarg :notehook)
   (decay-time :initform 1.0 :initarg decay-time :initarg :decay-time)
   (continue-old-file :initform nil :initarg continue-old-file
		       :initarg continue-old-file)))

(eval-when (load eval)
  (utils:initialize-resource 'scores 1 'clm-soundfile))

;;;
;;; the around method on realize-score for clm-soundfile implements identical
;;; code to clm-load and with-sound.
;;;

(defmethod schedule-score-events :around ((score clm-soundfile))
  (unwind-protect
      (let ((file (merge-pathnames (slot-value score 'sndfile)
				   default-sound-file-name))
	    (play (slot-value score 'play))
	    (srate (slot-value score 'srate))
	    (reverb (slot-value score 'reverb))
	    (channels (slot-value score 'channels))
	    (statistics (slot-value score 'statistics))
	    (notehook (slot-value score 'notehook))
	    (comment (slot-value score 'comment))
	    (*reverb* nil))

	(c56-reset-cleanup)
	(clm-cleanup)
	(set-srate srate)
	(setf last-dac-file-name (namestring file))
	(when statistics
	  (clm-initialize-statistics statistics file))
	(when notehook
	  (clm-initialize-notehook notehook))
	(if (slot-value score 'continue-old-file)
	    (reopen-output file)
	  (open-output file (make-header 
				 :channels channels
				 :sampling-rate srate
				 :info (if comment
					   (format nil "#| ~A |#" comment)
					 (clm-get-default-header)))))
	(when reverb
	  (setf *reverb*
	    (open-output default-reverb-file-name
			     (make-header 
			      :channels 1 
			      :sampling-rate srate
			      :info ";temporary reverb stream"))))
	;; this call-next-method invokes the scheduler.
	(call-next-method)
	(when reverb
	  (close-output *reverb*)
	  (open-input default-reverb-file-name)
	  (apply reverb 0 (+ (slot-value score 'decay-time)
			     (clm-get-duration *current-input-file*))
		 (slot-value score 'reverb-data))
	  (close-input *reverb*))
     (close-output)
     (when statistics
       (clm-print-statistics statistics))
     (clm-cleanup)
     (when play
       (unless (numberp play)
	 (setf play 1))
       (dac-n :file file :times play))
     (when (slot-value score 'wait)
       (wait-for-dac)))
   (progn
    (c56-reset-cleanup)
    (clm-end-run)
    (clm-cleanup))))

;;;
;;;
;;;

(defclass clm-part (part)
  ((syntax :initform :clm)))

(defmethod initialize-instance :after ((part clm-part) &rest initargs)
  (declare (ignore initargs))
  (unless (slot-boundp part 'name)
    (setf (slot-value part 'name) (class-name (class-of part)))))

;;;
;;; defpart calls make-part-methods to return a list of methods that implement
;;; score output and parsing for the new class of parts.  score-classes is 
;;; either a list of score objects or the symbol t, which indicates that
;;; methods for all scores appropriate to the syntax should be returned.
;;;

(defmethod make-part-methods ((part clm-part) new-part-name score-classes
			       parameters)
  (unless (listp score-classes)
    (setf score-classes (list (find-class 'clm-scorefile))))
  (loop for class in score-classes 
        collect (make-score-event-method (class-prototype class)
                                         part new-part-name parameters)))

;(defmethod make-part-methods ((part clm-part) new-part-name score-classes
;			      parameters)
;  (unless (listp score-classes)
;    (setf score-classes (list
;			 (class-prototype (find-class 'sequencer))
;			 (class-prototype (find-class 'clm-soundfile))
;			 (class-prototype (find-class 'clm-scorefile)))))
;  (cons (make-parse-to-part-method part new-part-name parameters)
;	(loop for score in score-classes
;	 collect (make-score-event-method score part new-part-name
;					  parameters))))

;;;
;;; compute the output method for clm parts in clm-scorefile
;;;

(defmethod make-score-event-method ((score clm-scorefile) (part clm-part)
				    new-part-name parameters)
  
  `(defmethod score-event ((part ,new-part-name) (score clm-scorefile))
     (declare (optimize (speed 3) (safety 0)))
     (write-char #\( *common-music-output*)
     (block $parameter-block$
       ,.(loop with delimiters = '((write-char #\space *common-music-output*))
	  for par in parameters
	  for num from 0
	  append (parameter-printing-forms part par num delimiters)))
     (write-char #\) *common-music-output*)
     (terpri *common-music-output*)))

;;;
;;; the make-score-event-method for sound file scores computes a score-event
;;; method that invokes a clm instrument directly, without writing an
;;; intermediate scorefile.
;;;

(defmethod make-score-event-method ((score clm-soundfile) (part clm-part)
				    new-part-name parameters)
  (let (positional message rest)
    (loop for p in parameters 
     when (member (pinfo-type p) '(:required :optional))
     collect `(slot-value part ',(pinfo-name p)) into l1
     else when (eq (pinfo-type p) ':message)
     collect `(let ((value (slot-value part ',(pinfo-name p))))
		(when value
		  (push value args)
		  (push ',(pinfo-keyword p) args))) into l2
     else when (eq (pinfo-type p) ':rest)
     collect `(setf args (slot-value part ',(pinfo-name p))) into l3
     finally (setf positional l1 message l2 rest l3))
  `(defmethod score-event ((part ,new-part-name) (score clm-soundfile))
     (let ((args '()))
       ,.rest
       ,.message
       (apply ,.positional args)))))

;;;
;;; parameter messages for clm are keyword forms of parameter names.
;;;

(defmethod default-message-string ((part clm-part) pinfo)
  (format nil ":~A " (pinfo-name pinfo)))

;;;
;;; clm parts support &required, &optional, &message and &rest parameters.
;;;

(defmethod parameter-printing-forms ((part clm-part) pinfo position delimiters)
  (ecase (pinfo-type pinfo)
    (:required
      (let ((access `(slot-value part ',(pinfo-name pinfo))))
	`(,@(if (> position 0) delimiters nil) 
	    ,(value-print-form access pinfo))))
    (:message
      (let ((var (gentemp "V")))
	`((let ((,var (slot-value part ',(pinfo-name pinfo))))
	    (when ,var
	      ,@(if (> position 0) delimiters nil)
	      (write-string ,(or (pinfo-message pinfo)
				 (default-message-string part pinfo))
			    *common-music-output*)
	      ,(value-print-form var pinfo))))))
    (:optional
      (let ((var (gentemp "V")))
	`((let ((,var (slot-value part ',(pinfo-name pinfo))))
	    (unless ,var 
	     (return-from $parameter-block$))
	    ,@(if (> position 0) delimiters nil)
	    ,(value-print-form var pinfo)))))

    (:rest
      (let ((var1 (gentemp "V"))
	    (var2 (gentemp "V")))
	`((let ((,var1 (slot-value part ',(pinfo-name pinfo)))
		 ,var2)
	    (when ,var1
	      ,@(if (> position 0) delimiters nil)
	      (loop 
	       (setf ,var2 (pop ,var1))
	       ,(value-print-form var2 pinfo)
	       (if ,var1
		   (progn ,@delimiters)
		   (return))))))))))

