;;; -*- Syntax: Common-Lisp; Package: COMMON-MUSIC; Base: 10; Mode: Lisp -*-
;;; **********************************************************************
;;; Copyright (c) 1989, 1990 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  No warranty is made about this software, its performance
;;; or its conformity to any specification.  Any distribution of this 
;;; software must comply with all applicable United States export control
;;; laws. Suggestions, comments and bug reports are welcome.  Please 
;;; address electronic correspondence to: hkt@ccrma.stanford.edu
;;; **********************************************************************

(in-package "COMMON-MUSIC")

;;;
;;; Ampitude streams map logical amplitudes onto actual amplitude values.  Logical
;;; amplitudes are numbers between 0 and 1.0 or symbols from the set of amplitude
;;; symbols maintained in the global variable *amplitude-symbols*.  As of now there
;;; is not a item stream mixin for amplitude streams, they are implemented as basic
;;; item streams whose elements are preparsed as logical amplitude values.
;;;

(defparameter *amplitude-minimum* .000
  "The minimum amplitude when logical amplitude is 0")

(defparameter *amplitude-maximum* 1.0 
  "The maximum amplitude when logical amplitude is 1.0")

(defparameter *amplitude-power* 1
  "The power curve for computing amplitudes.")

(defprop niente :amplitude-weight 0)
(defprop pppp :amplitude-weight .1)
(defprop ppp :amplitude-weight .2)
(defprop pp :amplitude-weight .3)
(defprop p :amplitude-weight .4)
(defprop mp :amplitude-weight .5)
(defprop mf :amplitude-weight .6)
(defprop f :amplitude-weight .7)
(defprop ff :amplitude-weight .8)
(defprop fff :amplitude-weight .9)
(defprop ffff :amplitude-weight 1)

;;;
;;;
;;;

(defclass AMPLITUDE-MIXIN ()
  ((ampmin :initarg :ampmin :initform *amplitude-minimum*)
   (ampmax :initarg :ampmax :initform *amplitude-maximum*)
   (power :initarg :power :initform *amplitude-power*)))

;;;
;;;
;;;

(defmethod AMPLITUDE ((amp T) &key (min *amplitude-minimum*)
		      (max *amplitude-maximum*) (power *amplitude-power*))
  (amplitude-aux (- max min) (parse-amplitude amp) min power))

(defmethod AMPLITUDE ((amp ITEM-STREAM) &key (min *amplitude-minimum*)
		      (max *amplitude-maximum*) (power *amplitude-power*))
  (declare (ignore min max power))
  amp)

(defmethod amplitude-aux ((range INTEGER) amp min power)
  (values (floor (+ min (* range (expt amp power))))))

(defmethod amplitude-aux ((range T) amp min power)
  (+ min (* range (expt amp power))))

;;;
;;; Amplitude stream parsing methods
;;;

(defmethod parse-amplitude ((amp SYMBOL))
  (or (get amp ':amplitude-weight)
      (error "Illegal amplitude reference: ~s." amp)))

(defmethod parse-amplitude ((amp NUMBER))
  (unless (and (>= amp 0.0)
	       (<= amp 1.0))
    (error "Illegal amplitude reference: ~s." amp))
  amp)

(defmethod parse-amplitude ((amp ITEM-STREAM))
  amp)

(defmethod parse-amplitude ((amp T))
  (error "Illegal amplitude reference: ~s." amp))





