;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(defclass number-pattern (item-pattern)
  ((from :initform 0 :initarg :from)
   (to   :initarg :to :initarg :below :initarg :downto :initarg :above)
   (by   :initform 1 :initarg :by)))
   
(defmethod initialize-instance :after ((object number-pattern) &rest args)
  (let ((from (slot-value object 'from))
        (by (slot-value object 'by))
        (flag 0) 
        to)
    (loop for arg in args by #'cddr
	  while (= flag 0)
          do
      (case arg
        (:below  (setf flag +numbers-below+))
        (:downto (setf flag +numbers-downto+))
        (:above  (setf flag +numbers-above+))))
    (setf to
      (if (slot-boundp object 'to)
          (slot-value object 'to)
        (if (floatp from)
            most-positive-single-float
          most-positive-fixnum)))
    (setf (slot-value object 'flags)
      (logior (slot-value object 'flags) flag))
    (setf (slot-value object 'length)
      (cond ((and (numberp from) (numberp to))
             (floor (/ (abs (- to from))
                       (if (numberp by) (abs by) 1))))
            (t 1)))
    (setf (slot-value object 'data)
      (vector from to by))))

(defmethod reset-period :before ((object number-pattern))
  (let ((resets (slot-value object 'data)))
    (setf (slot-value object 'from) (item (svref resets 0)))
    (setf (slot-value object 'to)   (item (svref resets 1)))
    (setf (slot-value object 'by)   (item (svref resets 2)))
    (let ((flags (slot-value object 'flags))
          (by (slot-value object 'by))
          from to)
      (if (or (logtest flags +numbers-downto+) 
              (logtest flags +numbers-above+))
          (setf to (slot-value object 'from) 
                from (slot-value object 'to))
        (setf from (slot-value object 'from) 
              to (slot-value object 'to)))
      ;; numbers compute their period lengths. is this correct?
      (setf (period-length (slot-value object 'period))
        (if (or (logtest flags +numbers-below+)
                (logtest flags +numbers-above+))
            (ceiling (abs (/ (- to from) by)))
          (multiple-value-bind (int rem) 
                               (ceiling (abs (/ (- to from) by)))
              (if (= rem 0) (1+ int) int)))))))

(defclass linear-number-pattern (number-pattern) ())
(defclass random-number-pattern (number-pattern) ())

(defmethod next-in-pattern ((object linear-number-pattern))
  (let ((flags (slot-value object 'flags))
        (from (slot-value object 'from)))
    (cond ((or (logtest flags +numbers-downto+)
               (logtest flags +numbers-above+)) 
           (decf (slot-value object 'from) (slot-value object 'by)))
          (t
           (incf (slot-value object 'from) (slot-value object 'by))))
    from))

(defmethod next-in-pattern ((object random-number-pattern))
  (let ((flags (slot-value object 'flags)))
    (cond ((logtest flags +numbers-downto+) 
           (random-downto (slot-value object 'from)
                          (slot-value object 'to)
                          (slot-value object 'by)))
          ((logtest flags +numbers-below+) 
           (random-below (slot-value object 'from)
                         (slot-value object 'to)
                         (slot-value object 'by)))
          ((logtest flags +numbers-above+) 
           (random-above (slot-value object 'from)
                         (slot-value object 'to)
                         (slot-value object 'by)))
          (t 
           (random-to (slot-value object 'from)
                      (slot-value object 'to)
                      (slot-value object 'by))))))

(defun random-to (from to by)
  (+ from (* by (random (1+ (round (/ (- to from) by)))))))

(defun random-above (from to by)
  (- from (* by (random (round (/ (- from to) by))))))

(defun random-below (from to by)
  (+ from (* by (random (round (/ (- to from) by))))))

(defun random-downto (from to by)
  (- from (* by (random (1+ (round (/ (- from to) by)))))))

;;;
;;; numbers only supports two pattern types.
;;;

(defclass linear-number-stream (item-stream linear-number-pattern)
  ())

(defclass random-number-stream (item-stream random-number-pattern)
  ())



#|
(setf x (numbers from 1 to 10))
(setf x (numbers below 10))
(setf x (numbers downto -2 in random by .5))
(setf x (numbers  from 1 to 10 by .1))
|#
         
