;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(defclass scale-mixin ()
  ((scale :initform *standard-scale* :initarg :scale :initarg :of)))

;;;
;;; note streams return symbolic note name, degree stream return integer
;;; scale degrees, pitch streams return floating point frequencies.
;;;

(defclass note-stream (item-stream scale-mixin)
  ())

(defclass degree-stream (item-stream scale-mixin)
  ())

(defclass pitch-stream (item-stream scale-mixin) 
  ())

;;;
;;; the after methods convert the stream's value to a scale reference.
;;;

(defmethod item :after ((object note-stream))
  ;(declare (optimize (speed 3) (safety 0)))
  (setf (slot-value object 'value)
    (scale-note (slot-value object 'value)
                (slot-value object 'scale))))

(defmethod item :after ((object pitch-stream))
  ;(declare (optimize (speed 3) (safety 0)))
  (setf (slot-value object 'value)
    (scale-pitch (slot-value object 'value)
                 (slot-value object 'scale))))

(defmethod item :after ((object degree-stream))
  ;(declare (optimize (speed 3) (safety 0)))
  (setf (slot-value object 'value)
    (scale-degree (slot-value object 'value)
                  (slot-value object 'scale))))

;;;
;;; note, degree and pitch stream definitions
;;;

(defclass cyclic-note-stream (note-stream cycle-pattern) ())
(defclass cyclic-degree-stream (degree-stream cycle-pattern) ())
(defclass cyclic-pitch-stream (pitch-stream cycle-pattern) ()) 

(defclass palindromic-note-stream (note-stream palindrome-pattern) ())
(defclass palindromic-degree-stream (degree-stream palindrome-pattern) ())
(defclass palindromic-pitch-stream (pitch-stream palindrome-pattern) ()) 

(defclass sequential-note-stream (note-stream sequence-pattern) ())
(defclass sequential-degree-stream (degree-stream sequence-pattern) ())
(defclass sequential-pitch-stream (pitch-stream sequence-pattern) ()) 

(defclass heap-note-stream (note-stream heap-pattern) ())
(defclass heap-degree-stream (degree-stream heap-pattern) ())
(defclass heap-pitch-stream (pitch-stream heap-pattern) ())

(defclass random-note-stream (note-stream random-pattern) ())
(defclass random-degree-stream (degree-stream random-pattern) ())
(defclass random-pitch-stream (pitch-stream random-pattern) ())

(defclass graph-note-stream (note-stream graph-pattern) ())
(defclass graph-degree-stream (degree-stream graph-pattern) ())
(defclass graph-pitch-stream (pitch-stream graph-pattern) ())

(defclass functional-note-stream (note-stream function-pattern) ())
(defclass functional-degree-stream (degree-stream function-pattern) ())
(defclass functional-pitch-stream (pitch-stream function-pattern) ())

(defclass rotational-note-stream (note-stream rotation-pattern) ())
(defclass rotational-degree-stream (degree-stream rotation-pattern) ())
(defclass rotational-pitch-stream (pitch-stream rotation-pattern)  ())

(defclass accumulating-note-stream (note-stream accumulation-pattern) ())
(defclass accumulating-degree-stream (degree-stream accumulation-pattern) ())
(defclass accumulating-pitch-stream (pitch-stream accumulation-pattern) ()) 

;;;
;;; chord streams return the state :chording if not at end of period.
;;;

(defclass chord-mixin () ())

(defclass chord-stream (chord-mixin item-stream cycle-pattern)
  ())

(defclass note-chord-stream (chord-mixin note-stream cycle-pattern)
  ())

(defmethod item :around ((stream chord-mixin)) 
  ;(declare (optimize (speed 3)(safety 0)))
  (multiple-value-bind (value state) (call-next-method)
    (values value (or state ':chording))))

;;;
;;; note parsing. 
;;;

(defvar *standard-octave* 4)
(defparameter *respect-note-spelling* #+cmn t #-cmn nil)

(defmethod datum-parser ((object note-stream))
  (let ((scale (slot-value object 'scale))
        (*standard-octave* *standard-octave*))
    #'(lambda (x) (parse-note x scale))))

(defmethod datum-parser ((object degree-stream))
  (let ((scale (slot-value object 'scale))
        (*standard-octave* *standard-octave*))
    #'(lambda (x) (parse-note x scale))))

(defmethod datum-parser ((object pitch-stream))
  (let ((scale (slot-value object 'scale))
        (*standard-octave* *standard-octave*))
    #'(lambda (x) (parse-note x scale))))

(defmethod parse-note ((ref integer) (scale essential-scale))
  (if (gethash ref (slot-value scale 'entries))
      ref
    (error "~A is not a degree in ~A." ref scale)))

(defmethod parse-note ((ref float) (scale essential-scale))
  ref)

(defmethod parse-note ((ref item-stream) (scale essential-scale))
  ref)

(defmethod parse-note ((ref symbol) (scale essential-scale))
  (if (eq ref 'r)
    *rest-name*
    (let ((thing (cdr (gethash ref (slot-value scale 'entries))))
          (divisions (slot-value scale 'divisions-per-octave)))
      (if thing
        (progn
          (setf *standard-octave* 
                (gethash (+ (floor thing divisions) 1024)
                         (slot-value scale 'tokens)))
          (setf thing ref))
        ;; get position of token, add in default octave and return symbol
        ;; at that degree
        (let* ((tokens (slot-value scale 'tokens)) 
               (octave (gethash *standard-octave* tokens)) 
               (interval (gethash ref tokens)) 
               degree entry) 
          (unless (and interval octave)
            (error "~S is not a note in ~S." ref scale))
          (setf degree (degree (+ (* divisions octave) interval)))
          (setf entry (gethash degree (slot-value scale 'entries)))
          (if *respect-note-spelling*
            (let* ((string (symbol-name ref))
                   (length (length string)))
              (setf thing (find string (cdr entry) 
                                :key #'symbol-name
                                :test #'(lambda (x y)
                                          (string= x y :end2 length)))))
            (setf thing (cadr entry)))))
      thing)))

(defmethod parse-note ((ref symbol) (scale general-scale))
  (if (> (slot-value scale 'number-of-octaves) 0)
      (call-next-method)
    (if (scale-degree ref scale) 
        ref
      (error "~S is not a note in ~S." ref scale))))
       
      

