;;; -*- Mode: Lisp; Syntax: Common-lisp; Package: common-music; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 89, 90, 91, 92 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(defclass multiple-item () 
  ((items :accessor multiple-item-items :initarg :items)))

(defmethod make-multiple-item ((name symbol) &rest args)
  (let ((item (make-instance name)))
    (setf (slot-value item 'items)
      (parse-item-elements item args))
    item))
  
(defmethod parse-item-elements ((item multiple-item) args)
  args)

(defmethod filter-item-elements ((item multiple-item) &rest values)
  values)

(defmacro defmultiple-item-parser (name args &body body)
  (let ((namevar (gensym))
	(argsvar (gensym)))
  `(defmethod parse-item-elements ((, namevar ,name) ,argsvar)
     (apply (function (lambda ,args ,@body))
	    ,argsvar))))

(defmacro defmultiple-item-filter (name args &body body)
  (let ((namevar (gensym))
	(argsvar (gensym)))
  `(defmethod filter-item-elements ((,namevar ,name) &rest ,argsvar)
     (apply (function (lambda ,args ,@body))
	    ,argsvar))))

(defmacro multiple-item-bind (vars item &body body)
  `(multiple-value-bind ,vars (values-list ,item)
     ,@body))

#|
defmultiple-item
The :element-period option has now the following syntax:

	(:element-period :ANY &rest elements)
	means any element in elements reaches end of period 
        returns end of period

	(:element-period :ALL &rest elements)
	means end of period when all elements simultaneously agree.

        (:element-period element)
	the same as (:element-period :all element)
If there is only a single element being tracked, :all or :any have the same effect.
For example:
(defmultiple-item klangs 
                  (pitch rhythm accent amp gesture)
  (:element-period :all))

means: "period ends when both rhythm and accent reach end of period at the same time"
|#

(defmacro defmultiple-item (name elements &rest options)
  (flet ((define-item-method (name descriptors period-type 
                                   state-elements filter)
           (let ((param-var (gensym))
                 (items-var (gensym))
                 (state-var (gensym))
                 (x (gensym))
                 (y (gensym))
                 forms)
             (setf forms
               (loop for d in descriptors
                     for s = (if (find (first d) state-elements) y nil)
                     collect 
                 (cond ((and (eq (first period-type) ':all)
                             (or (not (cdr period-type))
                                 (find (first d) period-type)))
                         `(multiple-value-bind (,x ,y) (item (pop ,items-var))
                            (unless (eq ,y +end-of-stream-token+)
                              (setf ,state-var ,s))
                            ,x))
                       ((and (eq (first period-type) ':any)
                             (or (not (cdr period-type))
                                 (find (first d) period-type)))
                         `(multiple-value-bind (,x ,y) (item (pop ,items-var))
                            (if (eq ,y +end-of-stream-token+)
                                (setf ,state-var +end-of-stream-token+)
                              (unless ,state-var (setf ,state-var ,s)))
                            ,x))
                       (t 
                        `(item (pop ,items-var))))))
             `(defmethod item ((,param-var ,name))
                (let ((,items-var (slot-value ,param-var 'items))
                      (,state-var ,(if (eq (first period-type) ':all)
                                       '+end-of-stream-token+
                                     nil)))
                  ,(if filter
                        `(values (filter-item-elements ,param-var ., forms) 
                                 , state-var)
                    `(values (list .,forms) ,state-var))))))
	 (parse-elements (elements)	; dummy parser for now
	   (loop for e in elements collect (if (consp e) e (list e))))
	 (parse-lambda (option)
	   (let ((lambda? (cadr option)))
	     (when (cddr option)
	       (error "Found extraneous garbage in ~S" option))
	     (unless (and (listp lambda?)
			  (eq (car lambda?) 'lambda)
			  (listp (cadr lambda?))
			  (cddr lambda?))
	       (error "~S is not a well formed lambda body." lambda?))
	     lambda?)))
    (let ((descriptors (parse-elements elements))
	  (period nil) (parser nil) (filter nil) (state nil))
      (dolist (option options)
	(unless (consp option)
	  (error "Option ~S not a list." option))
	(macrolet ((without-duplicates ((var clause) &body body)
		     `(if ,var
			  (error "Found duplicate option: " ,clause)
			(progn ,@body))))
	  (ecase (car option)
	    (:element-period
	     (without-duplicates (period option)
	       (cond ((and (= (length (cdr option)) 1)
                              (find (second option) descriptors 
                                    :test #'member))
		      (setf period (cons ':all (cdr option))))
                     ((and (member (cadr option) '(:all :any))
			   (loop for x in (cddr option) 
                                 always (find x descriptors :test #'member)))
		      (setf period (cdr option)))
		     (t 
		      (error "Bogus :element-period ~S." (cdr option))))))
	    (:element-parser
	     (without-duplicates (parser option)
	       (setf parser (parse-lambda option))) )
	    (:element-state
	     (without-duplicates (state option)
               (if (find (second state) descriptors :test #'member)
                   (setf state (cdr option))
                 (error "Bogus :element-state ~S. Not in ~S" 
                        (cdr option) elements))))
	    (:element-filter
	     (without-duplicates (filter option)
	       (setf filter (parse-lambda option)))))))

      (when (and (not state)
                 (= (length (cdr period)) 1)
                 (find (second period) descriptors :test #'member))
        (setf state (cdr period)))

    `(progn
       (defprop ,name :item-expand t)
       (defclass ,name (multiple-item) ())
       (defmacro ,name (&rest items)
	 `(make-multiple-item ',',name
			      .,(mapcar #'(lambda (x)
					    (if (listp x)
						x
					      (quote-if-necessary x)))
					items)))
       ,(define-item-method name descriptors (or period '(:all)) state filter)
       ,@(if parser 
	     `((defmultiple-item-parser ,name ,(cadr parser) ,@(cddr parser)))
	   nil)
       ,@(if filter
	     `((defmultiple-item-filter ,name ,(cadr filter) 
                                        ,@(cddr filter))))))))
