;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

;;;
;;; midifile-play, reimplemented by tobias kunze.
;;;

(defun midifile-play (&optional (pathname *default-midi-pathname*)
                      &key (start 0) end (timescale 1.0) 
                           (headstart 1000) port &aux localopen)
  ;; if midi is not open, open if we can
  (unless (midi-open-p)
    (unless port
      (format t "Midi is not currently open on any port. Either open it ~
              or call midifile-play with its :port argument specified.~&")
      (return-from midifile-play nil))
    (unless (midi-port-reference-p port)
       (error "~S is not a legal port reference." port))
    (setf localopen (midi-open :port port)))
  (let* ((scale 1)
         (curtime (+ headstart (midi-get-time)))
         (starttime (+ curtime (* timescale (quanta-time start))))
         (endtime (and end (+ curtime (* timescale (quanta-time end)))))
         (alloff t)
         divis usecs channelfn )

    ;; shift time "left" to skip forward to starttime and start
    ;; playing immediately
    (let ((skiptime (- starttime curtime)))
        (decf curtime skiptime)  
        (decf starttime skiptime)
        (and end (decf endtime skiptime)))

    ;; optimize channelfn based on start and end requirements
    (setf channelfn
      (if (not end)
          (if (= start 0)
              #'(lambda (msg dtime)
                  (incf curtime (* dtime scale timescale))
                  (ff-midi-write-message msg (round curtime)))
            #'(lambda (msg dtime)
                (incf curtime (* dtime scale timescale))
                (unless (< curtime starttime)
                  (ff-midi-write-message msg (round curtime)))))
        (if (= start 0)
            #'(lambda (msg dtime)
                (incf curtime (* dtime scale timescale))
                (if (> curtime endtime)
                    (progn (setf alloff (ceiling curtime))
                           (throw :stop-play nil))    
                  (ff-midi-write-message msg (round curtime))))
          #'(lambda (msg dtime)
              (incf curtime (* dtime scale timescale))
              (cond ((< curtime starttime) nil)
                    ((> curtime endtime)
                     (setf alloff (ceiling curtime))
                     (throw :stop-play nil))
                    (t   
                     (ff-midi-write-message msg (round curtime))))))))

    ;; process the midifile. perform and allnotesoff if we crash while 
    ;; playing or reach a specified end time.
    (unwind-protect
      (progn
        (catch :stop-play
          (midifile-map pathname 
            :channel-message-fn channelfn  
            :header-fn #'(lambda (format nr-of-tracks divisions)
                           (declare (ignore format nr-of-tracks))
                           (setf divis divisions))
            :meta-message-fn
            #'(lambda (message dtime data-length message-data)
                (when (= (ldb (byte 8 8) message) 81) ; +midi-tempo-change+
                  (setf usecs 
                    (do ((temp 0)
                         (i data-length (decf i 1))
                         (j 0 (incf j 1)))
                        ((= i 0) temp)
                      (unless (= (ldb (byte 8 24) (elt message-data j)) 1)
                        (format t "Warning: wrong size of data packet ~
                                   to tempo-change message!"))
                      (incf temp 
                        (ldb (byte (* 8 i) (* 8 j)) 
                             (elt message-data j)))))
                  (setf scale (/ usecs (* divis 1000))))
                (incf curtime (* dtime scale timescale)))
            :system-message-fn
            #'(lambda (message dtime data-length message-data)
                (declare (ignore message data-length message-data))
                (incf curtime (* dtime scale timescale)))))
        ;; unless we threw out, we reached end of file and dont need noteoffs.
        (unless (numberp alloff) (setf alloff nil)))
    ;; always cleanup if terminated by interrupt or throw.
    (when alloff
      (if (eq alloff t)
          (ff-midi-hush)
        (ff-midi-all-notes-off alloff)))
    ; we dont block so we cant close a local open
    ;(when localopen  (midi-close))
    )))

;;;
;;; midifile-map maps functions over data in midifiles.
;;;

(defun midifile-map (pathname &key (tracks t) header-fn channel-message-fn
				   track-fn system-message-fn meta-message-fn)
  (with-open-file (file (merge-pathnames pathname *default-midi-pathname*)
			:direction :input :element-type '(unsigned-byte 8))
    (multiple-value-bind (format ntrks division) 
	(read-midi-file-header file) 
    (when (> format 0)
      (error "midifile-map doesn't support level ~S MIDI files." format))
    (when header-fn
	(funcall header-fn format ntrks division))
      
      (loop for i below ntrks
       for len = (read-track-header file)
       if (or (eq tracks t) (member i tracks :test #'=))
       do
	(when track-fn (funcall track-fn i len))
	(map-track file len system-message-fn
		   meta-message-fn channel-message-fn)
       else
       do (loop for i below len do (read-byte file))))))

;;;
;;; midifile-to-vector
;;;

(defun midifile-to-vector (file &key (tracks t) vector channel-message-fn 
				  meta-message-fn system-message-fn)
  (let ((v (or vector 
	       (make-array 0 :adjustable t :fill-pointer t))))
    (midifile-map file :tracks tracks
    		  :channel-message-fn 
		  (if channel-message-fn
		      #'(lambda (msg time)
			  (vector-push-extend
			    (funcall channel-message-fn msg time) v))
		    nil)
		  :meta-message-fn 
		  (if meta-message-fn
		      #'(lambda (msg time len data)
			  (vector-push-extend 
			     (funcall channel-message-fn msg time len data) v))
		    nil)
		  :system-message-fn
		  (if system-message-fn
		      #'(lambda (msg time len data)
			  (vector-push-extend 
			    (funcall channel-message-fn msg time len data) v))
		    nil))
    v))

;;;
;;; midifile printing
;;;

(defun midifile-print (pathname &key (stream *standard-output*) (tracks t))
  (let ((*standard-output* stream))
    (midifile-map pathname :tracks tracks
      :header-fn #'(lambda (format ntrks division)
                      (format t
                        "~%File: ~a ~%Format: ~a~%Tracks: ~a~%Division: ~s"
                        (namestring 
                          (merge-pathnames pathname *default-midi-pathname*))
                        format ntrks division))
      :track-fn #'(lambda (num len)
                    (format t "~%Track ~d, length ~S" num len))
      :channel-message-fn #'midi-print-message
      :system-message-fn #'(lambda (message time data-length message-data)
                             (declare (ignore data-length))
                             (midi-print-message 
                               message time :message-data message-data))
      :meta-message-fn #'(lambda (message time data-length message-data)
                           (declare (ignore data-length))
                           (midi-print-message
                             message time :message-data message-data)))))

;;;
;;; midifile header reading and writing 
;;;

(defun midifile-read-header (pathname)
  (with-open-file (file (merge-pathnames pathname *default-midi-pathname*)
		   :element-type '(unsigned-byte 8))  
    (read-midi-file-header file)))
    
(defun midifile-write-header (pathname format tracks division)
  (with-open-file (file (merge-pathnames pathname *default-midi-pathname*)
		   :element-type '(unsigned-byte 8))  
    (write-midi-file-header file format tracks division)))  

;;;
;;; low level midifile functions
;;;

(defun write-message (message stream &optional (time 0) message-data)
  (declare (optimize (speed 3)(safety 0))
           #-(or aclpc clisp) (fixnum message)
           )
  (macrolet ((write-msg (m l s)
	     `(loop for pos from 16 downto 0 by 8
		    repeat ,l
		    do (write-byte (ldb (byte 8 pos) ,m) ,s))))
    (let ((count (if time (write-variable-quantity time stream) 0))
	  (size (midi-message-size message)))
      (declare (fixnum count size))
      (write-msg message size stream)
      (incf count size)		 
      (when (= (ldb +midi-type-byte+ message) #xf) ; system, sysex, meta
        (incf count (write-variable-quantity (length message-data) stream))
	(loop for msg in message-data
	  do (setf size (midi-message-size msg))
	     (write-msg msg size stream)
	     (incf count size)))
      count)))
      
(defun read-bytes (stream number-of-bytes)
  (declare (optimize (speed 3)(safety 0)))
  (loop with unsigned = 0 
   for i below number-of-bytes
   do (setf unsigned (+ (ash unsigned 8) (logand (read-byte stream) #xff)))
   finally (return unsigned)))

(defun write-bytes (stream number number-of-bytes)
 (declare (optimize (speed 3)(safety 0)))
  (loop for pos from (* (1- number-of-bytes) 8) downto 0 by 8
   do (write-byte (ldb (byte 8 pos) number) stream)))

(defun read-variable-quantity (stream)
  (loop with result = 0 
   for byte = (logand (read-byte stream) #xff)
   for count from 1
   do (setf result (+ (ash result 7) (logand byte #x7f)))
   while (logtest #x80 byte)
   finally (return (values result count))))

(defun write-variable-quantity (number stream)
  (let ((encoded (logand number #x7f)))
    (loop while (> (setf number (ash number -7)) 0)
     do (setf encoded (+ (logior (ash encoded 8) #x80) (logand number #x7f))))
    (loop counting (prog1 t (write-byte (logand encoded #xff) stream))
     while (logtest #x80 encoded)
     do (setf encoded (ash encoded -8)))))

(defconstant +MThd+ 1297377380)  ; 'MThd'
(defconstant +MTrk+ 1297379947)  ; 'MTrk'

(defun read-midi-file-header (stream)
  (let ((type (read-bytes stream 4)))
    (unless (= type +MThd+)
      (error "Expected 'MThd' but got ~s. instead." type)))
  (read-bytes stream 4)			; ignore header length bytes
  (values (read-bytes stream 2)		; format
	  (read-bytes stream 2)		; ntracks
	  (read-bytes stream 2)))	; divisions

(defun write-midi-file-header (stream format tracks division)
  (write-bytes stream +MThd+ 4)
  (write-bytes stream 6 4)              ; header length stored 00 00 00 06
  (write-bytes stream format 2)
  (write-bytes stream tracks 2)
  (write-bytes stream division 2)
  (values))

(defun read-track-header (stream)
  (let ((type (read-bytes stream 4)))
    (unless (= type +MTrk+)
      (error "Expected 'MTrk' but got ~s. instead." type)))
  (read-bytes stream 4))		; return length

(defun write-track-header (stream length)
  (write-bytes stream +MTrk+ 4)
  (write-bytes stream length 4)
  (values))

(defun map-track (stream length systemfn metafn channelfn)
  (let ((count 0) time byte type chan data1 data2 size message)
    (loop while (< count length)
          do
      (setf time (read-variable-quantity stream))
      (setf byte (read-byte stream))
      (cond ((logtest byte #b10000000)	; status byte
             (cond ((< byte #xf0)	; channel message
                    (setf type (ash (logand byte #xf0) -4)
                          chan (logand byte #x0f)
                          data1 (read-byte stream)
                          data2 (if (or (= type +midi-program-change+)
                                        (= type +midi-channel-pressure+))
                                     nil
                                  (read-byte stream))
                          size (if data2 3 2))
                    (setf message 
                      (make-channel-message type chan data1 data2))
                    (when channelfn 
                      (funcall channelfn message time))
                    (incf count))
                   ((= byte +midi-meta+); meta message
                    (let (data-length message-data)
                      (setf type nil byte (read-byte stream))
                      (setf data-length (read-variable-quantity stream))
                      (multiple-value-setq (message message-data)
                        (apply #'make-meta-message byte
                      (loop repeat data-length
                            collect (read-byte stream)))) 
                      (when metafn 
                        (funcall metafn message time data-length
                                 message-data))
                      (when (= byte +midi-eot+)	; End of track
                        (return))
                      (incf count)))
                   (t			        ; sysex message ??? system too
                    (let (data-length message-data)
                      (setf type nil size nil)
                      (multiple-value-setq (message message-data) 
                        (apply #'make-sysex-message
                               (loop for b = (read-byte stream)
                                     repeat (setf data-length 
                                              (read-variable-quantity stream))
                                     until (= b +midi-eox+)
                                     collect b
                                     )))
                      (incf count data-length)
                      (when systemfn
                        (funcall systemfn message time data-length
                                 message-data))
                      (incf count)))))
            (t				; running status
             (setf data1 byte
                   data2 (if (or (= type +midi-program-change+)
                                 (= type +midi-channel-pressure+))
                             nil
                           (read-byte stream)))
             (setf message (make-channel-message type chan data1 data2))
             (funcall channelfn message time)
             (incf count))))))

;;;
;;; midifile-parse parses midievents as parameteter note information
;;; and maps a user specified function across the data.  The supplied
;;; function must accept 6 arguments:
;;; 	(channel begin rhythm duration frequency amplitude)
;;; where channel is the midi channel of the note, begin is the starting
;;; time (seconds) of the note, rhythm is the real time until the next
;;; note (if any), duration is the total length of the note (seconds),
;;; frequency is the midi key number of the note and amplitude is the
;;; midi velocity value of the original note on.  Note that the parsing
;;; process ignores note off velocity and sets the rhythic value of the
;;; last note to nil.
;;; The keyword argument :channels controls how the midifile is to 
;;; be "time lined".  If channels is t (the default), each channel is 
;;; parsed in its own time line, ie start times, rhythms and durations are
;;; calculated using only notes from the same channel.  If channels is nil,
;;; then all the channels are "collapsed" into one parsing time line, which
;;; is indentical to the time line represented by the midifile itself.
;;; Otherwise, :channels should be lists of lists; each sublist represents
;;; channels that are to be grouped in a single time line.  :Merge controls
;;; whether or not data in seperate time lines is sorted prior to
;;; function mapping.  The default value nil means that the seperate time
;;; lines are not merged, and the function is mapped over the notes in
;;; one time line before the next time line is considered.  If merge is
;;; t then the function is mapped over notes sorted by start time.
;;;

(defun midifile-parse (file fn &key (channels t) merge)
  (flet ((map-note (e fn)
	   (let ((m (first e)))
	     (let ((channel (note-on-channel m))
	  	   (begin (second e))
		   (duration (third e))
		   (rhythm (or (fourth e) (third e)))
		   (key (note-on-key m))
		   (velocity (note-on-velocity m)))
	       (funcall fn channel begin rhythm duration key velocity))))
          (channel-index (channel channels)
            (if (null channels)
	        0
	      (if (eq channels t) 
                  channel
		(position channel channels :test #'member)))))
    (let ((quanta 0)
	  (time 0.0)
	  (tempo-factor 1.0)
	events divisions )

      (cond ((eq channels t)
             (setf events (make-array 16)))
	    ((null channels)
	     (setf events (make-array 1)))
	    (t
             (setf channels (loop for c in channels 
                                  if (integerp c) collect (list c) 
			          else collect c))
	     (setf events (make-array (length channels)))))
      (midifile-map file
        :header-fn
        #'(lambda (format tracks div) 
            (declare (ignore format tracks))
            (setf divisions div))
        :meta-message-fn
        #'(lambda (message time length data)
            (declare (ignore time length))
            (when (tempo-change-p message)
              (setf tempo-factor  
                (/ (logior (ash (midi-message-data1 (pop data)) 16)
	                   (ash (midi-message-data1 (pop data)) 8)
                           (midi-message-data1 (pop data)))
	            1000000.0))))
        :channel-message-fn
        #'(lambda (message beats &aux channel index)
            (incf quanta beats)
            (setf time (* (/ quanta divisions) tempo-factor))
	    (setf channel (channel-message-channel message))
            (when (setf index (channel-index channel channels))
	      (cond ((or (note-off-p message)
	                 (and (note-on-p message)
	                      (= 0 (note-on-velocity message))))
                     (let ((chan+key (ldb (byte 12 8) message))
			   entry)
		       ;; find the matching noteOn and compute event's 
		       ;; duration. event is found if it has same channel
		       ;; and keynum and doesn't already have a duration
		       (setf entry (find message (aref events index)
				         :test #'(lambda (m e)
				                   (declare (ignore m))
				  	           (and (null (third e))
					                (= (ldb (byte 12 8)
						                (first e))
						           chan+key)))))
  	  	       (unless entry
		         (error "Can't find noteOn for noteOff ~S at ~S."
		       	        message time))
		       (setf (third entry) (- time (second entry)))))
                     ((note-on-p message)
		      (let* ((tail (last (aref events index)))
			     (last (car tail))
			     (next (list message time nil nil)))
		         (cond (tail
		     	        (setf (fourth last) (- time (second last)))
			        (rplacd tail (list next)))
		     	       (t
			        (setf (aref events index) (list next))))))))))
     (if merge
         (let (merged)
	   (setf events (loop for i from 0 below (length events)
	 	              when (aref events i) collect it))
           (setf merged (pop events))		            
	   (loop for list in events
	   	 do (setf merged (merge 'list merged list #'<
		 			:key #'second)))
	    (dolist (n merged)
	      (map-note n fn)))
        (let (notes)
          (dotimes (i (length events))
            (when (setf notes (aref events i))
              (dolist (n notes)
	        (map-note n fn)))))))))
#|

(defun midifile-map-as-notes (file fn)
  (flet ((map-as-note (e fn)
	   (let ((m (first e)))
	     (let ((channel (note-on-channel m))
	  	   (begin (second e))
		   (duration (third e))
		   (rhythm (fourth e))
		   (key (note-on-key m))
		   (velocity (note-on-velocity m)))
	       (funcall fn channel begin rhythm duration key velocity)))))
    (let ((channels (make-array 16))
	  (quanta 0)
	  (time 0.0)
	  divisions tempo-factor)
      (midifile-map file
        :header-fn
        #'(lambda (format tracks div) 
            (declare (ignore format tracks))
            (setf divisions div))
        :meta-message-fn
        #'(lambda (message time length data)
            (declare (ignore time length))
            (when (tempo-change-p message)
              (setf tempo-factor  
                (/ (logior (ash (midi-message-data1 (pop data)) 16)
	                   (ash (midi-message-data1 (pop data)) 8)
                           (midi-message-data1 (pop data)))
	            1000000.0))))
        :channel-message-fn
        #'(lambda (message beats)
            (incf quanta beats)
            (setf time (* (/ quanta divisions) tempo-factor))
            (cond ((or (note-off-p message)
	               (and (note-on-p message)
	                    (= 0 (note-on-velocity message))))
                   (let ((channel 0 ;(note-off-channel message)
		   			))
		     ;; find the matching noteOn and compute event's duration.
		     (let ((entry (find message (aref channels channel)
				       :test #'(lambda (m e)
				  	         (and (null (third e))
					             (= (note-on-key (first e))
						        (note-on-key m)))))))
		       (unless entry
		         (error "Can't find noteOn for noteOff ~S at ~S."
		     	        message time))
		       (setf (third entry) (- time (second entry))))
		     ;; output all events that have duration & rhythm computed.
		     (loop while (aref channels channel)
		           for e = (car (aref channels channel))
		           while (and (third e) (fourth e))
		           do 
		        (map-as-note e fn) 
		        (pop (aref channels channel)))))
                   ((note-on-p message)
		    (let* ((chan 0 ;(note-on-channel message)
		    		)
		           (tail (last (aref channels chan)))
			   (last (car tail))
			   (next (list message time nil nil)))
		       (cond (tail
		     	      (setf (fourth last) (- time (second last)))
			      (rplacd tail (list next)))
		     	     (t
			      (setf (aref channels chan) (list next)))))))))
      ;; flush last notes. these were not output because they have no rhythm
      (loop with notes for i below 1 ;16
    	    when (setf notes (aref channels i))
	    do (loop for n in notes do (map-as-note n fn)))
      (values))))

(defun integer-to-variable-length (value)
  (declare (optimize (speed 3)(safety 0)))
  (let ((encoded (logand value #x7f))
	(result 0))
    (loop while (> (setf value (ash value -7)) 0)
	  do
      (setf encoded (+ (logior (ash encoded 8) #x80) (logand value #x7f))))
    (loop do
      (setf result (logior result (logand encoded #xff)))
      (if (logtest #x80 encoded)
	  (setf result (ash result 8) encoded (ash encoded -8))
	(return result)))))

(defun variable-length-to-integer (value)
  (let ((decoded (logand value #x7f))
	(pos 0))
    (loop while (logtest #x80 (setf value (ash value -8)))
	  do
      (incf pos)
      (setf decoded (+ (ash decoded 7) (logand value #x7f))))
    (let ((result (logand decoded #x7f)))
      (loop for i from 0 below pos
	    do 
	(setf result (+ (ash result 7) (logand (setf decoded (ash decoded -7))
					       #x7f))))
      result)))


|#
