;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

;;;
;;; Any port of this midi code needs to implement a low level 
;;; interface of the following functions:
;;;
;;; +ff-midi-success+                 success/error value
;;; ff-midi-open (portnum)            open midi on port 0 or 1
;;; ff-midi-close ()                  close midi port
;;; ff-midi-stop-timer ()             stop the timer
;;; ff-midi-start-timer ()            start the timer
;;; ff-midi-get-time (array)          put quanta time and size in array[0,1]
;;; ff-midi-set-time (qtime)          set quanta time (integer)
;;; ff-midi-set-quanta-size (size)    set size (integer) of quanta 
;;; ff-midi-read-messages ()          perform lisp_call on event message
;;; ff-midi-write-message (msg qtime) write msg (integer) at time (integer)
;;; ff-midi-flush-transmit ()         clear pending output queue
;;; ff-midi-flush-receive  ()         clear pending input queue
;;; ff-midi-hush  ()                  shut it up
;;; ff-midi-all-notes-off (time)      schedule note offs at time
;;; ff-midi-find-port (name)          return name if legal port reference
;;; ff-midi-port-id   (name)          return port int for port
;;;

(defparameter *default-midi-pathname* 
   (namestring
     (make-pathname :directory  (pathname-directory (user-homedir-pathname))
                    :name #-dos "test" #+dos "TEST"
                    :type +midi-type+))
   "The default MIDI output file.")
				  
(defparameter *midi-port* nil
   "The default MIDI port to use.")
				  
(defun midi-port-reference-p (x)
  (ff-midi-find-port x))

(defparameter *quanta-size* 1000 
  "MIDI quanta size in micro-seconds.")

(defvar *quanta-time* 0 
  "MIDI quanta time.")

#+lispworks
(defun *null-midi-read-hook* (msg tim))

(defvar *midi-read-hook* nil 
  "A function to invoke when a MIDI event is read.")

;;;
;;; real and quanta time conversion
;;;

(defun quanta-time (real-time)
  (floor (* real-time 1000000) *quanta-size*))

(defun real-time (quanta-time)
  (/ (* quanta-time *quanta-size*) 1000000.0))

;;;
;;; opening/closing
;;;

(defvar *midi-open-p* nil "Midi port is currently open.")

(defun midi-open (&key (port *midi-port*))
     (unless (midi-port-reference-p port)
       (error "~S is not a legal port name." port))
     (or *midi-open-p*
         (let ((flg (ff-midi-open (ff-midi-port-id port))))
           (if (= flg +ff-midi-success+)
             (setf *midi-open-p* (or port #-mcl t #+mcl %midi-open-p%))
             ;; was: (setf *midi-open-p* (or port t))
             (progn (warn "ff-midi-open failed with error: ~s" flg)
                    nil)))))

(defun midi-close (&optional port)
  (declare (ignore port))
  (and *midi-open-p*
       (let ((flg (ff-midi-close)))
         (if (= flg +ff-midi-success+)
             (progn (setf *midi-open-p* nil) t)
           (progn (warn "ff-midi-close failed with error: ~s" flg)
                  nil)))))

(defun midi-open-p ()
  *midi-open-p*)

#+tl
(defun ask-port (&key prompt (input "") default (abort-ok t) (check t)
                      (type 'midi) (stream t))
  (let (port)
    (if (and check (setf port (midi-open-p)))
        port
      (progn
        (unless prompt
          (setf prompt (format nil "~:(~A~) port to open: " type)))
        (tl:ask-user :prompt prompt :input input :default default
                     :stream stream :null-ok (and default ':ask) 
                     :predicate #'midi-port-reference-p
                     :abort-ok abort-ok)))))
;;;
;;; timer functions
;;;

(defun midi-stop-timer ()
  (ff-midi-stop-timer))

(defun midi-start-timer ()
  (ff-midi-start-timer))
		
(defvar +get-time-args+ 
  (make-array 2 :element-type #-(or aclpc clisp) 'fixnum 
                              #+(or aclpc clisp) 'bignum
                :initial-element 0))

(defun midi-get-time ()
  (let ((flg (ff-midi-get-time +get-time-args+)))
    (if (= flg +ff-midi-success+)
        (values (setf *quanta-time* (elt +get-time-args+ 0))
                (setf *quanta-size* (elt +get-time-args+ 1)))
      (progn (warn "ff-midi-get-time failed with error: ~s" flg)
             (values nil nil)))))

(defun midi-set-time (quanta-time)
  (let ((flg (ff-midi-set-time quanta-time)))
    (if (= flg +ff-midi-success+)
	quanta-time
      (progn (warn "ff-midi-set-time failed with error: ~s" flg)
             nil))))

(defun midi-quanta-size ()
  *quanta-size*)

(defun midi-set-quanta-size (usecs-per-quantum)
  (let ((flg (ff-midi-set-quanta-size usecs-per-quantum)))
    (if (= flg +ff-midi-success+)
        (setf *quanta-size* usecs-per-quantum)
      (progn (warn "ff-midi-set-quanta-size failed with error: ~s" flg)
             nil))))

;;;
;;; resetting
;;;

(defun midi-hush (&optional (flush t))
  (declare (ignore flush))
  (let ((flag (ff-midi-hush)))
    (if (= flag +ff-midi-success+)
        t
      (progn (warn "ff-midi-hush failed with error: ~s" flag)
             nil))))

(defun midi-all-notes-off (&optional (time 0))
  (let ((flag (ff-midi-all-notes-off time)))
    (if (= flag +ff-midi-success+)
        t
      (progn (warn "ff-midi-all-notes-off failed with error: ~s" flag)
             nil))))
 
;;;
;;;  reading
;;;						 
						 
(defun midi-read-messages (&optional (fn *midi-read-hook*))
  (declare (optimize (speed 3)(safety 0)))
  (let (#+lispworks
	(*old-midi-read-hook* *midi-read-hook*)
	(*midi-read-hook* fn))
    #+lispworks
    (if fn
	(setf (symbol-function '*midi-read-hook*) fn)
      (when *old-midi-read-hook*
	    (setf (symbol-function '*midi-read-hook*) 
		  (symbol-function '*null-midi-read-hook*))))
    (let ((flg (ff-midi-read-messages)))
      (declare (fixnum flg))
      (if (= flg +ff-midi-success+)
	  t
	(progn (warn "ff-midi-read-messages failed with error: ~s" flg)
	       nil)))
    #+lispworks
    (when (or fn *old-midi-read-hook*)
	  (if *old-midi-read-hook*
	      (setf (symbol-function '*midi-read-hook*) *old-midi-read-hook*)
	    (setf (symbol-function '*midi-read-hook*) 
		  (symbol-function '*null-midi-read-hook*))))))

(defun midi-listen (&optional fn)
  (setf *midi-read-hook* fn)
  (ff-midi-listen))

(defun midi-stop-listening ()
  (let ((flg (ff-midi-stop-listening)))
   (if (= flg 0)
       (progn (setf *midi-read-hook* nil)
              t)
      (progn (warn "ff-midi-stop-listening failed with: ~S" flg)
      	     nil))))

;;;
;;; writing
;;;
						 
(defun midi-write-message (message &optional (time 0) message-data)
  (declare #-(or aclpc clisp) (fixnum message time)
           #+(or aclpc clisp) (bignum message time)
  	   (list message-data)
           (optimize (speed 3)(safety 0)))
  (let ((flg (ff-midi-write-message message time)))
    (declare (fixnum flg))
    (when message-data
      (loop for m in message-data
	    while (= flg +ff-midi-success+)
	    do
	 (setf flg (ff-midi-write-message m time))))
    (if (= flg +ff-midi-success+)
	t
      (progn (warn "ff-midi-write-message failed with error: ~s" flg)
	     nil))))  
  
;;;
;;; utilities
;;;

(defmacro with-midi-open ((&key port quanta-size) &body body)
  (declare (ignore quanta-size))
  (let ((var (gensym)))
    `(let ((,var nil))
       (unwind-protect
          (progn (unless (midi-open-p) 
	           (midi-open :port ,port)
		   (setf ,var t))
               ,@body)
          (when ,var
            (midi-close))))))  
  
(defun midi-receive-messages (&optional (fn *midi-read-hook*)
			      &key (port *midi-port*) prompt keynum length 
			           end duration stop-if stop-if-not start)
  (declare (optimize (speed 3)(safety 0)))
  (unless port 
    (unless (setf port (midi-open-p))
      (error "Midi port is not currently open.")))

  (let ((count 0) wrapper)
    (macrolet ((checkdup (&rest args)
                 `(when (or ,@args)
                      (error "keynum, length, end, duration, stop-if and ~
                          stop-if-not are mutually exclusive keywords.")))
               (stop-receiving ()
                 #+mcl '(locally (declare (special stopflag)) 
                          (setf stopflag t))
                 #-mcl '(throw :midi-stop-reading nil)))
      (cond (length
             (checkdup keynum end duration stop-if stop-if-not)
             (setf wrapper
               #'(lambda (msg time) 
                   (declare (fixnum msg time)
                            (optimize (speed 3)(safety 0)))
                   (funcall (the function fn)
                            (the fixnum msg) (the fixnum time))
                   (unless (< (incf (the fixnum count)) (the fixnum length))
                     (stop-receiving)))))
            ((or end duration)
             (checkdup keynum length (and end duration) stop-if stop-if-not)
             (if end (unless (integerp end) (setf end (quanta-time end)))
               (unless (integerp duration)
                 (setf duration (quanta-time duration))))
             (setf wrapper
               #'(lambda (msg time)
                   (declare (fixnum msg time)
                            (optimize (speed 3) (safety 0)))
                   (unless (< (the fixnum time) (the fixnum end))
                     (stop-receiving))
                   (funcall (the function fn)
                            (the fixnum msg) (the fixnum time)))))
            (keynum
             (checkdup length end duration stop-if-not)
             (unless (and (integerp keynum) (<= 0 keynum 127))
               (error "~S is not a keynum: 0<127" keynum))
             (setf wrapper 
               #'(lambda (msg time)
                   (declare (fixnum msg time)
                            (optimize (speed 3) (safety 0)))
                   (if (= (the fixnum (note-on-key msg))
                          (the fixnum keynum))
                       (if (or (= 0 (the fixnum (note-on-velocity msg))) 
                               (note-off-p msg))
                            nil
                          (stop-receiving))
                     (funcall (the function fn) (the fixnum msg) 
                              (the fixnum time))))))
            (stop-if
             (checkdup keynum length end duration stop-if-not)
             (setf wrapper
               #'(lambda (msg time) 
                   (declare (fixnum msg time)
                            (optimize (speed 3) (safety 0)))
                   (when (funcall (the function stop-if)
                                  (the fixnum msg) (the fixnum time))
                     (stop-receiving))
                   (funcall (the function fn)
                            (the fixnum msg) (the fixnum time)))))
            (stop-if-not
             (checkdup keynum length end duration stop-if)
             (setf wrapper
               #'(lambda (msg time) 
                   (declare (fixnum msg time)
                            (optimize (speed 3)(safety 0)))
                   (unless (funcall (the function stop-if-not)
                                    (the fixnum msg) (the fixnum time))
                     (stop-receiving))
                   (funcall (the function fn)
                            (the fixnum msg) (the fixnum time)))))
            (t
              (setf wrapper fn))))
    (with-midi-open (:port port)
      (ff-midi-read-messages)
      (let ((*midi-read-hook* wrapper)
            (stopflag nil))
        (declare #+mcl (special stopflag) #-mcl (ignore stopflag))
        (catch :midi-stop-reading
          (when prompt (format t prompt) (read-line))
          (when start 
	    (midi-set-time (if (integerp start) start (quanta-time start))))
          (when duration (setf end (+ (midi-get-time) duration)))
          (loop do
            #+mcl (when stopflag (setf stopflag nil) (return))
            (let ((flg (ff-midi-read-messages)))
              (declare (fixnum flg))
              (if (= flg +ff-midi-success+)
	          t
	        (progn (warn "ff-midi-read-messages failed with error: ~s"
		             flg)
	               nil)))))
        (ff-midi-read-messages)))))

