;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

;;;
;;; midi messages are (hopefully) fixnums containing 1 to 3 bytes of data
;;; positioned in a 28 bit field:
;;;
;;; Position:  27-26   25-24    23-16  15-08   07-00
;;; Contents:  type    #bytes   data1  data2   data3
;;;
;;; Bits 26 to 28 are reserved for type tagging. Bits 24 to 25 contain
;;; the number of bytes in the message (a number beween 1 and 3).  Bits
;;; 16 to 24 contain the first data byte, 8 to 16 the second data byte
;;; (if any), and 0 to 7 the third data byte (if any).
;;;

(defconstant +midi-message-type-byte+  (byte 3 26))
(defconstant +midi-message-size-byte+  (byte 2 24))
(defconstant +midi-message-data1-byte+ (byte 8 16))
(defconstant +midi-message-data2-byte+ (byte 8 8))
(defconstant +midi-message-data3-byte+ (byte 8 0))

(defconstant +midi-status-byte+     (byte 8 16))
(defconstant +midi-data1-byte+      (byte 8 8))
(defconstant +midi-data2-byte+      (byte 8 0))
(defconstant +midi-type-byte+       (byte 4 20))
(defconstant +midi-channel-byte+    (byte 4 16))
(defconstant +midi-meta-tempo-byte+ (byte 16 0))

(defconstant +midi-data-tag+    #b00)          ; message is raw data
(defconstant +midi-channel-tag+ #b01)
(defconstant +midi-system-tag+  #b10)
(defconstant +midi-meta-tag+    #b11)

(defconstant +midi-note-off+         #x8)      ; channel message types
(defconstant +midi-note-on+          #x9)
(defconstant +midi-key-pressure+     #xa)
(defconstant +midi-control-change+   #xb)
(defconstant +midi-program-change+   #xc)
(defconstant +midi-channel-pressure+ #xd)
(defconstant +midi-pitch-bend+       #xe)

(defconstant +midi-sysex+            #xf0)     ; system message types
(defconstant +midi-song-position+    #xf2)
(defconstant +midi-song-select+      #xf3)
(defconstant +midi-tune-request+     #xf6)
(defconstant +midi-eox+              #xf7)

(defconstant +midi-meta+             #xff)     ; midi meta event types
(defconstant +midi-eot+              #x2f)
(defconstant +midi-tempo-change+     #x51)
(defconstant +midi-time-signature+   #x58)

;;;
;;; low level message field accessors and predicates. dont use these,
;;; use their higher level sibings defined later in this file.
;;;

(defmacro midi-message-type (message)
  `(ldb +midi-message-type-byte+ ,message))

(defmacro midi-message-size (message)
  `(ldb +midi-message-size-byte+ ,message))

(defmacro midi-message-data1 (message)
  `(ldb +midi-message-data1-byte+ ,message))

(defmacro midi-message-data2 (message)
  `(ldb +midi-message-data2-byte+ ,message))

(defmacro midi-message-data3 (message)
  `(ldb +midi-message-data3-byte+ ,message))
		 		 
(defmacro midi-channel-message-p (message)
  `(= (ldb +midi-message-type-byte+ ,message) +midi-channel-tag+))

(defmacro midi-system-message-p (message)
  `(= (ldb +midi-message-type-byte+ ,message) +midi-system-tag+))


(defmacro midi-meta-message-p (message)
  `(= (ldb +midi-message-type-byte+ ,message) +midi-meta-tag+))

(defmacro midi-data-message-p (message)
  (let ((var (gensym)))
    `(let ((,var ,message))
       (and (= (ldb +midi-message-type-byte+ ,var) +midi-data-tag+)
            (< 0 (ldb +midi-message-size-byte+ ,var) 4)))))

;;;
;;; midi message constructors, predicates and accessors. though it would
;;; have been much easier to define all the accessors as macros, it is
;;; reasonable to want to funcall these when mapping over vectors of
;;; messages, hence the inline functions with hand made setf expansions.
;;;

(proclaim '(inline make-channel-message channel-message-status 
                                        channel-message-channel
					channel-message-data1 
		                        channel-message-data2
                   note-on-channel note-on-key note-on-velocity 
                   note-off-channel note-off-key note-off-velocity
                   key-pressure-channel key-pressure-key key-pressure-pressure
                   control-change-channel control-change-control
                                          control-change-change
                   program-change-channel program-change-program
                   channel-pressure-channel channel-pressure-pressure
                   pitch-bend-channel pitch-bend-lsb pitch-bend-msb
                   channel-mode-channel channel-mode-control
                                        channel-mode-change
                   make-system-message system-message-status 
		                       system-message-data1 
				       system-message-data2
	           song-position-lsb song-position-msb
		   song-select-song
		   make-sysex-message sysex-message-id
		   make-meta-message meta-message-type
		   make-midi-data midi-data-data1 midi-data-data2
		                                  midi-data-data3))

;;;
;;; if you understand this one you get an A+...
;;;

(eval-when (compile load eval)
  (defmacro define-message-setf (accessor bytespec)
    `(#-(and cltl2 excl) define-setf-method 
      #+(and cltl2 excl) define-setf-expander
              ,accessor (message)
       (multiple-value-bind (temps vals stores store-form access-form)
	   (#-(and cltl2 excl) get-setf-method 
            #+(and cltl2 excl) get-setf-expansion
                    message)
	 (let ((val (gensym))
	       (msg (first stores)))
       	   (values temps
		   vals
		   (list val)
		   ,`(list 'let* (list (list msg (list 'dpb val ',bytespec 
						       access-form)))
			   store-form
			   val)
		   ,`(list 'ldb ',bytespec access-form))))))
  )

;;;
;;; :channel-message
;;;

(defun make-channel-message (status channel data1 &optional data2)
  (dpb +midi-channel-tag+ +midi-message-type-byte+
      (dpb (if data2 3 2) +midi-message-size-byte+
           (dpb status +midi-type-byte+ 
	        (dpb channel +midi-channel-byte+
		     (dpb data1 +midi-message-data2-byte+
		          (dpb (or data2 0) +midi-message-data3-byte+ 0)))))))

(defun channel-message-p (message)
  (midi-channel-message-p message))

(defun channel-message-status (message)
  (ldb +midi-type-byte+ message))

(defun channel-message-channel (message)
  (ldb +midi-channel-byte+ message))

(defun channel-message-data1 (message)
  (ldb +midi-message-data2-byte+ message))

(defun channel-message-data2 (message)
  (ldb +midi-message-data3-byte+ message))

(define-message-setf channel-message-status +midi-type-byte+)
(define-message-setf channel-message-channel +midi-channel-byte+)
(define-message-setf channel-message-data1  +midi-message-data2-byte+)
(define-message-setf channel-message-data2 +midi-message-data3-byte+)

;;;
;;; :note-on
;;;

(defun make-note-on (channel key velocity)
  (make-channel-message +midi-note-on+ channel key velocity))

(defun note-on-p (message)
  (= (ldb +midi-type-byte+ message) +midi-note-on+))

(defun note-on-channel (message)
  (ldb +midi-channel-byte+ message))

(defun note-on-key (message)
  (ldb +midi-message-data2-byte+ message))

(defun note-on-velocity (message)
  (ldb +midi-message-data3-byte+ message))

(define-message-setf note-on-channel +midi-channel-byte+)
(define-message-setf note-on-key  +midi-message-data2-byte+)
(define-message-setf note-on-velocity +midi-message-data3-byte+)

;;;
;;; :note-off
;;;

(defun make-note-off (channel key velocity)
  (make-channel-message +midi-note-off+ channel key velocity))

(defun note-off-p (message)
  (= (ldb +midi-type-byte+ message) +midi-note-off+))
	
(defun note-off-channel (message)
  (ldb +midi-channel-byte+ message))

(defun note-off-key (message)
  (ldb +midi-message-data2-byte+ message))

(defun note-off-velocity (message)
  (ldb +midi-message-data3-byte+ message))

(define-message-setf note-off-channel +midi-channel-byte+)
(define-message-setf note-off-key  +midi-message-data2-byte+)
(define-message-setf note-off-velocity +midi-message-data3-byte+)

;;;
;;; :key-pressure
;;;

(defun make-key-pressure (channel key pressure)
  (make-channel-message +midi-key-pressure+ channel key pressure))

(defun key-pressure-p (message)
  (= (ldb +midi-type-byte+ message) +midi-key-pressure+))
	
(defun key-pressure-channel (message)
  (ldb +midi-channel-byte+ message))

(defun key-pressure-key (message)
  (ldb +midi-message-data2-byte+ message))

(defun key-pressure-pressure (message)
  (ldb +midi-message-data3-byte+ message))

(define-message-setf key-pressure-channel +midi-channel-byte+)
(define-message-setf key-pressure-key  +midi-message-data2-byte+)
(define-message-setf key-pressure-pressure +midi-message-data3-byte+)

;;;
;;; :control-change
;;;

(defun make-control-change (channel control change)
  (make-channel-message +midi-control-change+ channel control change))

(defun control-change-p (message)
  (= (ldb +midi-type-byte+ message) +midi-control-change+))

(defun control-change-channel (message)
  (ldb +midi-channel-byte+ message))

(defun control-change-control (message)
  (ldb +midi-message-data2-byte+ message))

(defun control-change-change (message)
  (ldb +midi-message-data3-byte+ message))

(define-message-setf control-change-channel +midi-channel-byte+)
(define-message-setf control-change-control +midi-message-data2-byte+)
(define-message-setf control-change-change +midi-message-data3-byte+)

;;;
;;; :program-change
;;;

(defun make-program-change (channel program)
  (make-channel-message +midi-program-change+ channel program))

(defun program-change-p (message)
  (= (ldb +midi-type-byte+ message) +midi-program-change+))

(defun program-change-channel (message)
  (ldb +midi-channel-byte+ message))

(defun program-change-program (message)
  (ldb +midi-message-data2-byte+ message))

(define-message-setf program-change-channel +midi-channel-byte+)
(define-message-setf program-change-program +midi-message-data2-byte+)

;;;
;;; :channel-pressure
;;;

(defun make-channel-pressure (channel pressure)
  (make-channel-message +midi-channel-pressure+ channel pressure))

(defun channel-pressure-p (message)
  (= (ldb +midi-type-byte+ message) +midi-channel-pressure+))
	
(defun channel-pressure-channel (message)
  (ldb +midi-channel-byte+ message))

(defun channel-pressure-pressure (message)
  (ldb +midi-message-data2-byte+ message))

(define-message-setf channel-pressure-channel +midi-channel-byte+)
(define-message-setf channel-pressure-pressure +midi-message-data2-byte+)

;;;
;;; :pitch-bend
;;;

(defun make-pitch-bend (channel lsb msb)
  (make-channel-message +midi-pitch-bend+ channel lsb msb))

(defun pitch-bend-p (message)
  (= (ldb +midi-type-byte+ message) +midi-pitch-bend+))

(defun pitch-bend-channel (message)
  (ldb +midi-channel-byte+ message))

(defun pitch-bend-lsb (message)
  (ldb +midi-message-data2-byte+ message))

(defun pitch-bend-msb (message)
  (ldb +midi-message-data3-byte+ message))

(define-message-setf pitch-bend-channel +midi-channel-byte+)
(define-message-setf pitch-bend-lsb +midi-message-data2-byte+)
(define-message-setf pitch-bend-msb +midi-message-data3-byte+)

;;;
;;; :channel-mode (control change)
;;;

(defun make-channel-mode (channel control change)
  (make-channel-message +midi-control-change+ channel control change))
			   
(defun channel-mode-p (message)
  (= (ldb +midi-type-byte+ message) +midi-control-change+))

(defun channel-mode-channel (message)
  (ldb +midi-channel-byte+ message))

(defun channel-mode-control (message)
  (ldb +midi-message-data2-byte+ message))

(defun channel-mode-change (message)
  (ldb +midi-message-data3-byte+ message))

(define-message-setf channel-mode-channel +midi-channel-byte+)
(define-message-setf channel-mode-control +midi-message-data2-byte+)
(define-message-setf channel-mode-change +midi-message-data3-byte+)

;;;
;;; :system-message
;;;

(defun make-system-message (status &optional data1 data2)
  (dpb +midi-system-tag+ +midi-message-type-byte+
       (dpb (if data2 3 (if data1 2 1)) +midi-message-size-byte+
            (dpb status +midi-message-data1-byte+ 
	         (dpb (or data1 0) +midi-message-data2-byte+
	              (dpb (or data2 0) +midi-message-data3-byte+ 0))))))

(defun system-message-p (message)
  (midi-system-message-p message))

(defun system-message-status (message)
  (ldb +midi-message-data1-byte+ message))

(defun system-message-data1 (message)
  (ldb +midi-message-data2-byte+ message))

(defun system-message-data2 (message)
  (ldb +midi-message-data3-byte+ message))

(define-message-setf system-message-status +midi-message-data1-byte+)
(define-message-setf system-message-data1  +midi-message-data2-byte+)
(define-message-setf system-message-data2  +midi-message-data3-byte+)

;;;
;;; :song-position
;;;

(defun make-song-position (lsb msb)
  (make-system-message +midi-song-position+ lsb msb))

(defun song-position-p (message)
  (= (ldb +midi-message-data1-byte+ message) +midi-song-position+))
  
(defun song-position-lsb (message)
  (ldb +midi-message-data2-byte+ message))

(defun song-position-msb (message)
  (ldb +midi-message-data3-byte+ message))

(define-message-setf song-position-lsb +midi-message-data2-byte+)
(define-message-setf song-position-msb +midi-message-data3-byte+)

;;;
;;; :song-select
;;;

(defun make-song-select (song)
  (make-system-message +midi-song-select+ song))

(defun song-select-p (message)
  (= (ldb +midi-message-data1-byte+ message) +midi-song-select+))

(defun song-select-song (message)
  (ldb +midi-message-data2-byte+ message))

(define-message-setf song-select-song +midi-message-data2-byte+)


;;;
;;; :tune-request
;;;

(defun make-tune-request ()
  (make-system-message +midi-tune-request+))

(defun tune-request-p (message)
  (= (ldb +midi-message-data1-byte+ message) +midi-tune-request+))

;;;
;;; :sysex-message
;;;

(defun make-sysex-message (&rest data-bytes)
  (let ((msg (dpb +midi-system-tag+ +midi-message-type-byte+
                  (dpb 1 +midi-message-size-byte+
                      (dpb +midi-sysex+ +midi-message-data1-byte+ 0))))
        (data (nconc (mapcar #'make-midi-data data-bytes)
                     (list (make-midi-data +midi-eox+)))))
    (values msg data)))

;(defun make-sysex-message (id &rest data-bytes)
;  (let ((msg (dpb +midi-system-tag+ +midi-message-type-byte+
;                  (dpb 2 +midi-message-size-byte+
;                      (dpb +midi-sysex+ +midi-message-data1-byte+ 
;                           (dpb id +midi-message-data2-byte+ 0)))))
;        (data (nconc (mapcar #'make-midi-data data-bytes)
;	             (list (make-midi-data +midi-eox+)))))
;    (values msg data)))
    
(defun sysex-message-p (message)
  (and (midi-system-message-p message)
       (= (ldb +midi-message-data1-byte+ message) +midi-sysex+)))

(defun sysex-message-id (message)
  (ldb +midi-message-data2-byte+ message))
  
(define-message-setf sysex-message-id +midi-message-data2-byte+)

;;;
;;; :meta-message
;;;

(defun make-meta-message (type &rest data-bytes)
  (let ((msg (dpb +midi-meta-tag+ +midi-message-type-byte+
                  (dpb 2 +midi-message-size-byte+
                      (dpb +midi-meta+ +midi-message-data1-byte+ 
                           (dpb type +midi-message-data2-byte+ 0)))))
        (data (mapcar #'make-midi-data data-bytes)))
    (values msg data)))

(defun meta-message-p (message)
  (midi-meta-message-p message))

(defun meta-message-type (message)
  (ldb +midi-message-data2-byte+ message))
  
(define-message-setf meta-message-type +midi-message-data2-byte+)

;;;
;;; :eot
;;;

(defun make-eot ()
  (make-meta-message +midi-eot+))

(defun eot-p (message)
  (= (ldb +midi-message-data2-byte+ message) +midi-eot+))

;;;
;;; :tempo-change
;;;

(defun make-tempo-change (tempo)
  (apply #'make-meta-message +midi-tempo-change+
          (loop for pos from 16 by 8 downto 0
	        collect (ldb (byte 8 pos) tempo))))

(defun tempo-change-p (message)
  (= (ldb +midi-message-data2-byte+ message) +midi-tempo-change+))
  
;;;
;;; :time-signature
;;;

(defun make-time-signature (numerator denominator &optional (clocks 24)
						 (32nds 8))
  (make-meta-message +midi-time-signature+ numerator 
  		     (floor (log denominator 2)) clocks 32nds))
		     
(defun time-signature-p (message)
  (= (ldb +midi-message-data2-byte+ message) +midi-time-signature+))

;;;
;;; :midi-data
;;;

(defun make-midi-data (data1 &optional data2 data3)
  (let ((size (if data3 3 (if data2 2 1))))
    (dpb +midi-data-tag+ +midi-message-type-byte+
         (dpb size +midi-message-size-byte+
              (dpb data1 +midi-message-data1-byte+      
                  (dpb (or data2 0) +midi-message-data2-byte+
                       (dpb (or data3 0) +midi-message-data3-byte+ 0)))))))

(defun midi-data-p (message)
  (midi-data-message-p message))
  
(defun midi-data-size(message)
  (ldb +midi-message-size-byte+ message))
  
(defun midi-data-data1 (message)
  (ldb +midi-message-data1-byte+ message))
 
(defun midi-data-data2 (message)
  (ldb +midi-message-data2-byte+ message))
 
(defun midi-data-data3 (message)
  (ldb +midi-message-data3-byte+ message))
 
(define-message-setf midi-data-size +midi-message-size-byte+)
(define-message-setf midi-data-data1 +midi-message-data1-byte+)
(define-message-setf midi-data-data2 +midi-message-data2-byte+)
(define-message-setf midi-data-data3 +midi-message-data3-byte+)

;;;
;;; make-midi-message provides a general interface to message construction
;;;

(defun make-midi-message (type &rest args)
  (ecase type
    ;; channel messages
    ((:note-on note-on)
     (apply #'make-note-on args))
    ((:note-off note-off)
     (apply #'make-note-off args))
    ((:key-pressure key-pressure)
     (apply #'make-key-pressure args))
    ((:control-change control-change)
     (apply #'make-control-change args))
    ((:program-change program-change)
     (apply #'make-program-change args))
    ((:channel-pressure channel-pressure)
     (apply #'make-channel-pressure args))
    ((:pitch-bend pitch-bend)
     (apply #'make-pitch-bend args))
    ((:channel-mode channel-mode)
     (apply #'make-channel-mode args))
    ((:channel-message channel-message)
     (apply #'make-channel-message args))
    ;; system messages
    ((:song-position song-position)
     (apply #'make-song-position args))
    ((:song-select song-select)
     (apply #'make-song-select args))
    ((:tune-request tune-request)
     (apply #'make-tune-request args))
    ((:system-message system-message)
     (apply #'make-system-message args))
    ;; system exclusive messages
    ((:sysex-message sysex-message)
     (apply #'make-sysex-message args))    
    ;; meta messages
    ((:meta-message meta-message)
     (apply #'make-meta-message args))
    ((:tempo-change tempo-change)
     (apply #'make-tempo-change args))
    ((:time-signature time-signature)
     (apply #'make-time-signature args))
    ((:eot eot)
     (apply #'make-eot args))
    ;; midi data messages
    ((:midi-data midi-data)
     (apply #'make-midi-data args))))

;;;
;;; midi message printing
;;;

(defun midi-print-message (message &optional time 
                                   &key (stream *standard-output*)
                                   (time-format "~%~D ") message-data)
  (when (and time time-format)
    (format stream time-format time))
  (cond ((midi-channel-message-p message)
         (let ((type (ldb +midi-type-byte+ message))
               (chan (ldb +midi-channel-byte+ message))
               (data1 (midi-message-data2 message))
               (data2 (midi-message-data3 message)))
           (cond ((= type +midi-note-on+)
                  (format stream "#<NoteOn: ~a, ~a, ~a>" 
                          chan data1 data2))
                 ((= type +midi-note-off+)
                  (format stream "#<NoteOff: ~a, ~a, ~a>" 
                          chan data1 data2))
                 ((= type +midi-key-pressure+)
                  (format stream "#<KeyPres: ~a, ~a, ~a>" 
                          chan data1 data2))
                 ((= type +midi-control-change+)
                  (format stream "#<CtrlChng: ~a, ~a, ~a>" 
                          chan data1 data2))
                 ((= type +midi-program-change+)
                  (format stream "#<ProgChng: ~a, ~a>" 
                          chan data1))
                 ((= type +midi-channel-pressure+)
                  (format stream "#<ChanPres: ~a, ~a>" 
                          chan data1))
                 ((= type +midi-pitch-bend+)
                  (format stream "#<PitchBend: ~a, ~a, ~a>" 
                          chan data1 data2))
                 (t
                  (format stream "#<ChanMsg: ~a, ~x, ~x>" 
                          chan data1 data2)))))
        ((midi-meta-message-p message)
         (let ((type (midi-message-data2 message)))
           (cond ((= type +midi-tempo-change+)
                  (let ((b1 (midi-message-data1 (or (pop message-data) 0)))
                        (b2 (midi-message-data1 (or (pop message-data) 0)))
                        (b3 (midi-message-data1 (or (pop message-data) 0))))
                     (format stream "#<TempoChng: usecs=~d>" 
                             (logior (ash b1 16) (ash b2 8) b3))))
                 ((= type +midi-time-signature+)
                  (let ((b1 (midi-message-data1 (or (pop message-data) 0)))
                        (b2 (midi-message-data1 (or (pop message-data) 0)))
                        (b3 (midi-message-data1 (or (pop message-data) 0)))
                        (b4 (midi-message-data1 (or (pop message-data) 0))))
                    (format stream "#<TimeSig: ~a/~a clocks=~a 32nds=~a>" 
                            b1 (expt 2 b2) b3 b4)))
                 ((= type +midi-eot+)
                  (format stream "#<Meta: End of Track>"))
                 (t
                  (format stream "#<Meta: type=~d>" type)))))
        ((midi-system-message-p message)
         (if (= (ldb +midi-message-data1-byte+ message) +midi-sysex+)
             (format stream "#<SysEx: length=~S>" (length message-data))
           (format stream "#<System: type=~x>" 
                   (midi-message-data1 message))))
        (t
         (format stream "<Midi?: ~s>" message)))
  message) 

					
(eval-when (load eval)
  (pushnew ':midi *features*))							
