;;;-*- Mode: Lisp; Syntax: COMMON-LISP; Package: COMMON-MUSIC -*-

(in-package :common-music)

;;;
;;; Low level interface to midi manager very loosely based on Joe Chung's
;;; Midi Manager Lisp package, with output message buffering help courtesy 
;;; of Dale Skrien and Jon Hallstrom at Colby. This file is used by midi.lisp.
;;; 
;;; Output message buffering currently exploits a rather complicated technique
;;; of dynamically allocating new buffers, which works roughly as follows:
;;;   o  Future messages are written immediately to the MidiManager for time
;;;      ordering and queueing via an invisible output port.
;;;   o  If a packet becomes current, the MidiManager writes it back to the 
;;;      invisible output's corresponding (connected) invisible input port
;;;      and calls a supplied InvisibleReadHook for further processing.
;;;   o  The InvisibleReadHook normally simply passes the packet on and writes 
;;;      it to the actual output port, but may also check for the packet type
;;;      to only pass note-off packets when the port dumps all pending packets
;;;      to it in response to a midiflush command. Flushing the output queue
;;;      this way ensures also that multiple notes with equal channel and key
;;;      hanging around get turned off properly.
;;;   o  Since the MidiManager imposes a limit of #xFFFF bytes on the size of 
;;;      its buffers, additional buffers - that is, invisible in/output-port
;;;      pairs - are allocated dynamically. Allocation of ports takes place
;;;      whenever the existing queues signal an overflow error. However, since
;;;      there is no other way to test for an empty buffer which could be
;;;      disposed of, a variable is associated with every buffer which holds
;;;      the time stamp of the most future message written to that buffer.
;;;      A periodic MidiWakeUp callback then allows an internal garbage 
;;;      collecting 'sniffer' routine to check at periodic times whether any
;;;      expired buffers should be disposed of (see mcl-midi.c). 
;;;      The first (main) pair of invisible ports, however, gets never removed 
;;;      and remains allocated all the time until ff-midi-close executed
;;;      successfully.
;;;   o  Only in the event of an out-of-memory error, when no additional buffer
;;;      can be allocated, the system gets blocked, i.e., it loops trying to
;;;      write the refused packet until it finally succeedes. This behaviour is
;;;      probably preferable to simply discarding the packet.  (This is not the
;;;      whole truth; see below: Limitations, for a more precise description.)
;;;
;;; Function naming conventions and function dispatch mechanism:
;;;   Since calling MidiManager traps inadvertently without knowledge about the 
;;;   current MidiManager environment's state may crash your machine at any 
;;;   time, a function dispatch mechanism is employed that automatically
;;;   swaps the contents of the parent function name's function slot according
;;;   to this state, thus keeping the parent function accessible all the time
;;;   and signaling errors, if appropriate.  Function names that are meant to
;;;   be accessed from outside this file typically start with a 'ff-midi-' 
;;;   prefix, whereas their lower-level siblings, if any, put a '%mm-' in front
;;;   of their name - as all other functions that access the MidiManager.
;;;   DON'T CALL ANY FUNCTIONS MARKED WITH A '%mm-'-PREFIX DIRECTLY, UNLESS YOU
;;;   KNOW EXACTLY WHAT YOU ARE DOING. ACCESS THEIR HIGHER-LEVEL COUNTERPARTS
;;;   INSTEAD.
;;;
;;;   The following table lists the 'ff-midi-'-function names along with the 
;;;   '%mm-'- functions they access, the minimum %mm-gestalt% value they
;;;   require, and whether the 'ff-midi-' function will be swapped
;;;   (%mm-gestalt% values in parens indicate  that the function depends on
;;;   others which require that level):
;;;
;;;    [ff-midi-open is the entry function and thus will never be swapped.]
;;;
;;;	ff-midi-find-port				  -
;;;	ff-midi-port-id					  -
;;;	ff-midi-open		%mm-sign-in		  -	 1
;;;				%mm-add-port			 2
;;;				%mm-get-midiidlist		 2
;;;				%mm-connect-data		 2
;;;	ff-midi-close		%mm-close		  2	 2	S	

;;;				%mm-unconnect-data		 3
;;;				%mm-remove-port			 3
;;;				%mm-sign-out			 2
;;;	ff-midi-start-timer	%mm-start-timer		  3	 3	S
;;;	ff-midi-stop-timer	%mm-stop-timer		  3	 3	S
;;;	ff-midi-get-time	%mm-get-time		  3	 3	S
;;;	ff-midi-set-time	%mm-set-time		  3	 3	S
;;;	ff-midi-read-messages	%mm-read-messages	  3	 3	S
;;;	ff-midi-write-message	%mm-write-message	  3	 3	S
;;;	ff-midi-all-notes-off	%mm-all-notes-off	  3	 3	S
;;;	ff-midi-flush-transmit	%mm-flush-transmit	  3	 3	S
;;;	ff-midi-flush-receive	%mm-flush-receive	  3	 3	S
;;;	ff-midi-hush		%mm-hush		  3	 3	S
;;;	ff-midi-set-quanta-size	  <dummy>
;;;	ff-midi-listen		  <dummy>
;;;	ff-midi-stop-listening	  <dummy>
;;;	
;;;   Other internal functions:
;;;				%mm-get-driver-and-serial-port	(2)
;;;				%mm-get-midiidlist [macro]	 2
;;;                             %mm-add-cm-ports		 2
;;;                             %mm-remove-cm-ports		 3
;;;
;;;   Internal functions you should _never_ call yourself:
;;;                             %mm-midi-open-aux
;;;				%mm-make-aux-buffer
;;;                             %mm-remove-aux-buffers
;;;				%mm-set-readhook
;;;				%mm-set-invreadhook-to-flush
;;;				%mm-read-messages-aux
;;;
;;; External functions this module makes use of:
;;;    cm:midi-open
;;;    cm:midi-close
;;;
;;; LIMITATIONS
;;;   o  The Midimanager imposes an - undocumented - limit on the number of
;;;      ports it allows a client to add. This maximum number of ports (16)
;;;      unfortunately also limits the number of auxiliary buffers to allocate,
;;;      and hence, the maximum number of midi messages to queue up for output.
;;;      In particular, since three ports are normal 'visible' ports (time,
;;;      input, and output), only six invisible output/input port pairs with a
;;;      maximum buffer size of #xFFFF bytes or roughly 6540 messages may be
;;;      open at the same time.  Thus, although the functionality of this code
;;;      provides for an unlimited (as long as enough RAM is available) amount
;;;      of messages to be queued for output, this MidiManager feature
;;;      (&!$*@^%!!) actually doesn't allow for more than #x5fffa or 393210
;;;      bytes or 40000 messages to be queued.
;;;   o  The c sniffer routine cannot remove ports at the time it detects that
;;;      they are idling around, since it can not call midiRemovePort at
;;;      interrupt level.  I may dig in mcl's event dispatch mechanism at some
;;;      later time to send it a custom event that calls remove_ports at
;;;      non-interrupt time, but this version simply relies upon the more or
;;;      less steadyly called ff-midi functions to do the additional work of
;;;      calling remove_port.  Therefore, it may be possible that one or more
;;;      empty buffers being removed but not yet actually deallocated (which is
;;;      what remove_ports actually does) until the next midi function is
;;;      called or midi is closed.

;;;
;;; MIDI USER VARIABLES
;;;
;;;   *midi-preferences*       <default value: '((:apple :oms :mtp) (:a :b))>
;;;      A list of the known midi drivers and ports, sorted in descending
;;;      order according to their relative preference.
;;;      The variable's primary use is to provide midi-open with the means
;;;      to discriminate between multiple possibilities, whenever it it is
;;;      called withoput parameters (as done by the midi menu's open item).
;;;
;;;   *midi-multitimbral-channels*                         <default value: 1>
;;;      The number of available multitimbral channels.
;;;      Used by the 'All Notes Off' and 'Test Midi' midi menu items and by
;;;      the function ff-midi-all-notes-off. Midi data are only output within
;;;      the range of channels from 0 to (*midi-multitimbral-channels* -1).
;;;
;;;   *midi-flush-timeout*	                        <default value: 3000>
;;;      The time in milliseconds, for which the queueing mechanism stays in
;;;      flushing mode, therefore ignoring newly scheduled events.
;;;
;;;      Since flushing is achieved by dumping all pending note-off messages, 
;;;      it is possible for _huge_ amounts of buffered output to leave messages
;;;      unprocessed. This may lead to hanging notes, if _very_ long notes 
;;;      that are already turned on happen to have their note-off message 
;;;      among this unprocessed chunk of midi data. Therefore, you may want to
;;;      increase or decrease the value of *midi-flush-timeout*, depending on 
;;;      the overall musical structure you tend to produce.
;;;      Note, too, that this parameter is semi-global only. That is, any 
;;;      changes to it come into effect only after the next time you open a
;;;      midi port.

;;;
;;; KNOWN PROBLEMS:
;;;   o  There is a weird bug lurking around which appears to show up only
;;;      after you left your machine alone for a long period of time. Then,
;;;      if you input *any* midi message, the Mac seems to be crashed.
;;;      Actually, it is only frozen, that is executing an infinite loop so
;;;      that if you're running MacsBug, you may follow these steps to recover:
;;;         1) hit the interrupt switch to enter MacsBug (the "Programmer's
;;;            Key" extension won't work)
;;;         2) type the letter 'S' and repeatedly <return>. This puts MacsBug
;;;            into step mode. - Step through the six-instruction-loop until
;;;            you've reached the BRA instruction marked by an asterisk below:
;;;
;;;            No procedure name
;;;                      000E5B3C   ...        ...
;;;                      000E5B3E   MOVEA.L    D0,A2
;;;                      000E5B40   CMP.L      $0006(A2),D1
;;;                      000E5B44   BLT.S      *+$0008          ; 000E5B4C
;;;                      000E5B46   MOVE.L     A2,D2
;;;                      000E5B48   MOVE.L     (A2),D0
;;;                      000E5B4A  *BNE.S      *-$000C          ; 000E5B3E
;;;                      000E5B4C   TST.L      D2 
;;;                      000E5B4E   ...        ...
;;;
;;;         3) Now type exactly (without the double quotes) "br pc ';g pc+2'"
;;;            and <return>. This tells MacsBug to "always break at pc 
;;;            (= Program Counter Register) and execute the command 'Go 
;;;            (= resume Execution) at Address stored in pc plus two'", which
;;;            denotes the TST.L instruction above.
;;;         4) type 'G' <return> or cmd-G to leave MacsBug.
;;;      MacsBug will swap the screen some few times after which your machine
;;;      will be free again.
;;;      You should now save your work and quit and restart CommonMusic.

;;;
;;;      Bug reports, suggestions, etc. concerning midi (mcl-midi.lisp and 
;;;      mcl-midi.c) are welcome. Please address electronic mail to 
;;;      tkunze@mvax.kgw.tu-berlin.de.
;;;

;;;
;;; Changes:
;;;
;;;   11-05-93 tk  - released
;;;   16-05-93 tk  - new connection mechanism to cooperate with OMS under its 
;;;                  MidiManager mode; removed the %mm-unconnect-data function.
;;;                - added an icon to be displayed in PatchBay.
;;;   17-05-93 tk  - added the missing MIDINoConErr error to with-midi-checked.
;;;   18-05-93 tk  - new variable *midi-multitimbral-channels*, defaults to 1
;;;   30-05-93 tk  - eval-enqueued midi-open/close menu item's action
;;;   07-06-93 tk  - improved error and cleanup handling while opening midi
;;;                - new functions %mm-add-cm-ports, %mm-remove-cm-ports, 
;;;                  %update-midi-menu, and %mm-midi-open-aux
;;;                - %mm-get-driver-and-serial-ports now allows to recover
;;;                  from its error detection
;;;                - added a missing call to ff-dispatch to %mm-sign-in
;;;                - %mm-close now maps over a varying %mm-close-stack%
;;;   08-06-93 tk  - fixed the bug in the midi-open/close menu item's action
;;;                  function that caused infinite looping when 
;;;                  %mm-get-driver-and-serial-ports encountered an error
;;;                - added a second return value to the midi-open menu 
;;;                  command that reports the newly opened port
;;;                - added a new +canceled+ result code to ff-midi-open
;;;   15-06-93 tk  - implemented new midi-hush design and hopefully fixed the
;;;                  MC68040 instruction cache problems with midi-hush
;;;                - optimized %mm-all-notes-off by using the global
;;;                  *midi-multitimbral-channels*
;;;                - made *midi-flush-timeout* a (semi-) global parameter
;;;   10-12-93 tk  - added support for the Midi Time Piece II Interface. Works
;;;                  also with standard interfaces through the mtp driver
;;;   19-02-94 tk  - ccl::toplevel-print midi open/close menu item's action 


(export '(*midi-preferences* *midi-multitimbral-channels* 
          *midi-flush-timeout*))

(eval-when (:compile-toplevel :load-toplevel :execute) (ccl:require :ff))

;;;
;;; These two constants are missing in the MIDI.lisp interface file.
;;; Thanks to Dale Skrien, who managed to get them from Apple.

(defconstant $midiInvisiblePort #x8000)
(defconstant $midiPortTypeMask #x0007)

;;;
;;; Make sure midi is closed before exiting and insert a Midi menu before the 
;;; Windows menu. Balloon help is also available.

(defvar %midi-menu% nil)
(defvar %midi-open-p% nil)

;;  used by the 'Test Midi' menu item and by %mm-all-notes-off
(defparameter *midi-multitimbral-channels* 1
  "The number of available multitimbral channels.")


(defconstant +open-name+ "Open")
(defconstant +close-name+ "Close")
(defconstant +hush-name+ "Hush")
(defconstant +flush-input-name+ "Flush Input")
(defconstant +flush-output-name+ "Flush Output")
(defconstant +notes-off-name+ "All Notes Off")
(defconstant +test-name+ "Test Midi")

(eval-when (:load-toplevel :execute)
  ;; make sure we close midi when exiting 
  (let ((quit (ccl:find-menu-item ccl:*file-menu* "Quit")))
    (defmethod ccl:menu-item-action :before ((i (eql quit))) 
      (when %midi-open-p%  (ff-midi-close))))
  ;; make a midi menu
  (unless (ccl:find-menu "Midi")
    (ccl:menu-deinstall ccl:*windows-menu*)
    (ccl:menu-install
     (setf %midi-menu%
           (make-instance 'ccl:menu
                          :menu-title "Midi"
                          :help-spec "Some useful midi commands."
                          :menu-items 
             (list (make-instance 'ccl:menu-item 
                     :menu-item-title (if %midi-open-p% 
                                          +close-name+ +open-name+)
                     :menu-item-action 
                     #'(lambda () 
                         (ccl:eval-enqueue 
                          `(when ccl:*top-listener*
                             (ccl::toplevel-print
                              (multiple-value-list 
                               (if ,%midi-open-p% 
                                 (midi-close) 
                                 (midi-open)))))))
                     :command-key #\m
                     :disabled nil
                     :help-spec "Opens or closes a midi port by calling midi-open or midi-close, depending on whether a port is already open.")
                   (make-instance 'ccl:menu-item :menu-item-title "-")
                   (make-instance 'ccl:menu-item 
                     :menu-item-title +hush-name+
                     :disabled (if %midi-open-p%  nil t) 
                     :command-key #\;
                     :menu-item-action #'(lambda () (ff-midi-hush))
                     :help-spec "Flushes both input and output queues and turns off hanging notes by calling midi-hush")
                   (make-instance 'ccl:menu-item 
                     :menu-item-title +flush-input-name+
                     :disabled (if %midi-open-p%  nil t) 
                     :command-key #\[
                     :menu-item-action #'(lambda () (ff-midi-flush-receive))
                     :help-spec "Flushes the Midi Input queue by calling 
ff-midi-flush-receive. You may want to flush received messages before calling midi-read-messages.")
                   (make-instance 'ccl:menu-item 
                     :menu-item-title +flush-output-name+
                     :disabled (if %midi-open-p%  nil t) 
                     :command-key #\]
                     :menu-item-action #'(lambda () (ff-midi-flush-transmit))
                     :help-spec "Flushes all pending messages from the Midi Output queue by calling ff-midi-flush-transmit.")
                   (make-instance 'ccl:menu-item 
                     :menu-item-title +notes-off-name+
                     :disabled (if %midi-open-p%  nil t) 
                     :command-key #\'
                     :menu-item-action #'(lambda () (ff-midi-all-notes-off))
                     :help-spec "Transmits a note-off message for every key and 
every channel by calling (ff-midi-all-notes-off)")
                   (make-instance 'ccl:menu-item :menu-item-title "-")
                   (make-instance 'ccl:menu-item 
                     :menu-item-title +test-name+
                     :disabled (if %midi-open-p%  nil t) 
                     :command-key #\t
                     :menu-item-action 
                     #'(lambda () 
                         (loop for x below 12
                               with time = (midi-get-time)
                               for note = (+ x (* 12 (+ 3 (random 5))))
                               for chan = (random *midi-multitimbral-channels*)
                               do 
                               (ff-midi-write-message 
                                (make-note-on chan note (+ 32 (random 64))) 
                                time)
                               (ff-midi-write-message 
                                (make-note-off chan note (random 128)) 
                                (+ time (* (expt (+ 2 (random 6)) 2) 30)))
                               (incf time (* (expt (random 5) 2) 30))))
                     :help-spec "Sends a small number of random notes and chords. Set *midi-multitimbral-channels* to a value greater than 1 if you have a multitimbral midi device.")))))
    (ccl:menu-install ccl:*windows-menu*)))

(defun %update-midi-menu ()
  (ccl:set-menu-item-title
   (or (ccl:find-menu-item %midi-menu% +open-name+)
       (ccl:find-menu-item %midi-menu% +close-name+))
   (if %midi-open-p% +close-name+ +open-name+))
  (map 'nil (if %midi-open-p%
              #'ccl:menu-item-enable
              #'ccl:menu-item-disable)
       (map 'list #'(lambda (name) (ccl:find-menu-item %midi-menu% name))
            `(,+hush-name+ ,+flush-input-name+ ,+flush-output-name+
              ,+notes-off-name+ ,+test-name+))))

;; No libraries are needed for this object file.
(defvar %ff-env% (ccl:ff-load (concatenate 
                                 'string 
                                 (namestring *common-music-directory*)
                                 "midi:mcl-midi.c.o") 
                              :ffenv-name 'mcl-midi :replace t))

(defconstant +ff-midi-success+ 0
  "Foreign function success value.")

(defconstant +canceled+
  ;; Result code of ff-midi-open, if canceled.
  ;; should become :canceled once midi-open test for return values with eql
   -1)

;; Error result flags for %mm-add-port.
(defconstant +memFullErr+ 'memFullErr)          
(defconstant +tooManyPortsErr+ 'tooManyPortsErr)

(defconstant +cm-midi+ :|CM  |
  "Common Music's clientid.")

(defconstant +input-buffer-size+ (* 1024 10))   ; 1024 input messages
(defconstant +main-buffer-size+   #xFFFF)       ; 6550 future output messages
(defconstant +aux-buffer-size+    #xFFFF)       ; 6550 more future messages

;;;
;;; reference numbers for open ports

(defvar $time-port$ nil)
(defvar $input-port$ nil)
(defvar $output-port$ nil)
(defvar $invisible-input-port$ nil)     ; inv. input port from the main queue
(defvar $invisible-output-port$ nil)    ; inv. output port to the main queue

(defvar $serial-port$ nil)              ; 0 or 1
(defvar $driver-id$ nil)                ; currently, :|amdr|, :|OmMd|, or 
                                        ;   :|MTPd|. Set only once, in
                                        ;   %mm-get-driver-and-serial-port


;;;
;;; A list of preferred midi drivers (currently :apple, :mtp, and :oms) and 
;;; ports (:a or :b), in descending order. 
;;; Other drivers may easily be added in the future.

(defparameter *midi-preferences* '((:apple :mtp :oms) (:a :b))
  "A list of all known midi drivers and ports, sorted in descending ~
   order according to their relative preference.")

;;;
;;; Information about a known driver is held in a 'driver' structure. The list
;;; of all known driver structures - *drivers* - is used in 
;;; get-driver-and-serial-port to parse *midi-preferences*

(defstruct (driver (:type list))
  name filename id can-connect a b)

;; order of port id's is: '(<time-port> <output-port> <input-port> ...)
;; but see the comment below about what input and output means here.

(defvar *drivers* (list (make-driver :name ':apple
                                     :filename :|Apple MIDI Driver|
                                     :id :|amdr|
                                     :can-connect t
                                     :a '(:|ATim| :|Ain | :|Aout|)
                                     :b '(:|BTim| :|Bin | :|Bout|))
                        (make-driver :name ':oms
                                     :filename :|OMS MIDI Manager Driver|
                                     :id :|OmMd|
                                     :can-connect nil
                                     :a ()      ; oms isn't very cooperative:
                                     :b ())     ; you have to connect manually.
                        (make-driver :name ':mtp
                                     :filename :|MTP Driver 2.0|
                                     :id :|MTPd|
                                     :can-connect t
                                     :a '(:|timA| :MTPA :|Mod |
                                          :|Mod!| :|Mod"| :|Mod#| :|Mod$| 
                                          :|Mod%| :|Mod&| :|Mod'|)
                                     :b '(:|timB| :MTPB :|Prn | 
                                          :|Prn!| :|Prn"| :|Prn#| :|Prn$| 
                                          :|Prn%| :|Prn&| :|Prn'|))))   

(defun find-driver-by-name (ref)
  (or (find ref *drivers* :key #'first)
      (error "Can't find driver named ~S." ref)))

(defun find-driver-by-id (ref)
  (or (find ref *drivers* :key #'third)
      (error "Can't find driver with id ~S." ref)))

(defun find-driver-ports (port &key name id)
  "Returns a driver's :a or :b slot, depending on the first argument."
  (unless (numberp port) (error "~a is not a number" port))
  (elt (cond (name (find-driver-by-name name))
             (id (find-driver-by-id id))
             (t (error "need a client's name or ostype" )))
       (+ port 4)))       ; port 0 or :a is element 5 (elt x 4) currently

;; The name confusion between portids and their access functions is not a bug:
;; Forget any knowledge about apps and drivers and think of it simply as
;; icons appearing in Patch-Bay; a triangle pointing inwards is by definition
;; an output port, no matter whether it represents incoming MIDI data received
;; by a driver or MIDI data output by an application (et vice versa).
;; Thus, in-id accesses rather :|Aout| or :|Bout| ids of the apple midi driver
;; than :|Ain | or :|Bin | as one might think. 

;(defun time-id (x) (first x))			; unused
(defun out-id (x)  (second x))
(defun in-id (x)   (third x))


;;;
;;; Serial port specification and handling
;;;
;;; Normally, we just look at the midi driver (if its can-connect slot is t) to
;;; see which port is enabled and take it. If two or more serial ports are open
;;; (for those who have a QuadraLink NuBus card with lots of interfaces
;;; connected to it :-) ), the preferred according to *midi-preferences* is
;;; choosen unless a port is explicitly given as a :port keyword parameter to
;;; midi-open.

(defvar *port-names* 
  ;; The order of names in these list should be:
  ;; number letter :letter name :name.
  '((nil)                         ; let Common Music decide which one is open.
    (0 a :a modem :modem) 
    (1 b :b printer :printer)))

(defun verbose-find-port (int &key (long t) (key t))
  (elt (elt *port-names* (+ 1 int)) (if long (if key 4 3) (if key 2 1))))

(defun ff-midi-find-port (name)
  ;; test for a 'nil' port specification [(midi-open) w/o arguments] has to
  ;; return t!
  (let ((flg (find name *port-names* :test #'member)))
    (if (equal '(nil) flg)
    t
    (when flg name))))

(defun ff-midi-port-id (port)
  (case (position port *port-names* :test #'member)
    (0 nil)                             ; 'nil' port specification
    (1 0)                               ; modem port is 0
    (2 1)                               ; printer port is 1
    (t nil)))                           ; return nil anyway


;;;
;;; Common Music's MidiManager ports.
;;;
;;; A "port" here is just a holder for random port information.
;;; The global list *ports* holds information for the permanent ports.

(defstruct (port (:type list))
  name 
  type 
  id 
  (size 1024) 
  (format  #$midiFormatBeats) 
  (offset #$midiGetCurrent) 
  refnum)                        

(defvar *ports* (list (make-port :name ':|Time|
                                 :type #$midiPortTypeTime
                                 :id ':|1Tim|
                                 :format #$midiFormatMSec)
                      (make-port :name ':|Input|
                                 :type #$midiPortTypeInput
                                 :id ':|2In |
                           ;; don't let midi-read-hook get called at interrupt
                           ;; level!!
                                 :offset #$midiGetNothing
                                 :size +input-buffer-size+)
                      (make-port :name ':|Output|
                                 :type #$midiPortTypeOutput
                                 :id ':|3Out|)
                      (make-port :name ':invisible-input
                                 :type (logior #$midiPortTypeInput 
                                               $midiInvisiblePort)
                                 :id ':|4Xin|
                                 :size +main-buffer-size+)
                      (make-port :name ':invisible-output
                                 :type (logior #$midiPortTypeOutput 
                                               $midiInvisiblePort)
                                 :id ':|5Xou|)))

(defun find-port-by-name (ref)
  (or (find ref *ports* :key #'first)
      (error "Can't find port named ~S." ref)))

(defun find-port-by-id (ref)
  (or (find ref *ports* :key #'third)
      (error "Can't find port with id ~S." ref)))

(defun find-port-by-refnum (ref)
  (or (find ref *ports* :key #'seventh)         ; same key in %mm-remove-port!
      (error "Can't find port with refnum ~S." ref)))


;;;
;;; MidiManager utilities

(defmacro with-midi-checked (form)
  (let ((var (gensym)))
    `(let ((,var ,form))
       (cond ((or (null ,var) (= ,var +ff-midi-success+)) 
              +ff-midi-success+)
             ((member ,var '(#$MIDIVConnectErr #$MIDIVConnectMade 
                             #$MIDIVConnectRmvd))
              +ff-midi-success+)
             ((= ,var #$MIDINoClientErr)
              (error "MIDI: no client with that ID found."))
             ((= ,var #$MIDINoPortErr)
              (error "MIDI: no port with that ID found."))
             ((= ,var #$MIDITooManyPortsErr)
              (error "MIDI: too many ports already installed in the system."))
             ((= ,var #$MIDITooManyConsErr)
              (error "MIDI: too many connections made."))
             ((= ,var #$MIDINoConErr)
              (error "MIDI: connection does not exist."))
             ((= ,var #$MIDIWriteErr)
              (error "MIDI: MIDIWritePacket couldn't write to all ~
                      connected ports."))
             ((= ,var #$MIDINameLenErr)
              (error "MIDI: name supplied is longer than 31 characters."))
             ((= ,var #$MIDIDupIDErr)
              (error "MIDI: duplicate client ID."))
             ((= ,var #$MIDIInvalidCmdErr)
              (error "MIDI: command not supported for port type."))
             ((= ,var #$memFullErr)
              (error "MidiManager: Not enough room in the mac heap zone."))
             (t
              (error "MIDI: failed with ~s" ,var))))))

(defmacro %mm-get-midiidlist (midimanager-call)
  `(let* ((hdl ,midimanager-call) list)
     (if (eq hdl nil) (error "There is not enough room to get client or~%~
                              port information")
         (setf list (loop for i from 0 
                          repeat (ccl:href hdl MidiIDList.numIDs) 
                          collect (ccl:href hdl (MidiIDList.list i)))))
     (ccl:require-trap #_disposhandle hdl)
     list))

(defun %mm-get-driver-and-serial-port (port)        ; is nil, 0, or 1
  (let (driver-ostypes port-ostypes)
    ;; get the installed drivers
    (setf driver-ostypes (%mm-get-midiidlist (#_midigetclients)))
    ;; try to satisfy the preferences in getting a known driver with open 
    ;; ports: first loop through the preferred drivers
    (loop for drvrkey in (first *midi-preferences*)
          for drvrid = (driver-id (find-driver-by-name drvrkey))
          for drvr = (first (member drvrid driver-ostypes))
          thereis 
          (when drvr                
            ;; if a known driver is available, check its can-connect field.
            (case (driver-can-connect (find-driver-by-id drvr))
              (nil (warn "Selecting a driver which does not support ~
                          automatic connecting. You will have to make~
                          your connections manually using PatchBay")
                   (setf $serial-port$          ; take the preferred one...
                         (position (first (second *midi-preferences*)) 
                                   *port-names* :test #'member))
                   (setf $driver-id$ drvr))                  ; we are done
              ;; else get an open port or return nil if none can be found.
              (t (setf port-ostypes (%mm-get-midiidlist (#_midigetports drvr)))
                 (if port-ostypes
                   (when (or (and (find port '(0 1)) ; if a port was passed
                                  (every #'(lambda (x) (find x port-ostypes)) 
                                         (find-driver-ports port :id drvr))
                                  (setf $serial-port$ port))
                             (loop for portkey in (second *midi-preferences*)
                                   for portnum = (- (position portkey 
                                                              *port-names* 
                                                              :test #'member)
                                                    1)
                                   for portid = (find-driver-ports portnum 
                                                                   :id drvr)    

                                   thereis (when (every 
                                                  #'(lambda (id) 
                                                      (find id port-ostypes)) 
                                                  portid)  
                                             (setf $serial-port$ portnum))))
                     (setf $driver-id$ drvr))
                   (setf drvr nil))))))     ; reset drvr to nil for next test
    (unless $driver-id$
      (warn "MIDI: no known midi driver with open ports available.~@
             Check the cm:*midi-preferences* parameter and/or PatchBay to~@
             see whether your driver is included and has any ports enabled.~@
             ~%Type y(es) if you could fix the problem, or n(o) to cancel.~%")
      (if (y-or-n-p "~%Retry opening/connecting to a midi driver?")
        (%mm-get-driver-and-serial-port port)
        (throw :midi-open-exit :cancel-midi-open)))))

;;;
;;; Add and remove ports
;;;
;;; %mm-add-port takes a port structure and returns its refnum or the constant
;;; +memFullErr+ or +tooManyPortsErr+ if the port could not be allocated.
;;; %mm-remove-port expects the refnum of the port that is to be removed and 
;;; returns NIL. 

(defun %mm-add-port (port)
  (ccl:rlet ((clkinfo :MIDIClkInfo 
                      :sync 0 
                      :curtime 0 
                      :format (port-format port))
             (portparams :MIDIPortParams 
                         :portid (port-id port)
                         :porttype (port-type port)
                         :timebase (if (eq (port-name port) :|Time|)
                                     0
                                     (let ((time (find-port-by-name :|Time|)))
                                       (or (port-refnum time)
                                           (error "Time port: no refnum"))))
                         :offsettime (port-offset port)
                         :readhook (ccl:%null-ptr)
                         :refcon 0
                         :initclock clkinfo
                         :name (string (port-name port)))
             (refnum :signed-integer))
    ;; The error handling here depends on the calling function: %mm-open 
    ;; signals an error, while %mm-make-aux-buffers continues w/o allocating
    ;; new ports.
    (let ((flg (#_midiaddport +cm-midi+ (port-size port) refnum portparams)))
      (case flg
        ((0 -255)                         ; #$noErr, #$MIDIvConnectMade
         (let ((refn (ccl:%get-signed-word refnum)))
           (setf (port-refnum port) refn)
           ;; Store the real output port's refnum in the refCon field, if 
           ;; it's an invisible input port, so that our invisible ReadHook can
           ;; simply pass it its packets.
           (when (eq (port-type port) 
                     (logior #$midiPortTypeInput $midiInvisiblePort))
             (with-midi-checked
               (#_midisetrefcon refn $output-port$)))
           refn))                       ; return refnum of new port
        ;; don't return numerical values for error codes, since it could be 
        ;; misunderstood and taken as a refNum! 
        (-108                           ; #$memFullErr
         +memFullErr+)
        ;; Now here comes undocumented feature of the MidiManager :-(
        ;; MidiManager can't take more than 16 ports.
        (-252                           ; #$MIDITooManyPortsErr!!
         +tooManyPortsErr+)
        (t (warn "#_midiaddport failed with result code ~a." flg)
           (ccl:%get-signed-word refnum))))))

(defun %mm-remove-port (refnum)
  ;; This functon is only called for the removal of one of the ports held
  ;; in *ports*.
  (if (numberp refnum)
    (#_midiremoveport refnum) 
    (error "can't remove a port with refNum ~s" refnum))
  ;; If the removed port was one of the main ports, clear its refNum.
  (when (find refnum *ports* :key #'seventh)
    (setf (port-refnum (find-port-by-refnum refnum)) nil)))

(defun %mm-add-cm-ports ()
  ;; Add ports only if neccessary, so ask the MidiManager first what our ports
  ;; are -- but don't ask why they are still around.
  (let ((port-ostypes (%mm-get-midiidlist (#_midigetports +cm-midi+))))
    (loop for port in *ports*
          for ref in '($time-port$ $input-port$ $output-port$ 
                       $invisible-input-port$ $invisible-output-port$)
          ;; NIL or an OSType, if the port is already present.
          for id = (find (port-id port) port-ostypes)
          with err = 0
          do 
          (unless id 
            ;; (symbol-value ref) is the refNum of the port.
            (setf (symbol-value ref) (setf err (%mm-add-port port)))
            ;; Ignore noErr and MIDIvConnectionMade result codes.
            (when (eq err +memFullErr+)
              (error "MIDI: There is not enough memory to create a port!")))))
  (%ff-dispatch +mmCMSetUp+))

(defun %mm-remove-cm-ports ()
  ;; Loop through the main ports and pass their refNum to %mm-remove-port, 
  ;; if any: this takes care not to remove ports that are already removed, 
  ;; since calling #_midiremoveport with a port that has already been 
  ;; removed can cause a bus error.
  (loop for ref in '($invisible-output-port$ $invisible-input-port$ 
                     $output-port$ $input-port$ $time-port$)
        ;; (symbol-value ref) is the refNum of the port.
        do (when (symbol-value ref) 
             (%mm-remove-port (symbol-value ref))
             ;; clear the refNum
             (setf (symbol-value ref) nil)))
  (%ff-dispatch +mmCMSignedIn+))


;;;
;;; Auxiliary buffer handling
;;;
;;; %mm-make-aux-buffers is the allocation routine. Invalidation takes place
;;; at interrupt level by myTimeProc, whereas remove_ports does the actual 
;;; deallocation work at non_interrupt time (see mcl-midi.c).

(defconstant +callback-period+ 
  ;; A time interval for the periodic MidiWakeUp callback. As long as Common
  ;; Music keeps signed in, myTimeProc checks at these time intervals (in
  ;; milliseconds), whether it finds empty buffers to invalidate.
  5000)

(defconstant +buffer-idle-timeout+ 
  ;; A threshold that determines after how long an idle time a buffer will be 
  ;; considered obsolete and thus will be removed from th elist of valid 
  ;; invisible outports (invalidated).
  (/ +callback-period+ 2))

(defvar %next-buffer-identifier% 
  ;; Simply counts up from 0. Only the low three bytes are needed.
  (the fixnum 0))

(defvar %myports-ptr%
  ;; Holds the pointer to the nonrelocatable block in the mac heap that is 
  ;; occupied by the global inv_ports structure myports points to and is
  ;; synonymously used as a pointer to the Remove_Not flag - the word located
  ;; at this address. Set up by ff-midi-open and disposed of by %mm-close.
  nil)

(defvar %oldports-ptr%
  ;; A pointer to the nonrelocatable block in the mac heap that is occupied by 
  ;; the global rem_ports structure oldports points to. This pointer is also a
  ;; pointer to the first field of *oldports, the word-sized nextfree value, 
  ;; which is nonzero if there are any port_pairs listed in *oldports. Set up
  ;; and disposed of together with %myports-ptr%.
  nil)

(defun make-port-ids (&key (increment nil))
  ;; Returns OSTypes for an pair of inports and an outports with #\i and #\o
  ;; in their most significant byte, respectively, and the low three bytes of 
  ;; %next-buffer-identifier% in their low three bytes, which rends them quite 
  ;; unreadable.
  (let ((intype #x69000000)             ; #x69 = ASCII for #\i
        (outtype #x6f000000)            ; #x6f = ASCII for #\o
        (num (logand %next-buffer-identifier% #xffffff)))
    (when increment (incf %next-buffer-identifier% 1))
    (values (logior intype num) (logior outtype num))))

(defun make-port-names ()
  ;; A similar name constructing function. This one gets called last, so it
  ;; increments %next-buffer-identifier%.
  (let ((str (princ-to-string (logand %next-buffer-identifier% #xffffff))))
    (incf %next-buffer-identifier% 1)
  (values (concatenate 'string '"i " str) (concatenate 'string '"o " str))))

(defvar %inv-outports% 
  ;; A list of all invisible outport refnums, which is maintained as a
  ;; precedence list with the current outport as its first element. If
  ;; %mm-write-message detects an overflow, it goes through this list while
  ;; trying to write to one of its outport refnums. A new buffer is allocated
  ;; only, if all outports signaled an overflow.  Actually, this list is a list
  ;; of macptrs, which get allocated by %mm-midi-open.  Since there are more
  ;; macptrs than valid inv-outports, a c-string like zero value of
  ;; (ccl:%get-word <macptr>) signals the end of the list (see
  ;; %mm-write-message).
  nil)

(ccl:deffcfun (%cycle-inv-outports "cycle_port_pairs")
  ;; Called only from %mm-write-message. Takes an zero-based index and reorders
  ;; the list of outport-inport pairs to start with the specified element, as
  ;; if it were a cycle (a rotation on the array's elements is performed).
  ((fixnum :long))
  :void)

(ccl:deffcfun (%add-to-inv-outports "add_port_pair")
  ;; Prepends a new outport-inport refNum pair to the inv_ports structure, 
  ;; thus implicitly prepending the new outport's refnum to %inv-outports%.
  ;; Since this is called only if all other buffers signaled overflow, no
  ;; reordering of the outport precedence list is required.
  ((fixnum :long) (fixnum :long))
  :void)

(ccl:deffcfun (%mm-set-readhook "setReadHook")
  ;; Takes the refNum of an invisible outport and sets it's readHook to 
  ;; myInvisibleReadHook.
  ((fixnum :long))
  :void)

(defun %mm-make-aux-buffer ()
  ;; Allocate a fresh buffer if possible and do all necessary housekeeping...
  ;; This function is called only from %mm-write-message.
  ;; Set up a pair of temporary port structures.
  (let ((new-inport (make-port :type (logior #$midiPortTypeInput 
                                             $midiInvisiblePort)
                               :size +aux-buffer-size+))
        (new-outport (make-port :type (logior #$midiPortTypeOutput 
                                              $midiInvisiblePort))))
    ;; Fill in the id slot
    (multiple-value-bind (in-id out-id) (make-port-ids)
      (setf (port-id new-inport) in-id
            (port-id new-outport) out-id))
    ;; Fill in the name slot
    (multiple-value-bind (in-name out-name) (make-port-names)
      (setf (port-name new-inport) in-name
            (port-name new-outport) out-name))
    ;; We only use refNums beyond this point.
    ;; Return +memFullErr+ to %mm-write-message, no refnum of new outport, if
    ;; failure.  [new-in/out-port come with their refnum slot already set by
    ;; %mm-add-port.]
    (let ((flg (%mm-add-port new-inport)))
      (if (or (eq flg +memFullErr+) (eq flg +tooManyPortsErr+))
        ;; No matter whether memory or MidiManager is full. Just 
        ;; return +memFullErr+.
        (return-from %mm-make-aux-buffer +memFullErr+)
        ;; else, if the second port does not yield a 'normal' result 
        ;; code (in which case a refNum would have been returned):
        (unless (numberp (setf flg (%mm-add-port new-outport)))
          ;; it's been a +memFullErr+ or +tooManyPortsErr+ now...
          (when (eq flg +memFullErr+)
            ;; undo allocation of new inport before returning.
            (%mm-remove-port (port-refnum new-inport)))
          (return-from %mm-make-aux-buffer +memFullErr+))))
    ;; No out-of-memory or too-many-ports error occured while trying to add
    ;; the new ports, so prepend the new pair to the c side list held in the
    ;; inv_ports structure's port_pairs array.
    (%add-to-inv-outports (port-refnum new-outport) (port-refnum new-inport))
    ;; Connect them new ports to each other
    (with-midi-checked 
      (#_midiconnectdata 
       +cm-midi+ (port-id new-outport) 
       +cm-midi+ (port-id new-inport)))
    ;; Set the new inport's ReadHook to our invisibleReadHook
    (%mm-set-readhook (port-refnum new-inport))
    ;; All done. Writing may continue, accessing the newly set up buffer
    ;; through:
    (port-refnum new-outport)))

(ccl:deffcfun (%mm-remove-aux-buffers "remove_ports")
  ;; All work is done on the c side to deallocate auxiliary buffers.
  ()
  :void)

    
;;;
;;; ff-midi function swapper
;;;
;;; The dispatch mechanism relies on the global semaphore %mm-gestalt%, which
;;; encodes security-relevant information about the current MidiManager
;;; environment and thus stipulates what may be done in midi at any given time.

(defconstant +mmUnavailable+  -1)       ; MM did not load at startup
(defconstant +mmUnknown+       0)       ; the initial value
(defconstant +mmAvailable+     1)       ; MM present
(defconstant +mmCMSignedIn+    2)       ; CM could successfully sign in
(defconstant +mmCMSetUp+       3)       ; CM added ports - All up and running

(defvar %mm-gestalt% +mmUnknown+)

(defun %mm-error (&rest rest)
  (declare (ignore rest))
  (case %mm-gestalt%
    (-1 
     (error "MIDI: MidiManager is required to run Common Music with ~%~
             Midi syntax.~%~
             Make sure the MidiManager extension and a known Midi driver~%~
             are at their proper locations inside the System Folder and~%~
             reboot.~%~
             Currently known Midi drivers:~%~
             ~{~3T~a~%~}" (mapcar #'driver-filename *drivers*)))
    ((0 1)
     (cerror "let Common Music open a midi port."
             "MIDI: There is no port open for midi.")
     (if (y-or-n-p "Do you want to specify a port?")
       (progn 
         (format t "~&Midi port to open (nil or a port name or number): ")
         (midi-open :port 
                    (loop for form = (read)
                          while (not (ff-midi-find-port form))
                          do 
                          (format t "~s is not a known port specifcation. ~
                                     Midi port to open: " form)
                          finally (return (ff-midi-port-id form)))))
       (midi-open)))
    (2 (error "MIDI: oops! Common Music has not yet any ports allocated.~%~
               Please try closing Midi and opening it again."))
    (t (error "MIDI: oops! <strange error: MidiManager Gestalt value: ~a>" 
              %mm-gestalt%))))
     

;;;
;;; Mapping ff-midi-functions onto %mm-functions

(defconstant %ff-midi/mm-siblings%
  '((ff-midi-close %mm-close) 
    (ff-midi-start-timer %mm-start-timer) 
    (ff-midi-stop-timer %mm-stop-timer) 
    (ff-midi-get-time %mm-get-time) 
    (ff-midi-set-time %mm-set-time) 
    (ff-midi-read-messages %mm-read-messages) 
    (ff-midi-write-message %mm-write-message) 
    (ff-midi-all-notes-off %mm-all-notes-off) 
    (ff-midi-flush-transmit %mm-flush-transmit) 
    (ff-midi-flush-receive %mm-flush-receive) 
    (ff-midi-hush %mm-hush)))

(defun %ff-level-3-siblings () (nthcdr 1 %ff-midi/mm-siblings%))
(defun %ff-level-2-siblings () (list (first %ff-midi/mm-siblings%)))


;;;
;;; Initialization of the swappable ff-midi- functions: none of them is valid 
;;; yet

(eval-when (:load-toplevel :execute)
  (map nil #'(lambda (x) (setf (symbol-function (first x)) #'%mm-error))
       %ff-midi/mm-siblings%))

(defun %ff-dispatch (new-gestalt)
  (case %mm-gestalt%
    (1                                  ; +mmAvailable+
     ;; new-gestalt is always 2 (can't go back to +mmUnknown+)
     (map nil #'(lambda (x) (setf (symbol-function (first x))
                                  (symbol-function (second x))))
          (%ff-level-2-siblings))
     (setf %mm-gestalt% new-gestalt))
    (2                                  ; +mmCMSignedIn+
     (case new-gestalt
       ;; new-gestalt is 1 if we signed out or 3 if all is up and running now
       (1 (map nil #'(lambda (x) 
                       (setf (symbol-function (first x)) #'%mm-error))
               (%ff-level-2-siblings))
        (setf %mm-gestalt% 1))
       (3 (map nil #'(lambda (x) (setf (symbol-function (first x)) 
                                       (symbol-function (second x))))
               (%ff-level-3-siblings))
        (setf %mm-gestalt% 3))))
    (3                                 ; +mmCMSetUp+
     ;; new-gestalt is always 2 (can't go further)
     (map nil #'(lambda (x) (setf (symbol-function (first x)) 
                                  (symbol-function '%mm-error)))
          (%ff-level-3-siblings))
     (setf %mm-gestalt% new-gestalt))))


;;;
;;; Sign in/sign out
;;;
;;; %mm-sign-in checks for the presence of some client with the same id as 
;;; itself. If such a client is found and its name is _not_ equal to 
;;; *common-music-version*, a continuable error is signaled, leaving the choice
;;; to the user. Else it assumes that for some reason Common Music couldn't
;;; sign out properly and simply returns  nil. If it did actually sign in, 
;;; however, the return value is t.

(defvar %iconhandle% nil)

(defvar %icon% 
  ;; the icon and its mask.
  '(#x7FFFFFFE #x80000001 #x9FFFFFF9 #xA0000005
    #xA0000005 #xA0000005 #xA0000005 #xA0000005
    #xA0000005 #xA0200805 #xA0400405 #xA0400405
    #xA0454405 #xA0200805 #xA0000005 #xA0000005
    #xA0000005 #xA0000005 #xA0000005 #xA0000005
    #xA0000005 #x9FFFFFF9 #x80000001 #x80000001
    #xFFFFFFFF #x030000C0 #x00FFFF00 #x00555600
    #x00400200 #x0FFFFFF0 #x10000008 #x1FFFFFF8
    #x7FFFFFFE #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF
    #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF
    #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF
    #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF
    #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF
    #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF #xFFFFFFFF
    #xFFFFFFFF #x03FFFFC0 #x00FFFF00 #x007FFE00
    #x007FFE00 #x0FFFFFF0 #x1FFFFFF8 #x1FFFFFF8))

(defun %mm-sign-in ()
  (let ((name (string *common-music-version*))
        (refcon 0))
    (setf %iconhandle% (#_newhandle 256))
    (ccl:with-dereferenced-handles ((hdl %iconhandle%))
  (let ((offset 0))
    (map nil #'(lambda (long) 
                 (ccl:%put-long hdl long offset) 
                 (incf offset 4)) 
         %icon%)))
    (ccl:with-pstrs ((name name))
      ;; At first, try to sign in normally.
      (case (#_midisignin  +cm-midi+ refcon %iconhandle% name)
        (0                             			; #$noErr
         ;; All is done. Return t.
         (%ff-dispatch +mmCMSignedIn+) 
         t)
        (-108                          			; #$memFullErr
         (error "MIDI: not enough memory to sign into the MidiManager"))
        (-260                          			; #$MIDIDupIDErr
         ;; If there is already another client with the same id, get its name.
         ;; To do so, we have to sign in temporarily with another id.
         (let (randid err)
           (loop for id = (make-port-ids :increment t) 	; get a fresh id
                 for flg = (#_midisignin id refcon %iconhandle% name)
                 thereis (when (/= flg #$MIDIDupIDErr)  ; if it's not already
                           (setf randid id)     	;   there, exit loop
                           (setf err flg)
                           (%ff-dispatch +mmCMSignedIn+)))
           (case err
             (0                         		; #$noErr
              ;; This time, we succeeded. Now get the conflicting client's
              ;; name and see whether it's Common Music.
              (if (ccl:%stack-block ((str255 256))
                    (#_midigetclientname +cm-midi+ str255)
                    (string-equal (ccl:%get-string name) 
                                  (ccl:%get-string str255)))
                ;; It's just Common Music still signed in. Don't do anything.
                nil
                ;; Else leave the choice to the user...
                (cerror "Assume that the client is ~a~@
                         (Which it does not seem to be)."
                 "MIDI: duplicate client ID - apparently not Common Music.~@
                  If there is another application with the same ID, you will~@
                  have to quit it before Common Music can sign in.~@
                  However, if you know that Common Music is just still ~@
                  signed in for some reason, continue at your own risk."
                  *common-music-version*))
              ;; Sign out again and return nil.
              (#_midisignout randid)
              nil)
             ;; Else give up.
             (t (error "MIDI: While trying to recover from a ~
                        DuplicateClientID error:~@
                        Failed to sign in using a different ID with ~a" err)
                ;; Sign out again and return nil.
                (#_midisignout randid)
                nil))))
        ;; There shouldn't be any other result codes, but make sure anyway.
        (t (error "MIDI: failed to sign in with an unknown error."))))))

(defun %mm-sign-out ()
  (#_midisignout +cm-midi+)
  (%ff-dispatch +mmAvailable+)
  (#_disposehandle %iconhandle%)
  t)


;;;
;;; MidiManager default connections.
;;;
;;; This functionality could be extended in the future to make MidiManager 
;;; connections scriptable.

(defun %mm-connect-data (portnum)            ; portnum is 0 or 1
    ;; Connect to driver, if appropriate.
  (when (driver-can-connect (find-driver-by-id $driver-id$))
    (case portnum
      ((0 1) 
       ;; MIDI input: connect output of driver to input of common music
       (with-midi-checked
         (#_midiconnectdata $driver-id$ 
          (out-id (find-driver-ports $serial-port$ :id $driver-id$))
          +cm-midi+ (port-id (find-port-by-name :|Input|))))
       ;; for MIDI output: connect output of common music to input of driver
       (with-midi-checked
         (#_midiconnectdata +cm-midi+ (port-id (find-port-by-name :|Output|))
          $driver-id$ (in-id (find-driver-ports $serial-port$ 
                                                :id $driver-id$)))))
      (t (error "MIDI: portnum specified is not 0 or 1: ~A" portnum))))
  ;; Connect the two invisible ports.
  (with-midi-checked
    (#_midiconnectdata +cm-midi+ (port-id (find-port-by-name 
                                            :invisible-output)) 
                       +cm-midi+ (port-id (find-port-by-name 
                                            :invisible-input)))))

#| 
;; Unused - disconnecting is achieved automatically by signing out.
(defun %mm-unconnect-data (&optional portnum)
  ;; Disconnect the two invisible ports.
  (with-midi-checked
    (#_midiunconnectdata 
     +cm-midi+ (port-id (find-port-by-name :invisible-output)) 
     +cm-midi+ (port-id (find-port-by-name :invisible-input))))
  ;; Disconnect from the driver.
  (case (or portnum $serial-port$)
    ((0 1) 
     (with-midi-checked
       (#_midiunconnectdata 
        $driver-id$ (out-id (find-driver-ports (or portnum $serial-port$) 
                                               :id $driver-id$))
        +cm-midi+ (port-id (find-port-by-name :|Input|))))
     (with-midi-checked
       (#_midiunconnectdata 
        +cm-midi+ (port-id (find-port-by-name :|Output|))
        $driver-id$ (in-id (find-driver-ports (or portnum $serial-port$) 
                                              :id $driver-id$)))))
    (t (if portnum (error "MIDI: portnum specified is not 0 or 1: ~A" portnum)
           (error "<oops!> serial-port global variable is not 0 or 1: ~A" 
                  $serial-port$)))))
|#


;;;
;;; Timing functions

(defun %mm-start-timer ()
  (#_midistarttime $time-port$)
  (values +ff-midi-success+))

(defun %mm-stop-timer ()
  (#_midistoptime $time-port$)
  (when (/= (ccl:%get-word %oldports-ptr%) 0) (%mm-remove-aux-buffers))
  (values +ff-midi-success+))

(defun %mm-get-time (array)
  (setf (elt array 0) (#_midigetcurtime $time-port$))
  (setf (elt array 1) 1000)
  (when (/= (ccl:%get-word %oldports-ptr%) 0) (%mm-remove-aux-buffers))
  (values +ff-midi-success+))

(defun %mm-set-time (time)
  (#_midisetcurtime $time-port$ time)
  (when (/= (ccl:%get-word %oldports-ptr%) 0) (%mm-remove-aux-buffers))
  (values +ff-midi-success+))


;;;
;;; Midi input

(ccl:deffcfun (%mm-read-messages-aux "midireadevent")
  ()
  :word)

(defun %mm-read-messages ()
  (%mm-read-messages-aux)
  (when (/= (ccl:%get-word %oldports-ptr%) 0) (%mm-remove-aux-buffers))
  (values (the fixnum 0)))

(ccl:defccallable midi-read-hook (:long msg :long tim :void)
  (declare (optimize (speed 3)(safety 0)))
  (when *Midi-Read-Hook*
    (funcall (the function *midi-read-hook*)
             (the fixnum msg) (the fixnum tim))))


;;;
;;; Midi output

(defun %mm-write-message (message time)
  (declare (optimize (speed 3)(safety 0))
           (fixnum message time))
  (let ((size (ash (logand message #x03000000) -24)))
    (declare (fixnum byte size))
    (when (or (< size 1) (> size 3)) (error "bad message size: ~A" size))
    ;; Allocate a stack block in the mac heap and put the midi packet in it.
    (ccl:%stack-block ((buf 16))
      (ccl:%put-byte buf 0 0)           ; flags
      (ccl:%put-byte buf (+ size 6) 1)  ; size
      (ccl:%put-long buf message 5)     ; offset 5 (msb of message) gets
      (ccl:%put-long buf time 2)        ;    overwritten by the lsb of time!
                                        ;    (msb of message is unused)
      ;; We may NEVER attempt to access a possibly removed port, or we will
      ;; almost certainly crash. Therefore, inhibit myTimeProc - which 
      ;; executes at interrupt level - to remove an aux outport until 
      ;; we're done.
      (ccl:%put-word %myports-ptr% #x0001)          ; points to Remove_Not
      ;; Now try to write to the current outport ((%get-signed-word 
      ;; (first %inv-outports%)) is guaranteed to never return an invalid
      ;; refnum).
      (loop 
        for err = (#_midiwritepacket 
                   (ccl:%get-word (first %inv-outports%)) buf)
        ;; If no more buffers can be allocated, looping should still continue
        ;; until #_midiwritepacket stops signaling overflows, in which case 
        ;; the OSErr is returned
        until (/= err #$midiWriteErr)
        ;; The first attempt to write failed. Thus, see if there are more 
        ;; outports
        do (when (/= 0 (ccl:%get-word (second %inv-outports%)))
             (loop 
               ;; Search once through the other outports until one is found 
               ;; which has room left (or whose refnum is zero in inv_ports).
               for ptr in (cdr %inv-outports%)
               for x = (ccl:%get-word ptr)
               until (= x 0)
               ;; Exit and store the new OSErr in err if #_midiwritepacket 
               ;; didn't signal an overflow. Midi output will continue to 
               ;; write to this outport as long as it is successfull, 
               ;; therefore cycle %inv-outports%, too
               thereis (when 
                         (/= #$midiWriteErr 
                             (setf err (#_midiwritepacket x buf)))
                         (%cycle-inv-outports (position ptr %inv-outports%)))))
        ;; If no further outport could be found or were available, try 
        ;; allocating and writing to a new buffer. Exit and store the new OSErr
        ;; returned by  #_midiwritepacket in err, if the new buffer could be
        ;; allocated.  Else, leave err untouched, thus causing the outer loop
        ;; to continue looping through all outports until it finally succeeds.
        (when (= err #$midiWriteErr)
          (let ((flg (%mm-make-aux-buffer)))    ; is a refnum or +memFullErr+
            (unless (eq flg +memFullErr+)
              (setf err (#_midiwritepacket flg buf)))))
        ;; Finally, store the message's time in the refCon field of the current
        ;; outport, if it's an auxiliary one, so that myTimeProc can easily 
        ;; check whether it's about time to release the buffer. Return err.
        finally (let ((refn (ccl:%get-word (first %inv-outports%))))
                  (unless (= refn $invisible-output-port$)
                    (with-midi-checked (#_midisetrefcon refn time))))
        ;; Finally, clear the Remove_Not field in the inv_out structure.        

        (ccl:%put-word %myports-ptr% #x0000)          ; points to Remove_Not
        (when (/= (ccl:%get-word %oldports-ptr%) 0) (%mm-remove-aux-buffers))
        (return err)))))


;;; Hard-coded all-notes-off.
;;; Outputs note-off messages for all keys and all channels from 0 to 
;;; *midi-multitimbral-channels* - 1.
;;; The stack-block memory is set up as a MidiPacket consisting of the fields:
;;;   Byte:   0     1     2-5    6-8 (= 3 lsb of data)
;;;   Field:  flags length time  data
;;; and is initialized to 0009 0000 0000 8xxx  7f.. .... .... ....
;;; Only the word at offset 6 (the 2 most significant bytes of data) is 
;;; changed.
;;;
;;; There are actually two versions of this function, depending on whether it
;;; is called with a time parameter or not. Calling ff-midi-all-notes-off with
;;; a time parameter takes up to a second longer (.18 vs. .12 on per channel
;;; on my se/30)

(defun %mm-all-notes-off (&optional (time 0))
  (declare (optimize (speed 3)(safety 0))
           (fixnum time))
  (ccl:%put-word %myports-ptr% #x0001)
  (ccl:%stack-block ((buf 16))
    (ccl:%put-word buf #x0009 0)
    (ccl:%put-long buf #x0080007f 5)
    (ccl:%put-long buf time 2)
    (loop for channel from #x80 to (logior #x80 
                                           (1- *midi-multitimbral-channels*))
          do
          (ccl:%put-byte buf channel 6)
          (loop for note from #x00 to #x7f do
                (ccl:%put-byte buf note 7)
                (if (/= time 0)
                  ;; take an invisible outport, so that the packets get 
                  ;; rescheduled at their proper time
                  (loop 
                    for err = (#_midiwritepacket 
                               (ccl:%get-word (first %inv-outports%)) buf)
                    until (/= err #$midiWriteErr)
                    do (when (/= 0 (ccl:%get-word (second %inv-outports%)))
                         (loop 
                           for ptr in (cdr %inv-outports%)
                           for x = (ccl:%get-word ptr)
                           until (= x 0)
                           thereis (when 
                                     (/= #$midiWriteErr 
                                         (setf err (#_midiwritepacket x buf)))
                                     (%cycle-inv-outports 
                                      (position ptr %inv-outports%)))))
                    (when (= err #$midiWriteErr)
                      (let ((flg (%mm-make-aux-buffer)))
                        (unless (eq flg +memFullErr+)
                          (setf err (#_midiwritepacket flg buf))))))
                  ;; else, write directly to the output port - 
                  ;; it has room enough for our 4048 packets.
                  (#_midiwritepacket $output-port$ buf))))
  (ccl:%put-word %myports-ptr% #x0000)
  (when (/= (ccl:%get-word %oldports-ptr%) 0) (%mm-remove-aux-buffers))
  (values +ff-midi-success+)))
    

;;;
;;; Flushing

(defparameter *midi-flush-timeout* 3000
  "The time in milliseconds, for which the queueing mechanism stays in ~
   flushing mode, therefore ignoring newly scheduled events.")

(ccl:deffcfun (%mm-set-invreadhook-to-flush "flushAllBuffers")
  ()
  :void)

(defun %mm-flush-transmit ()
  ;; Flush all midi output buffers:  the c-side flushAllBuffers routine puts 
  ;; myInvisibleReadHook into its flushing state, sets the offset time of all 
  ;; invisible input ports to MidiGetEverything, and schedules a resetting 
  ;; callback.
  (%mm-set-invreadhook-to-flush)
  (values +ff-midi-success+))

(defun %mm-flush-receive ()
  ;; Flush the midi input buffer.
  (#_midiflush $input-port$)
  (values +ff-midi-success+))

(defun %mm-hush ()
  ;; Flush all pending midi input and output events.
  (%mm-flush-transmit)
  (%mm-flush-receive)
  (values +ff-midi-success+))


;;;
;;; Foreign function environment initialization 

(ccl:deffcfun (midi-c-setup-aux "midicsetup")
  ((fixnum :long) (fixnum :long) (fixnum :long) 
   (fixnum :long) (fixnum :long)
   (t :ptr)
   (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long))
  :ptr)

(defun midi-c-setup ()
  ;; Compute the needed pointersize, initialize the c side and set up 
  ;; %inv-outports%.
  (let* ((mac-heap-size (let ((s (with-output-to-string (str)
                                   (let ((*standard-output* str)) (room)))))
                          (parse-integer s :junk-allowed t
                                           :start (+ (length "Mac Heap:")
                                                     (search "Mac Heap:" s)))))
         ;; The size of the pointer to allocate for the inv_ports structure
         ;; is the maximum number of buffers allocatable within the bounds of
         ;; the current mac heap size times 4 bytes plus 4 bytes for the 
         ;; first two fields. The size of the rem_ports structure is this
         ;; same size minus 2.
         (max-nr-of-buffers (floor (/ (- mac-heap-size 
                                         +main-buffer-size+ 
                                         +input-buffer-size+) 
                                      +aux-buffer-size+)))
         (rem_ports-offset (+ (* max-nr-of-buffers 4) 4))
         (pointersize (- (* rem_ports-offset 2) 2))
         (outportlist 
          (midi-c-setup-aux $time-port$ $input-port$ $output-port$
                            $invisible-input-port$ $invisible-output-port$
                            midi-read-hook
                            +callback-period+ +buffer-idle-timeout+ 
                            *midi-flush-timeout* pointersize)))
    (if (ccl:%null-ptr-p outportlist)
      ;; Notify and signal failure if allocation failed.
      (format t "Fatal error: MIDI: not enough memory in the mac heap.~&~
                 ~13TRequested size: ~a bytes.~&~
                 ~13TClosing midi again." pointersize)
      ;; %inv-outports% holds one macptr for every offset from the start of
      ;; the structure that is a multiple of 4 - thus, the word every macptr
      ;; points to is a inv_ports->port_pairs[x].outport component of the
      ;; myports structure.
      (setf %inv-outports% (loop for offset from 4 by 4
                                 repeat max-nr-of-buffers
                                 collect (ccl:%inc-ptr outportlist offset))))
    ;; In any case, return the pointer to the overall structure (which is 
    ;; either nil or a valid macptr) to signal success or failure.
    (setf %oldports-ptr% (ccl:%inc-ptr outportlist rem_ports-offset))
    (setf %myports-ptr% outportlist)))


;;;
;;; Opening and closing midi
;;;
;;; ff-midi-open is the midi entry function. It performs its own checks on 
;;; whether the MidiManager has been installed, etc. Therefore, this function
;;; is accessible all the time and will never swapped.


(defun check-for-mm-availability ()
  ;; Set it this one time directly - don't call ff-dispatch at this level.
  (setf %mm-gestalt% (if (= (#_snddispversion #$midiToolNum) 0) 
                       +mmUnavailable+ 
                       +mmAvailable+))
  ;; Don't return to %mm-open unless the MidiManager is available.
  (when (= %mm-gestalt% +mmUnavailable+) (%mm-error))
  t)
  
(defvar %mm-close-stack%
  ;; Holds a varying list of functions to be executed by %mm-close. 
  nil)

(defun %mm-midi-open-aux (int)
  (catch :midi-open-exit
    (setf %mm-close-stack% 
          `(,#'(lambda ()
                 (setf %midi-open-p% nil)
                 (%update-midi-menu)
                 (values +ff-midi-success+)))) 
    (%mm-sign-in)
    (push #'%mm-sign-out %mm-close-stack%)
    (%mm-add-cm-ports)
    (push #'(lambda () 
              (%mm-remove-cm-ports)
              ;; Make sure the pointers held in %inv-outports% can't be 
              ;; accessed any more.
              (setf %inv-outports% nil)
              ;; Fight memory leaks and dispose the block %myports-ptr% points
              ;; to, but never dispose a dead pointer.
              (when  %myports-ptr%
                (#_disposeptr %myports-ptr%)
                (setf %myports-ptr% nil))) %mm-close-stack%)
    ;; Check for the availability of drivers and ports. This function also
    ;; sets the globals $driver-id$ and $serial-port$.
    (%mm-get-driver-and-serial-port int)
    (push #'(lambda () (setf $serial-port$ nil $driver-id$ nil))
          %mm-close-stack%)
    (%mm-connect-data $serial-port$)
    (ff-midi-start-timer)
    ;; make sure %midi-open-p% holds the serial port variable
    (setf %midi-open-p% (verbose-find-port $serial-port$))
    (%update-midi-menu)
    (push #'%update-midi-menu %mm-close-stack%)
    ;; Initialize the c side; but close everything again, if it failed.
    (unless (midi-c-setup) 
      (throw :midi-open-exit :cancel-midi-open))
    ;; Now make the data area, jump table, function code segment, and our
    ;; structures ineligible for virtual memory page swapping, because 
    ;; they will be accessed at interrupt level (avoid a possible fatal 
    ;; double page fault error at paging time).
    (mapcar #'(lambda (x) (#_holdmemory x (#_getptrsize x)))
            (list 
             (cdr (first (last (ccl::ffenv-entries %ff-env%)))) ; data & JT
             (first (ccl::ffenv-seg-ptrs %ff-env%))             ; code segment
             %myports-ptr%))                                    ; structures
    (push #'(lambda ()   
              ;; Make data & jump table, code segment, and our structures 
              ;; eligible for page swapping again
              (mapcar #'(lambda (x) (#_unholdmemory x (#_getptrsize x)))
                      (list (cdr (first (last (ccl::ffenv-entries %ff-env%))))
                            (first (ccl::ffenv-seg-ptrs %ff-env%))))
              ;; ...but the structures only if their pointer is valid.
              (when %myports-ptr% (#_unholdmemory %myports-ptr% 
                                   (#_getptrsize %myports-ptr%))))
          %mm-close-stack%)
    (values +ff-midi-success+)))

(defun %mm-close ()
  (map nil #'funcall %mm-close-stack%)
  (setf %mm-close-stack% nil)
  (values +ff-midi-success+))

(defun ff-midi-open (int)
  ;; test for MidiManager's presence the first time only
  (case %mm-gestalt%
    (-1 (%mm-error))                    ; +mmUnavailable+
    ( 0 (check-for-mm-availability)))   ; +mmUnknown+
  (if (eq (%mm-midi-open-aux int) :cancel-midi-open)
    (progn
      (%mm-close)
      (values +canceled+))
    (values +ff-midi-success+)))



;;;
;;; Dummys

(defun ff-midi-set-quanta-size (size)
  (declare (ignore size))
  (values +ff-midi-success+))

(defun ff-midi-listen ()
  (values +ff-midi-success+))

(defun ff-midi-stop-listening ()
  (values +ff-midi-success+))




#|
;;; not used -- the lisp version is just as fast 
(ccl:deffcfun (ff-midi-write-message "midiwritemessage")
  ((fixnum :long) (fixnum :long) )
  :long)
|#

;;;
;;; -*- EOF -*-
