;;; -*- Syntax: Common-Lisp; Mode: Lisp; Package: common-music; Base: 10 -*-

(in-package :common-music)

;;;
;;; This file implements a simple "real time" scheduler in lisp.  The two
;;; entry points are with-real-time-scheduling and schedule-message.
;;; To schedule messages in real time, the schedule-message function is
;;; used inside the macro with-real-time-scheduling:
;;;
;;; (with-real-time-scheduling ()
;;;   (schedule-message (make-note-on 0 60 127) 1000)
;;;   (schedule-message (make-note-off 0 60 127) 1500))
;;;
;;; The function schedule-message is used to schedule the specified message
;;; at the specified millisecond time:
;;;
;;;        schedule-message (message time &optional function)
;;;
;;; Messages may be dynamically specified and also rescheduled by supplying
;;; an optional function to schedule-message. If supplied, the function must
;;; accept two arguments, the current message and the current run time,
;;; and is applied to the current message just before the message is output.
;;; The function must return two values: the current message to output and
;;; the next run time.  If the function decides that no message should be
;;; output in the current pass it may return nil as the first return value.
;;; If it decides that no more messages should be output, it must return nil
;;; as its second value.
;;;
;;; This file should be compiled before it is loaded.
;;; See the file real-time-example.lisp for a simple real-time example.
;;;;

(defvar *message-queue* () "The queue of message entries.")

(defun schedule-message (message time &optional function)
  (unless (integerp time)
    (error "time ~S not an integer." time))
  (let ((entry (list time message function)))
    (setf *message-queue* (insert-entry entry *message-queue*))
    entry))
    
(defun insert-entry (entry list)
  (declare (optimize (speed 3) (safety 0)))
  (let ((time (car entry))
  	(tail list)
	(last ()))
    (declare (fixnum time))
    (loop until (or (null tail) (> (the fixnum (caar tail)) time))
          do (setf last tail tail (cdr tail)))
    (if last (setf (cdr last) (cons entry tail))
      (setf list (cons entry tail)))
   list))

    
(defun do-real-time (queue &optional wait)
  (declare (optimize (speed 3) (safety 0))
           (ignore wait))
  (unless (midi-open-p)
    (error "Midi port not open!"))
  (let (entry real-time base-time this-time next-time 
  	function message)
    (declare (fixnum this-time next-time last-time message))
    (setf *message-queue* queue)
    (setf base-time (get-internal-real-time))
    (without-interrupts                          ; utilties.lisp
      (loop while *message-queue* do
        (setf entry (pop *message-queue*))
        (setf this-time (car entry))
        (if (setf function (caddr entry))
            (multiple-value-setq (message next-time)
	  		         (funcall function (cadr entry) this-time))
          (setf message (cadr entry) next-time nil))
        (setf real-time (+ base-time this-time -10))
        (loop while (< (get-internal-real-time) real-time) do nil)
        (when message
          (midi-write-message message))
        (when next-time
          (setf (car entry) next-time)
	  (setf (cadr entry) message)
	  (setf *message-queue* (insert-entry entry *message-queue*)))
	)))
   (values))
      
(defmacro with-real-time-scheduling (options &body body)
  (declare (ignore options))
  `(let ((*message-queue* nil))
    ,@body
    (do-real-time *message-queue*)))
    
