;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(defgeneric item-stream? (object))                     ; its a stream
(defgeneric item (object))                             ; return value
(defgeneric reset-period (object))                     ; reset period
(defgeneric skip-datum? (object))                      ; 0 period?
(defgeneric datum-parser (object))                     ; return parser fn

(defgeneric parse-data (object data &optional parser)) ; canonicalize data
(defgeneric default-period-length (object))            ; suggested length
(defgeneric next-in-pattern (object))                  ; increment pattern
(defgeneric map-data (function object))                ; map fn over data

(defconstant +constant-data+    (ash 1 0))  ; avoid hair when possible
(defconstant +constant-weights+ (ash 1 1))  ; avoid random index recalc
(defconstant +count-periods+    (ash 1 2))  ; period counts subperiods
(defconstant +count-values+     (ash 1 3))  ; period counts values
(defconstant +depth-first+      (ash 1 4))  ; pattern moves on eop
(defconstant +breadth-first+    (ash 1 5))  ; pattern moves each time
(defconstant +coerce-to-note+   (ash 1 6))  ; return pitches
(defconstant +coerce-to-pitch+  (ash 1 7))  ; return note names
(defconstant +parallel-offsets+ (ash 1 8))  ; voicing mode
(defconstant +elide-left+      (ash 1 9))   ; palindrome flags
(defconstant +elide-right+     (ash 1 10))   
(defconstant +numbers-below+   (ash 1 11))  ; number tests
(defconstant +numbers-downto+  (ash 1 12)) 
(defconstant +numbers-above+   (ash 1 13)) 
(defconstant +nad+ '#:|nad|)                ; "not a datum" marker
(defconstant +eop+ ':end-of-period)         ; "end of period" marker
(defconstant +end-of-stream-token+ +eop+)   ; backward compatibility

(defun decode-flags (f) ; testing hack
  (when (item-stream? f) (setf f (slot-value f 'flags)))
  (loop for s in '(+constant-data+ +constant-weights+ +count-periods+
                   +count-values+  +depth-first+ +breadth-first+ 
                   +coerce-to-note+ +coerce-to-pitch+
                   +parallel-offsets+ +elide-left+ +elide-right+
                   +numbers-below+ +numbers-downto+ +numbers-above+)
         when (logtest f (symbol-value s)) collect s))

(defvar *item-streams* (make-hash-table))

(defun find-stream (name &optional (mode :error))
  (or (gethash name *item-streams*)
      (ecase mode
	(:error (error "~A is not a named item stream." name))
	(:ask nil))))

(defclass flags-mixin ()                    ; stream and pattern need
  ((flags   :initarg :flags :initform 0)))

;;;
;;; the period struct holds information for various period calculations. 
;;; count is number of reads remaining in current period. when count=0
;;; the period is reinitialized. length is maximum count of the period,
;;; either a number or T if dynamic length. if stream is not nil it a
;;; new length will be read from it each time the period is initialized.
;;; omit is the number of times this stream is skipped in its parent's
;;; pattern, if dynamic. 
;;;

(defstruct period (count 0) length stream default (omit 0))

;;;
;;; item-pattern and item-stream are the main root classes. subclasses
;;; are mixed together to produce fully specified item streams, ie
;;; note-stream+cycle-pattern=cyclic-note-stream. the class precedence
;;; list of the various leaf classes must insure that initialize-instance
;;; happens to the pattern side first because the default period length
;;; of a stream usually depends in some way onits pattern's length.
;;;

(defclass item-pattern (flags-mixin)
  ((data    :initform nil :initarg :items)
   length))

(defclass item-stream (flags-mixin) 
  ((name    :initarg :name :initarg :named)
   (datum   :initform +nad+)
   (period  :initarg :for)
   (value   :initarg :value :initform +nad+)
   (state   :initarg :state :initform +nad+)))

(defmethod initialize-instance :after ((object item-pattern) &rest args)
  (declare (ignore args))
  ;(format t "~%initialize-instance :after item-pattern")
  (let ((flags (slot-value object 'flags))
        data length constant?)

    ;; parse external data into canonical form
    (multiple-value-setq (data length constant?) 
      (parse-data object (slot-value object 'data)
                  (datum-parser object)))
    (setf (slot-value object 'data) data)
    (setf (slot-value object 'length) length)
    (when constant?
      (setf flags (logior flags +constant-data+)))
    ;; if constant data and counting subperiods, switch to counting
    ;; values instead since its the same thing and we can avoid
    ;; resetting subperiods if period length is nevertheless expressed
    ;; dynamically.
    (cond ((logtest flags +count-values+)
           (setf flags (logandc2 flags +count-periods+)))
          (t
           (if (logtest flags +constant-data+)
               (setf flags (logior (logandc2 flags +count-periods+)
                           +count-values+))
             (setf flags (logior flags +count-periods+)))))
    (setf (slot-value object 'flags) flags)))

;;;
;;; parse-data is called by item-pattern's initialize instance to
;;; return data in canonical form. parse-data returns three values:
;;; a list of parsed data, the number of elements parsed, and a flag 
;;; indicating if the data is constant (contains no substreams) or not.
;;;

(defmethod object? ((object standard-object)) t)
(defmethod object? ((object t)) nil)

(defmethod parse-data ((stream item-pattern) data &optional parser)
  (if parser
      (loop for datum in data 
            count datum into length
            count (item-stream? datum) into streams
            collect (funcall parser datum) into list
            finally (return (values list length (= streams 0))))
    (values data (length data) (not (some #'object? data)))))

;;;
;;; default-period-length is called by item-pattern's initialize-instance
;;; to return the default period length for that class of stream, which
;;; defaults to the to the number of elements that the stream contains.
;;;

(defmethod default-period-length ((object item-pattern))
  (slot-value object 'length))


(defmethod initialize-instance :after ((object item-stream) 
                                       &rest args)
  (declare (ignore args))
  ;(format t "~%initialize-instance :after item-stream")
  (when (slot-boundp object 'name)
    (setf (gethash (slot-value object 'name) *item-streams*) object))
  ;; period length depends on pattern
  (let* ((default (default-period-length object))
         (period (if (slot-boundp object 'period)
                     (or (slot-value object 'period) default)
                   default)))
    (setf (slot-value object 'period)
      (if (or (numberp period) (eq period t))
          (make-period :length period :default default)
        (make-period :stream period :default default)))))

(defmethod print-object ((object item-stream) stream)
  (if (slot-boundp object 'name)
      (format stream "#<~A ~:(~A~)>" (class-name (class-of object))
              (slot-value object 'name))
    (call-next-method)))

;;;
;;; datum-parser returns a function to apply to each external datum
;;; as it is parsed by the pattern. the default method returns nil.
;;;

(defmethod datum-parser ((object item-stream)) nil)

;;;
;;; item-stream? is the type predicate for item-streams.
;;;

(defmethod item-stream? ((object item-stream)) object)
(defmethod item-stream? ((object t)) nil)

;;;
;;; item returns two values: the next value read from the object and
;;; a flag indicating if the object is at the end of its current period
;;; as a result of the read.
;;;

(defmethod item ((object t))
  (values object +eop+))

(defmethod item ((object item-stream)) 
  (multiple-value-bind (value state) (item (slot-value object 'datum))
    (setf (slot-value object 'value) value)
    (values value state)))

;;;
;;; this around method implements the basic behavior of item-streams.
;;; it first checks the stream's period length and calls reset-period 
;;; if at end. if the next period length is 0 it immediately returns
;;; +nad+, which causes a superior stream (if any) to skip over the
;;; current stream as it increments its pattern.  otherwise, the method
;;; then increments the streams pattern until it yields a datum that is 
;;; not +nad+ and that call-next-method does not return +nad+ from. if
;;; the stream's data is known to contain only constant values, ie no
;;; substreams, the testing loop is skipped. once call-next-method
;;; returns a value (not +nad+), the period and pattern of the stream
;;; are incremented according to their mode. for period incrementing, 
;;; +count-periods+ increments the period count only on +eop+, and
;;; +count-values+ increments the period count every time. for pattern
;;; incrementing, +depth-first+ increments the pattern only on +eop+,
;;; and +breadth-first+ increments the pattern every time.
;;;

(defmethod item :around ((object item-stream))
  (let ((period (slot-value object 'period))
        (flags (slot-value object 'flags))
        value state)
    ;; reset period
    (when (= (period-count period) 0)
      (when (= (reset-period object) 0)
        (return-from item (values +nad+ +eop+))))
    ;; increment datum until not +nad+
    (if (logtest flags +constant-data+)
        (progn
          (setf (slot-value object 'datum) (next-in-pattern object))
          (multiple-value-setq (value state) (call-next-method)))
      (loop with dynamic? = (and (logtest flags +count-periods+)
                                 (eq (period-length period) t))
            do  
            ;; increment over 0 length substreams
            (loop while (eq (slot-value object 'datum) +nad+)
                  do (setf (slot-value object 'datum) 
                       (if dynamic?
                           (skip-datum? (next-in-pattern object))
                         (next-in-pattern object))))
            (multiple-value-setq (value state) (call-next-method))
            ;; increment over +nad+ values returned by item.
            while (eq value +nad+) 
            do (setf (slot-value object 'datum) +nad+)))
    ;; increment period and pattern as appropriate.
    (cond ((eq state +eop+)
           (decf (period-count period))
           (setf (slot-value object 'datum) +nad+)
           (setf state nil))
          (t
           (when (logtest flags +breadth-first+)
             (setf (slot-value object 'datum) +nad+))
           (when (logtest flags +count-values+)
             (decf (period-count period)))))
    (setf state (if (= (period-count period) 0) +eop+ state))
    (setf (slot-value object 'state) state)
    (values (slot-value object 'value) state)))

;;;
;;; skip-datum? returns +nad+ if the current stream should be
;;; skipped in the pattern. this only happens if we have dynamic
;;; periodicity and the datum had a 0 length period when it was
;;; encountered by reset-period.
;;;

(defmethod skip-datum? ((object item-stream))
  (let ((period (slot-value object 'period)))
    ;(format t "~%~S omit:~S" object (period-omit period))
    (if (> (period-omit period) 0)
        (progn (decf (period-omit period)) +nad+)
        object)))

(defmethod skip-datum? ((object t)) object)

;;;
;;; reset-period sets and returns the length of the next period.
;;; period length of constant datum is always 1.
;;;

(defmethod reset-period ((object t)) 1)

(defmethod reset-period ((object item-stream))
  (let ((period (slot-value object 'period))
        dynamic length)
    ;; if period is supplied as a stream get next length via item
    (when (period-stream period)
      (setf (period-length period) (item (period-stream period))))
    (setf length
      (if (setf dynamic (eq (period-length period) 't))
          (period-default period)
         (period-length period)))
    ;; if we have dynamic period length we adjust next period length
    ;; for the number of 0 subperiods that this period will encounter.
    ;; in order for this to work, all substream periods must be reset
    ;; now, at the same that the super stream is reset. we can only
    ;; do this if we know that all subperiods are currently at end
    ;; of period, ie if we are counting by subperiods. if so, then by
    ;; definition all the substreams must be at end-of-period  or we
    ;; couldn't have gotton here in the first place. after resetting
    ;; substream period lengths we decrement our current stream's period
    ;; length by the number of zero periods found.
    (when (and dynamic 
               (logtest (slot-value object 'flags) +count-periods+))
       (let ((zeros 0))
         (map-data #'(lambda (x)
                       (when (= (reset-period x) 0) 
                         (let ((p (slot-value x 'period)))
                           (incf (period-omit p)))
                         (incf zeros)))
                   object)
         (when (> zeros 0) (setf length (max (- length zeros) 0)))
         ;(format t "~%reset subperiods. found ~A zeros." zeros)
         ))
    (setf (period-count period) length)
    length))

;;;
;;; pattern implementations.
;;;
;;; cycle-pattern continously loops over its data. the data are held
;;; in a list of the form: (data . data). successive elements are
;;; popped from the cdr, when the cdr is null it's reset to the car.
;;;

(defclass cycle-pattern (item-pattern)
  ())

(defmethod initialize-instance :after ((object cycle-pattern)
                                       &rest args)
  (declare (ignore args))
  (setf (slot-value object 'data)
    (list (slot-value object 'data))))

(defmethod next-in-pattern ((object cycle-pattern))
  (let ((cycle (slot-value object 'data)))
    (unless (cdr cycle)
      (setf (cdr cycle) (car cycle)))
    (pop (cdr cycle))))

(defmethod map-data (function (object cycle-pattern))
  (map nil function (car (slot-value object 'data))))

;;;
;;; palindrome-pattern is a cycle that adds the reverse of its data at
;;; initialization time. this could be made a "real" pattern by rewriting
;;; this code in terms of next-in-pattern.
;;;

(defclass palindrome-pattern (cycle-pattern)
  ())

(defmethod initialize-instance :after ((object palindrome-pattern) 
                                       &rest args)
  (declare (ignore args))
  (let* ((data (slot-value object 'data))
         (flags (slot-value object 'flags))
         (half (car data))
         (length (slot-value object 'length)))
    (cond 
      ((and (logtest flags +elide-left+)
            (logtest flags +elide-right+))
       (setf (slot-value object 'length) (- (* length 2) 2)) 
       (loop with rev for tail on (cdr half)
             while (cdr tail)
             do (push (car tail) rev)
             finally (setf (car data) (nconc half rev))))
      ((logtest flags +elide-left+)  
       (setf (slot-value object 'length) (1- (* length 2)))
       (loop with rev for i in (cdr half)
             do (push i rev)
             finally (setf (car data) (nconc half rev))))
      ((logtest flags +elide-right+)  
       (setf (slot-value object 'length) (1- (* length 2)))
       (loop with rev for tail on half
             while (cdr tail)
             do (push (car tail) rev)
             finally (setf (car data) (nconc half rev))))
      (t
       (setf (slot-value object 'length) (* length 2))
       (loop with rev for i in half
             do (push i rev)
             finally (setf (car data) (nconc half rev)))))))

;;;
;;; sequence-pattern sticks on the last element.
;;;

(defclass sequence-pattern (item-pattern) 
  ())

(defmethod next-in-pattern ((object sequence-pattern))
  (let ((seq (slot-value object 'data)))
    (if (cdr seq)
      (pop (slot-value object 'data))
      (car seq))))

(defmethod reset-period :before ((object sequence-pattern))
  (unless (cdr (slot-value object 'data))
    (setf (slot-value object 'length) 1)))

(defmethod map-data (function (object sequence-pattern))
  (map nil function (car (slot-value object 'data))))

;;;
;;; heap-pattern is a cycle that shuffles its elements 
;;; each time through
;;;

(defclass random-mixin ()
  ((random :initform *cm-state* :initarg using :initarg :using))) 

(defclass heap-pattern (cycle-pattern random-mixin)
  ())

(defmethod next-in-pattern ((object heap-pattern))
  (let ((cycle (slot-value object 'data)))
    (unless (cdr cycle)
      (setf (cdr cycle)
        (shuffle-heap (car cycle) (slot-value object 'length)
                      (slot-value object 'random) NIL)))
    (pop (cdr cycle))))

(defun shuffle-heap (heap length state first)
  (loop for tail on heap 
        do (rotatef (car tail) (elt heap (random length state))))
  (when first
    (let ((pos (position first heap)))
      (rotatef (elt heap 0) (elt heap pos))))
  heap)

;;;
;;; random-item-stream chooses using weighted selection. its data are
;;; kept in a list of the form: ((&rest choices) . last-choice).
;;;

(defclass random-pattern (item-pattern random-mixin)
  (range))

(defstruct (random-item (:type list))		
  datum index (weight 1.0) (min 1) max (count 0) id)

(defmethod initialize-instance :after ((object random-pattern) 
                                       &rest args)
  (declare (ignore args))
  (let ((pool (slot-value object 'data))
        (sum 0.0))
    ;; check the stream for constant weights. if true, calculate the
    ;; range now and set a flag so we dont recalulate each period.
    (loop while sum 
          for item in pool
          for weight = (random-item-weight item)
          if (numberp weight)
          do (progn (incf sum weight)
                    (setf (random-item-index item) sum))
          else do (setf sum nil))
    (when sum
      (setf (slot-value object 'range) sum)
      (setf (slot-value object 'flags)
        (logior +constant-weights+ (slot-value object 'flags))))
    ;; all routines treat pool as: ((&rest choices) . last-choice)
    ;; no initial last choice. a first choice for the stream could
    ;; be implemented as a last with min=1
    (setf (slot-value object 'data) (list pool))))

(defmethod reset-period :after ((object random-pattern))
  (let ((flags (slot-value object 'flags)))
    (unless (logtest flags +constant-weights+)
      (loop with sum = 0.0 
            for item in (car (slot-value object 'data))
            do (incf sum (item (random-item-weight item)))
               (setf (random-item-index item) sum)
            finally (setf (slot-value object 'range) sum)))))

(defmethod next-in-pattern ((object random-pattern))
  ;; pool is ((&rest choices) . last-choice)
  (let* ((pool (slot-value object 'data))
         (last (cdr pool)))
    (if (and last (< (incf (random-item-count last)) 
                     (random-item-min last)))
        (random-item-datum last)
      (let ((range (slot-value object 'range))
            (state (slot-value object 'random))
            (choices (car pool))
            next)
        (setf next
          (loop for item = (loop with index = (random range state)
                                 for x in choices
                                 when (< index (random-item-index x))
                                 return x)
	        unless (and (random-item-max item)
	                    (= (random-item-count item)
	                       (random-item-max item)))
	        return item))
        (unless (eq next last)
          (loop for i in choices
                do (setf (random-item-count i) 0)))
        (setf (cdr pool) next)
        (random-item-datum next)))))

(defmethod map-data (function (object random-pattern))
  (map nil #'(lambda (x) (funcall function (random-item-datum x)))
       (car (slot-value object 'data))))

(defmethod parse-data ((stream random-pattern) data &optional parser)
  (let ((internal (mapcar #'(lambda (x) (parse-random-item x parser))
                          data))) 
    (values internal (length internal)
            (not (find-if #'item-stream? internal
                          :key #'random-item-datum)))))

(defun parse-random-item (external &optional parser)
  (apply 
   #'(lambda (datum &rest keys)
       (when parser (setf datum (funcall parser datum)))
       (loop with orig = keys and args = ()
             while keys 
             for key = (pop keys)
             for val = (if keys (pop keys)
                           (error "Uneven random keyword list: ~S." 
                                  orig))
             do
             (push val args)
             (ecase key
               ((weight :weight) (push ':weight args)) 
               ((min :min)  (push ':min args)) 
               ((max :max)  (push ':max args)))
             finally 
             (return (apply #'make-random-item :datum datum args))))
   (if (consp external) external (list external))))

;;;
;;; graph-item-stream traverses its nodes by applying a selection
;;; function to the graph. nodes are kept in a list of the form:
;;; (current . nodes). current is initialized to (first nodes)
;;;

(defclass graph-pattern (item-pattern) 
  ((selector :initform #'default-graph-node-select
             :initarg :selector :initarg :function)
   (last     :initform nil :initarg :last)
   (props    :initform nil :initarg :props)))

(defstruct (graph-node (:type list)) id datum to props)

(defmethod initialize-instance :after ((object graph-pattern) &rest args)
  (declare (ignore args))
  (let ((nodes (slot-value object 'data))
        (last (slot-value object 'last)))
    (when last
      (setf (slot-value object 'last)
        (if (consp last)
            (list* (length last) last)
          (list* last (make-list last :initial-element '*)))))
    (setf (slot-value object 'data)
      (cons (first nodes) nodes))))

(defun default-graph-node-select (stream node &rest lastids)
  (let ((to (graph-node-to node)))
    (if (consp to)
      (if (eq (car to) ':idsel)
        (markov-select stream node to lastids)
        (nth (random (length to) *cm-state*) to))
      (item to))))

(defun markov-select (stream node table lastids)
  (declare (ignore stream))
  ;;  table is a list (:idsel <id1> <stream1> ...)
  (let ((prob (loop for (i s) on (cdr table) by #'cddr
                    when (match-ids i lastids)
                    return s)))
    (unless prob
      (error "Node ~S has no entry for previous ids ~S in ~S."
             (graph-node-id node) lastids table))
    (item prob)))

(defun match-ids (user last)
  ;; match the user's ids with the past choices. * is a wildcard.
  ;; matching could get really fancy if we wanted it to.
  (cond ((null user) t)
        ((null last) nil)
        ((consp user)
         (and (match-ids (car user) (car last))
              (match-ids (cdr user) (cdr last))))
        ((eq user last) t)
        ((eq user '*) t)
        ((eq last '*) t)
        (t nil)))

(defmethod next-in-pattern ((stream graph-pattern))
  (let* ((last (slot-value stream 'last))
         (graph (slot-value stream 'data))
         (nodes (cdr graph))
         (this (car graph))
        next node)
    (if (setf next (apply (slot-value stream 'selector)
                          stream this (cdr last)))
      (if (setf node (find next nodes :key #'graph-node-id)) 
        (progn
          ;; next item becomes selected node. car of last is
          ;; the number of choices to remember.  push old 
          ;; selection onto the list and flush the the oldest
          ;; element. since we cant setf nthcdrin some lisps,
          ;; we setf the cdr of nthcdr-1...
          (setf (car graph) node)
          (when last
            (push (graph-node-id this) (cdr last))
            (setf (cdr (nthcdr (1- (car last)) (cdr last))) nil)))
        (error "Missing node for id ~S." next))
      (error "No next id returned from node ~A and function ~S"
             (graph-node-id node) (slot-value stream 'selector)))
    (graph-node-datum this)))

(defmethod map-data (function (object graph-pattern))
  (map nil #'(lambda (x) (funcall function (graph-node-datum x)))
       (slot-value object 'data)))

(defmethod parse-data ((stream graph-pattern) data &optional parser)
  (let ((internal (mapcar #'(lambda (i) (parse-graph-item i parser)) 
                          data)))
    (values internal
            (length internal)
            (not (find-if #'item-stream? internal
                          :key #'graph-node-datum)))))

(defun parse-graph-item (external &optional parser)
  (unless (consp external) (error "Graph node ~S not list." external))
  (apply
   #'(lambda (datum &rest keys)
       (when parser (setf datum (funcall parser datum)))
       (loop with orig = keys and args = ()
             and id = nil and to = nil
             while keys 
             for key = (pop keys)
             for val = (if keys (pop keys)
                           (error "Uneven graph node keyword list: ~S." 
                                  orig))
             do
             (push val args)
             (ecase key
               ((id :id) (setf id t) (push ':id args)) 
               ((to :to) (setf to t) (push ':to args))
               ((props :props) (push ':props args)))
             finally 
             (progn
               (unless id (push datum args) (push ':id args))
               (unless to (error "Missing :to nodes for ~S" id))
               (return (apply #'make-graph-node :datum datum args)))))
   external))

;;;
;;; accumulation-pattern adds the current item to the set of items
;;; selected so far: A A B A B C | A A B A B C 
;;;

(defclass accumulation-pattern (item-pattern)
  ((indices :initform (cons 0 0))))

(defmethod next-in-pattern ((stream accumulation-pattern))
  (macrolet ((minor (x) `(car ,x))
             (major (x) `(cdr ,x)))
    (let ((indices (slot-value stream 'indices)))      
      (prog1
        (elt (slot-value stream 'data) (minor indices))
        (if (= (minor indices) (major indices))
          (setf (minor indices) 0
                (major indices) (mod (1+ (major indices)) 
                                     (slot-value stream 'length)))
          (setf (minor indices) (1+ (minor indices))))))))

(defmethod map-data (function (object accumulation-pattern))
  (map nil function (slot-value object 'data)))

(defmethod default-period-length ((object accumulation-pattern))
  (let ((length (slot-value object 'length)))
    (loop for i from 1 to length sum i)))

;;;
;;; function-pattern calls a function to return the items
;;; constituting the data for the next period.
;;;

(defclass function-pattern (item-pattern)
  ((function :initarg :function)
   (period :initform 1)
   (funargs :initarg :args :initform nil)
   (states :initform nil)))

(defmethod item :around ((object function-pattern))
   (let ((value (call-next-method)))
     (values value
             (if (slot-value object 'states)
                 (pop (slot-value object 'states)) 
               (if (slot-value object 'data) nil +eop+)))))

(defmethod next-in-pattern ((object function-pattern))
  (unless (slot-value object 'data)
    (let (values states length)
      (multiple-value-setq (values states)
        (apply (slot-value object 'function) 
               (slot-value object 'funargs)))
      (if (null values)
        (error "No item list returned to functional item stream.")
        (unless (listp values) (setf values (list values))))
      (setf length (length values))
      (setf (slot-value object 'data) values)
      (setf (slot-value object 'length) length)
      (let ((period (slot-value object 'period)))
         (setf (period-count period) length 
               (period-length period) length))
      (setf (slot-value object 'states) states)))
  (pop (slot-value object 'data)))

(defmethod map-data (function (object function-pattern))
  (declare (ignore function))
  nil)

;;;
;;; rotation-pattern
;;;

(defclass rotation-pattern (cycle-pattern)
  ((change :initform 0 :initarg :change)))

(defmethod initialize-instance :after ((object rotation-pattern) 
                                &rest args)
  (declare (ignore args))
  ;; pattern is initialized now so that rotations only happen
  ;; after the first cycle.
  (let ((data (slot-value object 'data)))
    (setf (cdr data) (car data))))

(defmethod next-in-pattern ((object rotation-pattern))
  (let ((ring (slot-value object 'data)))
    (unless (cdr ring)
      (let ((change (item (slot-value object 'change)))
            start step width end)
        (if (consp change)
          (setf start (pop change) step (pop change) 
                width (pop change) end (pop change))
          (setf start change))
        (unless start (setf start 0))
        (unless step (setf step 1))
        (unless width (setf width 1))
        (unless end
          (setf end (- (slot-value object 'length) width)))
      (setf (cdr ring)
            (rotate-items (car ring) start
                          end step width))))
    (pop (cdr ring))))

(defun rotate-items (items start end step width)
  (loop for i from start below end by step 
        do (rotatef (elt items i) (elt items (+ i width))))
  items)

;;;
;;; the basic set of item streams.
;;;

(defclass cyclic-item-stream (item-stream cycle-pattern) ())
(defclass heap-item-stream (item-stream heap-pattern) ())
(defclass palindromic-item-stream (item-stream palindrome-pattern) ())
(defclass sequential-item-stream (item-stream sequence-pattern) ())
(defclass random-item-stream (item-stream random-pattern) ())
(defclass graph-item-stream (item-stream graph-pattern) ())
(defclass accumulating-item-stream (item-stream accumulation-pattern) ())
(defclass rotational-item-stream (item-stream rotation-pattern) ())
(defclass functional-item-stream (item-stream function-pattern) ())

;;;
;;; user utiltities
;;;

(defun read-items (stream &optional (length t) states?)
  ;; return length number of items in a list.
  (if (numberp length)
      (if states?
          (loop with x and y repeat length
                do (multiple-value-setq (x y) (item stream))
                collect x into values
                collect y into states
                finally (return (values values states)))
        (loop repeat length collect (item stream)))
    (if states?
        (loop with x and y
              do (multiple-value-setq (x y) (item stream))
              collect x into values 
              collect y into states
              until (eq y +eop+)
              finally (return (values values states)))
      (loop with x and y
            do (multiple-value-setq (x y) (item stream))
            collect x until (eq y +eop+)))))

(defmacro doitems ((var stream &key periods length) &body body)
  (declare (ignore length))
  (let ((streamvar (gensym))
        (periodvar (gensym))
        (countvar (gensym))
        itemvar statusvar)	
    (if (consp var)
        (setf itemvar (car var) statusvar (cadr var))
      (setf itemvar var statusvar (gensym)))	
    `(let ((,streamvar ,stream)
           (,periodvar (or ,periods 1))
           ,itemvar ,statusvar)
       (loop do 
             (multiple-value-setq (,itemvar ,statusvar)
 	                          (item ,streamvar))
             (progn ,@body)
             count (eq ,statusvar ':end-of-period) into ,countvar
             until (= ,countvar ,periodvar)))))

(defun last-item (stream &optional default)
  (let ((value (slot-value stream 'value)))
    (if (eq value +nad+)
      (if default
        (setf (slot-value stream 'value) default)
        (error "Stream ~S has no last value." stream))
      value)))

(defun last-state (stream &optional default)
  (let ((state (slot-value stream 'state)))
    (if (eq state +nad+)
      (if default
        (setf (slot-value stream 'state) default)
        (error "Stream ~S has no last state." stream))
      state)))
