;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

(defstruct offset 
  value    ; current offset value
  mode     ; one of :constant, :linked-to, :initially-from, :from
  source)  ; reinitialization source if mode not :constant.

(defclass interval-mixin (scale-mixin)
  ((offset :initform 0 :initarg :offset :initarg :from :initarg :on
           :initarg :linked-to :initarg :initially-from)))

(defclass essential-interval-stream (item-stream interval-mixin)
  ())

(defmethod initialize-instance :after ((object essential-interval-stream) 
                                       &rest args)
  (let ((offset (slot-value object 'offset))
        (mode ':from)
        (value nil)
        (source nil))
    (loop for arg in args by #'cddr
          do 
      (case arg
        (:initially-from 
         (setf mode arg value (read-offset offset)))
        (:linked-to 
         (setf mode arg source offset))
        (:on 
         (setf (slot-value object 'flags)
           (logior (slot-value object 'flags)
             +parallel-offsets+)))))
    (when (eq mode ':from)
      (if (or (numberp offset) (symbolp offset))
          (setf mode ':constant 
                source (scale-degree offset 
                                    (slot-value object 'scale)))
        (setf source offset)))
    (setf (slot-value object 'offset)
      (make-offset :mode mode :value value :source source))))

;;;
;;; interval streams return an interval added to an offset.
;;; step streams return an offset postincremented by the interval.
;;;

(defclass interval-stream (essential-interval-stream)
     ())

(defclass step-stream (essential-interval-stream)
     ())

;;;
;;; offsets are calculated once per period for interval and
;;; step streams, voicing streams calculate once per read.
;;;

(defmethod reset-period :after ((object essential-interval-stream))
  (unless (logtest (slot-value object 'flags)
                   +parallel-offsets+)
    (reset-offset (slot-value object 'offset))))

(defmethod next-in-pattern :before ((object essential-interval-stream))
    (when (logtest (slot-value object 'flags)
                   +parallel-offsets+)
      (reset-offset (slot-value object 'offset))))

;;;
;;; reset-offset computes a new offset value according to its mode.
;;;

(defun reset-offset (offset)
  (let ((source (offset-source offset))
        mode next)
    ;; if no source the offset was specified as :initially-from
    (when source
      (setf mode (offset-mode offset))
      (cond ((eq mode :constant)
             ;; just reset to constant value
             (setf next (offset-source offset)))
            ((eq mode ':linked-to)
             ;; get value from delegate stream.
             (let ((stream (find-stream (offset-source offset))))
               (setf next
                 (read-offset (linked-offset stream) ))))
            (t
             ;; recompute new value from source
             (setf next (read-offset source))))
      (setf (offset-value offset) 
        (if (restp next) *rest-degree* next)))))

(defmethod linked-offset ((object step-stream))
  (offset-value (slot-value object 'offset)))

(defmethod linked-offset ((object item-stream))
  (slot-value object 'value))

(defmethod read-offset ((offset integer))
  offset)

(defmethod read-offset ((offset t))
  (scale-degree offset *standard-scale*))

(defmethod read-offset ((offset scale-mixin))
  (scale-degree (item offset) (slot-value offset 'scale)))

(defmethod read-offset ((offset item-stream))
  (scale-degree (item offset) *standard-scale*))

(defmethod read-offset ((offset expr))
  (read-offset (item offset)))

;;;
;;; the after methods for interval and voicing streams are identical:
;;; the value of the stream is set to the sum of the current offset
;;; and the current value. step streams return the current offset
;;; and postincrement it by the current value. all three streams
;;; optionally coerce the returned value to scale notes or pitches.
;;; scale calculation would ideally be done by interval-mixin, but 
;;; can't be becauseit's after method would be called before these
;;; methods are called.
;;;;

(defmethod item :after ((object interval-stream))
  ;(declare (optimize (speed 3) (safety 0)))
  (let ((value (slot-value object 'value))
        (offset (slot-value object 'offset)))
    (if (or (eq value *rest-degree*)
            (eq (offset-value offset) *rest-degree*))
        (setf (slot-value object 'value) *rest-degree*)
     (let ((flags (slot-value object 'flags)))
       (declare (fixnum flags))
       (setf (slot-value object 'value)
         (cond ((logtest flags +coerce-to-note+)
                (scale-note (+ value (offset-value offset))
                            (slot-value object 'scale)))
               ((logtest flags +coerce-to-pitch+)
                (scale-pitch (+ value (offset-value offset))
                             (slot-value object 'scale)))
               (t
                (+ value (offset-value offset)))))))))

(defmethod item :after ((object step-stream))
  ;(declare (optimize (speed 3) (safety 0)))
  (let* ((value (slot-value object 'value))
         (offset (slot-value object 'offset))
         (current (offset-value offset)))
    (if (eq current *rest-degree*)
        (setf (slot-value object 'value) *rest-degree*))
      (if (not (eq value *rest-degree*))
        (let ((flags (slot-value object 'flags)))
         (declare (fixnum flags))
         (setf (slot-value object 'value)
           (cond ((logtest flags +coerce-to-note+)
                  (scale-note current (slot-value object 'scale)))
                 ((logtest flags +coerce-to-pitch+)
                  (scale-pitch current (slot-value object 'scale)))
                 (t current)))
         (incf (offset-value offset) value)))))


;;;
;;; interval stream definitions.
;;;

(defclass cyclic-interval-stream (interval-stream cycle-pattern) ())
(defclass cyclic-step-stream (step-stream cycle-pattern) ())

(defclass palindromic-interval-stream (interval-stream palindrome-pattern) ())
(defclass palindromic-step-stream (step-stream palindrome-pattern) ())

(defclass sequential-interval-stream (interval-stream sequence-pattern) ())
(defclass sequential-step-stream (step-stream sequence-pattern) ())

(defclass accumulating-interval-stream (interval-stream accumulation-pattern)
  ())
(defclass accumulating-step-stream (step-stream accumulation-pattern) ())

(defclass random-interval-stream (interval-stream random-pattern) ())
(defclass random-step-stream (step-stream random-pattern) ())

(defclass heap-interval-stream (interval-stream heap-pattern) ())
(defclass heap-step-stream (step-stream heap-pattern) ())

(defclass graph-interval-stream (interval-stream graph-pattern) ())
(defclass graph-step-stream (step-stream graph-pattern) ())

(defclass functional-interval-stream (interval-stream function-pattern) ())
(defclass functional-step-stream (step-stream function-pattern) ())

(defclass rotational-interval-stream (interval-stream rotation-pattern) ())
(defclass rotational-step-stream (step-stream rotation-pattern) ())

;;;
;;; the series stream is a cyclic interval stream that supports 
;;; most types of serial (twelve-tone) manipulations. series holds
;;; a canonical version of the series, which is processed each time
;;; the cyclic data repeats. the processing performed depends on the
;;; values of the form, mudulus, multiple and offset slots, which are
;;; read with item each time to reutunr new values. form is the row
;;; form to use, either prime, inversion ,retrograde or retrograde-
;;; inversion. multiple is a multiplicitive on each value in the series
;;; and modulus removes that interval from the computed value.
;;;


(defclass series-stream (cyclic-interval-stream)
  ((form     :initform nil :initarg forming :initarg :forming)
   (modulus  :initform nil :initarg :modulus)
   (multiple :initform nil :initarg multiple :initarg :multiple)
   (series)))

(defmethod initialize-instance :after ((object series-stream) &rest args)
  (declare (ignore args))
  (setf (slot-value object 'series)
    (copy-list (car (slot-value object 'data)))))

(defmethod next-in-pattern :before ((object series-stream))
  ;; recalculate series when data is reset.
  (let ((data (slot-value object 'data)))
    (when (null (cdr data))
      (let ((set (slot-value object 'series))
            (typ (slot-value object 'form))
            (mul (slot-value object 'multiple))
            (mod (slot-value object 'modulus))
            (inv nil)
            (rev nil))
        (when typ (setf typ (item typ)))
        (when mul (setf mul (item mul)))
        (when mod (setf mod (item mod)))
        (ecase typ
          ((nil p prime :p :prime))
          ((r retrograde :r :retrograde)
           (setf rev t))
          ((i :i inversion :inversion)
           (setf inv t))
          ((ri :ri retrograde-inversion :retrograde-inversion)
           (setf inv t rev t)))
        (loop for i in set
              for tail on (car data)
              do
              (when mul (setf i (* i mul)))
              (when inv (setf i (- i)))
              (when mod
                (setf i (mod i (if (minusp i) (- mod) mod))))
              (setf (car tail) i))
        (when rev
          (setf (car data) (nreverse (car data))))))))
